/* GIF Loader
   by Paul Bartrum
   Animation by Connelly Barnes */

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <allegro.h>
#include <string.h>
#include <limits.h>

#define MEMORY_BITMAP 0
#define VIDEO_BITMAP  1
#define SYSTEM_BITMAP 2

#define HAS_ANIMATE_FLAG(bmp) ((bmp)->clip == 2)
#define SET_ANIMATE_FLAG(bmp) ((bmp)->clip = 2)
#define CLEAR_ANIMATE_FLAG(bmp) ((bmp)->clip = 0)

#if (ALLEGRO_DATE<20000000)
#define ALLEGRO_312
#else
#define ALLEGRO_WIP
#endif

#ifdef ALLEGRO_WIP
/* An internal Allegro function, needed to find proper color conversion */
int _color_load_depth(int depth, int hasalpha);
#else
int _color_load_depth(int depth);
#endif

/* Allegro 3.12 doesn't use 'const char *' when it should.  Use ALLEGRO_CHAR *
   in place of 'const char *' to fix this problem */
#ifdef ALLEGRO_WIP
#define ALLEGRO_CHAR const char
#else
#define ALLEGRO_CHAR char
#endif

struct LZW_STRING
{
	short base;
  char n;
	short length;
};

PACKFILE *gif_file;
int global_empty_string, curr_bit_size, bit_overflow;
int bit_pos, data_pos, data_len, gif_entire, gif_code;
int gif_cc, string_length, global_i, bit_size;
unsigned char string[4096];
struct LZW_STRING gif_str[4096];
BITMAP *global_bmp;
int image_x, image_y, image_w, image_h, global_x, global_y;
int interlace;

volatile int _gif_timer = 0;
void _inc_gif_timer(void) { _gif_timer++; }
END_OF_FUNCTION(_inc_gif_timer);

int _local_timer = 0;
int t_color = -1, _auto_update = 1, _disposal, gif_registered = 0;

void clear_table(void)
{
  global_empty_string = gif_cc + 2;
	curr_bit_size = bit_size + 1;
	bit_overflow = 0;
}


void get_code(void)
{
	if(bit_pos + curr_bit_size > 8) {
    if(data_pos >= data_len) { data_len = pack_getc(gif_file); data_pos = 0; }
    gif_entire = (pack_getc(gif_file) << 8) + gif_entire;
		data_pos ++;
	}
	if(bit_pos + curr_bit_size > 16) {
    if(data_pos >= data_len) { data_len = pack_getc(gif_file); data_pos = 0; }
    gif_entire = (pack_getc(gif_file) << 16) + gif_entire;
		data_pos ++;
	}
  gif_code = (gif_entire >> bit_pos) & ((1 << curr_bit_size) - 1);
	if(bit_pos + curr_bit_size > 8)
    gif_entire >>= 8;
	if(bit_pos + curr_bit_size > 16)
    gif_entire >>= 8;
	bit_pos = (bit_pos + curr_bit_size) % 8;
	if(bit_pos == 0) {
    if(data_pos >= data_len) { data_len = pack_getc(gif_file); data_pos = 0; }
    gif_entire = pack_getc(gif_file);
		data_pos ++;
	}
}


void get_string(int num)
{
  if(num < gif_cc)
	{
		string_length = 1;
    string[0] = gif_str[num].n;
	}
	else
	{
    global_i = gif_str[num].length;
    string_length = global_i;
    while(global_i > 0)
		{
      global_i --;
      string[global_i] = gif_str[num].n;
      num = gif_str[num].base;
		}
		/* if(num != -1) **-{[ERROR]}-** */
	}
}


void output_string(void)
{
  for(global_i = 0; global_i < string_length; global_i ++)
	{
    if (t_color >= 0) {
      if (string[global_i] == t_color) {
        /* do nothing */
      } else if (string[global_i] == 0) {
        putpixel(global_bmp, global_x, global_y, t_color);
      } else {
        putpixel(global_bmp, global_x, global_y, string[global_i]);
      }
    } else {
      putpixel(global_bmp, global_x, global_y, string[global_i]);
    }
    global_x ++;
    if(global_x >= image_x + image_w)
		{
      global_x = image_x;
      global_y += interlace;
			if(interlace)
			{
        if(global_y >= image_y + image_h)
				{
          if(interlace == 8 && (global_y - image_y) % 8 == 0) {
						interlace = 8;
            global_y = image_y + 4;
					}
          else if(interlace == 8  && (global_y - image_y) % 8 == 4) {
						interlace = 4;
            global_y = image_y + 2;
					}
					else if(interlace == 4) {
						interlace = 2;
            global_y = image_y + 1;
					}
				}
			}
		}
	}
}


struct ANIMATION {
  BITMAP **frame;
  int frame_count;
  unsigned short *delay;

  BITMAP *bmp_struct;
  int cframe, next_update, completed;

  struct ANIMATION *next, *prev;
};

struct ANIMATION *first_anim = 0;
int _gif_initialized = 0;

void gif_init();

/* removes animation from linked list */
void unchain_animation(struct ANIMATION *a) {
  if (a->prev) { a->prev->next = a->next; }
  if (a->next) { a->next->prev = a->prev; }
  if (first_anim == a) { first_anim = a->next; }
}

/* inserts animation into linked list at proper (next_update) time */
void insert_animation(struct ANIMATION *a) {
  struct ANIMATION *current = first_anim;
  if (!current) {
    first_anim = a;
    a->next = a->prev = 0;
    return;
  }
  for (;;) {
    if (current->next_update >= a->next_update) { break; }
    if (!current->next) {
      a->prev = current;
      a->next = 0;
      current->next = a;
      return;
    }
    current = current->next;
  }
  a->prev = current->prev;
  a->next = current;
  if (current->prev) {
    current->prev->next = a;
    current->prev = a;
  } else {
    current->prev = a;
    first_anim = a;
  }
}

/*
void check_animations() {
  struct ANIMATION *current = first_anim;
  int t = 0;
  while (current) {
    if (current->next_update < t) { set_gfx_mode(GFX_TEXT, 0, 0, 0, 0); allegro_message("Sorting error\n"); exit(0); }
    t = current->next_update;
    current = current->next;
  }
}
*/

/* Returns size of bitmap structure, needed because Allegro uses ugly dynamic
   structure sizing. */

#define bmp_struct_size(bmp) (sizeof(BITMAP) + sizeof(char *) * (bmp)->h)

void destroy_animation_ex(struct ANIMATION *a) {
  int global_i;
  unchain_animation(a);
  for (global_i = 0; global_i < a->frame_count; global_i++) {
    if (a->frame[global_i]) {
      destroy_bitmap(a->frame[global_i]);
    }
  }
  free((void *) a->frame);
  free((void *) a->delay);
  free((void *) a);
}

void destroy_animation(BITMAP *bmp) {
  struct ANIMATION *current = first_anim;
  if (HAS_ANIMATE_FLAG(bmp)) {
    while (current) {
      if (current->bmp_struct == bmp) {
        destroy_animation_ex(current);
        return;
      }
      current = current->next;
    }
  }
  destroy_bitmap(bmp);
}

void update_single_animation(struct ANIMATION *a) {
  unchain_animation(a);

  a->cframe++;
  if (a->cframe >= a->frame_count) {
    a->cframe = 0;
    a->completed = 1;
  }

  memcpy((void *) a->bmp_struct, (void *) a->frame[a->cframe], bmp_struct_size(a->frame[a->cframe]));

  /* I think this is how .GIFs make 'infinite' delay */
  if (a->delay[a->cframe] > 32000) {
    a->next_update = INT_MAX;
    a->completed = 1;
  } else {
    a->next_update = _local_timer + a->delay[a->cframe];
  }
  insert_animation(a);
}

void update_animation_ex(void) {
  struct ANIMATION *current = first_anim, *next;
  if (_auto_update) { _gif_timer++; }
  _local_timer = _gif_timer;
  while (current) {
    next = current->next;
    if (_local_timer < current->next_update) {
      return;
    } else {
      update_single_animation(current);
    }
    current = next;
  }
}

END_OF_FUNCTION(update_animation_ex);
END_OF_FUNCTION(update_single_animation);
END_OF_FUNCTION(unchain_animation);
END_OF_FUNCTION(insert_animation);

void update_animation() {
  if (_auto_update) {
    _auto_update = 0;
    remove_int(update_animation_ex);
    install_int_ex(_inc_gif_timer, BPS_TO_TIMER(100));
  }

  update_animation_ex();
}

void reset_animation(BITMAP *bmp) {
  struct ANIMATION *current = first_anim;
  if (!bmp) { return; }
  while (current) {
    if (current->bmp_struct == bmp) {
      unchain_animation(current);

      memcpy((void *) current->bmp_struct, (void *) current->frame[0], bmp_struct_size(current->frame[0]));

      current->cframe = 0;
      current->completed = 0;
      current->next_update = _gif_timer + current->delay[0];

      insert_animation(current);

      return;
    }
    current = current->next;
  }
}

int get_frame_count(BITMAP *bmp) {
  struct ANIMATION *current = first_anim;
  while (current) {
    if (current->bmp_struct == bmp) {
      return current->frame_count;
    }
    current = current->next;
  }
  return -1;
}
int get_frame_data(BITMAP *bmp, int* data) {
  struct ANIMATION *current = first_anim;
  while (current) {
    if (current->bmp_struct == bmp) {
		int count = current->frame_count;
		while (count--) {
			data[count] = current->delay[count];
		}
		return 1;
    }
    current = current->next;
  }
  return 0;	// failure
}

BITMAP* get_frame(BITMAP *bmp, int framenum) {
  BITMAP *toRet = 0;
  struct ANIMATION *current = first_anim;
  while (current) {
    if (current->bmp_struct == bmp) {
      if (current->frame_count <= framenum) return 0;

      toRet = create_bitmap( current->bmp_struct->w , current->bmp_struct->h );
      memcpy((void *) toRet, (void *) current->frame[framenum], bmp_struct_size(current->frame[framenum]));
      
	  return toRet;
    }
    current = current->next;
  }
  return toRet;
}

int animation_completed(BITMAP *bmp) {
  struct ANIMATION *current = first_anim;
  while (current) {
    if (current->bmp_struct == bmp) {
      return current->completed;
    }
    current = current->next;
  }
  return 1;
}

void gif_init() {
  if (!_gif_initialized) {
    _gif_initialized = 1;

    install_timer();

    LOCK_FUNCTION(_inc_gif_timer);
    LOCK_VARIABLE(_gif_timer);
    LOCK_VARIABLE(_local_timer);

    LOCK_FUNCTION(update_animation_ex);
    LOCK_FUNCTION(update_single_animation);
    LOCK_FUNCTION(unchain_animation);
    LOCK_FUNCTION(insert_animation);

    LOCK_VARIABLE(first_anim);
    LOCK_VARIABLE(_auto_update);

    install_int_ex(update_animation_ex, BPS_TO_TIMER(100));
  }
}

/* load_gif:
    Loads a 2-256 colour GIF file onto a bitmap, returning the bitmap
    structure and storing the pallete data in the specified pallete (this
    should be an array of at least 256 RGB structures).

    Now also binds an ANIMATION into a global linked list so the returned
    BITMAP struct can be updated behind your back - creating animation...

    Be careful to NOT call destroy_bitmap() on the returned bitmap.
 */

BITMAP *load_gif_ex(ALLEGRO_CHAR *, RGB *, int);

BITMAP *load_gif(ALLEGRO_CHAR *filename, RGB *pal) {
  return load_gif_ex(filename, pal, MEMORY_BITMAP);
}

BITMAP *load_video_gif(ALLEGRO_CHAR *filename, RGB *pal) {
  return load_gif_ex(filename, pal, VIDEO_BITMAP);
}

#ifdef ALLEGRO_WIP
BITMAP *load_system_gif(ALLEGRO_CHAR *filename, RGB *pal) {
  return load_gif_ex(filename, pal, SYSTEM_BITMAP);
}
#else
BITMAP *load_system_gif(ALLEGRO_CHAR *filename, RGB *pal) {
  return load_gif_ex(filename, pal, VIDEO_BITMAP);
}
#endif

BITMAP *load_gif_ex(ALLEGRO_CHAR *filename, RGB *pal, int bmp_type)
{
	int width, height, depth;
  int old, ocolor, yval, xval, pcolor;
	int dest_depth;
  struct ANIMATION *a;
  int frame_count = 0, bsize, index;

  PALETTE swap_pal;

  BITMAP *frame[4096];
  unsigned short delay[4096];

  int current_delay = 0;
  int temp_pal = 0;

  _disposal = 0;

  if (!pal) {
    pal = (RGB *) malloc(sizeof(RGB) * 256);
    temp_pal = 1;
  }

  gif_init();

  gif_file = pack_fopen(filename, F_READ);
  if (!gif_file) /* can't open file */
		return NULL;

  global_i  = pack_mgetw(gif_file) << 8;
  global_i += pack_getc(gif_file);
  if(global_i != 0x474946) /* is it really a GIF? */
	{
    pack_fclose(gif_file);
		return NULL;
	}
  pack_fseek(gif_file, 3); /* skip version */

  width = pack_igetw(gif_file);
  height = pack_igetw(gif_file);

  global_i = pack_getc(gif_file);
  if(global_i & 128) /* no global colour table? */
    depth = (global_i & 7) + 1;
	else
		depth = 0;

  pack_fseek(gif_file, 2); /* skip background colour and aspect ratio */

	if(pal && depth) /* only read palette if pal and depth are not 0 */
	{
    for(global_i = 0; global_i < (1 << depth); global_i ++)
		{
      pal[global_i].r = pack_getc(gif_file) / 4;
      pal[global_i].g = pack_getc(gif_file) / 4;
      pal[global_i].b = pack_getc(gif_file) / 4;
		}
	}
	else
		if(depth)
      pack_fseek(gif_file, (1 << depth) * 3);

  global_bmp = 0;
  t_color = -1;
  do
	{
    global_i = pack_getc(gif_file);
    if (pack_feof(gif_file)) { global_i = 0x3B; }
    switch(global_i)
		{
			case 0x2C: /* Image Descriptor */
        image_x = pack_igetw(gif_file);
        image_y = pack_igetw(gif_file); /* individual image dimensions */
        image_w = pack_igetw(gif_file);
        image_h = pack_igetw(gif_file);

        if (!global_bmp) {
          global_bmp = create_bitmap_ex(8, width, height);
          if (global_bmp == NULL) {
            pack_fclose(gif_file);
            return NULL;
          }
          clear_to_color(global_bmp, bitmap_mask_color(global_bmp));
        }

        if (_disposal == 2 || _disposal == 3) {
          set_clip(global_bmp, image_x, image_y, image_x+image_w-1, image_y+image_h-1);
          clear_to_color(global_bmp, bitmap_mask_color(global_bmp));
        } else {
          set_clip(global_bmp, 0, 0, global_bmp->w - 1, global_bmp->h - 1);
        }

        global_i = pack_getc(gif_file);
        if(global_i & 64)
					interlace = 8;
				else
					interlace = 1;

        if(global_i & 128)
				{
          /* FIXME: Per-frame palettes will be butchered */
          depth = (global_i & 7) + 1;
          if(pal)
					{
            for(global_i = 0; global_i < (1 << depth); global_i ++)
						{
              pal[global_i].r = pack_getc(gif_file) / 4;
              pal[global_i].g = pack_getc(gif_file) / 4;
              pal[global_i].b = pack_getc(gif_file) / 4;
						}
					}
					else
            pack_fseek(gif_file, (1 << depth) * 3);
				}

				/* lzw stream starts now */
        bit_size = pack_getc(gif_file);
        gif_cc = 1 << bit_size;

				/* initialise string table */
        for(global_i = 0; global_i < gif_cc; global_i ++)
				{
          gif_str[global_i].base = -1;
          gif_str[global_i].n = global_i;
          gif_str[global_i].length = 1;
				}

				/* initialise the variables */
				bit_pos = 0;
        data_len = pack_getc(gif_file); data_pos = 0;
        gif_entire = pack_getc(gif_file); data_pos ++;
        string_length = 0; global_x = image_x; global_y = image_y;

        /* starting gif_code */
				clear_table();
				get_code();
        if(gif_code == gif_cc)
					get_code();
        get_string(gif_code);
				output_string();
        old = gif_code;

				while(TRUE)
				{
					get_code();

          if(gif_code == gif_cc)
					{
            /* starting gif_code */
						clear_table();
						get_code();
            get_string(gif_code);
						output_string();
            old = gif_code;
					}
          else if(gif_code == gif_cc + 1)
					{
						break;
					}
          else if(gif_code < global_empty_string)
					{
            get_string(gif_code);
						output_string();

						if(bit_overflow == 0) {
              gif_str[global_empty_string].base = old;
              gif_str[global_empty_string].n = string[0];
              gif_str[global_empty_string].length = gif_str[old].length + 1;
              global_empty_string ++;
              if(global_empty_string == (1 << curr_bit_size))
								curr_bit_size ++;
							if(curr_bit_size == 13) {
								curr_bit_size = 12;
								bit_overflow = 1;
							}
						}

            old = gif_code;
					}
					else
					{
						get_string(old);
            string[gif_str[old].length] = string[0];
						string_length ++;

						if(bit_overflow == 0) {
              gif_str[global_empty_string].base = old;
              gif_str[global_empty_string].n = string[0];
              gif_str[global_empty_string].length = gif_str[old].length + 1;
              global_empty_string ++;
              if(global_empty_string == (1 << curr_bit_size))
								curr_bit_size ++;
							if(curr_bit_size == 13) {
								curr_bit_size = 12;
								bit_overflow = 1;
							}
						}

						output_string();
            old = gif_code;
					}
				}

        if (pal) {
          memcpy((void *) swap_pal, (void *) pal, sizeof(RGB) * 256);
          if (t_color >= 0) {
            swap_pal[t_color] = swap_pal[0];
            swap_pal[0].r = 63;
            swap_pal[0].g = 0;
            swap_pal[0].b = 63;
          }
        }

        if (bmp_type == MEMORY_BITMAP) {
          /* convert to correct colour depth */
#ifdef ALLEGRO_WIP
          dest_depth = _color_load_depth(8, 0);
#else
          dest_depth = _color_load_depth(8);
#endif

          if (!dest_depth) { dest_depth = 8; }

          /* copy global (drawing) bitmap to more permanent memory. */
          frame[frame_count] = create_bitmap_ex(dest_depth, global_bmp->w, global_bmp->h);
        } else if (bmp_type == VIDEO_BITMAP) {
          frame[frame_count] = create_video_bitmap(global_bmp->w, global_bmp->h);
#ifdef ALLEGRO_WIP
        } else if (bmp_type == SYSTEM_BITMAP) {
          frame[frame_count] = create_system_bitmap(global_bmp->w, global_bmp->h);
#endif
        }

        if (!frame[frame_count]) {
          pack_fclose(gif_file);
          for (index = 0; index < frame_count; index++) {
            destroy_bitmap(frame[index]);
          }
          destroy_bitmap(global_bmp);
          return NULL;
        }
       
        select_palette(swap_pal);
        blit(global_bmp, frame[frame_count], 0, 0, 0, 0, global_bmp->w, global_bmp->h);
        unselect_palette();

        delay[frame_count] = current_delay;
        frame_count++;

				break;
			case 0x21: /* Extension Introducer */
        global_i = pack_getc(gif_file);
        if(global_i == 0xF9) /* Graphic Control Extension */
				{
          pack_fseek(gif_file, 1); /* skip size (it's 4) */
          global_i = pack_getc(gif_file);
          current_delay = pack_igetw(gif_file);

          if(global_i & 1) /* is transparency enabled? */
					{
            ocolor = t_color;
            t_color = pack_getc(gif_file); /* transparent colour */

            /* was transparency just enabled?
               if so, swap around transparent colors */
            if (global_bmp && ocolor < 0) {
              for (yval = 0; yval < global_bmp->h; yval++) {
                for (xval = 0; xval < global_bmp->w; xval++) {
                  pcolor = _getpixel(global_bmp, xval, yval);
                  if (pcolor == 0) { _putpixel(global_bmp, xval, yval, t_color); }
                }
              }
            }
					}
          else
          {
            t_color = -1;
            pack_fseek(gif_file, 1);
          }

          _disposal = (global_i >> 2) & 7;

				}
        global_i = pack_getc(gif_file);
        while(global_i) /* skip Data Sub-blocks */
				{
          pack_fseek(gif_file, global_i);
          global_i = pack_getc(gif_file);
				}
				break;
			case 0x3B: /* Trailer - end of data */
        pack_fclose(gif_file);

        if (frame_count > 1) {
          /* create the animation and return the first frame of it.
             subsequent frames will be updated by update_animation_ex */

          a = (struct ANIMATION *) malloc(sizeof(struct ANIMATION));
          if (!a) {

            /* malloc failed...ugly cleanup gif_code */
            pack_fclose(gif_file);
            for (index = 0; index < frame_count; index++) {
              destroy_bitmap(frame[index]);
            }
            destroy_bitmap(global_bmp);
            free((void *) a);
            if (temp_pal) { free((void *) pal); }
            return NULL;
          }

          a->delay = (unsigned short *) malloc(sizeof(int) * frame_count);
          a->frame = (BITMAP **) malloc(sizeof(BITMAP *) * frame_count);

          bsize = sizeof(BITMAP) + sizeof(char *) * height;
          a->bmp_struct = (BITMAP *) malloc(bsize);

          if (!a->delay || !a->frame || !a->bmp_struct) {

            /* malloc failed...ugly cleanup gif_code */
            pack_fclose(gif_file);
            for (index = 0; index < frame_count; index++) {
              destroy_bitmap(frame[index]);
            }
            destroy_bitmap(global_bmp);
            free((void *) a->delay);
            free((void *) a->frame);
            free((void *) a->bmp_struct);
            free((void *) a);
            if (temp_pal) { free((void *) pal); }
            return NULL;
          }

          a->frame_count = frame_count;

          memcpy((void *) a->frame, (void *) frame, sizeof(BITMAP *) * frame_count);
          memcpy((void *) a->delay, (void *) delay, sizeof(unsigned short) * frame_count);

          memcpy((void *) a->bmp_struct, (void *) a->frame[0], bmp_struct_size(a->frame[0]));

          a->cframe = 0;
          a->completed = 0;
          a->next_update = _gif_timer + a->delay[0];

          insert_animation(a);

          if (_auto_update) {
            /* memory needs to be locked if updating via interrupt */
            LOCK_VARIABLE(a);
            LOCK_VARIABLE(*a);
            LOCK_VARIABLE(*(a->bmp_struct));
            for (index = 0; index < a->frame_count; index++) {
              LOCK_VARIABLE(a->frame[index]);
              LOCK_VARIABLE(a->delay[index]);
              LOCK_VARIABLE(*(a->frame[index]));
            }
          }

          for (index = 0; index < a->frame_count; index++) {
            SET_ANIMATE_FLAG(a->frame[index]);
          }

          if (temp_pal) { free((void *) pal); }
          destroy_bitmap(global_bmp);

          return a->bmp_struct;
        } else if (frame_count == 1) {
          /* there's only 1 frame, so return it, and don't bother to create
             an ANIMATION structure */

          if (temp_pal) { free((void *) pal); }
          destroy_bitmap(global_bmp);

          return frame[0];
        } else {

          if (temp_pal) { free((void *) pal); }
          destroy_bitmap(global_bmp);
          return NULL;

        }
		}

	} while(TRUE);

	/* this is never executed but DJGPP complains if you leave it out */
	return NULL;
}


/* #defined replacements for destroy_bitmap and load_bitmap (gif.h). */

void __destroy_bitmap(BITMAP *bmp) {
  destroy_animation(bmp);
}

BITMAP *__load_bitmap(ALLEGRO_CHAR *filename, RGB *pal) {
  if (!gif_registered) {
    register_bitmap_file_type("gif", load_gif, 0);
  }
  return load_bitmap(filename, pal);
}
