#include "prototypes.h"
#include "globals.h"

double square(double n);
double dist(keeVector p1, keeVector p2);
void repelPoint(int ballIndex1, int ballIndex2);
void resolveBallWithJoints(int mm);
void doBalls();

//------------------------------------------------------------------------

static double theta = 0;	// Demo. Rotating wheel's angle.

//------------------------------------------------------------------------

// Convenience function.
double absoluteValue(double d)
{
	if(d < 0)
		d *= -1;

	return d;
}

//------------------------------------------------------------------------

void doPhysics()
{

	// Demo 1. Rotating wheel.
		int arms = 8;
		double twoPI = 6.28;
		double ratio = twoPI/arms;

		double length = 100;

		for(int i = 2; i < 2+arms; i++)
		// Update the rotating "wheel."
		{
			g_joints[i].p2	= g_joints[i].p1
											+ keeVector(cos(theta+(i-2)*(ratio)), sin(theta+(i-2)*(ratio))) * length;
		}
		theta += 0.005;	// Update the angle.
	//~

	// Demo 2. Keyboard controls the green ball.
	g_points[0].velocity += g_accControl;



	// Heart of the physics implementation. Resolve ball collisions.
		doBalls();
	//~



	// Part of demo 2. Reset, otherwise it will be continuous.
	g_accControl = keeVector(0, 0);	
}

//------------------------------------------------------------------------

// Square a number.  Convenience function because I don't want to type so much.
double square(double n)
{
	return n * n;
}

//------------------------------------------------------------------------




// The rest of the code in this file is taken from James Lohr's snookers demo.
// I merely renamed and hacked his code; if you have questions please direct them to James
// because I have pretty much no idea how the physics for this thing works.  In fact, the
// names and interpretations I make of his code are purely guess-work.
// Big thanks to James for his contribution.

// Detect and resolve collisions between balls and other balls.
void doBalls()
{
	for(int iteration = 0; iteration < 10; iteration++)
	// Controls the stability of the system.
	// The more iterations = improved stability but compromises performance.
	{
		// Check against other balls.  We can comment out this entire block of code
		// if all we care about is collision between balls and lines.
			for(unsigned ballIndex1 = 0; ballIndex1 < g_points.size(); ballIndex1++)
			// Loop through balls.
			{
				// Check collision against balls other than itself.
				for(unsigned ballIndex2 = 0; ballIndex2 < g_points.size(); ballIndex2++)
				// Check ball 1 against the other balls.
				{
					if(ballIndex2 != ballIndex1)
					// Only check against other balls.
					{
						double squaredDistance = dist(g_points[ballIndex1].position, g_points[ballIndex2].position);
						double combinedRadius = g_points[ballIndex1].size + g_points[ballIndex2].size;
							
						if(squaredDistance <= square(combinedRadius))
						// Bounding boxes collide.
						{
							// Push ball 1 away from ball 2.
							repelPoint(ballIndex1, ballIndex2);
						}
					}
				}
			}
		//~

		// Update balls.
		for(unsigned ballIndex = 0; ballIndex < g_points.size(); ballIndex++)
		// Loop through balls.
		{
			// Apply gravity.
			g_points[ballIndex].velocity += g_accGravity;	

			// While we're at it, interact with lines.
			resolveBallWithJoints(ballIndex);

			// Move the ball.
			g_points[ballIndex].position += g_points[ballIndex].velocity;

			// Dampen velocity, or we'll have perpetual motion.
			g_points[ballIndex].velocity *= 0.999;
		}
	}
}

//------------------------------------------------------------------------

// Squared distance.  I suspect James Lohr works with squares so as to reduce the processor load.
double dist(keeVector p1, keeVector p2)
{
	return square(p2.x-p1.x) + square(p2.y-p1.y);
}

//------------------------------------------------------------------------

// Repels two colliding balls.  Accepts indices of the two colliding balls.
void repelPoint(int ballIndex1, int ballIndex2)
{
	// This represents the line segment between the two balls, relative to the origin (0,0).
	keeVector lineSegment = g_points[ballIndex1].position - g_points[ballIndex2].position;

	keeVector repelForce;		// Force that will push the balls away from each other.

	keeVector repelDirection(1, 1);	// I think this takes care of the alignment (direction) of
																	// the line segment, so it shouldn't matter which point came
																	// first because this will fix it.

	if(lineSegment.x < 0)
	// Ball 1 is left of ball 2.
	{
		repelDirection.x = -1;	// Ball 1 should be repelled leftward.
		lineSegment.x *= -1;
	}

	if(lineSegment.y < 0)
	// Ball 1 is above ball 2.
	{
		repelDirection.y = -1;	// Ball 1 should be repelled upward.
		lineSegment.y	*= -1;
	}

	// Not sure, or don't want to think...
	if((lineSegment.x+lineSegment.y > 0.01) || (lineSegment.x+lineSegment.y < -0.01))
	{
		repelForce = lineSegment / (lineSegment.x+lineSegment.y);
	}

	repelForce *= repelDirection;	// Should now be the final fixed orientation.

	double squaredDistance = dist(g_points[ballIndex1].position, g_points[ballIndex2].position);
	double combinedRadius = g_points[ballIndex1].size + g_points[ballIndex2].size;

	repelForce *= squaredDistance - square(combinedRadius);

	// Repel.  I don't know the math behind it.
	g_points[ballIndex1].velocity -= repelForce*0.002 / g_points[ballIndex1].mass;
	g_points[ballIndex1].velocity *= 0.999; // 0.95 in original demo, but it limited motion.

}

//------------------------------------------------------------------------

// Resolve collision between a ball and lines.
// I don't understand much of this function.
void resolveBallWithJoints(int ballIndex)
{
	double squaredRadius = square(g_points[ballIndex].size);

	keeVector lineSegment, repelForce;

	// Not sure about this, but it I guess it's something like a temporary position.
	// I just know it's needed in the calculations.
	keeVector xy; 

	double xMin, xMax, yMin, yMax;
	
	// Aliases for ball position, promotes readability.
		double c = g_points[ballIndex].position.x;
		double d = g_points[ballIndex].position.y;
	//~

	for(int n = 0; n < g_joints.size(); n++)
	{
		keeVector slopeSegment = g_joints[n].p2 - g_joints[n].p1;
			slopeSegment.normalize();

		// Aliases for joint's first vertex.
			double a = g_joints[n].p1.x;
			double b = g_joints[n].p1.y;
		//~

		// Used to correctly orient the segments?
		keeVector repelDirection(1, 1);

		double slope = 10;	// Don't know why it's so large...

		if( square(g_joints[n].p2.x - a) > 0.1)
			slope = (g_joints[n].p2.y - b) / (g_joints[n].p2.x - a);

		if((square(slope)) < 1)
		// Slope is fairly small.
		{
			xy.x = (a*square(slope) + c + d*slope - b*slope) / (1+square(slope));
			xy.y = b + slope*(xy.x-a);

			if(a > g_joints[n].p2.x)
			// Point 1 is to the right of point 2.
			{
				xMax = a;
				xMin = g_joints[n].p2.x;
			}
			else
			{
				xMax = g_joints[n].p2.x;
				xMin = a;
			}

			if(xy.x < xMin)
			{
				xy.x = xMin;
				xy.y = b + slope*(xy.x-a);
			}
			if(xy.x > xMax)
			{
				xy.x = xMax;
				xy.y = b + slope*(xy.x-a);
			}

			if(dist(xy, g_points[ballIndex].position) < squaredRadius)
			// Within bounding box.
			{
				lineSegment = g_points[ballIndex].position - xy;

				if(lineSegment.x < 0)
				// Ball is left of xy.
				{
					repelDirection.x = -1;	// Ball should repel leftward.
					lineSegment.x *= -1;
				}
				if(lineSegment.y < 0)
				// Ball is above xy.
				{
					repelDirection.y = -1;	// Ball should repel upward.
					lineSegment.y *= -1;
				}

				double sumXY = lineSegment.x + lineSegment.y;
				if((sumXY > 0.01) || (sumXY < -0.01))
				{
					repelForce = lineSegment / sumXY;
				}

				repelForce *= repelDirection * (dist(g_points[ballIndex].position, xy) - squaredRadius);

				g_points[ballIndex].velocity -= repelForce*0.06;

				g_points[ballIndex].velocity *= 0.96;	// General friction.
			}
		}
		else
		// Slope is fairly steep.
		{
			slope = (g_joints[n].p2.x - a) / (g_joints[n].p2.y - b);

			xy.y = (b*square(slope) + d + c*slope - a*slope) / (1+square(slope));
			xy.x = a + (xy.y-b)*slope;

			if(b > g_joints[n].p2.y)
			// Point 1 is above point 2.
			{
				yMax = b;
				yMin = g_joints[n].p2.y;
			}
			else
			{
				yMax = g_joints[n].p2.y;
				yMin = b;
			}

			if(xy.y < yMin)
			{
				xy.y = yMin;
				xy.x = a + slope*(xy.y-b);
			}
			if(xy.y > yMax)
			{
				xy.y = yMax;
				xy.x = a + slope*(xy.y-b);
			}

			if(dist(xy, g_points[ballIndex].position) < squaredRadius)
			// Within bounding box.
			{
				lineSegment = g_points[ballIndex].position - xy;

				if(lineSegment.x < 0)
				// Ball is left of xy.
				{
					repelDirection.x = -1;	// Ball should repel leftward.
					lineSegment.x *= -1;
				}
				if(lineSegment.y < 0)
				// Ball is above xy.
				{
					repelDirection.y = -1;	// Ball should repel upward.
					lineSegment.y *= -1;
				}

				double sumXY = lineSegment.x + lineSegment.y;
				if((sumXY > 0.01) || (sumXY < -0.01))
				{
					repelForce = lineSegment / sumXY;
				}

				repelForce *= repelDirection * (dist(g_points[ballIndex].position, xy) - squaredRadius);

				g_points[ballIndex].velocity -= repelForce*0.06;

				g_points[ballIndex].velocity *= 0.96;	// General friction.
			}
		}
	}
}

//------------------------------------------------------------------------

