/* music-ad.m,
 *
 * Display the current songs ID3 tags when in game.  Uses TagLib.
 */

#ifndef NO_ID3TAGS

#include <allegro.h>
#include <assert.h>
#ifndef NO_FBLEND
# include "fblend/include/fblend.h"
#endif
#include "libid3tag/field.h"
#include "libid3tag/id3tag.h"
#include "libid3tag/latin1.h"
#include "common.h"
#include "music-ad.h"
#include "seborrhea/seborrhea-font.h"


MusicAd *music_ad;
BOOL music_ad_enabled = YES;


@implementation MusicAd
static void get_field(struct id3_file *file, struct id3_tag *tag, 
		      const char *frame_str, char *dest)
{
    struct id3_frame *frame = NULL;
    union id3_field *field = NULL;
    const id3_ucs4_t *data = NULL;
    assert(file && tag && frame_str && dest);

    frame = id3_tag_findframe(tag, frame_str, 0);
    if (not frame)
	goto exit;

    field = id3_frame_field(frame, 1);
    if (not field)
	goto exit;

    if (id3_field_getnstrings(field) < 1)
	goto exit;

    data = id3_field_getstrings(field, 0);
    id3_latin1_encode(dest, data);

 exit:
    if (field)
	id3_field_finish(field);
    if (frame)
	id3_frame_delete(frame);
}

- initWithTagsFrom:(const char *)filename
{
    char album[128] = "", title[128] = "", artist[128] = "";
    SebFont *font;

    [self init];

    /* Font. */
    font = (SebFont *)[base_sebum getSebumByName:"fonts/elle16"];
    if (not font)
	goto error;

    {
        struct id3_file *file;
        struct id3_tag *tag;
	
        file = id3_file_open(filename, ID3_FILE_MODE_READONLY);
        if (not file)
	    goto error;
	
        tag = id3_file_tag(file);
        if (not tag) {
	    id3_file_close(file);
	    goto error;
	}
	
        /* Title/album/artist. */
        get_field(file, tag, ID3_FRAME_TITLE, title);
        get_field(file, tag, ID3_FRAME_ALBUM, album);
        get_field(file, tag, ID3_FRAME_ARTIST, artist);

	id3_tag_delete(tag);
	id3_file_close(file);
    }

    {
	unsigned int w, h;

	w = [font textLength:album];
	w = MAX(w, [font textLength:title]);
	w = MAX(w, [font textLength:artist]);
	w = MIN(w, screen_w);
	h = [font textHeight];

	alpha_text = create_bitmap(w, h*3);
	assert(alpha_text);
	clear_to_color(alpha_text, makecol(0x00, 0x00, 0x00));

	[font putString:title  To:alpha_text X:0 Y:0   Colour:0xff:0xff:0xff];
	[font putString:artist To:alpha_text X:0 Y:h   Colour:0xff:0xff:0xff];
	[font putString:album  To:alpha_text X:0 Y:h*2 Colour:0xff:0xff:0xff];
    }

    t = 2.5*SECONDS;
    return self;

 error:
    return [self free];
}

- free
{
    FREE_BITMAP(alpha_text);
    return [super free];
}

- update
{
    if (t > 0) {
	t--;
	return self;
    }
    else
	return [self free];
}

- (void) drawTo:(BITMAP *)dest
{
    unsigned int y, alpha;
    assert(dest);

    y = dest->h - alpha_text->h;
    alpha = MIN(63, 64 * t / (1.5*SECONDS));

#ifndef NO_FBLEND
    fblend_add(alpha_text, dest, 0, y, alpha);
#else  /* Allegro routines. */
    set_add_blender(0xff, 0xff, 0xff, alpha);
    draw_trans_sprite(dest, alpha_text, 0, y);
#endif /* NO_FBLEND. */
}
@end

#endif /* NO_ID3TAGS. */
