#!/bin/sh
#
# Sets up the Adime package for building with the specified compiler,
# and if possible converts text files to the desired target format.

proc_help()
{
   echo
   echo "Usage: ./fix.sh <platform> [--quick|--dtou|--utod|--utom|--mtou]"
   echo "Where <platform> is one of:"
   echo "  djgpp       DOS/DJGPP"
   echo "  watcom      DOS/Watcom"
   echo "  mingw32     Win32/MinGW32"
   echo "  msvc        Win32/MSVC"
   echo "  bcc32       Win32/Borland C++"
   echo "  unix        Linux/gcc"
   echo "  macosx      Mac OS X/gcc"
   echo "  qnx         QNX/gcc"
   echo "The other parameters control conversion of text files:"
   echo "  --quick     Turn of text file conversion"
   echo "  --dtou      Convert from DOS/Win32 format to Unix"
   echo "  --utod      Convert from Unix to DOS/Win32 format"
   echo "  --utom      Convert from Unix to Macintosh format"
   echo "  --mtou      Convert from Macintosh to Unix format"
   echo "If no parameter is specified, --dtou is assumed."
   echo

   ADIME_NOCONV="1"
}

proc_fix()
{
   echo "Configuring Adime for $1..."

   if [ "$2" != "none" ]; then
      echo "# generated by fix.sh" > makefile
      echo "MAKEFILE_INC = $2" >> makefile
      echo "include misc/makefile.all" >> makefile
   fi
}

proc_noexist()
{
   echo "Sorry, Adime has not been ported to $1."
   echo "Want to help?"
   ADIME_NOCONV="1"
}

proc_filelist()
{
   # common files.
   ADIME_FILELIST=`find . -type f "(" \
      -name "*.c" -o -name "*.cfg" -o -name "*.cpp" -o -name "*.def" -o \
      -name "*.h" -o -name "*.hin" -o -name "*.in" -o -name "*.inc" -o \
      -name "*.m4" -o -name "*.mft" -o -name "*.s" -o -name "*.rc" -o \
      -name "*.spec" -o -name "*.pl" -o -name "*.txt" -o -name "*._tx" -o \
      -name "makefile*" -o -name "readme.*" -o \
      -name "CHANGES" -o -name "AUTHORS" -o -name "THANKS" \
   ")"`

   # touch unix shell scripts?
   if [ "$1" != "omit_sh" ]; then
      ADIME_FILELIST="$ADIME_FILELIST `find . -type f -name '*.sh'`"
   fi
}

proc_utod()
{
   echo "Converting files from Unix to DOS/Win32..."
   proc_filelist "omit_sh"
   for file in $ADIME_FILELIST; do
      echo "$file"
      cp $file _tmpfile
      perl -p -i -e "s/([^\r]|^)\n/\1\r\n/" _tmpfile
      touch -r $file _tmpfile
      mv _tmpfile $file
   done
}

proc_dtou()
{
   echo "Converting files from DOS/Win32 to Unix..."
   proc_filelist
   for file in $ADIME_FILELIST; do
      echo "$file"
      mv $file _tmpfile
      tr -d '\015' < _tmpfile > $file
      touch -r _tmpfile $file
      rm _tmpfile
   done
   chmod +x *.sh misc/*.sh
}

proc_utom()
{
   echo "Converting files from Unix to Macintosh..."
   proc_filelist "omit_sh"
   for file in $ADIME_FILELIST; do
      echo "$file"
      mv $file _tmpfile
      tr '\012' '\015' < _tmpfile > $file
      touch -r _tmpfile $file
      rm _tmpfile
   done
}

proc_mtou()
{
   echo "Converting files from Macintosh to Unix..."
   proc_filelist
   for file in $ADIME_FILELIST; do
      echo "$file"
      mv $file _tmpfile
      tr '\015' '\012' < _tmpfile > $file
      touch -r _tmpfile $file
      rm _tmpfile
   done
}

# prepare adime for the given platform.

case "$1" in
   "bcc32"   ) proc_fix     "Windows (BCC32)"   "makefile.bcc";;
   "beos"    ) proc_noexist "BeOS"              "none";;
   "djgpp"   ) proc_fix     "DOS (djgpp)"       "makefile.dj";;
   "mingw32" ) proc_fix     "Windows (Mingw32)" "makefile.mgw";;
   "msvc"    ) proc_fix     "Windows (MSVC)"    "makefile.vc";;
   "qnx"     ) proc_fix     "QNX"               "makefile.qnx";;
   "unix"    ) proc_fix     "Unix"              "makefile.uni";;
   "macosx"  ) proc_fix     "Mac OS X"          "makefile.osx";;
   "mac"     ) proc_noexist "Mac"               "none";;
   "watcom"  ) proc_fix     "DOS (Watcom)"      "makefile.wat";;
   "none"    ) ;;
   "help"    ) proc_help;;
   *         ) proc_help;;
esac

# convert all text-file line endings.

if [ "$ADIME_NOCONV" != "1" ]; then
   case "$2" in
      "--utod"  ) proc_utod;;
      "--dtou"  ) proc_dtou;;
      "--utom"  ) proc_utom;;
      "--mtou"  ) proc_mtou;;
      "--quick" ) echo "No text file conversion...";;
      *         ) proc_dtou;;
   esac
fi

echo "Done!"
