/*         ______   ___    ___ 
 *        /\  _  \ /\_ \  /\_ \ 
 *        \ \ \L\ \\//\ \ \//\ \      __     __   _ __   ___ 
 *         \ \  __ \ \ \ \  \ \ \   /'__`\ /'_ `\/\`'__\/ __`\
 *          \ \ \/\ \ \_\ \_ \_\ \_/\  __//\ \L\ \ \ \//\ \L\ \
 *           \ \_\ \_\/\____\/\____\ \____\ \____ \ \_\\ \____/
 *            \/_/\/_/\/____/\/____/\/____/\/___L\ \/_/ \/___/
 *                                           /\____/
 *                                           \_/__/
 *
 *      DirectInput joystick driver.
 *
 *      By Marcello Basta-Forte.
 *
 *      See readme.txt for copyright information.
 */


#define DIRECTINPUT_VERSION 0x0500

#include "allegro.h"
#include "allegro/internal/aintern.h"
#include "allegro/platform/aintwin.h"

#ifndef SCAN_DEPEND
   #include <process.h>
   #include <dinput.h>
#endif

#include "diutil.h"

#ifndef ALLEGRO_WINDOWS
#error something is wrong with the makefile
#endif




static int joy_directx_init(void);
static void joy_directx_exit(void);
static int joy_directx_poll(void);

JOYSTICK_DRIVER joystick_directx =
{
   JOY_DIRECTX,
   empty_string,
   empty_string,
   "DirectInput joystick",
   joy_directx_init,
   joy_directx_exit,
   joy_directx_poll,
   NULL,
   NULL,
   NULL,
   NULL
};

/* bitflags for hat directions */
#define HAT_FORWARD    0x0001
#define HAT_RIGHT      0x0002
#define HAT_BACKWARD   0x0004
#define HAT_LEFT       0x0008

/* the directions of the POV are based on degrees (0-359)
 */
#define DIJOY_POVFORWARD             0
#define DIJOY_POVRIGHT            9000
#define DIJOY_POVBACKWARD        18000
#define DIJOY_POVLEFT       	 27000
#define DIJOY_POVFORWARDRIGHT     4500
#define DIJOY_POVBACKWARDRIGHT   13500
#define DIJOY_POVBACKWARDLEFT    22500
#define DIJOY_POVFORWARDLEFT     31500
#define DIJOY_POVCENTERED	((WORD)-1)


/* Win32 joystick state */
#define DI_MAX_AXES    16
#define DI_MAX_HATS		4

struct DI_JOYSTICK {
   LPDIRECTINPUTDEVICE2	device;
   int axes_num;
   int min[DI_MAX_AXES];
   int max[DI_MAX_AXES];
   int axis[DI_MAX_AXES];
   int hats_num;
   int hat[DI_MAX_HATS];
   int buttons_num;
   int button[MAX_JOYSTICK_BUTTONS];
};

struct DI_JOYSTICK directx_joystick[MAX_JOYSTICKS];

static int directx_joy_num = 0;
static int directx_joy_attached = 0;

static char name_x[] = "X";
static char name_y[] = "Y";
static char name_stick[] = "stick";
static char name_throttle[] = "throttle";
static char name_rudder[] = "rudder";
static char name_slider[] = "slider";
static char name_hat[] = "hat";
static char *name_b[MAX_JOYSTICK_BUTTONS] = {
   "Button 1", "Button 2", "Button 3", "Button 4", "Button 5", "Button 6", "Button 7", "Button 8",
   "Button 9", "Button 10", "Button 11", "Button 12", "Button 13", "Button 14", "Button 15", "Button 16"
};



/* poll_directx_joysticks:
 *  poll the Win32 joystick devices
 */
static void poll_directx_joysticks(void)
{
	int n_joy, n_axis, n_but, h, j;



	HRESULT     hr;
	DIJOYSTATE  dijs;
	DWORD       dwInput = 0;

	directx_joy_attached = 0;

	for (n_joy = 0; n_joy < directx_joy_num; n_joy++) {
		if (directx_joystick[n_joy].device == NULL) continue;

		/* Poll the joystick to read the current state */
		hr = IDirectInputDevice2_Poll(directx_joystick[n_joy].device);

		hr = IDirectInputDevice_GetDeviceState(directx_joystick[n_joy].device, sizeof(DIJOYSTATE), &dijs );
		if( FAILED(hr) ) {
			if( hr == DIERR_INPUTLOST )
				IDirectInputDevice_Acquire(directx_joystick[n_joy].device);
			
			for(n_axis = 0; n_axis<directx_joystick[n_joy].axes_num; n_axis++) 
				directx_joystick[n_joy].axis[n_axis] = 0;

			for (h=0; h<directx_joystick[n_joy].hats_num; h++)
				directx_joystick[n_joy].hat[h] = 0;

			for (n_but = 0; n_but < directx_joystick[n_joy].buttons_num; n_but++)
				directx_joystick[n_joy].button[n_but] = FALSE;
		}
		directx_joy_attached++;

		/* axes */
		directx_joystick[n_joy].axis[0] = dijs.lX;
		directx_joystick[n_joy].axis[1] = dijs.lY;
		n_axis = 2;

		if (directx_joystick[n_joy].axes_num > 3) directx_joystick[n_joy].axis[n_axis++] = dijs.lZ;
		if (directx_joystick[n_joy].axes_num > 4) directx_joystick[n_joy].axis[n_axis++] = dijs.lRx;
		if (directx_joystick[n_joy].axes_num > 5) directx_joystick[n_joy].axis[n_axis++] = dijs.lRy;
		if (directx_joystick[n_joy].axes_num > 6) directx_joystick[n_joy].axis[n_axis++] = dijs.lRz;

		for (j=0; n_axis < directx_joystick[n_joy].axes_num; j++)
			directx_joystick[n_joy].axis[n_axis++] = dijs.rglSlider[j];

		/* hat */
		for (h=0; h<directx_joystick[n_joy].hats_num; h++)
			switch (dijs.rgdwPOV[h]) {
			case DIJOY_POVFORWARD:				directx_joystick[n_joy].hat[h] = HAT_FORWARD; break;
			case DIJOY_POVFORWARDRIGHT:			directx_joystick[n_joy].hat[h] = HAT_FORWARD | HAT_RIGHT; break;
			case DIJOY_POVRIGHT:				directx_joystick[n_joy].hat[h] = HAT_RIGHT; break;
			case DIJOY_POVBACKWARDRIGHT:		directx_joystick[n_joy].hat[h] = HAT_BACKWARD | HAT_RIGHT; break;
			case DIJOY_POVBACKWARD:				directx_joystick[n_joy].hat[h] = HAT_BACKWARD; break;
			case DIJOY_POVBACKWARDLEFT:			directx_joystick[n_joy].hat[h] = HAT_BACKWARD | HAT_LEFT; break;
			case DIJOY_POVLEFT:					directx_joystick[n_joy].hat[h] = HAT_LEFT; break;
			case DIJOY_POVFORWARDLEFT:			directx_joystick[n_joy].hat[h] = HAT_FORWARD | HAT_LEFT; break;
			case DIJOY_POVCENTERED: default:	directx_joystick[n_joy].hat[h] = 0; break;
		}

		/* buttons */
		for (n_but = 0; n_but < directx_joystick[n_joy].buttons_num; n_but++)
			directx_joystick[n_joy].button[n_but] =  dijs.rgbButtons[n_but] & 0x80;
	}
}



/* joy_directx_poll:
 *  updates the joystick status variables
 */
static int joy_directx_poll(void)
{
   int directx_axis, num_sticks;
   int n_joy, n_stick, n_axis, n_but, p, range, h;

   /* update the Win32 joysticks status */
   poll_directx_joysticks();

   /* translate Win32 joystick status to Allegro joystick status */
   for (n_joy = 0; n_joy < num_joysticks; n_joy++) {

      /* sticks */
      n_stick = 0;
      directx_axis = 0;
      while (directx_axis < directx_joystick[n_joy].axes_num) {
	 /* skip hat at this point, it's handled later */
	 num_sticks = joy[n_joy].num_sticks - directx_joystick[n_joy].hats_num;

	 for (n_stick = 0; n_stick < num_sticks; n_stick++) {
	    for (n_axis = 0; n_axis < joy[n_joy].stick[n_stick].num_axis; n_axis++) {
	       /* map Windows axis range to 0-256 Allegro range */
	       p = directx_joystick[n_joy].axis[directx_axis] - directx_joystick[n_joy].min[directx_axis];
	       range = directx_joystick[n_joy].max[directx_axis] - directx_joystick[n_joy].min[directx_axis];
	       if (range > 0)
			p = p * 256 / range;
	       else
		  p = 0;

	       /* set pos of analog stick */
	       if (joy[n_joy].stick[n_stick].flags & JOYFLAG_ANALOGUE) {
		  if (joy[n_joy].stick[n_stick].flags & JOYFLAG_SIGNED)
		     joy[n_joy].stick[n_stick].axis[n_axis].pos = p - 128;
		  else
		     joy[n_joy].stick[n_stick].axis[n_axis].pos = p;
	       }

	       /* set pos of digital stick */
	       if (joy[n_joy].stick[n_stick].flags & JOYFLAG_DIGITAL) {
		  if (p < 64)
		     joy[n_joy].stick[n_stick].axis[n_axis].d1 = TRUE;
		  else
		     joy[n_joy].stick[n_stick].axis[n_axis].d1 = FALSE;

		  if (p > 192)
		     joy[n_joy].stick[n_stick].axis[n_axis].d2 = TRUE;
		  else
		     joy[n_joy].stick[n_stick].axis[n_axis].d2 = FALSE;
	       }

	       directx_axis++;
	    }
	 }
      }

      /* hat */
      for (h=0; h<directx_joystick[n_joy].hats_num; h++) {
         /* emulate analog joystick */
         joy[n_joy].stick[n_stick].axis[0].pos = 0;	
         joy[n_joy].stick[n_stick].axis[1].pos = 0;

         if (directx_joystick[n_joy].hat[h] & HAT_LEFT) {
            joy[n_joy].stick[n_stick].axis[0].d1 = TRUE;
            joy[n_joy].stick[n_stick].axis[0].pos = -128;
         }
         else
            joy[n_joy].stick[n_stick].axis[0].d1 = FALSE;

         if (directx_joystick[n_joy].hat[h] & HAT_RIGHT) {
            joy[n_joy].stick[n_stick].axis[0].d2 = TRUE;
            joy[n_joy].stick[n_stick].axis[0].pos = +128;
         }
         else
            joy[n_joy].stick[n_stick].axis[0].d2 = FALSE;

         if (directx_joystick[n_joy].hat[h] & HAT_FORWARD) {
            joy[n_joy].stick[n_stick].axis[1].d1 = TRUE;
            joy[n_joy].stick[n_stick].axis[1].pos = -128;
         }
         else
            joy[n_joy].stick[n_stick].axis[1].d1 = FALSE;

         if (directx_joystick[n_joy].hat[h] & HAT_BACKWARD) {
            joy[n_joy].stick[n_stick].axis[1].d2 = TRUE;
            joy[n_joy].stick[n_stick].axis[1].pos = +128;
         }
         else
            joy[n_joy].stick[n_stick].axis[1].d2 = FALSE;
         n_stick++;
      }

      /* buttons */
      for (n_but = 0; n_but < directx_joystick[n_joy].buttons_num; n_but++)
	 joy[n_joy].button[n_but].b = directx_joystick[n_joy].button[n_but];
   }

   return 0;
}



/* init_directx_joysticks:
 *  initialises the Direct X joystick devices
 */
static int init_directx_joysticks(void)
{
	int n_joyat, n_joy, n_axis, j;
	
	DWORD i;
	DIDEVICEINSTANCE* pDevices;
	DWORD             dwNumDevices;

	/* Initialize DirectInput */
	if(FAILED(DIUtil_Initialize(win_get_window()))) return -1;

	/* Get the list of DirectInput devices */
	DIUtil_GetDevices( &pDevices, &dwNumDevices );


	/* retrieve joystick infos */
	n_joy = 0;
   
   directx_joy_attached=0;
   
	/* Search through the list, looking for a joystick */
	for (i=0; i < dwNumDevices; i++) {
		LPDIRECTINPUTDEVICE2 DIdevice;
		DIDEVCAPS  dicaps; 
		HRESULT    hr;
		if ((pDevices[i].dwDevType & 0x000000ff) != DIDEVTYPE_JOYSTICK) continue;
      	if (n_joy == MAX_JOYSTICKS) break;
		
		DIdevice = DIUtil_CreateDevice(win_get_window(), &pDevices[i]);
		if (DIdevice == NULL) continue;
		directx_joy_num++;

		if (DIUtil_IsForceFeedback(DIdevice));

 
		dicaps.dwSize = sizeof(DIDEVCAPS); 
		hr = IDirectInputDevice_GetCapabilities(DIdevice,&dicaps); 

		directx_joystick[n_joy].device = DIdevice;
		directx_joystick[n_joy].buttons_num = MIN((int)dicaps.dwButtons, MAX_JOYSTICK_BUTTONS);
		directx_joystick[n_joy].axes_num = MIN((int)dicaps.dwAxes, DI_MAX_AXES);
		
		for (j=0; j<directx_joystick[n_joy].axes_num; j++) {
			directx_joystick[n_joy].min[j] = RANGE_MIN;
			directx_joystick[n_joy].max[j] = RANGE_MAX;
		}
		directx_joystick[n_joy].hats_num = dicaps.dwPOVs;

		if (dicaps.dwFlags & DIDC_ATTACHED) directx_joy_attached++;
		
		n_joy++;
	}

	directx_joy_num = n_joy;

	return (directx_joy_attached == 0);
}


/* joy_directx_init:
 *  initialises the driver
 */
static int joy_directx_init(void)
{
	int n_stick, n_joy, n_axis, n_but, n_hat;

	if (init_directx_joysticks() != 0)
		return -1;

	num_joysticks = directx_joy_attached;

	/* fill in the joystick structure */
	for (n_joy = 0; n_joy < num_joysticks; n_joy++) {
		joy[n_joy].flags = JOYFLAG_ANALOGUE | JOYFLAG_DIGITAL;

		/* how many sticks ? */
		n_stick = 0;

		if (directx_joystick[n_joy].axes_num > 0) {
			n_axis = 0;

			/* main analogue stick */
			if (directx_joystick[n_joy].axes_num > 1) {
				joy[n_joy].stick[n_stick].flags = JOYFLAG_DIGITAL | JOYFLAG_ANALOGUE | JOYFLAG_SIGNED;
				joy[n_joy].stick[n_stick].axis[0].name = name_x;
				joy[n_joy].stick[n_stick].axis[1].name = name_y;
				joy[n_joy].stick[n_stick].name = name_stick;
				joy[n_joy].stick[n_stick].num_axis = 2;
				n_axis=2;

				if (directx_joystick[n_joy].axes_num > 2) {
					joy[n_joy].stick[n_stick].num_axis++;
					joy[n_joy].stick[n_stick].axis[2].name = name_throttle;
					n_axis++;
				}

				n_stick++;
			}

			/* other 1-axis sticks */
			while (n_axis < directx_joystick[n_joy].axes_num) {
				joy[n_joy].stick[n_stick].flags = JOYFLAG_DIGITAL | JOYFLAG_ANALOGUE;
				joy[n_joy].stick[n_stick].num_axis = 1;
				joy[n_joy].stick[n_stick].axis[0].name = "";
				joy[n_joy].stick[n_stick].name = name_slider;
				n_stick++;
				n_axis++;
			}


			n_hat = 0;
			
			/* hat */
			while (n_hat < directx_joystick[n_joy].hats_num) {
				joy[n_joy].stick[n_stick].flags = JOYFLAG_DIGITAL | JOYFLAG_SIGNED;
				joy[n_joy].stick[n_stick].num_axis = 2;
				joy[n_joy].stick[n_stick].axis[0].name = "left/right";
				joy[n_joy].stick[n_stick].axis[1].name = "up/down";
				joy[n_joy].stick[n_stick].name = name_hat;
				n_stick++;
				n_hat++;
			}
		}

		joy[n_joy].num_sticks = n_stick;

		/* how many buttons ? */
		joy[n_joy].num_buttons = directx_joystick[n_joy].buttons_num;

		/* fill in the button names */
		for (n_but = 0; n_but < joy[n_joy].num_buttons; n_but++)
			joy[n_joy].button[n_but].name = name_b[n_but];
	}

	return 0;
}



/* joy_directx_exit:
 *  shuts down the driver
 */
static void joy_directx_exit(void)
{
	int n_joy;
	
	/* Unacquire and release joystick */
	for (n_joy = 0; n_joy < directx_joy_num; n_joy++) {
		if (directx_joystick[n_joy].device == NULL) continue;
		IDirectInputDevice_Unacquire(directx_joystick[n_joy].device);
		IDirectInputDevice_Release(directx_joystick[n_joy].device);
		directx_joystick[n_joy].device = NULL;
	}
	
	/* Release DirectInput */
	DIUtil_CleanupDirectInput();
	
	directx_joy_num = 0;
}



int ff_joystick_has(int n) {
	return DIUtil_IsForceFeedback( directx_joystick[n].device );
}
FFEFFECT * ff_create_effect() {
	if (directx_joy_num==0) return NULL;
	return DIUtil_CreateEffect();
}
void ff_destroy_effect(FFEFFECT *f) {
	if (directx_joy_num==0) return;
	DIUtil_DeleteEffect(f);
}
int ff_play_effect(FFEFFECT *f) {
	if (directx_joy_num==0) return -1;
	if (FAILED(DIUtil_PlayEffect(f))) return -1;
	return 0;
}
int ff_play_dir_effect(FFEFFECT *f, long direction) {
	if (directx_joy_num==0) return -1;
	if (FAILED(DIUtil_PlayDirectionalEffect(f,direction))) return -1;
	return 0;
}
int ff_make_periodic_effect(int n, FFEFFECT *f) {
	if (directx_joy_num==0) return -1;
	if (FAILED(DIUtil_SetupPeriodicEffect(f,directx_joystick[n].device))) return -1;
	return 0;
}
int ff_make_constant_force_effect(int n, FFEFFECT *f, int fudge) {
	if (directx_joy_num==0) return -1;
	if (FAILED(DIUtil_SetupConstantForceEffect(f,directx_joystick[n].device,fudge?TRUE:FALSE))) return -1;
	return 0;
}
int ff_make_fire_force_effect(int n, FFEFFECT *f) {
	if (directx_joy_num==0) return -1;
	if (FAILED(DIUtil_SetupFireForceEffect(f,directx_joystick[n].device))) return -1;
	return 0;
}