////////////////////////////////////////////////////////////////////////////////
// Deity - By Carl Olsson 2000
// sprite.h Sprite, Spriteset and Anim classes
////////////////////////////////////////////////////////////////////////////////

#ifndef _SPRITE_H
#define _SPRITE_H

#include <allegro.h>
#include <string.h>

// Sprite Class
class Sprite {
public:
   Sprite(void) : bitmap(NULL), x(0), y(0), w(0), h(0) {}
   ~Sprite(void) { destroyBitmap(); }
   BITMAP *getBitmap(void) { return bitmap; }
   void setBitmap(BITMAP *const _bitmap) { bitmap = _bitmap; }
   void createBitmap(void) {
      destroyBitmap();
      bitmap = create_bitmap(getW(), getH());
   }
   void createBitmap(const int depth) {
      destroyBitmap();
      bitmap = create_bitmap_ex(depth, getW(), getH());
   }
   void destroyBitmap(void) {
      if (bitmap != NULL) {
         destroy_bitmap(bitmap);
         bitmap = NULL;
      }
   }
   void setX(const int _x) { x = _x; }
   int getX(void) { return x; }
   void setY(const int _y) { y = _y; }
   int getY(void) { return y; }
   void setW(const int _w) { w = _w; }
   int getW(void) { return w; }
   void setH(const int _h) { h = _h; }
   int getH(void) { return h; }
   void draw(BITMAP *const destination, const int x, const int y);
   void drawTrans(BITMAP *const destination, const int x, const int y, const COLOR_MAP *const map);
   void drawLit(BITMAP *const destination, const int x, const int y, const int lightLevel, const COLOR_MAP *const map);
private:
   BITMAP *bitmap;
   int x;
   int y;
   int w;
   int h;
};

// Spriteset Class
class SpriteSet {
public:
   SpriteSet(void) : name(NULL), spriteCount(0), sprites(NULL) {}
   ~SpriteSet(void) {
      destroySprites();
      destroyName();
   }
   char *getName(void) { return name; }
   void setName(const char *const _name) {
      destroyName();
      name = new char[strlen(_name) + 1];
      strcpy(name, _name);
   }
   void destroyName(void) {
      if (name != NULL) {
         delete [] name;
         name = NULL;
      }
   }
   Sprite *getSprite(const int position) { return &sprites[position]; }
   void setSpriteCount(const int _spriteCount) { spriteCount = _spriteCount; }
   int getSpriteCount(void) { return spriteCount; }
   void createSprites(void) {
      destroySprites();
      sprites = new Sprite[getSpriteCount()];
   }
   void destroySprites(void) {
      if (sprites != NULL) {
         delete [] sprites;
         sprites = NULL;
      }
   }
private:
   char *name;
   int spriteCount;
   Sprite *sprites;
};

// Frame Class
class Frame {
public:
   Frame(void) : sprite(0), time(0) {}
   ~Frame(void) {}
   void setTime(const int _time) { time = _time; }
   int getTime(void) { return time; }
   void setSprite(const int _sprite) { sprite = _sprite; }
   int getSprite(void) { return sprite; }
private:
   int sprite;
   int time;
};

// Anim Class
class Anim {
public:
   Anim(void) : spriteSetName(NULL), spriteSet(NULL), frames(NULL), frameCount(0), name(NULL), time(0) {}
   ~Anim(void) {
      destroyName();
      destroyFrames();
   }
   char *getName(void) { return name; }
   void setName(const char *const _name) {
      destroyName();
      name = new char[strlen(_name) + 1];
      strcpy(name, _name);
   }
   void destroyName(void) {
      if (name != NULL) {
         delete [] name;
         name = NULL;
      }
   }
   char *getSpriteSetName(void) { return spriteSetName; }
   void setSpriteSetName(const char *const _spriteSetName) {
      destroySpriteSetName();
      spriteSetName = new char[strlen(_spriteSetName) + 1];
      strcpy(spriteSetName, _spriteSetName);
   }
   void destroySpriteSetName(void) {
      if (spriteSetName != NULL) {
         delete [] spriteSetName;
         spriteSetName = NULL;
      }
   }
   SpriteSet *getSpriteSet(void) { return spriteSet; }
   void setSpriteSet(SpriteSet *const _spriteSet) { spriteSet = _spriteSet; }
   Frame *getFrame(const int position) { return &frames[position]; }
   void setFrameCount(const int _frameCount) { frameCount = _frameCount; }
   int getFrameCount(void) { return frameCount; }
   void setTime(const int _time) { time = _time; }
   int getTime(void) { return time; }
   void createFrames(void) {
      destroyFrames();
      frames = new Frame[frameCount];
   }
   void destroyFrames(void) {
      if (frames != NULL) {
         delete [] frames;
         frames = NULL;
      }
   }
private:
   char *spriteSetName;
   SpriteSet *spriteSet;
   Frame *frames;
   int frameCount;
   char *name;
   int time;
};

class Animator {
public:
   Animator(void) : anim(NULL), frame(0), time(0) {}
   Animator(Anim *const _anim, const unsigned int _frame = 0,  const unsigned int _time = 0) {
      Animator();
      anim = _anim;
      frame = _frame;
      time = _time;
   }
   void setAnim(Anim *const _anim) { anim = _anim; }
   void setFrame(const int _frame) { frame = _frame; }
   void setTime(const int _time) { time = _time; }
   Anim *getAnim(void) { return anim; }
   int getFrame(void) { return frame; }
   int getTime(void) { return time; }
   void set(const int _frame, const int _time) {
      frame = _frame;
      while (frame < 0) frame += anim->getFrameCount();
      while (frame >= anim->getFrameCount()) frame -= anim->getFrameCount();
      time = _time;
      while (time < 0) {
         frame--;
         while (frame < 0) frame += anim->getFrameCount();
         while (frame >= anim->getFrameCount()) frame -= anim->getFrameCount();
         time += anim->getFrame(frame)->getTime();
      }
      while (time >= anim->getFrame(frame)->getTime()) {
         frame++;
         while (frame < 0) frame += anim->getFrameCount();
         while (frame >= anim->getFrameCount()) frame -= anim->getFrameCount();
         time -= anim->getFrame(frame)->getTime();
      }
   }
   void shift(const int amount) {
      time += amount;
      while (time < 0) {
         frame--;
         while (frame < 0) frame += anim->getFrameCount();
         while (frame >= anim->getFrameCount()) frame -= anim->getFrameCount();
         time += anim->getFrame(frame)->getTime();
      }
      while (time >= anim->getFrame(frame)->getTime()) {
         frame++;
         while (frame < 0) frame += anim->getFrameCount();
         while (frame >= anim->getFrameCount()) frame -= anim->getFrameCount();
         time -= anim->getFrame(frame)->getTime();
      }
   }
   Sprite *getSprite() {
      return anim->getSpriteSet()->getSprite(getSpriteIndex());
   }
   int getSpriteIndex() {
      return anim->getFrame(frame)->getSprite();
   }
   void draw(BITMAP *const destination, const int x, const int y) {
      getSprite()->draw(destination, x, y);
   }
   void drawTrans(BITMAP *const destination, const int x, const int y, const COLOR_MAP *const map) {
      getSprite()->drawTrans(destination, x, y, map);
   }
   void drawLit(BITMAP *const destination, const int x, const int y, const int lightLevel, const COLOR_MAP *const map) {
      getSprite()->drawLit(destination, x, y, lightLevel, map);
   }
private:
   Anim *anim;
   int frame;
   int time;
};

#endif // _SPRITE_H
