////////////////////////////////////////////////////////////////////////////////
// Deity - By Carl Olsson 2003
// script.cpp Script class
////////////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <allegro.h>
#include "script.h"
#include "sprite.h"
#include "main.h"



#define TOK_IDENT     0
#define TOK_CMNT_BLK  1
#define TOK_CMNT_LINE 2
#define TOK_STR_LIT   3
#define TOK_CHR_LIT   4
#define TOK_BRC_BLK   5
#define TOK_BRKT_BLK  6
#define TOK_PAREN_BLK 7

typedef struct TOKEN_TYPE_LIST {
   char *name;
   int id;
   char *blockStart;
   char *blockFinish;
} TOKEN_TYPE_LIST;

TOKEN_TYPE_LIST tokenTypes[] = {
   {"block comment",     TOK_CMNT_BLK,  "/*", "*/"},
   {"line comment",      TOK_CMNT_LINE, "//", "\n"},
   {"string literal",    TOK_STR_LIT,   "\"", "\""},
   {"character literal", TOK_CHR_LIT,   "\'", "\'"},
   {"braces block",      TOK_BRC_BLK,   "{",  "}"},
   {"brackets block",    TOK_BRKT_BLK,  "[",  "]"},
   {"parentheses block", TOK_PAREN_BLK, "(",  ")"},
   {NULL,                0,             NULL, NULL}
};

char *tokenSeperators[] = {
   " ",    "\n",   "\v",   "\f",
   "\t",   "\r",   "+",    "=",
   "-",    "/",    "*",    ".",
   "->",   ",",    ";",    "(",
   "[",    "{",
   "\0"
};

int processAnim(char *name, char *block)
{
   char *spriteSetString = NULL;
   int spriteSetType = 0;
   char *spriteNumString = NULL;
   int spriteNumType = 0;
   char *timeString = NULL;
   int timeType = 0;
   char **blockPos = NULL;
   char *framesPos = NULL;
   int frameCount;
   Anim *anim = NULL;

   blockPos = &block;

   readToken(blockPos, &spriteSetString, &spriteSetType);
   if (spriteSetString != NULL) {
      if (spriteSetType != TOK_STR_LIT) {
         // ERROR
         printf("sprite set name expected!\n");
      }
   }
   framesPos = *blockPos;

   frameCount = 0;
   readToken(blockPos, &spriteNumString, &spriteNumType);
   readToken(blockPos, &timeString, &timeType);
   while (spriteNumString != NULL) {
      frameCount++;

      delete [] spriteNumString;
      delete [] timeString;

      readToken(blockPos, &spriteNumString, &spriteNumType);
      readToken(blockPos, &timeString, &timeType);
   }

   anim = new Anim;
   anim->setName(name);
   anim->setSpriteSetName(spriteSetString);
   anim->setFrameCount(frameCount);
   anim->createFrames();

   blockPos = &framesPos;

   int currentFrame = 0;
   readToken(blockPos, &spriteNumString, &spriteNumType);
   readToken(blockPos, &timeString, &timeType);
   while (spriteNumString != NULL) {
      anim->getFrame(currentFrame)->setSprite(atoi(spriteNumString));
      anim->getFrame(currentFrame)->setTime(atoi(timeString));
      currentFrame++;

      delete [] spriteNumString;
      delete [] timeString;

      readToken(blockPos, &spriteNumString, &spriteNumType);
      readToken(blockPos, &timeString, &timeType);
   }

   delete [] spriteSetString;

   animRegister->add(anim);

   anim->setSpriteSet(spriteSetRegister->find(anim->getSpriteSetName()));/////////////////////

   return 0;
}

int processSpriteSet(char *name, char *block)
{
   char *imageFileString = NULL;
   int imageFileType = 0;
   char *spriteBitmapXString = NULL;
   int spriteBitmapXType = 0;
   char *spriteBitmapYString = NULL;
   int spriteBitmapYType = 0;
   char *spriteWidthString = NULL;
   int spriteWidthType = 0;
   char *spriteHeightString = NULL;
   int spriteHeightType = 0;
   char *spriteXString = NULL;
   int spriteXType = 0;
   char *spriteYString = NULL;
   int spriteYType = 0;
   char **blockPos = NULL;
   char *spritesPos = NULL;
   int spriteCount;
   SpriteSet *spriteSet = NULL;
   BITMAP *image = NULL;

   blockPos = &block;

   readToken(blockPos, &imageFileString, &imageFileType);
   if (imageFileString != NULL) {
      if (imageFileType != TOK_STR_LIT) {
         // ERROR
         printf("image name expected!\n");
      }
   }

   spritesPos = *blockPos;

   spriteCount = 0;

   readToken(blockPos, &spriteBitmapXString, &spriteBitmapXType);
   readToken(blockPos, &spriteBitmapYString, &spriteBitmapYType);
   readToken(blockPos, &spriteWidthString, &spriteWidthType);
   readToken(blockPos, &spriteHeightString, &spriteHeightType);
   readToken(blockPos, &spriteXString, &spriteXType);
   readToken(blockPos, &spriteYString, &spriteYType);
   while (spriteBitmapXString != NULL) {
      spriteCount++;

      delete [] spriteBitmapXString;
      delete [] spriteBitmapYString;
      delete [] spriteWidthString;
      delete [] spriteHeightString;
      delete [] spriteXString;
      delete [] spriteYString;

      readToken(blockPos, &spriteBitmapXString, &spriteBitmapXType);
      readToken(blockPos, &spriteBitmapYString, &spriteBitmapYType);
      readToken(blockPos, &spriteWidthString, &spriteWidthType);
      readToken(blockPos, &spriteHeightString, &spriteHeightType);
      readToken(blockPos, &spriteXString, &spriteXType);
      readToken(blockPos, &spriteYString, &spriteYType);
   }

   spriteSet = new SpriteSet;
   spriteSet->setName(name);
   spriteSet->setSpriteCount(spriteCount);
   spriteSet->createSprites();
   set_color_conversion(COLORCONV_NONE);////////////////////////
   image = load_bitmap(imageFileString, NULL);
   delete [] imageFileString;

   blockPos = &spritesPos;

   readToken(blockPos, &spriteBitmapXString, &spriteBitmapXType);
   readToken(blockPos, &spriteBitmapYString, &spriteBitmapYType);
   readToken(blockPos, &spriteWidthString, &spriteWidthType);
   readToken(blockPos, &spriteHeightString, &spriteHeightType);
   readToken(blockPos, &spriteXString, &spriteXType);
   readToken(blockPos, &spriteYString, &spriteYType);
   int currentSprite = 0;
   while (spriteBitmapXString != NULL) {
      Sprite *sprite = spriteSet->getSprite(currentSprite);
      int bitmapX = atoi(spriteBitmapXString);
      int bitmapY = atoi(spriteBitmapYString);
      int w = atoi(spriteWidthString);
      int h = atoi(spriteHeightString);
      int x = atoi(spriteXString);
      int y = atoi(spriteYString);
      sprite->setX(x);
      sprite->setY(y);
      sprite->setW(w);
      sprite->setH(h);
      sprite->createBitmap();
      blit(image, sprite->getBitmap(), bitmapX, bitmapY, 0, 0, w, h);
      currentSprite++;

      delete [] spriteBitmapXString;
      delete [] spriteBitmapYString;
      delete [] spriteWidthString;
      delete [] spriteHeightString;
      delete [] spriteXString;
      delete [] spriteYString;

      readToken(blockPos, &spriteBitmapXString, &spriteBitmapXType);
      readToken(blockPos, &spriteBitmapYString, &spriteBitmapYType);
      readToken(blockPos, &spriteWidthString, &spriteWidthType);
      readToken(blockPos, &spriteHeightString, &spriteHeightType);
      readToken(blockPos, &spriteXString, &spriteXType);
      readToken(blockPos, &spriteYString, &spriteYType);
   }

   destroy_bitmap(image);

   spriteSetRegister->add(spriteSet);

   return 0;
}

typedef struct TYPE_LIST {
   char *name;
   int (*func)(char *name, char *block);
   TYPE_LIST *typeProcess;
} TYPE_LIST;

TYPE_LIST typeProcess[] = {
   {"SpriteSet", processSpriteSet, NULL},
   {"Anim",    processAnim,    NULL},
   {NULL,      NULL,           NULL}
};

int readScript(char *fileName)
{
   FILE *diskFile = NULL;
   char *memFile = NULL;
   int fileSize = 0;
   char inChar = 0;

   // Read file into memory
   if ((diskFile = fopen(fileName, "rb")) == NULL) { // Open file
      return -1;
   }
   while (!feof(diskFile)) { // Find out soize of file
      fread(&inChar, 1, 1, diskFile);
      fileSize++;
   }
   fileSize--;
   rewind(diskFile);
   memFile = new char[fileSize + 1]; // Allocate memory
   fread(memFile, 1, fileSize, diskFile); // Copy file to memory
   memFile[fileSize] = '\0'; // Terminate string
   if (fclose(diskFile) == EOF) { // Close file
      return -1;
   }

   // Process
   if (processScript(memFile) < 0) {
      return -1;
   }

   // Free allocated memory
   delete [] memFile;

   return 0;
}

int isSpace(char testChar)
{
   if (testChar == ' ' ||
       testChar == '\t' ||
       testChar == '\v' ||
       testChar == '\n' ||
       testChar == '\r' ||
       testChar == '\f') {
      return 1;
   }
   return 0;
}

int isSeperator(char *string)
{
   for (int i = 0; *tokenSeperators[i] != '\0'; i++) {
      if (strncmp(tokenSeperators[i], string, strlen(tokenSeperators[i])) == 0) {
         return 1;
      }
   }
   return 0;
}

int skipSpace(char **memFile)
{
   int found = 0;

   while (isSpace(**memFile) != 0 && **memFile != '\0') { // Skip whitespace
      found++;
      (*memFile)++;
   }

   return found;
}

int readToken(char **memFile, char **token, int *type)
{
   char *tokenStart = NULL;
   char *tokenFinish = NULL;
   int tokenLength;
   char *tokenBuffer = NULL;
   int blockFound;
   int typeBuffer = 0;

   skipSpace(memFile); // Skip whitespace

   // No token found
   if (**memFile == '\0') {
      *token = NULL;
      return 1;
   }
   blockFound = FALSE;
   for (int i = 0; tokenTypes[i].name != NULL && blockFound == FALSE; i++) {
      if (strncmp(*memFile, tokenTypes[i].blockStart, strlen(tokenTypes[i].blockStart)) == 0) {
         typeBuffer = tokenTypes[i].id;
         (*memFile) += strlen(tokenTypes[i].blockStart);
         tokenStart = *memFile;
         while (strncmp(*memFile, tokenTypes[i].blockFinish, strlen(tokenTypes[i].blockFinish)) != 0 && **memFile != '\0') {
            (*memFile)++;
         }
         tokenFinish = *memFile;
         (*memFile) += strlen(tokenTypes[i].blockFinish);
         blockFound = TRUE;
      }
   }
   if (blockFound != TRUE) {
      typeBuffer = TOK_IDENT;
      tokenStart = *memFile;
      (*memFile)++;
      while (isSeperator(*memFile) == 0 && **memFile != '\0') {
        (*memFile)++;
      }
      tokenFinish = *memFile;
   }
   tokenLength = tokenFinish - tokenStart;
   tokenBuffer = new char[tokenLength + 1];
   memcpy(tokenBuffer, tokenStart, tokenLength);
   tokenBuffer[tokenLength] = '\0';
//   printf("readToken: %s, %d\n", tokenBuffer, tokenLength);

   *token = tokenBuffer;
   *type = typeBuffer;

   return 0;
}

int processScript(char *memFile)
{
   char **memFilePos = &memFile;
   int identifierType;
   char *typeString = NULL;
   char *nameString = NULL;
   char *blockString = NULL;
   int typeType;
   int nameType;
   int blockType;

   // Read type
   readToken(memFilePos, &typeString, &typeType);
   readToken(memFilePos, &nameString, &nameType);
   readToken(memFilePos, &blockString, &blockType);
   while (typeString != NULL) {
      if (typeString != NULL) {
         if (typeType != TOK_IDENT) {
            // ERROR
            printf("identifier expected!\n");
         }
         identifierType = -1;
         for (int i = 0; typeProcess[i].name != NULL; i++) {
            if (strcmp(typeProcess[i].name, typeString) == 0) {
               identifierType = i;
            }
         }
         if (identifierType < 0) { // valid type not found
            // ERROR
            printf("unknown type!\n");
         }
      }
      // Read name
      if (nameString != NULL) {
         if (nameType != TOK_STR_LIT) {
            // ERROR
            printf("string expected!\n");
         }
      }
      // Read block
      if (blockString != NULL) {
         if (blockType != TOK_BRC_BLK) {
            // ERROR
            printf("block expected!\n");
         }
      }

      if (nameString != NULL && blockString != NULL) {
         typeProcess[identifierType].func(nameString, blockString);
      }
      else {
         // ERROR
         printf("incomplete type!\n");
      }

      // Clean up
      delete [] blockString;
      delete [] nameString;
      delete [] typeString;

      // Read next
      readToken(memFilePos, &typeString, &typeType);
      readToken(memFilePos, &nameString, &nameType);
      readToken(memFilePos, &blockString, &blockType);
   }

   return 0;
}


