////////////////////////////////////////////////////////////////////////////////
// Deity - By Carl Olsson 2003
// input.cpp Input class
////////////////////////////////////////////////////////////////////////////////

#include <allegro.h>
#include "main.h"
#include "input.h"
#include "misc.h"
#include "interface.h"
#include "application.h"

#define KEY_SIZE 128

const int keyWaitTime = 4;
const bool allwaysKeyWait = false;

const int doubleClickWaitTime = 6;
const int dragWaitTime = 4;

int buildMode = BUILD_SQUARE;
int viewMode = 0;

int leftDoubleClick = FALSE;
int leftSingleClick = FALSE;
int leftDrag = FALSE;
int rightDoubleClick = FALSE;
int rightSingleClick = FALSE;
int rightDrag = FALSE;

enum inputType {
   inputState,
   inputPress,
   inputLMB,
   inputRMB
};

const double scrollAmount = 0.125;

void keyScroll(Input *const input, const int k)
{
   int xScroll = 0, yScroll = 0;

   switch (k) {
   case KEY_UP:
   case KEY_8_PAD:
      xScroll++;
      yScroll++;
      break;
   case KEY_DOWN:
   case KEY_2_PAD:
      xScroll--;
      yScroll--;
      break;
   case KEY_LEFT:
   case KEY_4_PAD:
      xScroll++;
      yScroll--;
      break;
   case KEY_RIGHT:
   case KEY_6_PAD:
      xScroll--;
      yScroll++;
      break;
   case KEY_HOME:
   case KEY_7_PAD:
      xScroll++;
      break;
   case KEY_END:
   case KEY_1_PAD:
      yScroll--;
      break;
   case KEY_PGUP:
   case KEY_9_PAD:
      yScroll++;
      break;
   case KEY_PGDN:
   case KEY_3_PAD:
      xScroll--;
      break;
   }
   input->getInterface()->view.setFocusX(input->getInterface()->view.getFocusX() - (xScroll * scrollAmount));
   input->getInterface()->view.setFocusY(input->getInterface()->view.getFocusY() - (yScroll * scrollAmount));
}

void keyWaterLevel(Input *const input, const int k)
{
   int change = 0;

   switch (k) {
   case KEY_EQUALS:
      change++;
      break;
   case KEY_MINUS:
      change--;
      break;
   }
   input->getWorld()->setWaterLevel(input->getWorld()->getWaterLevel() + change);
}

void keyWorldRegenerate(Input *const input, const int k)
{
   input->getWorld()->destroyPoints();
   input->getWorld()->createPoints();
   input->getWorld()->generatePointMap(200, 8);
   input->getWorld()->generateSquareMap();
   input->getWorld()->depopulate();
   input->getWorld()->populate(worldWorshipers, worldTrees);
}

void keyCursorMode(Input *const input, const int k)
{
   input->getInterface()->view.setCursorMode(!input->getInterface()->view.getCursorMode());
}

void keyScreenCapture(Input *const input, const int k)
{
   save_bitmap("scrsave.pcx", input->getInterface()->getApplication()->getDisplay()->getBitmap(), *input->getInterface()->getApplication()->gamePalette);
}

void keyBuildMode(Input *const input, const int k)
{
   if (buildMode == BUILD_SQUARE) buildMode = BUILD_DIAMOND;
   else buildMode = BUILD_SQUARE;
}

void keyViewMode(Input *const input, const int k)
{
   viewMode++;
   if (viewMode > 3) viewMode = 0;
}

void keyPause(Input *const input, const int k)
{
   input->getInterface()->getApplication()->setPaused(!input->getInterface()->getApplication()->getPaused());
}

void keyVideoMode(Input *const input, const int k)
{
/*   font = gameFontSmall;
   gui_fg_color = makecol_depth(videoDepth, 255, 255, 255);
   gui_bg_color = makecol_depth(videoDepth, 0, 0, 0);
   int driver = videoDriver;
   int w = videoW;
   int h = videoH;
   int depth = videoDepth;
   if (my_gfx_mode_select_ex(&driver, &w, &h, &depth)) {
      if (setVideoMode(driver, w, h, 0, 0, depth) < 0) { // Initialise video mode
         error("Select video mode failed!", EXIT_FAILURE);
      }
      set_palette(*gamePalette);
   }
   clear_keybuf();*/
}

void keyFont(Input *const input, const int k)
{
   if (input->getInterface()->getApplication()->gameFont == input->getInterface()->getApplication()->gameFontSmall) {
      input->getInterface()->getApplication()->gameFont = input->getInterface()->getApplication()->gameFontLarge;
   }
   else {
      input->getInterface()->getApplication()->gameFont = input->getInterface()->getApplication()->gameFontSmall;
   }
}

void keyGameExit(Input *const input, const int k)
{
   input->getInterface()->getApplication()->setExitFlag(true);
}

typedef struct INPUT_EVENT {
  void (*func)(Input *const input, const int k);
  int type;
  int key;
} INPUT_EVENT;

INPUT_EVENT inputEvents[] = {
   { keyGameExit,        inputPress, KEY_ESC },
   { keyWaterLevel,      inputPress, KEY_EQUALS },
   { keyWaterLevel,      inputPress, KEY_MINUS },
   { keyWorldRegenerate, inputPress, KEY_R },
   { keyCursorMode,      inputPress, KEY_C },
   { keyScreenCapture,   inputPress, KEY_S },
   { keyBuildMode,       inputPress, KEY_B },
   { keyViewMode,        inputPress, KEY_V },
   { keyVideoMode,       inputPress, KEY_M },
   { keyPause,           inputPress, KEY_P },
   { keyFont,            inputPress, KEY_F },
   { keyScroll,          inputState, KEY_PGUP },
   { keyScroll,          inputState, KEY_9_PAD },
   { keyScroll,          inputState, KEY_PGDN },
   { keyScroll,          inputState, KEY_3_PAD },
   { keyScroll,          inputState, KEY_END },
   { keyScroll,          inputState, KEY_1_PAD },
   { keyScroll,          inputState, KEY_HOME },
   { keyScroll,          inputState, KEY_7_PAD },
   { keyScroll,          inputState, KEY_UP },
   { keyScroll,          inputState, KEY_8_PAD },
   { keyScroll,          inputState, KEY_DOWN },
   { keyScroll,          inputState, KEY_2_PAD },
   { keyScroll,          inputState, KEY_LEFT },
   { keyScroll,          inputState, KEY_4_PAD },
   { keyScroll,          inputState, KEY_RIGHT },
   { keyScroll,          inputState, KEY_6_PAD },
   { NULL,               0,          0 }
};

World *Input::getWorld(void)
{
   return getInterface()->getWorld();
}

Input::Input(void)
{
   for (int i = 0; i < KEY_SIZE; i++) {
      keyPress[i] = FALSE;
      keyWait[i] = 0;
      keyState[i] = FALSE;
   }
   mouseX = 0;
   mouseY = 0;
   leftClick = 0;
   rightClick = 0;
   mouseChangeX = 0;
   mouseChangeY = 0;
}

Input::~Input(void)
{
}

void Input::poll(void)
{
   int i;

   // Mouse
   // Get mouse clicks
   leftClick = 0;
   rightClick = 0;

   // Update mouse vars
   poll_mouse();

   static int leftLastState = FALSE;
   static int leftDragTime = 0;
   static int leftFirstClick = FALSE;
   static int leftSecondClick = FALSE;
   static int leftClickWait = FALSE;

   static int rightLastState = FALSE;
   static int rightDragTime = 0;

   // Mouse button states
   leftState = (mouse_b & 1) ? TRUE : FALSE;
   rightState = (mouse_b & 2) ? TRUE : FALSE;

   // Left mouse drag
   if ((leftState == TRUE) && (leftLastState == TRUE)) {
      if (leftDragTime < dragWaitTime) leftDragTime++;
   }
   else {
      leftDragTime = 0;
   }
   if (leftDragTime >= dragWaitTime) {
      leftClick = DRAG;
   }

   // Right mouse drag
   if ((rightState == TRUE) && (rightLastState == TRUE)) {
      if (rightDragTime < dragWaitTime) rightDragTime++;
   }
   else {
      rightDragTime = 0;
   }
   if (rightDragTime >= dragWaitTime) {
      rightClick = DRAG;
   }

   // Left mouse click
   if ((leftState == TRUE) && (leftLastState == FALSE)) {
      if (leftFirstClick == FALSE) {
         leftFirstClick = TRUE;
         leftClickWait = doubleClickWaitTime;
      }
      else if (leftSecondClick == FALSE) {
         leftSecondClick = TRUE;
         leftClickWait = 0;
      }
   }
   leftLastState = leftState;

   rightLastState = rightState;

   if (leftClickWait == 0) {
      if (leftSecondClick == TRUE) {
         leftClick = DOUBLE_CLICK;
      }
      else if (leftFirstClick == TRUE) {
         leftClick = SINGLE_CLICK;
      }
      leftFirstClick = FALSE;
      leftSecondClick = FALSE;
   }

   if (leftClickWait > 0) leftClickWait--;

   // Get mouse mickeys
   get_mouse_mickeys(&mouseMickeysX, &mouseMickeysY);

   // Get mouse position change
   int tX = mouseX + mouseMickeysX;
   int tY = mouseY + mouseMickeysY;
   if (tX < 0) tX = 0;
   else if (tX >= SCREEN_W) tX = SCREEN_W - 1;
   if (tY < 0) tY = 0;
   else if (tY >= SCREEN_H) tY = SCREEN_H - 1;
   mouseChangeX = tX - mouseX;
   mouseChangeY = tY - mouseY;

   // Update mouse position
   mouseX += mouseChangeX;
   mouseY += mouseChangeY;

//   mouse_x = mouseX;
//   mouse_y = mouseY;
   position_mouse(mouseX, mouseY);

   // Keyboard
   // Update keyboard vars
   poll_keyboard();

   keyShift = key_shifts;
   // Store key states
   for (i = 0; i < KEY_SIZE; i++) key[i] ? keyState[i] = true : keyState[i] = false;
   // Validate key presses
   for (i = 0; i < KEY_SIZE; i++) {
      if ((keyState[i] == true) && (keyWait[i] == 0)) {
         keyPress[i] = true;
         keyWait[i] = keyWaitTime;
      }
      else {
         keyPress[i] = false;
         if ((keyState[i] == false) && !allwaysKeyWait) {
             keyWait[i] = 0;
         }
      }
   }
   // Reduce key wait
   for (i = 0; i < KEY_SIZE; i++) {
      if (keyWait[i] > 0) keyWait[i]--;
   }
}

//void Input::process(World *world, View *view, Map *map)
void Input::process(void)
{
   int i = 0;
   while (inputEvents[i].func != NULL) {
      switch (inputEvents[i].type) {
      case inputState:
         if (keyState[inputEvents[i].key] == true) {
            inputEvents[i].func(this, inputEvents[i].key);
         }
         break;
      case inputPress:
         if (keyPress[inputEvents[i].key] == true) {
            inputEvents[i].func(this, inputEvents[i].key);
         }
         break;
      }
      i++;
   }

#define TERRAIN_DRAG_CHANGE 8
   // Mouse
   // Click and drag testing
   if (leftClick == DRAG) leftDrag = TRUE;
   else leftDrag = FALSE;
   if (rightClick == DRAG) rightDrag = TRUE;
   else rightDrag = FALSE;
   if (leftClick == SINGLE_CLICK) {
      if (leftSingleClick == FALSE) leftSingleClick = TRUE;
      else leftSingleClick = FALSE;
   }
   if (leftClick == DOUBLE_CLICK) {
      if (leftDoubleClick == FALSE) leftDoubleClick = TRUE;
      else leftDoubleClick = FALSE;
   }

   if ((leftClick == DRAG) && // Raise/lower terrain
       (getInterface()->view.getCursorX() >= 0) && (getInterface()->view.getCursorX() < getWorld()->getW() + 1) &&
       (getInterface()->view.getCursorY() >= 0) && (getInterface()->view.getCursorY() < getWorld()->getH() + 1)) {
      mouseX -= mouseChangeX;
      mouseY -= mouseChangeY;
      if (mouseMickeysY < -TERRAIN_DRAG_CHANGE) {
         if (buildMode == BUILD_SQUARE) {
            getWorld()->raiseSquare(getInterface()->view.getCursorX(), getInterface()->view.getCursorY(), 1);
         }
         else if (buildMode == BUILD_DIAMOND) {
            getWorld()->raiseDiamond(getInterface()->view.getCursorX(), getInterface()->view.getCursorY(), 1);
         }
         mouseY += (int)getInterface()->view.getGameZPixelY();
      }
      if (mouseMickeysY > TERRAIN_DRAG_CHANGE) {
         if (buildMode == BUILD_SQUARE) {
            getWorld()->lowerSquare(getInterface()->view.getCursorX(), getInterface()->view.getCursorY(), 1);
         }
         else if (buildMode == BUILD_DIAMOND) {
            getWorld()->lowerDiamond(getInterface()->view.getCursorX(), getInterface()->view.getCursorY(), 1);
         }
         mouseY += (int)-getInterface()->view.getGameZPixelY();
      }
   }
   else if (rightClick == DRAG) { // Mouse scroll
      getInterface()->view.setFocusX(getInterface()->view.getFocusX() + mouseMickeysX / getInterface()->view.getGameXPixelX() + mouseMickeysY / getInterface()->view.getGameXPixelY());
      getInterface()->view.setFocusY(getInterface()->view.getFocusY() + mouseMickeysX / getInterface()->view.getGameYPixelX() + mouseMickeysY / getInterface()->view.getGameYPixelY());
      mouseX -= mouseChangeX;
      mouseY -= mouseChangeY;
   }
   
   if (getInterface()->view.getFocusX() < 0) getInterface()->view.setFocusX(0);
   else if (getInterface()->view.getFocusX() > getWorld()->getW()) getInterface()->view.setFocusX(getWorld()->getW());
   if (getInterface()->view.getFocusY() < 0) getInterface()->view.setFocusY(0);
   else if (getInterface()->view.getFocusY() > getWorld()->getH()) getInterface()->view.setFocusY(getWorld()->getH());

   double square_x, square_y;
//   view->pixelToSquareWithZ(input.mouseX, input.mouseY, &square_x, &square_y);
   getInterface()->view.pixelToPointWithZ(mouseX, mouseY, &square_x, &square_y);
//   view->pixelToSquare(input.mouseX, input.mouseY, &square_x, &square_y);

   if ((square_x >= 0) && (square_x < getWorld()->getW() + 1) &&
       (square_y >= 0) && (square_y < getWorld()->getH() + 1) &&
       (mouseX >= getInterface()->view.getX()) && (mouseX < getInterface()->view.getX() + getInterface()->view.getW()) &&
       (mouseY >= getInterface()->view.getY()) && (mouseY < getInterface()->view.getY() + getInterface()->view.getH())) {
      getInterface()->view.setCursorX(square_x);
      getInterface()->view.setCursorY(square_y);
   }
   else {
      getInterface()->view.setCursorX(-1);
      getInterface()->view.setCursorY(-1);
   }
}
