#ifndef _DISPLAY_H
#define _DISPLAY_H

#include <allegro.h>

#ifdef __cplusplus
   extern "C" {
#endif /* __cplusplus */

typedef struct display_t {
   int method;
   int driver;
   int w, h;
   int bpp;
   void *data;
   long flags;
} display_t;

AL_FUNC(void, display_init, (display_t *AL_CONST display));
AL_FUNC(int, display_set, (display_t *AL_CONST display, AL_CONST int method, AL_CONST int driver, AL_CONST int w, AL_CONST int h, AL_CONST int bpp));
AL_FUNC(void, display_clean, (display_t *AL_CONST display));
AL_FUNC(void, display_predraw, (display_t *AL_CONST display));
AL_FUNC(void, display_postdraw, (display_t *AL_CONST display));
AL_FUNC(void, display_update, (display_t *AL_CONST display));
AL_FUNC(BITMAP *, display_get_bitmap, (display_t *AL_CONST display));
AL_FUNC(const char *, display_get_name, (display_t *AL_CONST display));
AL_FUNC(void, display_mark_draw_region, (display_t *AL_CONST display, AL_CONST int x, AL_CONST int y, AL_CONST int w, AL_CONST int h));
AL_FUNC(int, display_flag_get, (display_t *AL_CONST display, AL_CONST long flag));
AL_FUNC(void, display_flag_set, (display_t *AL_CONST display, AL_CONST long flag, AL_CONST int state));
AL_FUNC(void, display_flag_toggle, (display_t *AL_CONST display, AL_CONST long flag));
AL_FUNC(void, display_flag_on, (display_t *AL_CONST display, AL_CONST long flag));
AL_FUNC(void, display_flag_off, (display_t *AL_CONST display, AL_CONST long flag));

#define FLAG_TRUE TRUE
#define FLAG_FALSE FALSE
#define FLAG_TOGGLE 3

typedef struct display_method_t {
   AL_CONST char *name;
   AL_METHOD(int, construct, (display_t *AL_CONST display));
   AL_METHOD(void, destruct, (display_t *AL_CONST display));
   AL_METHOD(void, predraw, (display_t *AL_CONST display));
   AL_METHOD(void, postdraw, (display_t *AL_CONST display));
   AL_METHOD(void, update, (display_t *AL_CONST display));
   AL_METHOD(void, mark_draw_region, (display_t *AL_CONST display, AL_CONST int x, AL_CONST int y, AL_CONST int w, AL_CONST int h));
   AL_METHOD(BITMAP *, get_bitmap, (display_t *AL_CONST display));
} display_method_t;

AL_CONST extern display_method_t display_methods[];



/* Display Flags **********************************************************/
#define FLAG_VSYNC             1
#define FLAG_SYSTEM_BUFFER     2
#define FLAG_SORT_DIRTY        4
#define FLAG_RETRACE_FLIPPING  8



/* Display Update Methods *************************************************/
#define METHOD_NONE -1
#define METHOD_DIRECT_TO_SCREEN 0
#define METHOD_DOUBLE_BUFFERING 1
#define METHOD_PAGE_FLIPPING    2
#define METHOD_TRIPLE_BUFFERING 3
#define METHOD_DIRTY_RECTANGLES 4
#define METHOD_COUNT 5



/* C++ Wrapper Class ******************************************************/
#ifdef __cplusplus
}

class Display {
public:
   Display(void) { init(); }
   Display(int method, int driver, int w, int h, int bpp) {
      init();
      if (set(method, driver, w, h, bpp)) {
         clean();
      }
   }
   ~Display(void) { clean(); }
   void init(void) { display_init(&display); }
   int set(const int method, const int driver, const int w, const int h, const int bpp) { return display_set(&display, method, driver, w, h, bpp); }
   void clean(void) { display_clean(&display); }
   void predraw(void) { display_predraw(&display); }
   void postdraw(void) { display_postdraw(&display); }
   void update(void) { display_update(&display); }
   BITMAP *getBitmap(void) { return display_get_bitmap(&display); }
   const char *getName(void) { return display_get_name(&display); }
   void markDrawRegion(const int x, const int y, const int w, const int h) { display_mark_draw_region(&display, x, y, w, h); }
   int flagGet(const long flag) { return display_flag_get(&display, flag); }
   void flagSet(const long flag, const int state) { display_flag_set(&display, flag, state); }
   void flagToggle(const long flag) { display_flag_toggle(&display, flag); }
   void flagOn(const long flag) { display_flag_on(&display, flag); }
   void flagOff(const long flag) { display_flag_off(&display, flag); }
//   int getW(const void) { return display.w; }
//   int getH(const void) { return display.h; }
//   int getBpp(const void) { return display.bpp; }
//   int getDriver(const void) { return display.driver; }
//   int getMethod(const void) { return display.method; }
   int getW(void) { return display.w; }
   int getH(void) { return display.h; }
   int getBpp(void) { return display.bpp; }
   int getDriver(void) { return display.driver; }
   int getMethod(void) { return display.method; }
private:
   display_t display;
};

extern "C" {
#endif /* __cplusplus */

#ifdef __cplusplus
   }
#endif /* __cplusplus */

#endif /* _DISPLAY_H */
