#include "application.h"

#include <allegro.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>
#include "gamedata.h"
#include "main.h"
#include "misc.h"
#include "world.h"
#include "input.h"
#include "view.h"
#include "register.h"
#include "script.h"
#include "display.h"
#include "sprite.h"
#include "interface.h"
#include "application.h"

#define VIDEO_X 0
#define VIDEO_Y 0
#define VIDEO_W display->getW()
#define VIDEO_H display->getH()
#define VIEW_X 0
#define VIEW_Y 0
#define VIEW_W VIDEO_W
#define VIEW_H (VIDEO_W / 2)
#define INFO_X (VIDEO_H - (VIDEO_W / 2))
#define INFO_Y (VIDEO_W / 2)
#define INFO_W (VIDEO_W - (VIDEO_H - (VIDEO_W / 2)))
#define INFO_H (VIDEO_H - (VIDEO_W / 2))

const int worldW = 64;
const int worldH = 64;
const int worldWorshipers = sqrt(worldW * worldH);
const int worldTrees = sqrt(worldW * worldH) * 10;

int waterFrame = 0;
int objectFrame = 0;
int waterFrameWait = 0;
int objectFrameWait = 0;

volatile int milliseconds = 0;
volatile int seconds = 0;
volatile int minutes = 0;
volatile int hours = 0;

volatile int frame = 0;
volatile long int frameTotal = 0;
volatile int fps = 0;

volatile int cycle = 0;
volatile long int cycleTotal = 0;
volatile int cps = 0;

volatile int speedCounter = 0;

RGB_MAP rgbToIndexMap;
COLOR_MAP lightMap;
COLOR_MAP transMap;

Register<SpriteSet> *spriteSetRegister = NULL;
Register<Anim> *animRegister = NULL;

void fpsUpdate(void)
{
   fps = frame;
   frame = 0;
}
END_OF_FUNCTION(fpsUpdate);

void cpsUpdate(void)
{
   cps = cycle;
   cycle = 0;
}
END_OF_FUNCTION(cpsUpdate);

void timerUpdate(void)
{
   milliseconds++;
   while (milliseconds >= 1000) {
      fpsUpdate();
      cpsUpdate();
      seconds++;
      milliseconds -= 1000;
   }
   while (seconds >= 60) {
      minutes++;
      seconds -= 60;
   }
   while (minutes >= 60) {
      hours++;
      minutes -= 60;
   }
}
END_OF_FUNCTION(timerUpdate);

void incrementSpeedCounter(void)
{
   speedCounter++;
}
END_OF_FUNCTION(incrementSpeedCounter);

////////////////////////////////////////////////////////////////////////////////
// Draw Info
// Draws the info display
////////////////////////////////////////////////////////////////////////////////
void Application::drawInfo(BITMAP *destination, World *world, View *view, int _x, int _y, int _w, int _h)
{
   int lineHeight = text_height(gameFont);
   int xx, yy, ww, hh;

   xx = _x + 2;
   yy = _y + 2;
   ww = _w - 4;
   hh = _h - 4;

   set_clip(destination, _x, _y, _x + _w - 1, _y + _h - 1);
   rectfill(destination, _x, _y, _x + _w - 1, _y + _h - 1, makecol_depth(display->getBpp(), 31, 31, 127));
   rect(destination, _x, _y, _x + _w - 1, _y + _h - 1, makecol_depth(display->getBpp(), 0, 127, 255));
   set_clip(destination, xx, yy, xx + ww - 1, yy + hh - 1);

   text_mode(-1);

   long int secs = ((hours * 60) + minutes) * 60 + seconds;
   double frmsecs = (double)frameTotal / (double)secs;
   double cycsecs = (double)cycleTotal / (double)secs;

   int textColour = makecol_depth(display->getBpp(), 255, 255, 255);

   textprintf(destination, gameFont, xx + 0, yy + ( 0 * lineHeight), textColour, "INFORMATION");
   textprintf(destination, gameFont, xx + 0, yy + ( 1 * lineHeight), textColour, "Font (F), Save screen (S), Exit (Esc),");
   textprintf(destination, gameFont, xx + 0, yy + ( 2 * lineHeight), textColour, "Regenerate (R), Cursor Mode (C), Water level (-, =), V-Sync (Y),");
   textprintf(destination, gameFont, xx + 0, yy + ( 3 * lineHeight), textColour, "Raise/lower (LMB Y drag), Scroll (RMB drag), Build mode (B)");
   textprintf(destination, gameFont, xx + 0, yy + ( 4 * lineHeight), textColour, "Time: %02d:%02d:%02d.%03d", hours, minutes, seconds, milliseconds);
   textprintf(destination, gameFont, xx + 0, yy + ( 5 * lineHeight), textColour, "FPS: %3d, %7.3f, (%ld)", fps, frmsecs, frameTotal);
   textprintf(destination, gameFont, xx + 0, yy + ( 6 * lineHeight), textColour, "CPS: %3d, %7.3f, (%ld)", cps, cycsecs, cycleTotal);
   textprintf(destination, gameFont, xx + 0, yy + ( 7 * lineHeight), textColour, "View: %3.2f, %3.2f", interface.view.getFocusX(), interface.view.getFocusY());
   textprintf(destination, gameFont, xx + 0, yy + ( 8 * lineHeight), textColour, "Mouse: %4d, %4d", interface.input.mouseX, interface.input.mouseY);
   textprintf(destination, gameFont, xx + 0, yy + ( 9 * lineHeight), textColour, "Cursor: %2d, %2d, (%s)", interface.view.getCursorX(), interface.view.getCursorY(), interface.view.getCursorMode() ? "Point" : "Square");
   textprintf(destination, gameFont, xx + 0, yy + (10 * lineHeight), textColour, "Squares drawn: %d", squaresDrawn);
   textprintf(destination, gameFont, xx + 0, yy + (11 * lineHeight), textColour, "Objects drawn: %d", objectsDrawn);
   textprintf(destination, gameFont, xx + 0, yy + (12 * lineHeight), textColour, "Tiles drawn: %d", spritesDrawn);
   textprintf(destination, gameFont, xx + 0, yy + (13 * lineHeight), textColour, "Water level: %d", interface.getWorld()->getWaterLevel());
   textprintf(destination, gameFont, xx + 0, yy + (14 * lineHeight), textColour, "Build mode: %s", (buildMode == BUILD_SQUARE) ? "Square" : "Diamond");
   textprintf(destination, gameFont, xx + 0, yy + (15 * lineHeight), textColour, "Display: %s, %d x %d (%d bit)", display->getName(), display->getW(), display->getH(), display->getBpp());
}

////////////////////////////////////////////////////////////////////////////////
// Force Draw
// Force the screen to update on next loop
////////////////////////////////////////////////////////////////////////////////
void Application::forceDraw(void)
{
   speedCounter = 1;
}

////////////////////////////////////////////////////////////////////////////////
// Draw Mouse
// Draw the mouse cursor on the screen
////////////////////////////////////////////////////////////////////////////////
void drawMouse(BITMAP *destination, Input *input, int sprite)
{
   set_clip(destination, 0, 0, destination->w - 1, destination->h - 1);

   spriteSetRegister->find("Cursors")->getSprite(sprite)->draw(destination, input->mouseX, input->mouseY);
}

////////////////////////////////////////////////////////////////////////////////
// Startup
// Mess up for playing
////////////////////////////////////////////////////////////////////////////////
void Application::startup(void)
{
   int videoMethod = METHOD_DOUBLE_BUFFERING;
   int videoDriver = GFX_SAFE;
   int videoW = 320;
   int videoH = 240;
   int videoBpp = 8;

   // Open Up
   allegro_init(); // Initialise Allegro
   install_timer();
      
   char windowTitle[128];
   sprintf(windowTitle, "%s Stage %d Version %4.2f", GAME_TITLE, GAME_STAGE, GAME_VERSION);
   set_window_title(windowTitle);

   // Input devices
   install_keyboard();
   if (!install_mouse()) {
      error("Install mouse failed!", EXIT_FAILURE);
   }

   // Data files
   if ((gameData = load_datafile("gamedata.dat")) == NULL) {
      error("Load datafile failed!", EXIT_FAILURE);
   }
   gamePalette = (PALETTE *)gameData[GAME_PALETTE].dat;
   set_palette(*gamePalette);
   gameFontSmall = (FONT *)gameData[GAME_FONT_SMALL].dat;
   gameFontLarge = (FONT *)gameData[GAME_FONT_LARGE].dat;
   gameFont = gameFontSmall;

   // Configuration
   videoMethod = METHOD_DOUBLE_BUFFERING;
//   videoMethod = METHOD_PAGE_FLIPPING;
//   videoMethod = METHOD_TRIPLE_BUFFERING;
   videoDriver = GFX_AUTODETECT;
//   videoDriver = GFX_VESA3;
//   videoDriver = GFX_DIRECTX;
   videoW = 640;
   videoH = 480;
   videoBpp = 8;
   set_config_file("deity.cfg");
//   videoMethod = get_config_int("video", "method", videoMethod);
   videoDriver = get_config_int("video", "driver", videoDriver);
   videoW = get_config_int("video", "width", videoW);
   videoH = get_config_int("video", "height", videoH);
   videoBpp = get_config_int("video", "bpp", videoBpp);

   // Display
   display = new Display;
   if (display->set(videoMethod, videoDriver, videoW, videoH, videoBpp)) { // Initialise safe video mode
      error("Set video mode failed!", EXIT_FAILURE);
   }
   set_palette(*gamePalette);
   font = gameFontSmall;
   gui_fg_color = makecol_depth(display->getBpp(), 255, 255, 255);
   gui_bg_color = makecol_depth(display->getBpp(), 0, 0, 0);

   // Timers
   LOCK_VARIABLE(milliseconds);
   LOCK_VARIABLE(seconds);
   LOCK_VARIABLE(minutes);
   LOCK_VARIABLE(hours);
   LOCK_VARIABLE(frame);
   LOCK_VARIABLE(frameTotal);
   LOCK_VARIABLE(fps);
   LOCK_VARIABLE(cycle);
   LOCK_VARIABLE(cycleTotal);
   LOCK_VARIABLE(cps);
   LOCK_VARIABLE(speedCounter);
   LOCK_FUNCTION(timerUpdate);
   LOCK_FUNCTION(incrementSpeedCounter);
   install_int(timerUpdate, 1);
   install_int_ex(incrementSpeedCounter, BPS_TO_TIMER(30));

   // Colour mapping
   create_rgb_table(&rgbToIndexMap, *gamePalette, NULL);
   rgb_map = &rgbToIndexMap;
//   create_light_table(&light_map, *gamePalette, 0, 0, 127, NULL);
   create_light_table(&lightMap, *gamePalette, 0, 0, 0, NULL);
//   color_map = &lightMap;
   create_trans_table(&transMap, *gamePalette, 127, 127, 127, NULL);
   color_map = &transMap;

   // Randomise
   srand(time(NULL));
}

////////////////////////////////////////////////////////////////////////////////
// Shutdown
// Clean up for closing
////////////////////////////////////////////////////////////////////////////////
void Application::shutdown(void)
{
   // Configuration
   set_config_int("video", "method", display->getMethod());
   set_config_int("video", "driver", display->getDriver());
   set_config_int("video", "width", display->getW());
   set_config_int("video", "height", display->getH());
   set_config_int("video", "bpp", display->getBpp());

   // destroy display
   delete display;
      
   // Timers
   remove_int(timerUpdate);
   remove_int(incrementSpeedCounter);

   // Data files
   unload_datafile(gameData);

   // Shut Down
   allegro_exit(); // Deinitialise Allegro
}

////////////////////////////////////////////////////////////////////////////////
// Title Screen
// Display the title screen
////////////////////////////////////////////////////////////////////////////////
int Application::titleScreen(void)
{
   BITMAP *title = NULL;
   PALETTE palette;

   // Display title
   set_color_conversion(COLORCONV_TOTAL);
   if ((title = load_bitmap("title.pcx", palette)) == NULL) {
      error("Load bitmap title.pcx failed!", EXIT_FAILURE);
   }
   set_palette(black_palette);
   acquire_screen();
   vsync();
   stretch_blit(title, screen, 0, 0, title->w, title->h, 0, 0, SCREEN_W, SCREEN_H);
   release_screen();
   destroy_bitmap(title);
   text_mode(-1);
   fade_in(palette, 8);
   int secs = seconds;
   while (!keypressed() && !mouse_b && ((secs + 8) > ((seconds < secs) ? (seconds + 60) : seconds)));
   fade_out(8);
   acquire_screen();
   clear(screen);
   release_screen();

   // Make sure correct palette is in use
   set_palette(*gamePalette);

   if (key[KEY_ESC]) {
      return 1;
   }

   return 0;
}

////////////////////////////////////////////////////////////////////////////////
// Run
// Game is called from here
////////////////////////////////////////////////////////////////////////////////
int Application::run(void)
{
   World *world = NULL;
   
   world = new World(worldW, worldH);

   spriteSetRegister = new Register<SpriteSet>;
   animRegister = new Register<Anim>;

   set_config_file("deity.cfg");

   // Open tilesets
   readScript("cursors.txt");
   readScript("grass.txt");
   readScript("dirt.txt");
   readScript("sand.txt");
   readScript("tree.txt");
   readScript("human.txt");
   readScript("water0.txt");
   readScript("water1.txt");
   readScript("water2.txt");
   readScript("water3.txt");

//   animRegister->display();
//   while (!keypressed());

   // Display title screen
   if (titleScreen()) {
      exitFlag = TRUE;
   }

   world->generatePointMap(200, 8);
   world->generateSquareMap();
   world->populate(worldWorshipers, worldTrees);

   interface.setWorld(world);
   
   // Set up view
   interface.view.setSpatialRelationship(32, 16, -32, 16, 0, -12);

   // Make it always draw a frame no matter how slow
   forceDraw();

   // Body
   while (!exitFlag) {
      while (speedCounter > 0) {
         // Get input
         interface.input.poll();
         interface.input.process();
         if (!paused) {
            // Update frame animation
            if (++waterFrameWait >= 4) {
               waterFrameWait = 0;
               if (++waterFrame >= 4) waterFrame = 0;
            }
            if (++objectFrameWait >= 4) {
               objectFrameWait = 0;
               if (++objectFrame >= 4) objectFrame = 0;
            }
            // Do processing
            world->update();
            // One more game cycle has been processed
            cycle++;
            cycleTotal++;
         }
         if (++interface.view.cursorColourWait >= 8) {
            interface.view.cursorColourWait = 0;
            if (interface.view.cursorColourState == true) {
               interface.view.cursorColour = makecol_depth(display->getBpp(), 255, 255, 255);
               interface.view.cursorColourState = false;
            }
            else {
               interface.view.cursorColour =  makecol_depth(display->getBpp(), 0, 0, 0);
               interface.view.cursorColourState = true;
            }
         }
         speedCounter--;
      }
      // Draw screen
      display->predraw();
//      clear_to_color(display->getBitmap(), 1);
      interface.view.set(VIEW_X, VIEW_Y, VIEW_W, VIEW_H, VIEW_W / 2, VIEW_H / 2);
      interface.view.draw(display->getBitmap());
      drawInfo(display->getBitmap(), world, &interface.view, 0, INFO_Y, VIDEO_W, INFO_H);
      drawMouse(display->getBitmap(), &interface.input, 0);
      if (paused) {
         textprintf_centre(display->getBitmap(), gameFontLarge, display->getW() / 2, display->getH() / 2,  makecol_depth(display->getBpp(), 255, 255, 255), "Paused");
      }

      display->postdraw();
      display->update();
      // One more frame has been drawn
      frame++;
      frameTotal++;
   }
   world->depopulate();

   delete world;

   delete spriteSetRegister;
   delete animRegister;

   return EXIT_SUCCESS; // Send success message to OS
}

