#include "display.h"



/* Direct To Screen Method ************************************************/
/* Predraw */
void direct_to_screen_predraw(display_t *const display)
{
   acquire_bitmap(screen);
   if (display_flag_get(display, FLAG_VSYNC)) vsync();
}

/* Postdraw */
void direct_to_screen_postdraw(display_t *const display)
{
   release_bitmap(screen);
}

/* Get Bitmap */
BITMAP *direct_to_screen_get_bitmap(display_t *const display)
{
   return screen;
}



/* Double Buffering Method ************************************************/
typedef struct double_buffering_data_t {
   BITMAP *buffer;
} double_buffering_data_t;

/* Construct */
int double_buffering_construct(display_t *const display)
{
   if ((display->data = malloc(sizeof(double_buffering_data_t))) == NULL) {
      return 1;
   }

   ((double_buffering_data_t *)display->data)->buffer = NULL;
   
   if (display_flag_get(display, FLAG_SYSTEM_BUFFER)) {
      if ((((double_buffering_data_t *)display->data)->buffer = create_bitmap(display->w, display->h)) == NULL) {
         return 1;
      }
   }
   else {
      if ((((double_buffering_data_t *)display->data)->buffer = create_system_bitmap(display->w, display->h)) == NULL) {
         return 1;
      }
   }

   return 0;
}

/* Destruct */
void double_buffering_destruct(display_t *const display)
{
   if (display->data != NULL) {
      destroy_bitmap(((double_buffering_data_t *)display->data)->buffer);
      free(display->data);
      display->data = NULL;
   }
}

/* Update */
void double_buffering_update(display_t *const display)
{
   acquire_bitmap(screen);
   if (display_flag_get(display, FLAG_VSYNC)) vsync();
   blit(((double_buffering_data_t *)display->data)->buffer, screen, 0, 0, 0, 0, display->w, display->h);
   release_bitmap(screen);
}

/* Get Bitmap */
BITMAP *double_buffering_get_bitmap(display_t *const display)
{
   return ((double_buffering_data_t *)display->data)->buffer;
}



/* Page Flipping Method ***************************************************/
typedef struct page_flipping_data_t {
   int page;
   BITMAP *pages[2];
} page_flipping_data_t;

/* Construct */
int page_flipping_construct(display_t *const display)
{
   if (display_flag_get(display, FLAG_RETRACE_FLIPPING)) {
      if (!timer_can_simulate_retrace()) {
         return 1;
      }
      timer_simulate_retrace(TRUE);
   }

   if ((display->data = malloc(sizeof(page_flipping_data_t))) == NULL) {
      return 1;
   }

   ((page_flipping_data_t *)display->data)->pages[0] = NULL;
   ((page_flipping_data_t *)display->data)->pages[1] = NULL;
   
   if ((((page_flipping_data_t *)display->data)->pages[0] = create_video_bitmap(display->w, display->h)) == NULL) {
      return 1;
   }

   if ((((page_flipping_data_t *)display->data)->pages[1] = create_video_bitmap(display->w, display->h)) == NULL) {
      return 1;
   }

   ((page_flipping_data_t *)display->data)->page = 0;

   return 0;
}

/* Destruct */
void page_flipping_destruct(display_t *const display)
{
   if (display->data != NULL) {
      destroy_bitmap(((page_flipping_data_t *)display->data)->pages[0]);
      destroy_bitmap(((page_flipping_data_t *)display->data)->pages[1]);
      free(display->data);
      display->data = NULL;
   }
}

/* Predraw */
void page_flipping_predraw(display_t *const display)
{
   acquire_bitmap(((page_flipping_data_t *)display->data)->pages[((page_flipping_data_t *)display->data)->page]);
}

/* Postdraw */
void page_flipping_postdraw(display_t *const display)
{
   textprintf(((page_flipping_data_t *)display->data)->pages[((page_flipping_data_t *)display->data)->page], font, 0, 0, 255, "Page %d", ((page_flipping_data_t *)display->data)->page);
   release_bitmap(((page_flipping_data_t *)display->data)->pages[((page_flipping_data_t *)display->data)->page]);
}

/* Update */
void page_flipping_update(display_t *const display)
{
   show_video_bitmap(((page_flipping_data_t *)display->data)->pages[((page_flipping_data_t *)display->data)->page]);
   ((page_flipping_data_t *)display->data)->page = 1 - ((page_flipping_data_t *)display->data)->page;
}

/* Get Bitmap */
BITMAP *page_flipping_get_bitmap(display_t *const display)
{
   return ((page_flipping_data_t *)display->data)->pages[((page_flipping_data_t *)display->data)->page];
}



/* Triple Buffering Method ************************************************/
typedef struct triple_buffering_data_t {
   BITMAP *active_page;
   int page;
   BITMAP *pages[3];
} triple_buffering_data_t;

/* Construct */
int triple_buffering_construct(display_t *const display)
{
   if (!(gfx_capabilities & GFX_CAN_TRIPLE_BUFFER))
      enable_triple_buffer();

   if (!(gfx_capabilities & GFX_CAN_TRIPLE_BUFFER)) {
      return 1;
   }

   if ((display->data = malloc(sizeof(triple_buffering_data_t))) == NULL) {
      return 1;
   }

   ((triple_buffering_data_t *)display->data)->pages[0] = NULL;
   ((triple_buffering_data_t *)display->data)->pages[1] = NULL;
   ((triple_buffering_data_t *)display->data)->pages[2] = NULL;
   
   if ((((triple_buffering_data_t *)display->data)->pages[0] = create_video_bitmap(SCREEN_W, SCREEN_H)) == NULL) {
      return 1;
   }
   if ((((triple_buffering_data_t *)display->data)->pages[1] = create_video_bitmap(SCREEN_W, SCREEN_H)) == NULL) {
      return 1;
   }
   if ((((triple_buffering_data_t *)display->data)->pages[2] = create_video_bitmap(SCREEN_W, SCREEN_H)) == NULL) {
      return 1;
   }

   ((triple_buffering_data_t *)display->data)->page = 0;

   return 0;
}

/* Destruct */
void triple_buffering_destruct(display_t *const display)
{
   if (display->data != NULL) {
      destroy_bitmap(((triple_buffering_data_t *)display->data)->pages[0]);
      destroy_bitmap(((triple_buffering_data_t *)display->data)->pages[1]);
      destroy_bitmap(((triple_buffering_data_t *)display->data)->pages[2]);
      free(display->data);
      display->data = NULL;
   }
}

/* Predraw */
void triple_buffering_predraw(display_t *const display)
{
   acquire_bitmap(((triple_buffering_data_t *)display->data)->pages[((triple_buffering_data_t *)display->data)->page]);
}

/* Postdraw */
void triple_buffering_postdraw(display_t *const display)
{
   release_bitmap(((triple_buffering_data_t *)display->data)->pages[((triple_buffering_data_t *)display->data)->page]);
}

/* Update */
void triple_buffering_update(display_t *const display)
{
   do {
   } while (poll_scroll());
   request_video_bitmap(((triple_buffering_data_t *)display->data)->pages[((triple_buffering_data_t *)display->data)->page]);
   ((triple_buffering_data_t *)display->data)->page = (((triple_buffering_data_t *)display->data)->page + 1) % 3;
}

/* Get Bitmap */
BITMAP *triple_buffering_get_bitmap(display_t *const display)
{
   return ((triple_buffering_data_t *)display->data)->pages[((triple_buffering_data_t *)display->data)->page];
}



/* Dirty Rectangles Method ************************************************/
typedef struct dirty_t {
   int x, y;
   int w, h;
} dirty_t;

typedef struct dirty_list_t {
   int size;
   int count;
   dirty_t *list;
} dirty_list_t;

typedef struct dirty_rectangles_data_t {
   BITMAP *buffer;
   dirty_list_t current;
   dirty_list_t old;
} dirty_rectangles_data_t;

/* Construct */
int dirty_rectangles_construct(display_t *const display)
{
   if ((display->data = malloc(sizeof(dirty_rectangles_data_t))) == NULL) {
      return 1;
   }

   ((dirty_rectangles_data_t *)display->data)->old.size = 1024;
   if ((((dirty_rectangles_data_t *)display->data)->old.list = (dirty_t *)malloc(sizeof(dirty_t) * ((dirty_rectangles_data_t *)display->data)->old.size)) == NULL) {
      return 1;
   }
   ((dirty_rectangles_data_t *)display->data)->old.count = 0;

   ((dirty_rectangles_data_t *)display->data)->current.size = 1024;
   if ((((dirty_rectangles_data_t *)display->data)->current.list = (dirty_t *)malloc(sizeof(dirty_t) * ((dirty_rectangles_data_t *)display->data)->current.size)) == NULL) {
      return 1;
   }
   ((dirty_rectangles_data_t *)display->data)->current.count = 0;

   ((dirty_rectangles_data_t *)display->data)->buffer = NULL;

   if (display_flag_get(display, FLAG_SYSTEM_BUFFER)) {
      if ((((dirty_rectangles_data_t *)display->data)->buffer = create_bitmap(display->w, display->h)) == NULL) {
         return 1;
      }
   }
   else {
      if ((((dirty_rectangles_data_t *)display->data)->buffer = create_system_bitmap(display->w, display->h)) == NULL) {
         return 1;
      }
   }

   return 0;
}

/* Destruct */
void dirty_rectangles_destruct(display_t *const display)
{
   if (display->data != NULL) {
      free(((dirty_rectangles_data_t *)display->data)->old.list);
      free(((dirty_rectangles_data_t *)display->data)->current.list);
      destroy_bitmap(((dirty_rectangles_data_t *)display->data)->buffer);
      free(display->data);
      display->data = NULL;
   }
}

static int _dirty_sorter(const void *e1, const void *e2)
{
   return ((dirty_t *)e1)->y - ((dirty_t *)e2)->y;
}

/* Update */
void dirty_rectangles_update(display_t *const display)
{
   int i;

   acquire_bitmap(screen);
   if (display_flag_get(display, FLAG_VSYNC)) vsync();
   for (i = 0; i < ((dirty_rectangles_data_t *)display->data)->old.count; i++) {
      blit(((dirty_rectangles_data_t *)display->data)->buffer, screen, ((dirty_rectangles_data_t *)display->data)->old.list[i].x, ((dirty_rectangles_data_t *)display->data)->old.list[i].y, ((dirty_rectangles_data_t *)display->data)->old.list[i].x, ((dirty_rectangles_data_t *)display->data)->old.list[i].y, ((dirty_rectangles_data_t *)display->data)->old.list[i].w, ((dirty_rectangles_data_t *)display->data)->old.list[i].h);
   }
   for (i = 0; i < ((dirty_rectangles_data_t *)display->data)->current.count; i++) {
      ((dirty_rectangles_data_t *)display->data)->old.list[i].x = ((dirty_rectangles_data_t *)display->data)->current.list[i].x;
      ((dirty_rectangles_data_t *)display->data)->old.list[i].y = ((dirty_rectangles_data_t *)display->data)->current.list[i].y;
      ((dirty_rectangles_data_t *)display->data)->old.list[i].w = ((dirty_rectangles_data_t *)display->data)->current.list[i].w;
      ((dirty_rectangles_data_t *)display->data)->old.list[i].h = ((dirty_rectangles_data_t *)display->data)->current.list[i].h;
   }
   ((dirty_rectangles_data_t *)display->data)->old.count = ((dirty_rectangles_data_t *)display->data)->current.count;

   if (display_flag_get(display, FLAG_SORT_DIRTY))
      qsort(((dirty_rectangles_data_t *)display->data)->old.list, ((dirty_rectangles_data_t *)display->data)->old.count, sizeof(dirty_t), _dirty_sorter);

   for (i = 0; i < ((dirty_rectangles_data_t *)display->data)->old.count; i++) {
      blit(((dirty_rectangles_data_t *)display->data)->buffer, screen, ((dirty_rectangles_data_t *)display->data)->old.list[i].x, ((dirty_rectangles_data_t *)display->data)->old.list[i].y, ((dirty_rectangles_data_t *)display->data)->old.list[i].x, ((dirty_rectangles_data_t *)display->data)->old.list[i].y, ((dirty_rectangles_data_t *)display->data)->old.list[i].w, ((dirty_rectangles_data_t *)display->data)->old.list[i].h);
   }
   release_bitmap(screen);
   ((dirty_rectangles_data_t *)display->data)->current.count = 0;
}

/* Mark Draw Region */
void dirty_rectangles_mark_draw_region(display_t *const display, const int x, const int y, const int w, const int h)
{
   ((dirty_rectangles_data_t *)display->data)->current.list[((dirty_rectangles_data_t *)display->data)->current.count].x = x;
   ((dirty_rectangles_data_t *)display->data)->current.list[((dirty_rectangles_data_t *)display->data)->current.count].y = y;
   ((dirty_rectangles_data_t *)display->data)->current.list[((dirty_rectangles_data_t *)display->data)->current.count].w = w;
   ((dirty_rectangles_data_t *)display->data)->current.list[((dirty_rectangles_data_t *)display->data)->current.count].h = h;
   ((dirty_rectangles_data_t *)display->data)->current.count++;
}

/* Get Bitmap */
BITMAP *dirty_rectangles_get_bitmap(display_t *const display)
{
   return ((dirty_rectangles_data_t *)display->data)->buffer;
}



/* Display Method Declarations ********************************************/
const display_method_t display_methods[] = {
 { "Direct to Screen",
   NULL,
   NULL,
   direct_to_screen_predraw,
   direct_to_screen_postdraw,
   NULL,
   NULL,
   direct_to_screen_get_bitmap
 },
 { "Double Buffering",
   double_buffering_construct,
   double_buffering_destruct,
   NULL,
   NULL,
   double_buffering_update,
   NULL,
   double_buffering_get_bitmap
 },
 { "Page Flipping",
   page_flipping_construct,
   page_flipping_destruct,
   page_flipping_predraw,
   page_flipping_postdraw,
   page_flipping_update,
   NULL,
   page_flipping_get_bitmap
 },
 { "Triple Buffering",
   triple_buffering_construct,
   triple_buffering_destruct,
   triple_buffering_predraw,
   triple_buffering_postdraw,
   triple_buffering_update,
   NULL,
   triple_buffering_get_bitmap
 },
 { "Dirty Rectangles",
   dirty_rectangles_construct,
   dirty_rectangles_destruct,
   NULL,
   NULL,
   dirty_rectangles_update,
   dirty_rectangles_mark_draw_region,
   dirty_rectangles_get_bitmap
 }
};



/* Display Functions ******************************************************/
void display_init(display_t *const display)
{
   display->method = METHOD_NONE;
   display->driver = -1;
   display->w = display->h = 0;
   display->bpp = -1;
   display->data = NULL;
   display->flags = 0;
}

int display_set(display_t *const display, const int method, const int driver, const int w, const int h, const int bpp)
{
   int old_method;
   int old_driver;
   int old_w, old_h;
   int old_bpp;
   void *old_data;
   int success = FALSE;

   old_method = display->method;
   old_driver = display->driver;
   old_w = display->w;
   old_h = display->h;
   old_bpp = display->bpp;
   old_data = display->data;

   display->method = method;
   display->driver = driver;
   display->w = w;
   display->h = h;
   display->bpp = bpp;

   set_color_depth(display->bpp);
   if (set_gfx_mode(display->driver, display->w, display->h, 0, 0) >= 0) {
      if (display_methods[display->method].construct != NULL) {
         if (display_methods[display->method].construct(display) == 0) {
            success = TRUE;
         }
      }
      else {
         success = TRUE;
      }
   }

   if (success == TRUE) {
      return 0;
   }

   display->method = old_method;
   display->driver = old_driver;
   display->w = old_w;
   display->h = old_h;
   display->bpp = old_bpp;
   display->data = old_data;

   return 1;
}

void display_clean(display_t *const display)
{
   if (display_methods[display->method].destruct != NULL)
      display_methods[display->method].destruct(display);
}

void display_predraw(display_t *const display)
{
   if (display_methods[display->method].predraw != NULL)
      display_methods[display->method].predraw(display);
}

void display_postdraw(display_t *const display)
{
   if (display_methods[display->method].postdraw != NULL)
      display_methods[display->method].postdraw(display);
}

void display_update(display_t *const display)
{
   if (display_methods[display->method].update != NULL)
      display_methods[display->method].update(display);
}

BITMAP *display_get_bitmap(display_t *const display)
{
   return display_methods[display->method].get_bitmap(display);
}

const char *display_get_name(display_t *const display)
{
   return display_methods[display->method].name;
}

void display_mark_draw_region(display_t *const display, const int x, const int y, const int w, const int h)
{
   if (display_methods[display->method].mark_draw_region != NULL)
      display_methods[display->method].mark_draw_region(display, x, y, w, h);
}

int display_flag_get(display_t *const display, const long flag)
{
   return (display->flags & flag);
}

void display_flag_set(display_t *const display, const long flag, const int state)
{
   switch (state) {
   case FLAG_TRUE:
      display_flag_on(display, flag);
      break;
   case FLAG_FALSE:
      display_flag_off(display, flag);
      break;
   case FLAG_TOGGLE:
      display_flag_toggle(display, flag);
      break;
   }
}

void display_flag_toggle(display_t *const display, const long flag)
{
   display->flags ^= flag;
}

void display_flag_on(display_t *const display, const long flag)
{
   display->flags |= flag;
}

void display_flag_off(display_t *const display, const long flag)
{
   display->flags |= flag;
   display->flags ^= flag;
}

