// link to: C:\DevCpp\Lib\liballeg.a
// or where ever your DevCpp Allegro library build file is located

// ALSO, use these compiler optimizations: -O -O3 -fexpensive-optimizations
// They're helpful for cranking out as much speed as possible in any graphic game
// Don't compile without'em!

/*
///////// Educational Jacks/Hoops/TargetPass Engine 0.4 ///////////
AUTHOR: Chris DeLeon
EDITED BY: [add your name here]
CREATED FOR: Game Creation Society
DEVELOPMENT TIME: One night
CREATED IN: Dev-C++ 4 with Allegro
  For help on getting this environment configured, see:
  http://www.allegro.cc/files/install-devcpp.html
LAST UPDATED: 6:00 pm Central Time, July 20, 2004
PURPOSE: Sample of basic game physics programming, displayed using Allegro
GAME:
  This is a basic ball bouncing engine, in which the player interacts
by using the mouse to control a simple pseudo-3D arm.  It's set up for
playing jacks, shooting hoops, and passing at a moving target.

Suggested modifications:

ENGINE ADVANCEMENTS: Important practice in the technical side
-Flexible gameplay variables
  Alter the game so that gameplay physics constants and locations are loaded
  from the settings.ini file, instead of using hard coded defines.  See jacks.h
  for the variables that affect gravity, ball size, bounce speed, and hoop height.
  Replace these with globally available integers (or package them all in one nice
  struct/class) and load them at run-time from their own section of the ini file.
  This is the first step in writing games that non-technical designers on the team
  can participate in developing, beyond a conceptual level.
-More realistic physics
  Perhaps when the hand brushes against jacks, or the ball bounces on them, the
  jacks should jump and roll.  Or maybe the ball should be able to bank off the rim.
  What are some ways this could be coded flexibly and efficiently?  Implement!

GAMEPLAY CHANGES: Important practice on the conceptual side
-Scoring goals (X number of jacks, hoops, and/or passing targets for Y amount of
  throws or bounces) and a level format, to give the game purpose and challenge.
  Begin by hard coding a level or two to see what variables distinguish one level
  from another, then challenge yourself to soft code and load these variables
  from a data file - either settings.ini or another level.ini file.  It would
  follow that once a level structure is in place, some minor reward screen/animation
  would be in order for completing the game.
-Simple save game feature.  A game of this nature really doesn't justify a save/load
  feature, but it's good practice and easy to do.  Just export all important
  variables to an ini file of choice, and then load them back on demand.
-Enemies
  Maybe there's a bouncing obstacle that blocks your basketball goals, or a more 
  complicated entity that runs around scrambling your jacks and jumps to swat your
  free throws.  A second player could even use the keyboard to attempt to interfere
  with the player's goals, wherein the players switch roles to keep matches fair.

BELLS AND WHISTLES: Important practice on the presentation
-Bitmapped graphics
  The game could use a background image, ground, basketball hoop, real jack images,
  and a bouncy ball.  Replacing the arm and hand with bitmapped graphics requires
  a little more creativity, but there are plenty of interesting ways to get it done.
-Flashy graphical effects
  Screen transitions, particle effects of dust and sparks, a ball that stretches
  when it bounces... these won't make or break a game, but they're nice.
-Sound effects
  Add sfx during events that score numbers, like swishing hoops and boinging bounces
-Title/Menu screen
  Allow the player to adjust the game's ini variables, game physics options, and
  gameplay modes through either a keyboard based text menu, or a mouse GUI.

ANYTHING ELSE: As important as anything I've listed
-Use your imagination.  I stake no claim on this code; make it into a game of your own!
*/

#include "jacks.h"
#include "allegengcommon.h"

double fastsqrt(double r) // fast squareroot approx. function, handy for distance finding
  {
  double x,y;
  double tempf;
  unsigned long *tfptr = ((unsigned long *)&tempf)+1;

  tempf = r;
  *tfptr=(0xbfcd4600-*tfptr)>>1;
  x=tempf;
  y=r*0.5;
  x*=1.5-x*x*y;  // I only need to estimate
  return x*r;
  }

inline float mag(float x,float y) // magnitude (distance) function using the fastsquareroot
  {
  return fastsqrt(x*x+y*y);
  }

int g_liveJacks=0; // one of the few gameplay globals used.  Just tracks the # of visible jacks.

class jack_typ // Jacks, as in, the little metal things you pick up between ball bounces.
  {
  public:
  float x, y, xv, yv; // x,y pos and x,y velocity.  xv/yv aren't used, but are ready to be implemented
  int Ix, Iy; // cached integer coords, for more rapid display.  Avoids float->int cast every frame
  jack_typ() // constructor
    {
    Ix=Iy=0;
    x=y=0.0;
    }
  void placeJack() // called to randomly place the jack at the bottom of the screen
    {
    x=(float)(JACK_PLACEMENT_MARGIN+(rand()%(SCREEN_W-(JACK_PLACEMENT_MARGIN<<1))));
    y=(float)(SCREEN_H-JACK_PLACEMENT_MARGIN-JACK_RADIUS);
    Ix=(int)x;
    Iy=(int)y;
    g_liveJacks++; // increment static jack tracker
    }
  void moveJack()
    {
    if(xv==0.0 && yv==0.0) // no updates.
      {
      ; // note that this if statement checks and disregards the usual case
        // besides being a minor optimization habit, sometimes this is a more
        // clear way of saying, "If it's sitting still... don't move it."
      }
    else
      {
      x+=xv; // move it based on it's current xv & yv
      y+=yv;

      if(y>SCREEN_H-JACK_PLACEMENT_MARGIN-JACK_RADIUS && yv>0.0) // bounce on ground
        {
        yv=yv*(-BALL_BOUNCE_CONSTANT); // reverse and dampen vertical component
        xv*=BALL_BOUNCE_CONSTANT_X_WHEN_VERT_BOUNCE; // apply friction
        }
      else // not on the ground, so it's in the air
        {
        yv+=GRAVITY_CONSTANT; // only apply gravity if it's not on the ground
        }

      if(x<JACK_PLACEMENT_MARGIN+JACK_RADIUS && xv<0.0) // bounce (left side)
        xv=xv*(-BALL_BOUNCE_CONSTANT);
      else if(x>SCREEN_W-JACK_PLACEMENT_MARGIN-JACK_RADIUS && xv>0.0) // bounce (right side)
        xv=xv*(-BALL_BOUNCE_CONSTANT);

      // zero velocity components if they're going slow enough
      if(xv<-STOP_MIN_V || xv>STOP_MIN_V) ;
      else xv=0.0;
      if(yv<-STOP_MIN_V || yv>STOP_MIN_V) ;
      else yv=0.0;

      Ix=(int)x; // update cached integers for display; it moved
      Iy=(int)y;
      }
    }
  void drawJack()
    { // displays a jack at it's cached coordinates.  A white circle with a black plus sign
    circlefill(DB,Ix,Iy,JACK_RADIUS,COLOR_WHITE);
    rectfill(DB,Ix-(JACK_RADIUS>>1),Iy-(JACK_RADIUS>>2),Ix+(JACK_RADIUS>>1),Iy+(JACK_RADIUS>>2),0);
    rectfill(DB,Ix-(JACK_RADIUS>>2),Iy-(JACK_RADIUS>>1),Ix+(JACK_RADIUS>>2),Iy+(JACK_RADIUS>>1),0);
    }
  jack_typ(jack_typ &copyJack) // copy constructor
    { // this is used when removing a jack, because they're kept in a linear array. see takeJack()
    x=copyJack.x;
    y=copyJack.y;
    Ix=copyJack.Ix;
    Iy=copyJack.Iy;
    }
  };

jack_typ jack[JACK_MAX]; // declaration of linear array of jacks

void placeJacks(int numToPlace) // places multiple jacks
  {
  g_liveJacks=0; // zero out the jack counter
  int happyPlacement; // flag used in making sure jacks don't overlap

  if(numToPlace>=JACK_MAX) // safety cap to avoid accessing outside the jack array
    numToPlace=JACK_MAX;

  for(int i=0;i<numToPlace;i++)
    {
    happyPlacement=0; // set it to unhappy so we enter the loop
    while(!happyPlacement) // continue trying to place the jack until it fits
      {
      jack[i].placeJack();
      happyPlacement=1;
      for(int ii=0;ii<i;ii++) // make sure it's not too close to the others
        {
        if((jack[i].x-jack[ii].x)<MIN_JACK_DIST && (jack[i].x-jack[ii].x)>-MIN_JACK_DIST)
          { // it's too close
          happyPlacement=0; // mark it as an unhappy solution spot
          g_liveJacks--; // un-tally that one
          break; // stop checking for space, we know it didn't fit
          }
        }
      }
    }
  };

void takeJack(int indexToTake) // removes a jack from the screen
  {
  g_liveJacks--; // cut one from the counter
  // switch it's properties with the jack that's off the stack
  if(g_liveJacks>0 && indexToTake<g_liveJacks)
    jack[indexToTake]=jack[g_liveJacks];
  };

class hand_typ
  {
  public:
  float x, y; // position of hand
  int mickey_x,mickey_y; // how much has the mouse moved in x,y pixels?
  int gripOpen; // flag for if the hand is open (nonzero) or closed (0)
  int jackGrabs; // keeps score for jacks grabbed up

  hand_typ() // constructor
    {
    mickey_x=mickey_y=0;
    x=y=0;
    gripOpen=0;
    jackGrabs=0;
    }
  void placeHand() // called when program begins
    {
    mickey_x=mickey_y=0;
    x=mouse_x;
    y=mouse_y;
    gripOpen=0;
    }
  void moveHand() // called every frame
    {
    get_mouse_mickeys(&mickey_x,&mickey_y); // they build up if not taken
    x+=mickey_x; // update the hand's position
    y+=mickey_y;

    // don't let the hand go offscreen
    if(x>(SCREEN_W-HAND_WIDTH_HALF))
      x=SCREEN_W-HAND_WIDTH_HALF;
    else if(x<HAND_WIDTH_HALF)
      x=HAND_WIDTH_HALF;
    if(y>(SCREEN_H-HAND_HEIGHT_HALF))
      y=SCREEN_H-HAND_HEIGHT_HALF;
    else if(y<HAND_HEIGHT_HALF)
      y=HAND_HEIGHT_HALF;

    // don't let the hand overlap the shoulder joint
    if(y>=HAND_HEIGHT) // which way was it coming in from? Side or bottom?
      {
      if(x>(SCREEN_W-HAND_WIDTH-HAND_WIDTH_HALF))
        if(y<HAND_HEIGHT+HAND_HEIGHT_HALF)
          y=HAND_HEIGHT+HAND_HEIGHT_HALF;
      }
    else
      {
      if(y<HAND_HEIGHT+HAND_HEIGHT_HALF)
        if(x>(SCREEN_W-HAND_WIDTH-HAND_WIDTH_HALF))
          x=SCREEN_W-HAND_WIDTH-HAND_WIDTH_HALF;
      }
    }
  void drawHand()
    {
    int Ix=(int)x; // cache integer equiv to position
    int Iy=(int)y;
    int Ex=SCREEN_W-((SCREEN_W-Ix)>>1); // elbow location
    int Ey=(SCREEN_H-((SCREEN_H-Iy)>>1))>>1;
    int elbowScaled_w,elbowScaled_h; // shrink it as it moves into background

    float scale=mag((float)(SCREEN_W-Ix),y)/mag((float)SCREEN_W,(float)SCREEN_H);
    scale=(scale+2.0)/3.0;
    elbowScaled_w=(int)(HAND_WIDTH_HALF*scale);
    elbowScaled_h=(int)(HAND_HEIGHT_HALF*scale); // elbow bends back, shrinking

    // shoulder
    rectfill(DB,SCREEN_W-HAND_WIDTH,0,SCREEN_W,HAND_HEIGHT,COLOR_WHITE);

    // bicep (shoulder to elbow):
    line(DB,SCREEN_W,HAND_HEIGHT,Ex+elbowScaled_w,Ey+elbowScaled_h,COLOR_WHITE);
    line(DB,SCREEN_W-HAND_WIDTH,HAND_HEIGHT,Ex-elbowScaled_w,Ey+elbowScaled_h,COLOR_WHITE);
    line(DB,SCREEN_W,0,Ex+elbowScaled_w,Ey-elbowScaled_h,COLOR_WHITE);
    line(DB,SCREEN_W-HAND_WIDTH,0,Ex-elbowScaled_w,Ey-elbowScaled_h,COLOR_WHITE);

    // elbow
    rect(DB,Ex-elbowScaled_w,Ey-elbowScaled_h,Ex+elbowScaled_w,Ey+elbowScaled_h,COLOR_WHITE);

    // forearm (elbow to hand):
    line(DB,Ex+elbowScaled_w,Ey+elbowScaled_h,Ix+HAND_WIDTH_HALF,Iy+HAND_HEIGHT_HALF,COLOR_WHITE);
    line(DB,Ex-elbowScaled_w,Ey+elbowScaled_h,Ix-HAND_WIDTH_HALF,Iy+HAND_HEIGHT_HALF,COLOR_WHITE);
    line(DB,Ex+elbowScaled_w,Ey-elbowScaled_h,Ix+HAND_WIDTH_HALF,Iy-HAND_HEIGHT_HALF,COLOR_WHITE);
    line(DB,Ex-elbowScaled_w,Ey-elbowScaled_h,Ix-HAND_WIDTH_HALF,Iy-HAND_HEIGHT_HALF,COLOR_WHITE);

    // hand (draw filled in if hand is closed, empty if open)
    if(gripOpen)
      rect(DB,Ix-HAND_WIDTH_HALF,Iy-HAND_HEIGHT_HALF,Ix+HAND_WIDTH_HALF,Iy+HAND_HEIGHT_HALF,COLOR_WHITE);
    else
      rectfill(DB,Ix-HAND_WIDTH_HALF,Iy-HAND_HEIGHT_HALF,Ix+HAND_WIDTH_HALF,Iy+HAND_HEIGHT_HALF,COLOR_WHITE);
    }
  void tryToGrabJacks() // check hand against jack positions to pick them up
    {
    float distance;

    for(int i=0;i<g_liveJacks;i++) // check against all jacks
      {
      distance=mag(x-jack[i].x,y-jack[i].y);
      if(distance<JACK_RADIUS) // if it's close enough
        {
        takeJack(i); // remove it
        jackGrabs++; // and count it
        }
      }
    }
  };

hand_typ hand; // hand declaration

class ball_typ // the player's ball
  {
  public:
  float x,y, xv,yv;  // x,y position, x & y velocities
  float cumulative_float_mickey_x,cumulative_float_mickey_y;
    // built up mouse moves
  int inHand; // flag - is the ball in the player's hand?
  int bounces, releases; // keeps track of bounces and tosses

  ball_typ() // constructor
    {
    x=y=xv=yv=0;
    cumulative_float_mickey_x=cumulative_float_mickey_y=0;
    inHand=1;
    bounces=releases=0;
    }
  void placeBall() // reset ball, called at program startup
    {
    x=hand.x;
    y=hand.y;
    cumulative_float_mickey_x=cumulative_float_mickey_y=0;
    inHand=1;
    }
  void throwBall() // ball gets thrown
    {
    inHand=0; // the ball is no longer in the hand
    releases++; // keep a throw tally
    xv=cumulative_float_mickey_x*BALL_THROW_DAMPEN;
    yv=cumulative_float_mickey_y*BALL_THROW_DAMPEN;
    }
  void landInHand() // call when the player's hand closes, to try & grab it
    {
    if(!inHand)
      {
      if(mag(x-hand.x,y-hand.y)<BALL_RADIUS) // if it's close enough to the hand
        inHand=1; // take it
      }
    }
  void moveBall() // called every frame
    {
    // mickey's (mouse position deltas) are tracked here as cumulative floats
    // and they decay constantly so that the player can "throw" the ball by
    // swinging the arm and releasing the ball
    cumulative_float_mickey_x*=CUMULATIVE_MICKEY_DECAY;
    cumulative_float_mickey_y*=CUMULATIVE_MICKEY_DECAY;
    cumulative_float_mickey_x+=(float)hand.mickey_x; // taken every frame
    cumulative_float_mickey_y+=(float)hand.mickey_y;

    // limit the throwing power, to prevent unrealistic speeds
    if(cumulative_float_mickey_x<-MAX_CUMULATIVE_MICKEY)
      cumulative_float_mickey_x=-MAX_CUMULATIVE_MICKEY;
    else if(cumulative_float_mickey_x>MAX_CUMULATIVE_MICKEY)
      cumulative_float_mickey_x=MAX_CUMULATIVE_MICKEY;
    if(cumulative_float_mickey_y<-MAX_CUMULATIVE_MICKEY)
      cumulative_float_mickey_y=-MAX_CUMULATIVE_MICKEY;
    else if(cumulative_float_mickey_y>MAX_CUMULATIVE_MICKEY)
      cumulative_float_mickey_y=MAX_CUMULATIVE_MICKEY;

    if(inHand) // if it's in the player's hand, make it go where the hand goes
      {
      x=hand.x;
      y=hand.y;
      }
    else // otherwise, handle the ball's physics
      {
      x+=xv; // add the velocities to the position variables
      y+=yv;

      if(y>SCREEN_H-JACK_PLACEMENT_MARGIN-BALL_RADIUS) // bounce on ground
        {
        if(yv>0.0) // if it's still moving downward, bounce it
          yv=yv*(-BALL_BOUNCE_CONSTANT);

        // friction
        xv*=BALL_BOUNCE_CONSTANT_X_WHEN_VERT_BOUNCE;

        // zero velocity vectors if it's moving slow enough
        if(xv<-STOP_MIN_V || xv>STOP_MIN_V) ;
        else xv=0.0;
        if(yv<-STOP_MIN_V || yv>STOP_MIN_V) bounces++; // only count if moving
        else yv=0.0;
        }
      else // not bouncing off the ground, so it's in the air
        {
        yv+=GRAVITY_CONSTANT; // only apply gravity if it's not on the ground

        /*if(y<JACK_PLACEMENT_MARGIN+BALL_RADIUS && yv<0.0) // bounce
          yv=yv*(-BALL_BOUNCE_CONSTANT); don't put a ceiling on the ball*/
        }

      if(x<JACK_PLACEMENT_MARGIN+BALL_RADIUS && xv<0.0) // bounce (left side)
        xv=xv*(-BALL_BOUNCE_CONSTANT);
      else if(x>SCREEN_W-JACK_PLACEMENT_MARGIN-BALL_RADIUS && xv>0.0) // bounce (right side)
        xv=xv*(-BALL_BOUNCE_CONSTANT);
      }
    }
  void drawBall()
    { // simple circle
    int Ix=(int)x;
    int Iy=(int)y;

    circlefill(DB,Ix,Iy,BALL_RADIUS,COLOR_WHITE);
    }
  };

ball_typ ball; // ball declaration

class hoop_typ // basketball hoop
  {
  public:
  float x,y; // position of hoop
  int Ix, Iy; // cached integers, handy since x,y usually won't change
  int netStretch, activated; // netStretch is for swishing effect
                  // activated is flagged when player is far enough away to shoot
  int score; // number of shots made

  hoop_typ() // constructor
    {
    Ix=Iy=0;
    x=y=0;
    netStretch=0;
    score=0;
    }
  void placeHoop() // called when program starts
    {
    x=HOOP_X;
    y=HOOP_Y;
    Ix=(int)x;
    Iy=(int)y;
    netStretch=0;
    activated=0;
    }
  void moveHoop() // called every frame
    {
    if(netStretch>0) netStretch--; // if the net is stretched, tuck it back

    if(ball.inHand) // only update if ball is in hand
      {
      if(hand.x>HOOP_RANGE_X) // only open the goal if they're too far to drop
        activated=1;
      else activated=0; // they're trying to drop it in for an easy shot
      }

    int xDist,yDist; // get integer values for the distance from ball to hoop
    xDist=(int)(ball.x-x);
    if(xDist<0)xDist=-xDist; // geometric distances; positive only
    yDist=(int)(ball.y-y);
    if(yDist<0)yDist=-yDist;

    // If the rim is open (activated), the ball is close enough, and a shot
    // hasn't just been counted in the last frame (!netStretch) then we count it
    if(activated && xDist<HOOP_RADIUS+BALL_RADIUS && yDist<HOOP_HEIGHT && !netStretch)
      { // considering checking if shot hit rim here, or if it's a smooth shot
      netStretch=HOOP_SWOOSH_STRETCH; // move the net to indicate a good shot
      score++; // add one point
      }
    }
  void drawHoop()
    {
    // pole
    rectfill(DB,JACK_PLACEMENT_MARGIN-HOOP_BACKBOARD_DRAW_WIDTH-(HOOP_BACKBOARD_DRAW_WIDTH>>1),Iy-5,JACK_PLACEMENT_MARGIN-HOOP_BACKBOARD_DRAW_WIDTH,SCREEN_H-JACK_PLACEMENT_MARGIN,COLOR_WHITE);
    
    // backboard
    rectfill(DB,JACK_PLACEMENT_MARGIN-HOOP_BACKBOARD_DRAW_WIDTH,Iy-HOOP_BACKBOARD_DRAW_HEIGHT,JACK_PLACEMENT_MARGIN,Iy,COLOR_WHITE);
    rectfill(DB,JACK_PLACEMENT_MARGIN-HOOP_BACKBOARD_DRAW_WIDTH,Iy-HOOP_BACKBOARD_DRAW_HEIGHT,JACK_PLACEMENT_MARGIN,Iy,COLOR_WHITE);
    // draw net:
    line(DB,Ix-HOOP_RADIUS,Iy,Ix-HOOP_RADIUS+HOOP_NET_PULLIN-netStretch,Iy+HOOP_NET_LENGTH+netStretch,COLOR_WHITE);
    line(DB,Ix+HOOP_RADIUS,Iy,Ix+HOOP_RADIUS-HOOP_NET_PULLIN+netStretch,Iy+HOOP_NET_LENGTH+netStretch,COLOR_WHITE);

    if(activated) // open rim
      ellipse(DB,Ix,Iy,HOOP_RADIUS,HOOP_HEIGHT,COLOR_WHITE);
    else // too close, no goal.  Close it up.
      ellipsefill(DB,Ix,Iy,HOOP_RADIUS,HOOP_HEIGHT,COLOR_WHITE);
    }
  };

hoop_typ hoop; // hoop declaration

// Football passing target.  Just a sideways hoop that slides up and down the screen.
class passTarget_typ
  {
  public:
  float x,y; // location of moving target
  int Ix, Iy; // cached integers for position
  int netStretch, activated; // netStretch is for swishing
                    // activated is flagged when player is far enough to shoot
  int goingUp; // directional flag for the moving target
  int score; // how many points have been made on this target?

  passTarget_typ() // constructor
    {
    Ix=Iy=0;
    x=y=0;
    netStretch=0;
    goingUp=1;
    score=0;
    }
  void placePassTarget() // called at program startup
    {
    x=PASSTARGET_X;
    y=PASSTARGET_Y;
    Ix=(int)x;
    Iy=(int)y;
    netStretch=0;
    activated=0;
    goingUp=rand()&1;
    }
  void movePassTarget() // called every frame
    {
    if(netStretch>0) netStretch--; // if the net is stretched, tuck it in

    if(ball.inHand) // only update if ball is in hand
      {
      if(hand.x<(SCREEN_W-HOOP_RANGE_X)) // open the goal
        activated=1;
      else activated=0; // too close; they're trying an easy shot/drop
      }

    if(activated) // if the goal is open/active
      {
      if(goingUp) // the goal is one the rise
        {
        y--;
        if(y<JACK_PLACEMENT_MARGIN+HOOP_RADIUS) // at the top, change direction
          goingUp=0;
        }
      else // !goingUp, i.e. NOT going up.  Going down.
        {
        y++;
        if(y>SCREEN_H-JACK_PLACEMENT_MARGIN-HOOP_RADIUS)
          goingUp=1; // at the bottom, change direction
        }
      Iy=(int)y; // update the cached integer y for display
      }

    int xDist,yDist; // distances are geometric; positive only
    xDist=(int)(ball.x-x);
    if(xDist<0)xDist=-xDist;
    yDist=(int)(ball.y-y);
    if(yDist<0)yDist=-yDist;

    // !netStretch ensures that the same toss won't be counted multiple times if it's moving slowly
    // target must be open (activated) and the ball must be close enough
    if(activated && xDist<HOOP_HEIGHT && yDist<HOOP_RADIUS+BALL_RADIUS && !netStretch)
      { // considering checking if shot hit rim here, or if it's in the middle
      netStretch=HOOP_SWOOSH_STRETCH;
      score++;
      }
    }
  void drawPassTarget() // draw the passing target
    {
    // draw net:
    line(DB,Ix,Iy-HOOP_RADIUS,Ix+HOOP_NET_LENGTH+netStretch,Iy-HOOP_RADIUS+HOOP_NET_PULLIN-netStretch,COLOR_WHITE);
    line(DB,Ix,Iy+HOOP_RADIUS,Ix+HOOP_NET_LENGTH+netStretch,Iy+HOOP_RADIUS-HOOP_NET_PULLIN+netStretch,COLOR_WHITE);

    if(activated) // draw open goal target
      ellipse(DB,Ix,Iy,HOOP_HEIGHT,HOOP_RADIUS,COLOR_WHITE);
    else // too close, no goal.  Close it up.
      ellipsefill(DB,Ix,Iy,HOOP_HEIGHT,HOOP_RADIUS,COLOR_WHITE);
    }
  };

passTarget_typ passTarget; // pass target declared

void MoveAll() // called every frame
  {
  // gameplay mouse inputs
  if(hand.gripOpen && !LMB) // hand WAS open, just closed
    {
    if(!ball.inHand)
      {
      ball.landInHand();
      hand.tryToGrabJacks();
      }
    }
  else if(!hand.gripOpen && LMB) // hand WAS closed, just opened
    {
    if(ball.inHand)
      ball.throwBall();
    }
  hand.gripOpen=LMB; // trapped this variable lets us remember the last frame
   // without this we would know the state of the mouse, but not when it changed

  hoop.moveHoop();
  passTarget.movePassTarget();
  hand.moveHand();
  ball.moveBall();
  for(int i=0;i<g_liveJacks;i++)
    jack[i].moveJack();
  }

void DrawAll() // called every frame
  {
  clear(DB); // clears entire screen every frame.  This isn't the fastest for
             // a game like this with limited graphical implementation, but it's
             // fast enough.  Also note that a full screen bitmapped game would
             // be refreshing the screen every frame with the background image.

  //ground
  rectfill(DB,0,SCREEN_H-JACK_PLACEMENT_MARGIN,SCREEN_W,SCREEN_H,COLOR_WHITE);

  hand.drawHand();
  hoop.drawHoop();
  passTarget.drawPassTarget();
  ball.drawBall();
  for(int i=0;i<g_liveJacks;i++)
    jack[i].drawJack();

  // display tracked score values
  textprintf(DB, font, 0, 2, COLOR_WHITE,
                "Hoops:%i", hoop.score);
  textprintf(DB, font, 0, 12, COLOR_WHITE,
                "Throws:%i", ball.releases);
  textprintf_centre(DB, font, SCREEN_W>>1, 2, COLOR_WHITE,
                "Jacks:%i", hand.jackGrabs);
  textprintf_centre(DB, font, SCREEN_W>>1, 12, COLOR_WHITE,
                "Bounces:%i", ball.bounces);
  textprintf_right(DB, font, SCREEN_W-HAND_WIDTH, 2, COLOR_WHITE,
                "Passes:%i", passTarget.score);

  DB2SCREEN // macro from the jacks.h: blit(DB,screen,0,0,0,0,screenW,screenH);
         // that's a full screen copy from the DB bitmap memory to screen
         // topleft corner (0,0) to topleft corner, full screen's width & height
  }

////////////////////////////////////////////////////////////////////
int main(int argc, char *argv[])////////////////////////////////////
  {
  int inprogram=1, gameon=1, pause=0;

   // open if possible, or else close all pieces; from allegengcommon.cpp
  if(InitAll()) {CloseAll(); return(1);}
  
  while(inprogram) // the program is still on
    {
    inprogram=MenuScreens(); // menuscreen enters menus and returns an int
     // MenuScreens is found in allegengcommong.cpp

    if(inprogram)
      {
      gameon=1;
      // initialize game positions & states:
      hand.placeHand();
      hoop.placeHoop();
      passTarget.placePassTarget();
      ball.placeBall();
      placeJacks(10);
      }
    else gameon=0; // if the player quit from the menu, skip playing the game

    while(gameon) // the game is still on
      {
      // key inputs:
      if(key[KEY_ESC]) // quit
        {
        while(key[KEY_ESC]) ; // freeze while player holds the button
        gameon=0;
        }
      if(key[KEY_F11]) // screenshot
        {
        while(key[KEY_F11]) ; // freeze while player holds the button
        ScreenShot(); // in allegengcommon.cpp
        }
      if(key[KEY_P]) // pause
        {
        while (key[KEY_P]) ; // freeze while player holds the button

        if(pause) // flip states
          pause=0;
        else
          pause=1;
        }
      if(key[KEY_R]) // reset game
        {
        // zero all scores
        hoop.score=hand.jackGrabs=ball.bounces=ball.releases=passTarget.score=0;
        ball.inHand=1; // put the ball in the player's hand
        hand.gripOpen=0; // close the player's hand
        // note that a curious bug arises from this implementation:
        // Hold R while moving the mouse and holding the left mouse button
        // Magic floating ball!  Don't let this type of bug get overlooked in YOUR
        // games.  Always try combinations, moves, and positions that no sane
        // gamer would ever try, or you'll have a program ridden with imperfections
        // like this.
        }
      ////

      if(g_liveJacks==0) // reset jacks if they've all been picked up:
        placeJacks(10);

      if(!pause) MoveAll(); // only update positions if not paused

      DrawAll(); // render the frame

      rest(PROG_REST); // rest between frames. PROG_REST set in settings.ini
      } // close gameon
    } // close inprogram
  CloseAll(); // close all pieces; from allegengcommon.cpp
  return(0); // no errors
  } // close main
END_OF_MAIN();
