/************************************
 *
 * Kickle 0.81
 * 2004 Drew Willcoxon
 * http://www.cs.uga.edu/~adw/
 * dripfeed@uga.edu
 *
 ************************************/


#include <stdlib.h>
#include <stdio.h>  // printf
#include <string.h>

#include "baddie.h"
#include "entity.h"
#include "list.h"
#include "log.h"
#include "manager.h"
#include "map.h"
#include "projctl.h"
#include "video.h"




/* the game's log file */
//extern type_log *manager.log;

extern Manager manager;




Projectile *construct_projectile(const projectile_type type,
                                 const unsigned char row,
                                 const unsigned char col,
                                 const signed char heading_x,
                                 const signed char heading_y,
                                 Map *const map)
{

    Projectile *projectile=
        (Projectile *) malloc(sizeof(Projectile));

    if (projectile == 0)
    {
        write_to_log(manager.log, 1, "Could not allocate projectile memory.\n");
        return 0;
    }

    /* set the breaths pointer ahead of anything else, in case we have to
       destroy the projectile we're making */
    projectile->player_num_breaths_ptr= 0;

    projectile->entity= construct_entity();
    if (projectile->entity == 0)
    {
        write_to_log(manager.log, 1,
                     "Could not construct projectile's entity.\n");
        destruct_projectile(projectile);
        return 0;
    }

    /* set entity members */
    projectile->entity->heading_x= heading_x;
    projectile->entity->heading_y= heading_y;

    /* set projectile members */
    projectile->type= type;
    projectile->status= PROJECTILE_S_MOVING;

    switch (type)
    {

        /* other values of an ice breath should be set outside this function
           on creation of the breath */
        case PROJECTILE_T_ICE_BREATH:
            projectile->status= PROJECTILE_S_MOVING;
            break;

    }

    /* animate projectile */
    animate_projectile(projectile);

    /* add projectile to map */
    if (map != 0)
    {
        set_bb_size(projectile->entity, map);
        if (add_projectile_to_map(projectile, map, row, col) > 0)
        {
            write_to_log(manager.log, 1, "Could not add projectile to map.\n");
            destruct_projectile(projectile);
            return 0;
        }
    }

    return projectile;

} /* end construct projectile */




void destruct_projectile(Projectile *projectile)
{

    /* if projectile is an ice breath, decrease the number of the player's
       current ice breaths */
    if (projectile->type == PROJECTILE_T_ICE_BREATH
        && projectile->player_num_breaths_ptr != 0)
    {
        --(*(projectile->player_num_breaths_ptr));
    }
    if (projectile->entity != 0) destruct_entity(projectile->entity);
    free(projectile);
    projectile= 0;
}




void do_projectile_logic(Projectile *const projectile, Map *const map,
                         const signed short target_row,
                         const signed short target_col)
{

    //type_baddie *coll_baddie;
    //type_feature *coll_feature;
    //type_tile *coll_tile;
    //unsigned short i;

    if (projectile->entity->dx == 0 && projectile->entity->dy == 0)
    {
        do_projectile_square_logic(projectile, map, target_row,
                                   target_col);
    }

    /* collision detection */
    if (projectile->entity->c_speed > 0)
        find_projectile_to_tile(projectile, map);
    if (projectile->entity->c_speed > 0)
        find_projectile_to_feature(projectile, map);
    if (projectile->entity->c_speed > 0)
        find_projectile_to_baddie(projectile, map);



    #if 0
    if (projectile->entity->c_speed > 0)
    {

        /* find projectile-to-baddie collision */
        coll_baddie= 0;
        for (i= 0; i < map->max_num_baddies && coll_baddie == 0; i++)
        {
            if (map->baddies[i] != 0
                && will_entities_collide(projectile->entity,
                                        map->baddies[i]->entity, map,
                                        COLLISION_FUDGE_X,
                                        COLLISION_FUDGE_Y))
            {
                coll_baddie= map->baddies[i];
            }
        }

        /* find projectile-to-feature collision */
        coll_feature= 0;
        for (i= 0; i < map->max_num_features; i++)
        {
            if (map->features[i] != 0
                && will_entities_collide(projectile->entity,
                                              map->features[i]->entity, map,
                                              COLLISION_FUDGE_X,
                                              COLLISION_FUDGE_Y))
            {
                coll_feature= map->features[i];
            }
        }

        /* find projectile-to-tile collision */
        coll_tile= 0;
        if (coll_baddie == 0)
            coll_tile= projectile_coll_tile(projectile, map);

        /* collision response logic */
        if (coll_baddie != 0)
                do_projectile_to_baddie_coll(projectile, map, coll_baddie);
        if (coll_feature != 0)
                do_projectile_to_feature_coll(projectile, map, coll_feature);
        if (coll_tile != 0)
            do_projectile_to_tile_coll(projectile, map, coll_tile);

    }
    #endif

    /* if projectile is allowed to step, step it */
    if (projectile->entity->c_speed > 0)
        step_entity(projectile->entity, projectile->entity->c_speed, map);

} /* end do_projectile_logic */




void do_projectile_square_logic(Projectile *const projectile, Map *const map,
                                const signed short target_row,
                                const signed short target_col)
{

    if (projectile->entity->logic_counter >= projectile->life)
        projectile->status= PROJECTILE_S_KAPUT;

    ++projectile->entity->logic_counter;

}




void do_projectile_to_baddie(Projectile *const projectile, const Map *const map,
                             Baddie *const baddie)
{

    //printf("projectile-baddie collision\n");

    /* update projectile's and baddie's logic based on the type of projectile */
    switch (projectile->type)
    {

        /* if projectile is an ice breath */
        case PROJECTILE_T_ICE_BREATH:

            /* set baddie status depending on baddie type */
            switch (baddie->type)
            {

                /* baddie5 */
                case BADDIE_T_BADDIE5:
                    /* do nothing */
                    break;

                /* default, freeze baddie */
                default:

                    if (baddie->status == BADDIE_S_MOVING
                        || baddie->status == BADDIE_S_STOPPED
                        || baddie->status == BADDIE_S_FROZEN
                        || baddie->status == BADDIE_S_MELTING)
                    {

                        set_baddie_status(baddie, BADDIE_S_FROZEN, map);

                        /* make sure baddie ends up on the same row/column as
                           the breath */
                        if (projectile->entity->heading_y != 0
                            && baddie->entity->col != projectile->entity->col)
                        {
                            square_entity(baddie->entity, baddie->entity->row,
                                          projectile->entity->col, map);
                        }

                        if (projectile->entity->heading_x != 0
                            && baddie->entity->row != projectile->entity->row)
                        {
                            square_entity(baddie->entity,
                                          projectile->entity->row,
                                          baddie->entity->col, map);
                        }

                    }
                    break;

            } /* end switch on baddie type */

        break; /* end if projectile is an ice breath */

    } /* end switch on projectile type */

    /* kill the projectile */
    projectile->status= PROJECTILE_S_KAPUT;

} /* end do_projectile_to_baddie */




void do_projectile_to_feature(Projectile *const projectile,
                              const Map *const map, Feature *const feature)
{

    switch (feature->type)
    {

        case FEATURE_T_PILLAR:
        case FEATURE_T_BUMPER:
        case FEATURE_T_HIVE:
        case FEATURE_T_POW:
            projectile->status= PROJECTILE_S_KAPUT;
            break;

        default:
            break;

    }

}




void do_projectile_to_tile(Projectile *const projectile, const Map *const map,
                           const unsigned char row, const unsigned char col)
{


}




void find_projectile_to_baddie(Projectile *const projectile,
                               const Map *const map)
{

    Baddie *coll_baddie;

    //unsigned short i;

    List_iterator *iterator;

    if (projectile->status == PROJECTILE_S_KAPUT) return;

    /* loop while there's more baddies and projectile wants to move */
    coll_baddie= 0;
    #if 0
    for (i= 0; i < map->max_num_baddies && coll_baddie == 0
         && projectile->entity->c_speed > 0; i++)
    {
        if (map->baddies[i] != 0
            && will_entities_collide(projectile->entity,
                                     map->baddies[i]->entity,
                                     map, COLL_FUDGE_X, COLL_FUDGE_Y))
        {
            coll_baddie= map->baddies[i];
            //do_projectile_to_baddie(projectile, map, map->baddies[i]);
        }
    }
    #endif

    iterator= get_iterator(map->baddie_list);
    while (iterator != 0 && coll_baddie == 0 && projectile->entity->c_speed > 0)
    {

        coll_baddie= (Baddie *) iterator->dat;

        if (will_entities_collide(projectile->entity, coll_baddie->entity,
                                  map, COLL_FUDGE_X, COLL_FUDGE_Y) == 0
            || projectile_baddie_coll(projectile->type, projectile->status,
                                      coll_baddie->type, coll_baddie->status)
                                      == 0)
        {
            coll_baddie= 0;
            //do_projectile_to_baddie(projectile, map, map->baddies[i]);
        }

        iterator= next_iterator(iterator);

    }

    if (coll_baddie != 0) do_projectile_to_baddie(projectile, map, coll_baddie);

} /* end find_projectile_to_tile */




void find_projectile_to_feature(Projectile *const projectile,
                                const Map *const map)
{

    //unsigned short i;

    Feature *feature;
    List_iterator *iterator;

    if (projectile->status == PROJECTILE_S_KAPUT) return;

    /* loop while there's more features and projectile wants to move */
    #if 0
    for (i= 0; i < map->max_num_features
         && projectile->entity->c_speed > 0; i++)
    {
        if (map->features[i] != 0
            && will_entities_collide(projectile->entity,
                                     map->features[i]->entity,
                                     map, 1, 1))
        {
            do_projectile_to_feature(projectile, map, map->features[i]);
        }
    }
    #endif

    iterator= get_iterator(map->feature_list);
    while (iterator != 0 && projectile->entity->c_speed > 0)
    {

        feature= (Feature *) iterator->dat;

        if (will_entities_collide(projectile->entity, feature->entity, map, 1,
                                  1)
            && projectile_feature_coll(projectile->type, projectile->status,
                                       feature->type, feature->status))
        {
            do_projectile_to_feature(projectile, map, feature);
        }

        iterator= next_iterator(iterator);

    }

} /* end find_projectile_to_feature */




void find_projectile_to_tile(Projectile *const projectile,
                             const Map *const map)
{

    unsigned char row;
    unsigned char col;

    if (projectile->status == PROJECTILE_S_KAPUT) return;

    if (projectile->entity->dx > 0 || projectile->entity->dy > 0) return;

    if (projectile->entity->heading_y != 0)
    {

        row= adj_row(projectile->entity);

        for (col= projectile->entity->col;
             col < projectile->entity->col + projectile->entity->bb_t_width
             && projectile->entity->c_speed > 0; col++)
        {
            if (row < map->m_height && col < map->m_width
                && projectile_tile_coll(projectile->type, projectile->status,
                                        map->tiles[row][col].type))
            {
                do_projectile_to_tile(projectile, map, row, col);
            }
        }

    }

    if (projectile->entity->heading_x != 0)
    {

        col= adj_col(projectile->entity);

        for (row= projectile->entity->row;
             row < projectile->entity->row + projectile->entity->bb_t_height
             && projectile->entity->c_speed > 0; row++)
        {
            if (row < map->m_height && col < map->m_width
                && projectile_tile_coll(projectile->type, projectile->status,
                                        map->tiles[row][col].type))
            {
                do_projectile_to_tile(projectile, map, row, col);
            }
        }

    }

} /* end find_projectile_to_tile */




void projectile_anim_done(Projectile *const projectile)
{

}




unsigned char projectile_baddie_coll(const projectile_type p_type,
                                     const projectile_status p_status,
                                     const baddie_type b_type,
                                     const baddie_status b_status)
{

    switch (p_type)
    {
        case PROJECTILE_T_ICE_BREATH:
            return 1;
            break;
    }

    /* no collision */
    return 0;

}




unsigned char projectile_feature_coll(const projectile_type p_type,
                                      const projectile_status p_status,
                                      const feature_type f_type,
                                      const feature_status f_status)
{

    switch (f_type)
    {

        case FEATURE_T_BLACK_ICE:
        case FEATURE_T_HOLE:
            return 0;
            break;

        default:
            return 1;
            break;

    }

    /* collision */
    return 1;

}




unsigned char projectile_tile_coll(const projectile_type p_type,
                                   const projectile_status p_status,
                                   const tile_type t_type)
{

    /* no collision */
    return 0;

}
