/************************************
 *
 * Kickle 0.81
 * 2004 Drew Willcoxon
 * http://www.cs.uga.edu/~adw/
 * dripfeed@uga.edu
 *
 ************************************/


#include <stdlib.h>
#include <string.h>

#include "baddie.h"
//#include "config.h"
//#include "common.h"
//#include "controls.h"
#include "entity.h"
#include "feature.h"
#include "goodie.h"
#include "list.h"
#include "log.h"
#include "player.h"
#include "projctl.h"
#include "manager.h"
#include "map.h"
#include "video.h"




/* the game's log file */
//extern type_log *manager.log;


/* the game's configurations */
//extern type_config kickle_config;

extern Manager manager;


Player *clone_player(const Player *const player, Map *const map,
                     const unsigned char row, const unsigned char col)
{

    Player *clone=
        construct_player(row, col, player->entity->heading_x,
                         player->entity->heading_y, player->status,
                         map);

    clone->num_lives= player->num_lives;

    if (clone == 0)
    {
        write_to_log(manager.log, 1, "Could not construct player clone.\n");
        return 0;
    }

    return clone;

} /* end clone_player */




Player *construct_player(const unsigned char row, const unsigned char col,
                         const signed char heading_x,
                         const signed char heading_y,
                         const player_status status, Map *const map)
{

    /* allocate the player and his entity */
    Player *player= (Player *) malloc(sizeof(Player));
    if (player == 0)
    {
        write_to_log(manager.log, 1, "Could not allocate player memory.\n");
        return 0;
    }

    player->entity= construct_entity();
    if (player->entity == 0)
    {
        write_to_log(manager.log, 1, "Could not construct player's entity.\n");
        destruct_player(player);
        return 0;
    }

    /* set entity members */
    player->entity->n_speed= 2;
    player->entity->heading_x= heading_x;
    player->entity->heading_y= heading_y;

    /* set player members */
    player->status= status;
    player->num_breaths= 0;
    player->max_num_breaths= 2;
    player->breath_speed= 8;
    player->breath_life= 5;
    player->kick_speed= 8;
    player->num_lives= 3;
    player->score= 0;

    /* animate player one frame and set his bounding box sizes */
    animate_player(player);

    /* set bounding box size and add to map */
    if (map != 0)
    {
        set_bb_size(player->entity, map);
        add_player_to_map(player, map, row, col);
    }

    return player;

} /* end construct_player */




void destruct_player(Player *player)
{
    if (player->entity != 0) destruct_entity(player->entity);
    free(player);
    player= 0;
}




void do_breath(Player *const player, Map *const map)
{

    Projectile *ice_breath;
    Baddie *adj_baddie;
    Feature *adj_feature;

    unsigned char breath_row;
    unsigned char breath_col;

    unsigned char is_in_range;

    if (player->status == PLAYER_S_STOPPED
        || player->status == PLAYER_S_MOVING
        || player->status == PLAYER_S_BREATHING)
    {

        /* figure out where we should look for a kickable entity */

        #if 0
        /* heading vertically */
        if ((player->entity->heading_y < 0
             && map->t_height - player->entity->dy > map->t_height / 2)
            || (player->entity->heading_y > 0
                && (player->entity->dy > map->t_height / 2
                    || player->entity->dy == 0)))
        {
            breath_row= adj_row(player->entity);
        }
        else
        {
            breath_row= adj_row(player->entity) - player->entity->heading_y;
        }

        /* heading horizontally */
        if ((player->entity->heading_x < 0
             && map->t_width - player->entity->dx > map->t_width / 2)
            || (player->entity->heading_x > 0
                && (player->entity->dx > map->t_width / 2
                    || player->entity->dx == 0)))
        {
            breath_col= adj_col(player->entity);
        }
        else
        {
            breath_col= adj_col(player->entity) - player->entity->heading_x;
        }
        #endif

        if (player->entity->dy == 0) breath_row= adj_row(player->entity);
        //else breath_row= adj_row(player->entity) - player->entity->heading_y;
        else breath_row= adj_row(player->entity);

        if (player->entity->dx == 0) breath_col= adj_col(player->entity);
        //else breath_col= adj_col(player->entity) - player->entity->heading_x;
        else breath_col= adj_col(player->entity);

        if (breath_col >= map->m_width || breath_row >= map->m_height) return;

        /* determine if player is within kicking range */
        if ((player->entity->dx == 0 && player->entity->dy == 0)
            || (player->entity->heading_y < 0
                && player->entity->dy <= manager.balk_y)
            || (player->entity->heading_x < 0
                && player->entity->dx <= manager.balk_x)
            || (player->entity->heading_y > 0
                && map->t_height - player->entity->dy <= manager.balk_y)
            || (player->entity->heading_x > 0
                && map->t_width - player->entity->dx <= manager.balk_x))
        {
            is_in_range= 1;
        }
        else is_in_range= 0;

        /* find adjacent frozen baddie */
        adj_baddie=
            baddie_on_tile(map, breath_row, breath_col, map->t_width,
                           map->t_height);

        /* if there's an adjacent baddie */
        if (//player->entity->dx == 0 && player->entity->dy == 0
            adj_baddie != 0
            && (adj_baddie->status == BADDIE_S_FROZEN
                || adj_baddie->status == BADDIE_S_MELTING)
            && is_in_range)
        {

            set_player_status(player, PLAYER_S_KICKING, map);

            /* update adjacent baddie based on type */
            switch (adj_baddie->type)
            {

                /* if baddie1, kick him */
                case BADDIE_T_BADDIE1:

                    adj_baddie->kicked_by_player= 1;
                    set_baddie_status(adj_baddie,
                        (adj_baddie->status == BADDIE_S_FROZEN
                         ? BADDIE_S_KICKED : BADDIE_S_KICKED_MELTING), map);
                    adj_baddie->entity->n_speed= player->kick_speed;
                    set_heading(adj_baddie->entity, player->entity->heading_x,
                                player->entity->heading_y);

                    break;

                /* if baddie4, unfreeze him */
                case BADDIE_T_BADDIE4:
                    set_baddie_status(adj_baddie, BADDIE_S_STOPPED, map);
                    break;

                /* shatter other baddies */
                default:
                    adj_baddie->kicked_by_player= 1;
                    set_baddie_status(adj_baddie, BADDIE_S_SHATTERING, map);
                    break;

            } /* end switch on baddie type */

        } /* end if there's an adjacent frozen baddie */

        /* else if there's no adjacent frozen baddie */
        else
        {

            /* find adjacent feature */
            adj_feature=
                feature_on_tile(map, breath_row, breath_col);

            /* if pow block */
            if (adj_feature != 0 && adj_feature->type == FEATURE_T_POW
                && is_in_range)
            {
                set_player_status(player, PLAYER_S_KICKING, map);
                do_pow(adj_feature, map);
            }

            /* else if there's no adjacent kickable entity, do an ice breath */
            else if (player->num_breaths < player->max_num_breaths)
            {

                /* correct the row or col if player is square */
                //if (player->entity->dx == 0) breath_col= player->entity->col;
                //if (player->entity->dy == 0) breath_row= player->entity->row;
                breath_col-= player->entity->heading_x;
                breath_row-= player->entity->heading_y;

                ice_breath=
                    construct_projectile(PROJECTILE_T_ICE_BREATH, breath_row,
                                         breath_col, player->entity->heading_x,
                                         player->entity->heading_y, map);

                if (ice_breath != 0)
                {
                    ++player->num_breaths;
                    ice_breath->player_num_breaths_ptr= &(player->num_breaths);
                    ice_breath->entity->n_speed= player->breath_speed;
                    ice_breath->entity->c_speed= player->breath_speed;
                    ice_breath->life= player->breath_life;
                    set_player_status(player, PLAYER_S_BREATHING, map);
                }

            } /* end if no adjacent kickable entity */

        } /* end if there's an adjacent frozen baddie */

    } /* end if player has proper status for an ice breath */

} /* end do_breath */




void do_pillar(Player *const player, Map *const map)
{

    unsigned char pillar_row;
    unsigned char pillar_col;

    unsigned char is_legal;

    Feature *adj_feature;

    if (player->status == PLAYER_S_STOPPED
        || player->status == PLAYER_S_MOVING)
    {

        if ((player->entity->dx == 0 && player->entity->dy == 0)
            || (player->entity->heading_y < 0
                && map->t_height - player->entity->dy > map->t_height / 2)
            || (player->entity->heading_y > 0
                && player->entity->dy > map->t_height / 2)
            || (player->entity->heading_x < 0
                && map->t_width - player->entity->dx > map->t_width / 2)
            || (player->entity->heading_x > 0
                && player->entity->dx > map->t_width / 2))
        {
            is_legal= 1;
            pillar_row= adj_row(player->entity);
            pillar_col= adj_col(player->entity);
        }
        else is_legal= 0;

        /* if we should make or destroy an ice pillar, do so */
        if (is_legal && pillar_row < map->m_height && pillar_col < map->m_width)
        {

            /* if there's already a pillar, destroy it */
            adj_feature= feature_on_tile(map, pillar_row, pillar_col);

            if (adj_feature != 0 && adj_feature->type == FEATURE_T_PILLAR)
                set_feature_status(adj_feature, FEATURE_S_KAPUT);

            /* else if there's not a pillar and the tile is empty */
            else if (is_tile_empty_feature(FEATURE_T_PILLAR, map, pillar_row,
                                           pillar_col))
            {
                construct_feature(FEATURE_T_PILLAR, pillar_row, pillar_col, 0,
                                  1, FEATURE_S_NORMAL, map);
            }

            set_player_status(player, PLAYER_S_PILLARING, map);

        } /* end if we should make or destroy an ice pillar */

    } /* end if player has appropriate status */

} /* end do_pillar */




void do_player_logic(Player *const player, Map *const map)
{

    //Map_tile *coll_tile;
    //unsigned short i;

    /* if player wants to move, do collision detection */
    if (player->entity->c_speed > 0) find_player_to_tile(player, map);
    if (player->entity->c_speed > 0) find_player_to_feature(player, map);
    if (player->entity->c_speed > 0 && manager.invincibility == 0)
        find_player_to_baddie(player, map);
    if (player->entity->c_speed > 0) find_player_to_goodie(player, map);

    /* check out of bounds */
    if (player->entity->c_speed > 0 && will_step_oob(player->entity, map))
        set_player_status(player, PLAYER_S_STOPPED, map);

    #if 0
    if (player->entity->current_speed > 0)
    {

        /* handle player-to-baddie collisions */
        for (i= 0; i < map->max_num_baddies; i++)
        {
            if (map->baddies[i] != 0
                && entity_to_entity_collision(player->entity,
                                              map->baddies[i]->entity,
                                              map, 1, 1))
            {
                do_player_to_baddie_coll_logic(player, map,
                                               map->baddies[i]);
            }
        }

        /* handle player-to-feature collisions */
        for (i= 0; i < map->max_num_features; i++)
        {
            if (map->features[i] != 0
                && entity_to_entity_collision(player->entity,
                                              map->features[i]->entity,
                                              map, 1, 1))
            {
                do_player_to_map_feature_coll_logic(player, map,
                                                    map->features[i]);
            }
        }

        /* handle player-to-tile collision */
        coll_tile= player_to_tile_coll_tile(player, map);
        if (coll_tile != 0)
            do_player_to_tile_coll_logic(player, map, coll_tile);

        /* check out of bounds */
        if (entity_out_of_bounds(player->entity, map))
            set_player_status(player, PLAYER_S_STOPPED);

    }
    #endif

    /* if player is allowed to move after collision detection, step him */
    if (player->entity->c_speed > 0)
        step_entity(player->entity, player->entity->c_speed, map);

} /* end do_player_logic */




void do_player_to_baddie(Player *const player, const Map *const map,
                         const Baddie *const baddie)
{

    /* update logic based on baddie status */
    switch (baddie->status)
    {

        /* a frozen or melting baddie */
        case BADDIE_S_FROZEN:
        case BADDIE_S_MELTING:
        case BADDIE_S_KICKED:
        case BADDIE_S_KICKED_MELTING:

            /* it's possible the player freezes baddie and baddie gets frozen on
               the tile onto which player is stepping; if this is the case,
               place player back onto the tile he was moving from */
            if (player->entity->dx > 0 || player->entity->dy > 0)
            {
                if (player->entity->heading_x > 0
                    || player->entity->heading_y > 0)
                {
                    square_entity(player->entity, player->entity->row,
                                  player->entity->col, map);
                }
                else if (player->entity->heading_x < 0)
                {
                    square_entity(player->entity, player->entity->row,
                                  player->entity->col + 1, map);
                }
                else if (player->entity->heading_y < 0)
                {
                    square_entity(player->entity, player->entity->row + 1,
                                  player->entity->col, map);
                }
            }

            /* stop player */
            set_player_status(player, PLAYER_S_STOPPED, map);

            break;

        /* steamrolled baddie */
        case BADDIE_S_STEAMROLLED:

        /* normal baddie */
        case BADDIE_S_STOPPED:
        case BADDIE_S_MOVING:
        case BADDIE_S_KICKING:

            set_player_status(player, PLAYER_S_DYING, 0);

            break;

        default:
            break;

    } /* end switch on baddie status */

} /* end do_player_to_baddie */




void do_player_to_feature(Player *const player, const Map *const map,
                          const Feature *const feature)
{
    set_player_status(player, PLAYER_S_STOPPED, map);
}




void do_player_to_goodie(Player *const player, const Map *const map,
                         Goodie *const goodie)
{
    collect_goodie(goodie, player, map, 1);
}




void do_player_to_tile(Player *const player, const Map *const map,
                       const unsigned char row, const unsigned char col)
{
    set_player_status(player, PLAYER_S_STOPPED, map);
}




void find_player_to_baddie(Player *const player, const Map *const map)
{

    //unsigned short i;

    unsigned char coll_fudge_x;
    unsigned char coll_fudge_y;

    List_iterator *iterator;
    Baddie *baddie;

    /* loop while there's more baddies and player wants to move */
    #if 0
    for (i= 0; i < map->max_num_baddies && player->entity->c_speed > 0; i++)
    {

        if (map->baddies[i] != 0)
        {

            if (map->baddies[i]->entity->status == BADDIE_S_MOVING)
            {
                coll_fudge_x= COLL_FUDGE_X;
                coll_fudge_y= COLL_FUDGE_Y;
            }
            else
            {
                coll_fudge_x= 1;
                coll_fudge_y= 1;
            }

            if (will_entities_collide(player->entity, map->baddies[i]->entity,
                                      map, coll_fudge_x, coll_fudge_y))
            {
                do_player_to_baddie(player, map, map->baddies[i]);
            }

        }

    } /* end for loop */
    #endif

    iterator= get_iterator(map->baddie_list);
    while (iterator != 0 && player->entity->c_speed > 0)
    {

        baddie= (Baddie *) iterator->dat;

        if (baddie->status == BADDIE_S_MOVING
            || baddie->status == BADDIE_S_STOPPED)
        {
            coll_fudge_x=
                (manager.balk_x > COLL_FUDGE_X ? manager.balk_x : COLL_FUDGE_X);
            coll_fudge_y=
                (manager.balk_y > COLL_FUDGE_Y ? manager.balk_y : COLL_FUDGE_Y);
        }
        else
        {
            coll_fudge_x= 1;
            coll_fudge_y= 1;
        }

        if (will_entities_collide(player->entity, baddie->entity, map,
                                  coll_fudge_x, coll_fudge_y)
            && player_baddie_coll(player->status, baddie->type,
                                  baddie->status))
        {
            do_player_to_baddie(player, map, baddie);
        }

        iterator= next_iterator(iterator);

    }

} /* end find_player_to_baddie */




void find_player_to_feature(Player *const player, const Map *const map)
{

    //unsigned short i;

    /* loop while there's more features and player wants to move */
    #if 0
    for (i= 0; i < map->max_num_features && player->entity->c_speed > 0; i++)
    {
        if (map->features[i] != 0
            && will_entities_collide(player->entity, map->features[i]->entity,
                                     map, 1, 1))
        {
            do_player_to_feature(player, map, map->features[i]);
        }
    }
    #endif

    Feature *feature;

    List_iterator *iterator= get_iterator(map->feature_list);
    while (iterator != 0 && player->entity->c_speed > 0)
    {

        feature= (Feature *) iterator->dat;

        if (will_entities_collide(player->entity, feature->entity, map, 1, 1)
            && player_feature_coll(player->status, feature->type,
                                   feature->status))
        {
            do_player_to_feature(player, map, feature);
        }

        iterator= next_iterator(iterator);

    }

} /* end find_player_to_feature */




void find_player_to_goodie(Player *const player, const Map *const map)
{

    //unsigned short i;

    unsigned char coll_fudge_x;
    unsigned char coll_fudge_y;

    Goodie *goodie;

    List_iterator *iterator;

    if (manager.balk_x >= map->t_width) coll_fudge_x= map->t_width;
    else coll_fudge_x= manager.balk_x + 1;

    if (manager.balk_y >= map->t_height) coll_fudge_y= map->t_height;
    else coll_fudge_y= manager.balk_y + 1;

    /* loop while there's more goodies and player wants to move */
    #if 0
    for (i= 0; i < map->max_num_goodies && player->entity->c_speed > 0; i++)
    {
        if (map->goodies[i] != 0
            && will_entities_collide(player->entity, map->goodies[i]->entity,
                                     map, coll_fudge_x, coll_fudge_y))
        {
            do_player_to_goodie(player, map, map->goodies[i]);
        }
    }
    #endif

    iterator= get_iterator(map->goodie_list);
    while (iterator != 0 && player->entity->c_speed > 0)
    {

        goodie= (Goodie *) iterator->dat;

        if (will_entities_collide(player->entity, goodie->entity, map,
                                  coll_fudge_x, coll_fudge_y)
            && player_goodie_coll(player->status, goodie->type,
                                  player->status))
        {
            do_player_to_goodie(player, map, goodie);
        }

        iterator= next_iterator(iterator);

    }

} /* end find_player_to_goodie */




void find_player_to_tile(Player *const player, const Map *const map)
{

    unsigned char row;
    unsigned char col;

    if (player->entity->dx > 0 || player->entity->dy > 0) return;

    if (player->entity->heading_y != 0)
    {

        row= adj_row(player->entity);

        for (col= player->entity->col;
             col < player->entity->col + player->entity->bb_t_width
             && player->entity->c_speed > 0; col++)
        {
            if (row < map->m_height && col < map->m_width
                && player_tile_coll(player->status,
                                    map->tiles[row][col].type))
            {
                do_player_to_tile(player, map, row, col);
            }
        }

    }

    if (player->entity->heading_x != 0)
    {

        col= adj_col(player->entity);

        for (row= player->entity->row;
             row < player->entity->row + player->entity->bb_t_height
             && player->entity->c_speed > 0; row++)
        {
            if (row < map->m_height && col < map->m_width
                && player_tile_coll(player->status,
                                    map->tiles[row][col].type))
            {
                do_player_to_tile(player, map, row, col);
            }
        }

    }

} /* end find_player_to_tile */




void move_player_down(Player *const player, const Map *const map)
{

    /* if player is not in mid-move */
    if (player->entity->dx == 0 && player->entity->dy == 0)
    {
        set_heading(player->entity, 0, 1);
        set_player_status(player, PLAYER_S_MOVING, map);
    }

    /* else if player is in mid-move */
    else
    {

        #if 0
        /* moving right and far enough into the tile to skip the tile */
        if (player->entity->heading_x > 0
            && player->entity->dx >= map->cornering_grace_w
            && is_tile_empty_player(map, player->entity->row + 1,
                                    player->entity->col + 1))
        {
            square_entity(player->entity, player->entity->row,
                          player->entity->col + 1, map);
            set_heading(player->entity, 0, 1);
            set_player_status(player, PLAYER_S_MOVING, map);
        }

        /* moving left and far enough into the tile to skip the tile */
        else if (player->entity->heading_x < 0
                 && map->t_width - player->entity->dx >= map->cornering_grace_w
                 && is_tile_empty_player(map, player->entity->row + 1,
                                         player->entity->col))
        {
            square_entity(player->entity, player->entity->row,
                          player->entity->col, map);
            set_heading(player->entity, 0, 1);
            set_player_status(player, PLAYER_S_MOVING, map);
        }
        #endif

        /* moving right */
        if (player->entity->heading_x > 0)
        {
            /* within balk */
            if (manager.use_balk && player->entity->dx <= manager.balk_x
                && is_tile_empty_player(player->status, map,
                                        player->entity->row + 1,
                                        player->entity->col))
            {
                square_entity(player->entity, player->entity->row,
                              player->entity->col, map);
                set_heading(player->entity, 0, 1);
                set_player_status(player, PLAYER_S_MOVING, map);
            }
            /* far enough into the tile to skip the tile */
            else if (player->entity->dx >= map->cornering_grace_w
                     && is_tile_empty_player(player->status, map,
                                             player->entity->row + 1,
                                             player->entity->col + 1))
            {
                square_entity(player->entity, player->entity->row,
                              player->entity->col + 1, map);
                set_heading(player->entity, 0, 1);
                set_player_status(player, PLAYER_S_MOVING, map);
            }
        }

        /* moving left */
        else if (player->entity->heading_x < 0)
        {
            /* within balk */
            if (manager.use_balk
                && map->t_width - player->entity->dx <= manager.balk_x
                && is_tile_empty_player(player->status, map,
                                        player->entity->row + 1,
                                        player->entity->col + 1))
            {
                square_entity(player->entity, player->entity->row,
                              player->entity->col + 1, map);
                set_heading(player->entity, 0, 1);
                set_player_status(player, PLAYER_S_MOVING, map);
            }
            /* far enough into the tile to skip the tile */
            else if (map->t_width - player->entity->dx >= map->cornering_grace_w
                     && is_tile_empty_player(player->status, map,
                                             player->entity->row + 1,
                                             player->entity->col))
            {
                square_entity(player->entity, player->entity->row,
                              player->entity->col, map);
                set_heading(player->entity, 0, 1);
                set_player_status(player, PLAYER_S_MOVING, map);
            }
        }

    }

} /* end move_player_down */




void move_player_left(Player *const player, const Map *const map)
{

    /* if player is not in mid-move */
    if (player->entity->dx == 0 && player->entity->dy == 0)
    {
        set_heading(player->entity, -1, 0);
        set_player_status(player, PLAYER_S_MOVING, map);
    }

    /* else if player is in mid-move */
    else
    {

        /* moving down */
        if (player->entity->heading_y > 0)
        {
            /* within balk */
            if (manager.use_balk && player->entity->dy <= manager.balk_y
                && is_tile_empty_player(player->status, map,
                                        player->entity->row,
                                        player->entity->col - 1))
            {
                square_entity(player->entity, player->entity->row,
                              player->entity->col, map);
                set_heading(player->entity, -1, 0);
                set_player_status(player, PLAYER_S_MOVING, map);
            }
            /* far enough into the tile to skip the tile */
            else if (player->entity->dy >= map->cornering_grace_h
                     && is_tile_empty_player(player->status, map,
                                             player->entity->row + 1,
                                             player->entity->col - 1))
            {
                square_entity(player->entity, player->entity->row + 1,
                              player->entity->col, map);
                set_heading(player->entity, -1, 0);
                set_player_status(player, PLAYER_S_MOVING, map);
            }
        }

        /* moving up */
        else if (player->entity->heading_y < 0)
        {
            /* within balk */
            if (manager.use_balk
                && map->t_height - player->entity->dy <= manager.balk_y
                && is_tile_empty_player(player->status, map,
                                        player->entity->row + 1,
                                        player->entity->col - 1))
            {
                square_entity(player->entity, player->entity->row + 1,
                              player->entity->col, map);
                set_heading(player->entity, -1, 0);
                set_player_status(player, PLAYER_S_MOVING, map);
            }
            /* far enough into the tile to skip the tile */
            else if (map->t_height - player->entity->dy
                     >= map->cornering_grace_h
                     && is_tile_empty_player(player->status, map,
                                             player->entity->row,
                                             player->entity->col - 1))
            {
                square_entity(player->entity, player->entity->row,
                              player->entity->col, map);
                set_heading(player->entity, -1, 0);
                set_player_status(player, PLAYER_S_MOVING, map);
            }
        }

    }

} /* end move_player_left */




void move_player_right(Player *const player, const Map *const map)
{

    /* if player is not in mid-move */
    if (player->entity->dx == 0 && player->entity->dy == 0)
    {
        set_heading(player->entity, 1, 0);
        set_player_status(player, PLAYER_S_MOVING, map);
    }

    /* else if player is in mid-move */
    else
    {

        /* moving down */
        if (player->entity->heading_y > 0)
        {
            /* within balk */
            if (manager.use_balk && player->entity->dy <= manager.balk_y
                && is_tile_empty_player(player->status, map,
                                        player->entity->row,
                                        player->entity->col + 1))
            {
                square_entity(player->entity, player->entity->row,
                              player->entity->col, map);
                set_heading(player->entity, 1, 0);
                set_player_status(player, PLAYER_S_MOVING, map);
            }
            /* far enough into the tile to skip the tile */
            else if (player->entity->dy >= map->cornering_grace_h
                     && is_tile_empty_player(player->status, map,
                                             player->entity->row + 1,
                                             player->entity->col + 1))
            {
                square_entity(player->entity, player->entity->row + 1,
                              player->entity->col, map);
                set_heading(player->entity, 1, 0);
                set_player_status(player, PLAYER_S_MOVING, map);
            }
        }

        /* moving up */
        else if (player->entity->heading_y < 0)
        {
            /* within balk */
            if (manager.use_balk
                && map->t_height - player->entity->dy <= manager.balk_y
                && is_tile_empty_player(player->status, map,
                                        player->entity->row + 1,
                                        player->entity->col + 1))
            {
                square_entity(player->entity, player->entity->row + 1,
                              player->entity->col, map);
                set_heading(player->entity, 1, 0);
                set_player_status(player, PLAYER_S_MOVING, map);
            }
            /* far enough into the tile to skip the tile */
            else if (map->t_height - player->entity->dy
                     >= map->cornering_grace_h
                     && is_tile_empty_player(player->status, map,
                                             player->entity->row,
                                             player->entity->col + 1))
            {
                square_entity(player->entity, player->entity->row,
                              player->entity->col, map);
                set_heading(player->entity, 1, 0);
                set_player_status(player, PLAYER_S_MOVING, map);
            }
        }

    }

} /* end move_player_right */




void move_player_up(Player *const player, const Map *const map)
{

    /* if player is not in mid-move */
    if (player->entity->dx == 0 && player->entity->dy == 0)
    {
        set_heading(player->entity, 0, -1);
        set_player_status(player, PLAYER_S_MOVING, map);
    }

    /* else if player is in mid-move */
    else
    {

        /* moving right */
        if (player->entity->heading_x > 0)
        {
            /* within balk */
            if (manager.use_balk && player->entity->dx <= manager.balk_x
                && is_tile_empty_player(player->status, map,
                                        player->entity->row - 1,
                                        player->entity->col))
            {
                square_entity(player->entity, player->entity->row,
                              player->entity->col, map);
                set_heading(player->entity, 0, -1);
                set_player_status(player, PLAYER_S_MOVING, map);
            }
            /* far enough into the tile to skip the tile */
            else if (player->entity->dx >= map->cornering_grace_w
                     && is_tile_empty_player(player->status, map,
                                             player->entity->row - 1,
                                             player->entity->col + 1))
            {
                square_entity(player->entity, player->entity->row,
                              player->entity->col + 1, map);
                set_heading(player->entity, 0, -1);
                set_player_status(player, PLAYER_S_MOVING, map);
            }
        }

        /* moving left */
        else if (player->entity->heading_x < 0)
        {
            /* within balk */
            if (manager.use_balk
                && map->t_width - player->entity->dx <= manager.balk_x
                && is_tile_empty_player(player->status, map,
                                        player->entity->row - 1,
                                        player->entity->col + 1))
            {
                square_entity(player->entity, player->entity->row,
                              player->entity->col + 1, map);
                set_heading(player->entity, 0, -1);
                set_player_status(player, PLAYER_S_MOVING, map);
            }
            /* far enough into the tile to skip the tile */
            else if (map->t_width - player->entity->dx >= map->cornering_grace_w
                     && is_tile_empty_player(player->status, map,
                                             player->entity->row - 1,
                                             player->entity->col))
            {
                square_entity(player->entity, player->entity->row,
                              player->entity->col, map);
                set_heading(player->entity, 0, -1);
                set_player_status(player, PLAYER_S_MOVING, map);
            }
        }

    }

} /* end move_player_up */




void player_anim_done(Player *const player)
{

    switch (player->status)
    {

        case PLAYER_S_DYING:
        case PLAYER_S_WON:
            set_player_status(player, PLAYER_S_KAPUT, 0);
            break;

        default:
            if (player->entity->c_speed > 0)
                //player->entity->status= PLAYER_S_MOVING;
                set_player_status(player, PLAYER_S_MOVING, 0);
            else
                set_player_status(player, PLAYER_S_STOPPED, 0);
                //player->entity->status= PLAYER_S_STOPPED;
            break;

    }

}




unsigned char player_baddie_coll(const player_status p_status,
                                 const baddie_type b_type,
                                 const baddie_status b_status)
{
    switch (p_status)
    {
        case PLAYER_S_DYING:
        case PLAYER_S_WON:
        case PLAYER_S_KAPUT:
            return 0;
            break;
        default:
            return 1;
            break;
    }
    return 1;
}





unsigned char player_feature_coll(const player_status p_status,
                                  const feature_type f_type,
                                  const feature_status f_status)
{
    if (f_type == FEATURE_T_BLACK_ICE) return 0;
    return 1;
}




unsigned char player_goodie_coll(const player_status p_status,
                                 const goodie_type g_type,
                                 const goodie_status g_status)
{
    return 1;
}




unsigned char player_tile_coll(const player_status p_status,
                               const tile_type t_type)
{
    if (t_type == TILE_T_ICE) return 0;
    return 1;
}




void set_player_status(Player *const player, player_status status,
                       const Map *const map)
{

    //printf("%s\n", __FUNCTION__);

    //player->entity->anim_counter= 0;
    //player->entity->anim_frame_counter= 0;
    //player->entity->anim_play_counter= 0;

    if (player->status != PLAYER_S_KAPUT)
    {

        switch (status)
        {

            case PLAYER_S_STOPPED:

                /* stop player if square */
                if (player->entity->dx == 0 && player->entity->dy == 0)
                {
                    player->entity->c_speed= 0;
                    player->status= status;
                    reset_anim(player->entity);
                }

                /* if player balking is on and player is moving */
                else if (manager.use_balk
                         && player->status == PLAYER_S_MOVING)
                {

                    /* heading down */
                    if (player->entity->heading_y > 0
                        && player->entity->dy <= manager.balk_y)
                    {

                        square_entity(player->entity, player->entity->row,
                                      player->entity->col, map);
                        player->entity->c_speed= 0;
                        player->status= status;

                        reset_anim(player->entity);

                    }

                    /* heading up */
                    else if (player->entity->heading_y < 0
                             && map->t_height - player->entity->dy
                                <= manager.balk_y)
                    {

                        square_entity(player->entity, player->entity->row + 1,
                                      player->entity->col, map);
                        player->entity->c_speed= 0;
                        player->status= status;

                        reset_anim(player->entity);

                    }

                    /* heading right */
                    if (player->entity->heading_x > 0
                        && player->entity->dx <= manager.balk_x)
                    {

                        square_entity(player->entity, player->entity->row,
                                      player->entity->col, map);
                        player->entity->c_speed= 0;
                        player->status= status;

                        reset_anim(player->entity);

                    }

                    /* heading left */
                    else if (player->entity->heading_x < 0
                             && map->t_width - player->entity->dx <=
                                manager.balk_x)
                    {

                        square_entity(player->entity, player->entity->row,
                                      player->entity->col + 1, map);
                        player->entity->c_speed= 0;
                        player->status= status;

                        reset_anim(player->entity);

                    }

                } /* end if player balking is on */

                break;

            case PLAYER_S_MOVING:
                player->entity->c_speed= player->entity->n_speed;
                if (player->status != status)
                {
                    reset_anim(player->entity);
                    player->status= status;
                }
                break;

            case PLAYER_S_DYING:
            case PLAYER_S_WON:
            case PLAYER_S_KAPUT:
                if (player->status != status)
                {
                    //printf("status == %i %s\n", status, __FUNCTION__);
                    player->status= status;
                    player->entity->c_speed= 0;
                    reset_anim(player->entity);
                }
                break;

            default:
                //if (player->entity->status == PLAYER_S_STOPPED
                //  || player->entity->status == PLAYER_S_MOVING)
                //{

                    player->status= status;

                    reset_anim(player->entity);

                //}
                break;

        }

    }

} /* end set_player_status */
