/************************************
 *
 * Kickle 0.81
 * 2004 Drew Willcoxon
 * http://www.cs.uga.edu/~adw/
 * dripfeed@uga.edu
 *
 ************************************/


#include <errno.h>
#include <limits.h>
#include <stdlib.h>
#include <string.h>

#include <allegro.h>

#include "baddie.h"
#include "common.h"
//#include "config.h"
#include "controls.h"
#include "entity.h"
#include "feature.h"
#include "ftext.h"
#include "goodie.h"
#include "list.h"
#include "log.h"
#include "manager.h"
#include "map.h"
#include "player.h"
#include "projctl.h"
#include "video.h"




/* water and ice sprite indices */
#define SPRITE_L            1
#define SPRITE_L_U          2
#define SPRITE_L_UL         3
#define SPRITE_U            4
#define SPRITE_UL_U         5
#define SPRITE_UL           6
#define SPRITE_ICE          7


/* default map time limit, used when constructing an empty map */
#define TIME_LIMIT_DEF      2000



extern Manager manager;



unsigned char add_baddie_to_map(Baddie *const baddie, Map *const map,
                                const unsigned char row,
                                const unsigned char col)
{


    #if 0
    unsigned short i;

    for (i= 0; i < map->max_num_baddies; i++)
    {
        /* if there's a spot open to add the baddie, add him */
        if (map->baddies[i] == 0)
        {
            map->baddies[i]= baddie;
            square_entity(baddie->entity, row, col, map);
            return 0;
        }
    }

    /* if we've reached here, no empty spots to add the baddie */
    write_to_log(manager.log, 1, "No room to add baddie to map.\n");
    return 1;
    #endif

    if (append_to_list(map->baddie_list, baddie, 0) == 0)
    {
        write_to_log(manager.log, 1, "Could not add baddie to map.\n");
        return 1;
    }

    square_entity(baddie->entity, row, col, map);
    return 0;

} /* end add_baddie_to_map */




unsigned char add_feature_to_map(Feature *const feature, Map *const map,
                                 const unsigned char row,
                                 const unsigned char col)
{

    #if 0
    unsigned short i;

    for (i= 0; i < map->max_num_features; i++)
    {
        /* if there's a spot open to add the feature, add it */
        if (map->features[i] == 0)
        {
            map->features[i]= feature;
            square_entity(feature->entity, row, col, map);
            return 0;
        }
    }

    /* if we've reached here, no empty spots to add the feature */
    write_to_log(manager.log, 1, "No room to add feature to map.\n");
    return 1;
    #endif

    if (append_to_list(map->feature_list, feature, 0) == 0)
    {
        write_to_log(manager.log, 1, "Could not add feature to map.\n");
        return 1;
    }

    square_entity(feature->entity, row, col, map);
    return 0;

} /* end add_feature_to_map */




unsigned char add_goodie_to_map(Goodie *const goodie, Map *const map,
                                const unsigned char row,
                                const unsigned char col)
{

    #if 0
    unsigned short i;

    for (i= 0; i < map->max_num_goodies; i++)
    {
        /* if there's a spot open to add the goodie, add it */
        if (map->goodies[i] == 0)
        {
            map->goodies[i]= goodie;
            square_entity(goodie->entity, row, col, map);
            return 0;
        }
    }

    /* if we've reached here, no empty spots to add the goodie */
    write_to_log(manager.log, 1, "No room to add goodie to map.\n");
    return 1;
    #endif

    if (append_to_list(map->goodie_list, goodie, 0) == 0)
    {
        write_to_log(manager.log, 1, "Could not add goodie to map.\n");
        return 1;
    }

    square_entity(goodie->entity, row, col, map);
    return 0;

} /* end add_goodie_to_map */




void add_player_to_map(Player *const player, Map *const map,
                       const unsigned char row, const unsigned char col)
{
    map->player= player;
    square_entity(player->entity, row, col, map);
}




unsigned char add_projectile_to_map(Projectile *const projectile,
                                    Map *const map, const unsigned char row,
                                    const unsigned char col)
{

    #if 0
    unsigned short i;

    for (i= 0; i < map->max_num_projectiles; i++)
    {
        /* if there's a spot open to add the projectile, add it */
        if (map->projectiles[i] == 0)
        {
            map->projectiles[i]= projectile;
            square_entity(projectile->entity, row, col, map);
            return 0;
        }
    }

    /* if we've reached here, no empty spots to add the projectile */
    write_to_log(manager.log, 1, "No room to add projectile to map.\n");
    return 1;
    #endif

    if (append_to_list(map->projectile_list, projectile, 0) == 0)
    {
        write_to_log(manager.log, 1, "Could not add projectile to map.\n");
        return 1;
    }

    square_entity(projectile->entity, row, col, map);
    return 0;

} /* end add_projectile_to_map */




Baddie *baddie_on_tile(const Map *const map, const unsigned char row,
                       const unsigned char col, const unsigned char fudge_x,
                       const unsigned char fudge_y)
{

    #if 0
    unsigned short i;

    for (i= 0; i < map->max_num_baddies; i++)
    {
        if (map->baddies[i] != 0
            && is_entity_on_tile(map->baddies[i]->entity, map, row, col,
                                 fudge_x, fudge_y))
        {
            return map->baddies[i];
        }
    }
    #endif

    Baddie *baddie;
    List_iterator *iterator= get_iterator(map->baddie_list);

    while (iterator != 0)
    {
        baddie= (Baddie *) iterator->dat;
        if (is_entity_on_tile(baddie->entity, map, row, col, fudge_x, fudge_y))
            return baddie;
        iterator= next_iterator(iterator);
    }

    /* no baddie found */
    return 0;

} /* end baddie_on_tile */




Map *clone_map(const Map *const map)
{

    Map *clone;

    //unsigned short i;

    unsigned char row;
    unsigned char col;

    Feature *clone_hive;

    List_iterator *iterator;
    Baddie *baddie;
    Goodie *goodie;
    Feature *feature;

    /* construct the empty map */
    clone=
        construct_empty_map(map->name, map->author, map->m_width,
                            map->m_height, map->t_width, map->t_height);

    if (clone == 0)
    {
        write_to_log(manager.log, 1,
                     "Could not construct empty map for clone map.\n");
        return 0;
    }

    clone->time_limit= map->time_limit;

    /* set up the tiles */
    for (row= 0; row < clone->m_height; row++)
    {
        for (col= 0; col < clone->m_width; col++)
        {
            setup_tile(&(clone->tiles[row][col]), map->tiles[row][col].type,
                         map->tiles[row][col].anim_index);
        }
    }

    /* clone the player */
    if (map->player != 0)
    {
        if (clone_player(map->player, clone, map->player->entity->row,
                         map->player->entity->col) == 0)
        {
            write_to_log(manager.log, 1, "Could not clone player.\n");
            destruct_map(clone);
            return 0;
        }
    }

    /* clone the features */
    #if 0
    for (i= 0; i < map->max_num_features; i++)
    {
        if (map->features[i] != 0)
        {
            if (clone_feature(map->features[i], clone,
                              map->features[i]->entity->row,
                              map->features[i]->entity->col) == 0)
            {
                write_to_log(manager.log, 1, "Could not clone feature.\n");
                destruct_map(clone);
                return 0;
            }
        }
    }
    #endif

    iterator= get_iterator(map->feature_list);
    while (iterator != 0)
    {

        feature= (Feature *) iterator->dat;

        if (clone_feature(feature, clone, feature->entity->row,
                          feature->entity->col) == 0)
        {
            write_to_log(manager.log, 1, "Could not clone feature.\n");
            destruct_map(clone);
            return 0;
        }

        iterator= next_iterator(iterator);

    }

    /* clone the baddies */
    #if 0
    for (i= 0; i < map->max_num_baddies; i++)
    {
        if (map->baddies[i] != 0)
        {

            /* determine the hive on the clone map which is baddie's hive */
            if (map->baddies[i]->hive != 0)
            {
                clone_hive=
                    feature_on_tile(clone, map->baddies[i]->hive->entity->row,
                                    map->baddies[i]->hive->entity->col);
            }
            else clone_hive= 0;

            if (clone_baddie(map->baddies[i], clone,
                             map->baddies[i]->entity->row,
                             map->baddies[i]->entity->col, clone_hive) == 0)
            {
                write_to_log(manager.log, 1, "Could not clone baddie.\n");
                destruct_map(clone);
                return 0;
            }

        }
    }
    #endif

    iterator= get_iterator(map->baddie_list);
    while (iterator != 0)
    {

        baddie= (Baddie *) iterator->dat;

        /* determine the hive on the clone map which is baddie's hive */
        if (baddie->hive != 0)
        {
            clone_hive=
                feature_on_tile(clone, baddie->hive->entity->row,
                                baddie->hive->entity->col);
        }
        else clone_hive= 0;

        if (clone_baddie(baddie, clone, baddie->entity->row,
                         baddie->entity->col, clone_hive) == 0)
        {
            write_to_log(manager.log, 1, "Could not clone baddie.\n");
            destruct_map(clone);
            return 0;
        }

        iterator= next_iterator(iterator);

    }

    /* clone the goodies */
    #if 0
    for (i= 0; i < map->max_num_goodies; i++)
    {
        if (map->goodies[i] != 0)
        {
            if (clone_goodie(map->goodies[i], clone,
                             map->goodies[i]->entity->row,
                             map->goodies[i]->entity->col) == 0)
            {
                write_to_log(manager.log, 1, "Could not clone goodie.\n");
                destruct_map(clone);
                return 0;
            }
        }
    }
    #endif

    iterator= get_iterator(map->goodie_list);
    while (iterator != 0)
    {

        goodie= (Goodie *) iterator->dat;

        if (clone_goodie(goodie, clone, goodie->entity->row,
                         goodie->entity->col) == 0)
        {
            write_to_log(manager.log, 1, "Could not clone goodie.\n");
            destruct_map(clone);
            return 0;
        }

        iterator= next_iterator(iterator);

    }

    #if 0
    /* clone the projectiles */
    for (i= 0; i < map->max_num_projectiles; i++)
    {
        if (map->projectiles[i] != 0)
        {
            if (clone_projectile(map->projectiles[i], clone,
                                 map->projectiles[i]->entity->map_row,
                                 map->projectiles[i]->entity->map_col) == 0)
            {
                strcat(kickle_err_msg,
                       "\nclone_map: Could not clone projectile.");
                destruct_map(clone);
                return 0;
            }
        }
    }
    #endif

    return clone;

} /* end clone_map */




Tile *clone_tile(const Tile *const tile)
{

    Tile *clone= construct_tile(tile->type, tile->anim_index);

    if (clone == 0)
        write_to_log(manager.log, 1, "Could not construct tile clone.\n");

    return clone;

}




Map *construct_empty_map(const char *map_name, const char *author,
                         const unsigned char map_width,
                         const unsigned char map_height,
                         const unsigned char tile_width,
                         const unsigned char tile_height)
{


    //unsigned char i;
    unsigned short map_area;

    unsigned char row;
    unsigned char col;

    Map *map= (Map *) malloc(sizeof(Map));
    if (map == 0)
    {
        write_to_log(manager.log, 1, "Could not allocate empty map memory.\n");
        return 0;
    }

    /* copy map name */
    if (map_name != 0)
    {
        map->name= (char *) malloc(sizeof(char) * (strlen(map_name) + 1));
        if (map->name != 0) strcpy(map->name, map_name);
    }
    else map->name= 0;

    /* copy author */
    if (author != 0)
    {
        map->author= (char *) malloc(sizeof(char) * (strlen(author) + 1));
        if (map->author!= 0) strcpy(map->author, author);
    }
    else map->author= 0;

    /* and the rest */
    map->m_width= map_width;
    map->m_height= map_height;
    map->t_width= tile_width;
    map->t_height= tile_height;

    /* set the other members which require malloc'ing */

    map_area= map_width * map_height;

    /* tile memory: rows */
    map->tiles= (Tile **) malloc(sizeof(Tile *) * map_height);

    if (map->tiles == 0)
    {
        write_to_log(manager.log, 1,
                     "Could not allocate empty map's row memory.\n");
        destruct_map(map);
        return 0;
    }

    /* tile memory: columns */
    for (row= 0; row < map_height; row++)
    {

        map->tiles[row]= (Tile *) malloc(sizeof(Tile) * map_width);

        if (map->tiles[row] == 0)
        {
            write_to_log(manager.log, 1,
                         "Could not allocate empty map's column memory.\n");
            destruct_map(map);
            return 0;
        }

        /* set the tile x's and y's */
        for (col= 0; col < map_width; col++)
        {
            map->tiles[row][col].x= COL_TO_X(col, map);
            map->tiles[row][col].y= ROW_TO_Y(row, map);
        }

    }

    /* player */
    map->player= 0;

    /* baddie memory */
    #if 0
    map->max_num_baddies= map_area;
    map->baddies=
        (Baddie **)
        malloc(sizeof(type_baddie *) * map->max_num_baddies);

    if (map->baddies == 0)
    {
        write_to_log(manager.log, 1,
                     "Could not allocate empty map's baddie memory.\n");
        destruct_map(map);
        return 0;
    }

    for (i= 0; i < map->max_num_baddies; i++)
    {
        map->baddies[i]= 0;
    }
    #endif

    map->baddie_list= construct_list();
    if (map->baddie_list == 0)
    {
        write_to_log(manager.log, 1,
                     "Could not construct empty map's baddie list.\n");
        destruct_map(map);
        return 0;
    }

    /* feature memory */
    #if 0
    map->max_num_features= map_area;
    map->features=
        (Feature **)
        malloc(sizeof(type_feature *) * map->max_num_features);

    if (map->features == 0)
    {
        write_to_log(manager.log, 1,
                     "Could not allocate empty map's feature memory.\n");
        destruct_map(map);
        return 0;
    }

    for (i= 0; i < map->max_num_features; i++)
    {
        map->features[i]= 0;
    }
    #endif

    map->feature_list= construct_list();
    if (map->feature_list == 0)
    {
        write_to_log(manager.log, 1,
                     "Could not construct empty map's feature list.\n");
        destruct_map(map);
        return 0;
    }

    /* goodie memory */
    #if 0
    map->max_num_goodies= map_area;
    map->goodies=
        (Goodie **)
        malloc(sizeof(type_goodie *) * map->max_num_goodies);

    if (map->goodies == 0)
    {
        write_to_log(manager.log, 1,
                     "Could not allocate empty map's goodie memory.\n");
        destruct_map(map);
        return 0;
    }

    for (i= 0; i < map->max_num_goodies; i++)
    {
        map->goodies[i]= 0;
    }
    #endif

    map->goodie_list= construct_list();
    if (map->goodie_list == 0)
    {
        write_to_log(manager.log, 1,
                     "Could not construct empty map's goodie list.\n");
        destruct_map(map);
        return 0;
    }

    /* projectile memory */
    #if 0
    map->max_num_projectiles= map_area;
    map->projectiles=
        (type_projectile **)
        malloc(sizeof(type_projectile *) * map->max_num_projectiles);

    if (map->projectiles == 0)
    {
        write_to_log(manager.log, 1,
                     "Could not allocate empty map's projectile memory.\n");
        destruct_map(map);
        return 0;
    }

    for (i= 0; i < map->max_num_projectiles; i++)
    {
        map->projectiles[i]= 0;
    }
    #endif

    map->projectile_list= construct_list();
    if (map->projectile_list == 0)
    {
        write_to_log(manager.log, 1,
                     "Could not construct empty map's projectile list.\n");
        destruct_map(map);
        return 0;
    }

    /* set the remaining members */

    map->point_list= construct_list();
    if (map->point_list == 0)
    {
        write_to_log(manager.log, 1,
                     "Could not construct empty map's point list.\n");
        destruct_map(map);
        return 0;
    }

    map->time_limit= TIME_LIMIT_DEF;
    map->cornering_grace_w= tile_width - (tile_width / 4);
    map->cornering_grace_h= tile_height - (tile_height / 4);

    return map;

} /* end construct_empty_map */




Map *construct_map_from_file(const char *filename,
                             const unsigned char compressed)
{


    Map *map;

    unsigned char buffer[14];

    unsigned char row;
    unsigned char col;

    Feature *hive;

    /* OPEN THE FILE */
    PACKFILE *file= pack_fopen(filename, ( compressed ? "rp" : "r" ));

    if (file == 0)
    {
        write_to_log(manager.log, 1,
                     "Could not open map file \"%s\" for reading.  ",
                     filename);
        /* if trying to read unpacked file as packed */
        if (errno == EDOM)
        {
            write_to_log(manager.log, 0,
                         "File is uncompressed, but it was attempted to be opened in compressed mode.\n");
        }
        /* some other error */
        else write_to_log(manager.log, 0, "%s.\n", strerror(errno));
        return 0;
    }



    /* READ THE HEADER */

    pack_fread(buffer, 13, file);
    buffer[13]= '\0';

    if (strcmp(buffer, "KiCkLeMaPfIlE") != 0)
    {
        write_to_log(manager.log, 1,
                     "File \"%s\" is not a valid map file.\n", filename);
        pack_fclose(file);
        return 0;
    }

    /* map and tile dimensions -- buffer[0] = map width,
       buffer[1] = map height, buffer[2] = tile width,
       buffer[3] = tile height */
    pack_fread(buffer, 4, file);

    /* construct the empty map */
    map= construct_empty_map(0, 0, buffer[0], buffer[1], buffer[2], buffer[3]);
    if (map == 0)
    {
        write_to_log(manager.log, 1, "Could not construct empty map.\n");
        pack_fclose(file);
        return 0;
    }

    /* map name -- buffer[0] = map name string length */
    pack_fread(buffer, 1, file);
    map->name= (char *) malloc(sizeof(char) * (buffer[0] + 1));
    pack_fread(map->name, sizeof(char) * buffer[0], file);
    map->name[buffer[0]]= '\0';

    /* map author -- buffer[0] = author string length */
    pack_fread(buffer, 1, file);
    map->author= (char *) malloc(sizeof(char) * (buffer[0] + 1));
    pack_fread(map->author, sizeof(char) * buffer[0], file);
    map->author[buffer[0]]= '\0';

    /* time limit */
    pack_fread(&(map->time_limit), sizeof(unsigned short), file);



    /* READ THE TILE DATA */
    for (row= 0; row < map->m_height; row++)
    {
        for (col= 0; col < map->m_width; col++)
        {
            /* buffer[0] = tile type, buffer[1] = tile anim_index */
            pack_fread(buffer, 2, file);
            setup_tile(&(map->tiles[row][col]), buffer[0], buffer[1]);
        }
    }



    /* READ THE PLAYER DATA: buffer[0] = map_row, buffer[1] = map_col,
       buffer[2] = heading_x, buffer[3] = heading_y, buffer[4] = status */
    pack_fread(buffer, 1, file);
    if (buffer[0] != UCHAR_MAX)
    {
        pack_fread(buffer + 1, 4, file);
        if (construct_player(buffer[0], buffer[1], buffer[2], buffer[3],
                             buffer[4], map) == 0)
        {
            write_to_log(manager.log, 1, "Could not construct player.\n");
            destruct_map(map);
            pack_fclose(file);
            return 0;
        }
    }


    /* READ THE FEATURE DATA: buffer[0] = feature type, buffer[1] = map_row,
       buffer[2] = map_col, buffer[3] = heading_x, buffer[4] = heading_y,
       buffer[5] = status */
    pack_fread(buffer, 1, file);
    while (buffer[0] != UCHAR_MAX)
    {
        pack_fread(buffer + 1, 5, file);
        if (construct_feature(buffer[0], buffer[1], buffer[2], buffer[3],
                              buffer[4], buffer[5], map) == 0)
        {
            write_to_log(manager.log, 1, "Could not construct feature.\n");
            destruct_map(map);
            pack_fclose(file);
            return 0;
        }
        pack_fread(buffer, 1, file);
    }


    /* READ THE BADDIE DATA: buffer[0] = baddie type, buffer[1] = map_row,
       buffer[2] = map_col, buffer[3] = heading_x, buffer[4] = heading_y,
       buffer[5] = status, buffer[6] = hive_row, buffer[7] = hive_col */
    pack_fread(buffer, 1, file);
    while (buffer[0] != UCHAR_MAX)
    {

        pack_fread(buffer + 1, 7, file);

        if (buffer[6] < UCHAR_MAX && buffer[7] < UCHAR_MAX)
        {
            hive= feature_on_tile(map, buffer[6], buffer[7]);
            if (hive != 0 && hive->type != FEATURE_T_HIVE) hive= 0;
        }
        else hive= 0;

        if (construct_baddie(buffer[0], buffer[1], buffer[2], buffer[3],
                             buffer[4], buffer[5], hive, map) == 0)
        {
            write_to_log(manager.log, 1, "Could not construct baddie.\n");
            destruct_map(map);
            pack_fclose(file);
            return 0;
        }

        pack_fread(buffer, 1, file);

    }


    /* READ THE GOODIE DATA: buffer[0] = goodie type, buffer[1] = map_row,
       buffer[2] = map_col, buffer[3] = heading_x, buffer[4] = heading_y,
       buffer[5] = status */
    pack_fread(buffer, 1, file);
    while (buffer[0] != UCHAR_MAX)
    {
        pack_fread(buffer + 1, 5, file);
        if (construct_goodie(buffer[0], buffer[1], buffer[2], buffer[3],
                             buffer[4], buffer[5], map) == 0)
        {
            write_to_log(manager.log, 1, "Could not construct goodie.\n");
            destruct_map(map);
            pack_fclose(file);
            return 0;
        }
        pack_fread(buffer, 1, file);
    }


    /* CLOSE THE FILE */
    pack_fclose(file);

    return map;

} /* end construct_map_from_file */




Tile *construct_tile(tile_type type, unsigned char anim_index)
{

    Tile *tile= (Tile *) malloc(sizeof(Tile));
    if (tile == 0)
    {
        write_to_log(manager.log, 1, "Could not allocate tile memory.\n");
        return 0;
    }

    setup_tile(tile, type, anim_index);

    return tile;

} /* end construct_tile */




void destruct_map(Map *map)
{

    unsigned char row;
    unsigned char col;
    //unsigned short i;

    List_iterator *iterator;

    /* free the tile memory */
    if (map->tiles != 0)
    {
        for (row= 0; row < map->m_height; row++)
        {
            if (map->tiles[row] != 0)
            {
                for (col= 0; col < map->m_width; col++)
                {
                    /* free map tile */
                    //free_tile(&(map->tiles[row][col]));
                }
                /* free the row */
                free(map->tiles[row]);
            }
        }
        /* free the memory holding the pointers to each row */
        free(map->tiles);
    }

    /* destroy the player */
    if (map->player != 0) destruct_player(map->player);

    #if 0
    /* destroy all the baddies and free the array memory */
    if (map->baddies != 0)
    {
        for (i= 0; i < map->max_num_baddies; i++)
        {
            if (map->baddies[i] != 0) destruct_baddie(map->baddies[i]);
        }
        free(map->baddies);
    }
    #endif

    iterator= get_iterator(map->baddie_list);
    while (iterator != 0)
    {
        destruct_baddie(((Baddie *) iterator->dat));
        iterator= next_iterator(iterator);
    }
    destruct_list(map->baddie_list);

    #if 0
    /* destroy all the features and free the array memory */
    if (map->features != 0)
    {
        for (i= 0; i < map->max_num_features; i++)
        {
            if (map->features[i] != 0) destruct_feature(map->features[i]);
        }
        free(map->features);
    }
    #endif

    iterator= get_iterator(map->feature_list);
    while (iterator != 0)
    {
        destruct_feature(((Feature *) iterator->dat));
        iterator= next_iterator(iterator);
    }
    destruct_list(map->feature_list);

    #if 0
    /* destroy all the goodies and free the array memory */
    if (map->goodies != 0)
    {
        for (i= 0; i < map->max_num_goodies; i++)
        {
            if (map->goodies[i] != 0) destruct_goodie(map->goodies[i]);
        }
        free(map->goodies);
    }
    #endif

    iterator= get_iterator(map->goodie_list);
    while (iterator != 0)
    {
        destruct_goodie(((Goodie *) iterator->dat));
        iterator= next_iterator(iterator);
    }
    destruct_list(map->goodie_list);

    #if 0
    /* destroy all the projectiles and free the array memory */
    if (map->projectiles != 0)
    {
        for (i= 0; i < map->max_num_projectiles; i++)
        {
            if (map->projectiles[i] != 0)
                destruct_projectile(map->projectiles[i]);
        }
        free(map->projectiles);
    }
    #endif

    iterator= get_iterator(map->projectile_list);
    while (iterator != 0)
    {
        destruct_projectile(((Projectile *) iterator->dat));
        iterator= next_iterator(iterator);
    }
    destruct_list(map->projectile_list);

    /* point displays */
    iterator= get_iterator(map->point_list);
    while (iterator != 0)
    {
        destruct_floating_text(((Floating_text *) iterator->dat));
        iterator= next_iterator(iterator);
    }
    destruct_list(map->point_list);

    /* free the map name and author */
    if (map->name != 0) free(map->name);
    if (map->author != 0) free(map->author);

    /* free the map */
    free(map);
    map= 0;

} /* end destruct_map */




void destruct_tile(Tile *tile)
{
    free(tile);
    tile= 0;
}




map_status do_map_logic(Map *const map)
{

    //unsigned short i;

    signed short baddie_target_row;
    signed short baddie_target_col;

    List_iterator *iterator;
    Baddie *baddie;
    Goodie *goodie;
    Feature *feature;
    Projectile *projectile;
    Floating_text *points;

    unsigned char bag_exists;

    /* player */
    if (map->player != 0)
    {
        if (map->time_limit > 0) do_player_logic(map->player, map);
        else if (map->player->status != PLAYER_S_WON
                 && map->player->status != PLAYER_S_DYING
                 && map->player->status != PLAYER_S_KAPUT)
        {
            set_player_status(map->player, PLAYER_S_DYING, map);
        }
        baddie_target_row= map->player->entity->row;
        baddie_target_col= map->player->entity->col;
    }
    else
    {
        baddie_target_row= -1;
        baddie_target_col= -1;
    }

    /* baddies */
    #if 0
    for (i= 0; i < map->max_num_baddies; i++)
    {
        if (map->baddies[i] != 0)
        {
            /* check first if we should clean up this baddie */
            if (map->baddies[i]->entity->status == BADDIE_S_KAPUT
                || map->baddies[i]->entity->row >= map->m_height
                || map->baddies[i]->entity->col >= map->m_width)
            {
                remove_baddie_from_map(map->baddies[i], map);
            }
            /* else if the baddie is still alive, update his logic */
            else
            {
                do_baddie_logic(map->baddies[i], map, baddie_target_row,
                                baddie_target_col);
            }
        }
    }
    #endif

    iterator= get_iterator(map->baddie_list);
    while (iterator != 0)
    {

        baddie= (Baddie *) iterator->dat;
        iterator= next_iterator(iterator);

        /* check first if we should clean up this baddie */
        if (baddie->status == BADDIE_S_KAPUT
            || baddie->entity->row >= map->m_height
            || baddie->entity->col >= map->m_width)
        {
            remove_baddie_from_map(baddie, map);
        }

        /* else if the baddie is still alive, update his logic */
        else
        {
            do_baddie_logic(baddie, map, baddie_target_row, baddie_target_col);
        }

    } /* end baddie logic */

    /* features */
    #if 0
    for (i= 0; i < map->max_num_features; i++)
    {
        if (map->features[i] != 0)
        {
            /* check first if we should clean up this feature */
            if (map->features[i]->entity->status == FEATURE_S_KAPUT
                || map->features[i]->entity->row >= map->m_height
                || map->features[i]->entity->col >= map->m_width)
            {
                remove_feature_from_map(map->features[i], map);
            }
            /* else if the feature is still alive, update its logic */
            else
            {
                do_feature_logic(map->features[i], map);
            }
        }
    }
    #endif

    iterator= get_iterator(map->feature_list);
    while (iterator != 0)
    {

        feature= (Feature*) iterator->dat;
        iterator= next_iterator(iterator);

        /* check first if we should clean up this feature */
        if (feature->status == FEATURE_S_KAPUT
            || feature->entity->row >= map->m_height
            || feature->entity->col >= map->m_width)
        {
            remove_feature_from_map(feature, map);
        }
        /* else if the feature is still alive, update its logic */
        else do_feature_logic(feature, map);

    } /* end feature logic */

    /* goodies */

    #if 0
    for (i= 0; i < map->max_num_goodies; i++)
    {
        if (map->goodies[i] != 0)
        {

            if (map->goodies[i]->type == GOODIE_T_BAG) map->won= 0;

            /* check first if we should clean up this goodie */
            if (map->goodies[i]->entity->status == GOODIE_S_KAPUT
                || map->goodies[i]->entity->row >= map->m_height
                || map->goodies[i]->entity->col >= map->m_width)
            {
                remove_goodie_from_map(map->goodies[i], map);
            }

            else
            {
                do_goodie_logic(map->goodies[i], map);
            }

        }
    }
    #endif

    bag_exists= 0;
    iterator= get_iterator(map->goodie_list);
    while (iterator != 0)
    {

        goodie= (Goodie *) iterator->dat;
        iterator= next_iterator(iterator);

        if (goodie->type == GOODIE_T_BAG) bag_exists= 1;

        /* check first if we should clean up this goodie */
        if (goodie->status == GOODIE_S_KAPUT
            || goodie->entity->row >= map->m_height
            || goodie->entity->col >= map->m_width)
        {
            remove_goodie_from_map(goodie, map);
        }

        else do_goodie_logic(goodie, map);

    } /* end goodie logic */

    /* projectiles */
    #if 0
    for (i= 0; i < map->max_num_projectiles; i++)
    {
        if (map->projectiles[i] != 0)
        {
            /* check first if we should clean up this projectile */
            if (map->projectiles[i]->entity->status == PROJECTILE_S_KAPUT
                || map->projectiles[i]->entity->row >= map->m_height
                || map->projectiles[i]->entity->col >= map->m_width)
            {
                remove_projectile_from_map(map->projectiles[i], map);
            }
            /* else if the projectile is still alive, update its logic */
            else
            {
                do_projectile_logic(map->projectiles[i], map,
                                    baddie_target_row, baddie_target_col);
            }
        }
    }
    #endif

    iterator= get_iterator(map->projectile_list);
    while (iterator != 0)
    {

        projectile= (Projectile *) iterator->dat;
        iterator= next_iterator(iterator);

        /* check first if we should clean up this projectile */
        if (projectile->status == PROJECTILE_S_KAPUT
            || projectile->entity->row >= map->m_height
            || projectile->entity->col >= map->m_width)
        {
            remove_projectile_from_map(projectile, map);
        }
        /* else if the projectile is still alive, update its logic */
        else
        {
            do_projectile_logic(projectile, map, baddie_target_row,
                                baddie_target_col);
        }

    } /* end projectile logic */

    /* point displays */
    iterator= get_iterator(map->point_list);
    while (iterator != 0)
    {

        points= (Floating_text *) iterator->dat;
        iterator= next_iterator(iterator);

        if (update_floating_text(points) > 0)
        {
            remove_from_list(map->point_list, points, 0);
            destruct_floating_text(points);
        }

    } /* end point display logic */

    /* time limit */
    if (map->player != 0 && map->player->status != PLAYER_S_KAPUT
        && map->player->status != PLAYER_S_DYING
        && map->time_limit > 0 && !manager.ignore_time)
    {
        //if (time_counter != manager.beats_counter)
        //{
            --map->time_limit;

        //  time_counter= manager.beats_counter;
        //}
    }

    /* determine the status of the map */
    if (map->player != 0 && manager.editing == 0)
    {
        if (map->player->status == PLAYER_S_KAPUT)
        {
            if (bag_exists == 0) return MAP_S_WON;
            else if (map->time_limit == 0) return MAP_S_TIME_UP;
            else return MAP_S_KILLED;
        }
        else
        {
            if (bag_exists == 0 && map->player->status != PLAYER_S_WON)
                set_player_status(map->player, PLAYER_S_WON, map);
        }
    }

    return MAP_S_PLAYING;

} /* end do_map_logic */




Feature *feature_on_tile(const Map *const map, const unsigned char row,
                         const unsigned char col)
{

    #if 0
    unsigned short i;

    for (i= 0; i < map->max_num_features; i++) {

        if (map->features[i] != 0
            && is_entity_on_tile(map->features[i]->entity, map, row, col, 1, 1))
        {
            return map->features[i];
        }

    }
    #endif

    Feature *feature;
    List_iterator *iterator= get_iterator(map->feature_list);

    while (iterator != 0)
    {
        feature= (Feature *) iterator->dat;
        if (is_entity_on_tile(feature->entity, map, row, col, 1, 1))
            return feature;
        iterator= next_iterator(iterator);
    }

    /* no feature found */
    return 0;

} /* end feature_on_tile */



#if 0
void free_tile(type_tile *tile)
{
    //free(tile->anim_frames);
}
#endif



Map *get_map(const unsigned short map_num)
{

    Map *map;
    char *filename;

    /* "+ 7" follows from the fact we need 5 digits to store an unsigned short,
       plus one for the magic datafile character "#", plus one for the null
       character */
    filename= (char *) malloc(sizeof(char) * (strlen(MAP_DAT_FILENAME) + 7));

    sprintf(filename, "%s#%i", MAP_DAT_FILENAME, map_num);

    map= construct_map_from_file(filename, 0);

    free(filename);

    return map;

} /* end get_map */




Goodie *goodie_on_tile(const Map *const map, const unsigned char row,
                       const unsigned char col)
{

    #if 0
    unsigned short i;

    for (i= 0; i < map->max_num_goodies; i++) {

        if (map->goodies[i] != 0
            && is_entity_on_tile(map->goodies[i]->entity, map, row, col, 1, 1))
        {
            return map->goodies[i];
        }

    }
    #endif

    Goodie *goodie;
    List_iterator *iterator= get_iterator(map->goodie_list);

    while (iterator != 0)
    {
        goodie= (Goodie *) iterator->dat;
        if (is_entity_on_tile(goodie->entity, map, row, col, 1, 1))
            return goodie;
        iterator= next_iterator(iterator);
    }

    /* no goodie found */
    return 0;

} /* end goodie_on_tile */




unsigned char is_tile_empty_baddie(const baddie_type type,
                                   const baddie_status status,
                                   const Map *const map,
                                   const unsigned char row,
                                   const unsigned char col)
{

    Baddie *baddie;
    Feature *feature;
    Goodie *goodie;

    /* check map bounds */
    if (row >= map->m_height || col >= map->m_width) return 0;

    /* check the tile */
    if (baddie_tile_coll(type, status, map, row, col))
        return 0;

    #if 0
    /* check for player */
    if (player_on_tile(map, row, col, COLL_FUDGE_X, COLL_FUDGE_Y) != 0)
    {
        return 0;
    }
    #endif

    /* check for baddies */
    baddie= baddie_on_tile(map, row, col, 1, 1);
    if (baddie != 0 && baddie_baddie_coll(type, status, baddie->type,
                                          baddie->status))
    {
        return 0;
    }

    /* check for features */
    feature= feature_on_tile(map, row, col);
    if (feature != 0 && baddie_feature_coll(type, status, feature->type,
                                            feature->status))
    {
        return 0;
    }

    /* check for goodies */
    goodie= goodie_on_tile(map, row, col);
    if (goodie != 0 && baddie_goodie_coll(type, status, goodie->type,
                                          goodie->status))
    {
        return 0;
    }

    /* empty */
    return 1;

} /* end is_tile_empty_baddie */




unsigned char is_tile_empty_feature(const feature_type f_type,
                                    const Map *const map,
                                    const unsigned char row,
                                    const unsigned char col)
{

    Player *player;
    Baddie *baddie;
    Feature *feature;
    Goodie *goodie;

    /* check map bounds */
    if (row >= map->m_height || col >= map->m_width) return 0;

    /* check the tile */
    if (feature_tile_coll(f_type, map->tiles[row][col].type)) return 0;

    /* check for player */
    player= player_on_tile(map, row, col, 1, 1);
    if (player != 0 && feature_player_coll(f_type,
                                           player->status))
    {
        return 0;
    }

    /* check for baddies */
    baddie= baddie_on_tile(map, row, col, 1, 1);
    if (baddie != 0 && feature_baddie_coll(f_type, baddie->type,
                                           baddie->status))
    {
        return 0;
    }

    /* check for features */
    feature= feature_on_tile(map, row, col);
    if (feature != 0 && feature_feature_coll(f_type, feature->type))
        return 0;

    /* check for goodies */
    goodie= goodie_on_tile(map, row, col);
    if (goodie != 0 && feature_goodie_coll(f_type, goodie->type,
                                           goodie->status))
    {
        return 0;
    }

    /* empty */
    return 1;

} /* end is_tile_empty_feature */




unsigned char is_tile_empty_goodie(const goodie_type g_type,
                                   const Map *const map,
                                   const unsigned char row,
                                   const unsigned char col)
{

    Player *player;
    Baddie *baddie;
    Feature *feature;
    Goodie *goodie;

    /* check map bounds */
    if (row >= map->m_height || col >= map->m_width) return 0;

    /* check the tile */
    if (goodie_tile_coll(g_type, map->tiles[row][col].type)) return 0;

    /* check for player */
    player= player_on_tile(map, row, col, 1, 1);
    if (player != 0 && goodie_player_coll(g_type, player->status))
        return 0;

    /* check for baddies */
    baddie= baddie_on_tile(map, row, col, map->t_width, map->t_height);
    if (baddie != 0 && goodie_baddie_coll(g_type, baddie->type,
                                          baddie->status))
    {
        return 0;
    }

    /* check for features */
    feature= feature_on_tile(map, row, col);
    if (feature != 0 && goodie_feature_coll(g_type, feature->type,
                                            feature->status))
    {
        return 0;
    }

    /* check for goodies */
    goodie= goodie_on_tile(map, row, col);
    if (goodie != 0 && goodie_goodie_coll(g_type, goodie->type)) return 0;

    /* empty */
    return 1;

} /* end is_tile_empty_goodie */




unsigned char is_tile_empty_player(const player_status p_status,
                                   const Map *const map,
                                   const unsigned char row,
                                   const unsigned char col)
{

    Baddie *baddie;
    Feature *feature;

    /* check map bounds */
    if (row >= map->m_height || col >= map->m_width) return 0;

    /* check the tile */
    if (player_tile_coll(p_status, map->tiles[row][col].type)) return 0;

    /* check for baddies */
    baddie= baddie_on_tile(map, row, col, COLL_FUDGE_X, COLL_FUDGE_Y);
    if (baddie != 0
        && player_baddie_coll(p_status, baddie->type, baddie->status))
    {
        return 0;
    }

    /* check for features */
    feature= feature_on_tile(map, row, col);
    if (feature != 0
        && player_feature_coll(p_status, feature->type, feature->status))
    {
        return 0;
    }

    /* empty */
    return 1;

} /* end is_tile_empty_player */




void make_new_ice(Map *const map, const unsigned char row,
                  const unsigned char col)
{

    /* destruct the map tile currently at the location, and replace it with a
       new ice tile */

    //free_tile(&(map->tiles[row][col]));
    // fix this!  don't rely on 1!
    setup_tile(&(map->tiles[row][col]), TILE_T_ICE, SPRITE_ICE);

    /* set the water sprites around the ice tile */

    /* right */
    if (col < map->m_width - 1 && map->tiles[row][col + 1].type == TILE_T_WATER)
    {

        if (row > 0 && map->tiles[row - 1][col + 1].type == TILE_T_ICE)
            setup_tile(&(map->tiles[row][col + 1]), TILE_T_WATER, SPRITE_L_U);
        else if (row > 0 && map->tiles[row - 1][col].type == TILE_T_ICE)
            setup_tile(&(map->tiles[row][col + 1]), TILE_T_WATER, SPRITE_L_UL);
        else
            setup_tile(&(map->tiles[row][col + 1]), TILE_T_WATER, SPRITE_L);
    }

    /* down */
    if (row < map->m_height - 1
        && map->tiles[row + 1][col].type == TILE_T_WATER)
    {
        if (col > 0 && map->tiles[row + 1][col - 1].type == TILE_T_ICE)
            setup_tile(&(map->tiles[row + 1][col]), TILE_T_WATER, SPRITE_L_U);
        else if (col > 0 && map->tiles[row][col - 1].type == TILE_T_ICE)
            setup_tile(&(map->tiles[row + 1][col]), TILE_T_WATER, SPRITE_UL_U);
        else
            setup_tile(&(map->tiles[row + 1][col]), TILE_T_WATER, SPRITE_U);
    }

    /* down right */
    if (col < map->m_width - 1 && row < map->m_height - 1
        && map->tiles[row + 1][col + 1].type == TILE_T_WATER)
    {
        if (map->tiles[row][col + 1].type != TILE_T_ICE
            || map->tiles[row + 1][col].type != TILE_T_ICE)
        {
            if (map->tiles[row][col + 1].type == TILE_T_ICE)
            {
                setup_tile(&(map->tiles[row + 1][col + 1]), TILE_T_WATER,
                           SPRITE_UL_U);
            }
            else if (map->tiles[row + 1][col].type == TILE_T_ICE)
            {
                setup_tile(&(map->tiles[row + 1][col + 1]), TILE_T_WATER,
                           SPRITE_L_UL);
            }
            else
            {
                setup_tile(&(map->tiles[row + 1][col + 1]), TILE_T_WATER,
                           SPRITE_UL);
            }
        }
    }

} /* end make_new_ice */




map_status play_map(Map *const map)
{

    unsigned char draw_needed;

    map_status status;

    unsigned char left;
    unsigned char right;
    unsigned char up;
    unsigned char down;
    unsigned char spin;

    //unsigned short time_counter;

    write_to_log(manager.log, 0, "Playing map.\n");

    manager.editing= 0;

    draw_needed= 1;

    manager.fps_aggregate= 0;
    manager.fps_counter= 0;
    manager.fps_average= 0.0;
    manager.logic_counter= 1;
    manager.beats_counter= 0;

    //time_counter= 0;

    status= MAP_S_PLAYING;
    while (status == MAP_S_PLAYING)
    {

        while (manager.logic_counter > 0 && status == MAP_S_PLAYING)
        {

            draw_needed= 1;

            /* poll controls should be outside the player null test so that user
               is allowed to pres Esc even if the map has no player */
            poll_controls();

            if (map->player != 0
                && map->player->status != PLAYER_S_DYING
                && map->player->status != PLAYER_S_WON
                && map->player->status != PLAYER_S_KAPUT)
            {

                if (controls_left(&(manager.controls))) left= 1;
                else left= 0;

                if (controls_right(&(manager.controls))) right= 1;
                else right= 0;

                if (controls_down(&(manager.controls))) down= 1;
                else down= 0;

                if (controls_up(&(manager.controls))) up= 1;
                else up= 0;

                if (controls_spin(&(manager.controls))) spin= 1;
                else spin= 0;

                if (left)
                {
                    if (spin && map->player->entity->c_speed == 0)
                        set_heading(map->player->entity, -1, 0);
                    else move_player_left(map->player, map);
                }
                if (right)
                {
                    if (spin && map->player->entity->c_speed == 0)
                        set_heading(map->player->entity, 1, 0);
                    else move_player_right(map->player, map);
                }
                if (up)
                {
                    if (spin && map->player->entity->c_speed == 0)
                        set_heading(map->player->entity, 0, -1);
                    else move_player_up(map->player, map);
                }
                if (down)
                {
                    if (spin && map->player->entity->c_speed == 0)
                        set_heading(map->player->entity, 0, 1);
                    else move_player_down(map->player, map);
                }

                if (map->player->entity->c_speed > 0
                    && !left && !right && !up && !down)
                {
                    set_player_status(map->player, PLAYER_S_STOPPED, map);
                }

                if (controls_breath(&(manager.controls)))
                    do_breath(map->player, map);

                if (controls_pillar(&(manager.controls)))
                    do_pillar(map->player, map);

            } /* end if player exists and is alive */

            //animate_map(map);

            status= do_map_logic(map);

            animate_map(map);

            /* check for user exit */
            if (key[KEY_ESC] && status == MAP_S_PLAYING
                && (map->player == 0
                    || (map->player->status != PLAYER_S_WON
                        && map->player->status != PLAYER_S_DYING
                        && map->player->status != PLAYER_S_KAPUT)))
            {
                status= MAP_S_QUIT;
            }

            --manager.logic_counter;

        } /* end logic update */

        yield_timeslice();
        //animate_map(map, sprite_datafile);

        if (draw_needed) draw_map(map);
        if (draw_needed || manager.draw_always)
        {
            draw_to_screen(0);
            draw_needed= 0;
        }

    } /* end while map is being played */

    write_to_log(manager.log, 0,
                 "Playing map paused or completed.  Average FPS:  %f.\n",
                 manager.fps_average);

    return status;

} /* end play_map */




Player *player_on_tile(const Map *const map, const unsigned char row,
                       const unsigned char col, const unsigned char fudge_x,
                       const unsigned char fudge_y)
{

    if (map->player != 0
        && is_entity_on_tile(map->player->entity, map, row, col, fudge_x,
                             fudge_y))
    {
        return map->player;
    }

    /* no player found */
    return 0;

} /* end player_on_tile */




Projectile *projectile_on_tile(const Map *const map, const unsigned char row,
                               const unsigned char col,
                               const unsigned char fudge_x,
                               const unsigned char fudge_y)
{

    #if 0
    unsigned short i;

    for (i= 0; i < map->max_num_projectiles; i++) {

        if (map->projectiles[i] != 0
            && is_entity_on_tile(map->projectiles[i]->entity, map, row, col,
                                 fudge_x, fudge_y))
        {
            return map->projectiles[i];
        }

    }
    #endif

    Projectile *projectile;
    List_iterator *iterator= get_iterator(map->projectile_list);

    while (iterator != 0)
    {
        projectile= (Projectile *) iterator->dat;
        if (is_entity_on_tile(projectile->entity, map, row, col, fudge_x,
                              fudge_y))
        {
            return projectile;
        }
        iterator= next_iterator(iterator);
    }

    /* no projectile found */
    return 0;

} /* end projectile_on_tile */




void remove_baddie_from_map(Baddie *baddie, Map *const map)
{

    //unsigned short i;

    /* if baddie was spawned from hive, spawn another baddie */
    if (baddie->hive != 0) notify_hive(baddie->hive, baddie->type);

    #if 0
    /* remove the baddie from the baddies array */
    for (i= 0; i < map->max_num_baddies && map->baddies[i] != baddie; i++);
    if (i < map->max_num_baddies) map->baddies[i]= 0;
    #endif

    remove_from_list(map->baddie_list, baddie, 0);

    /* destroy baddie */
    destruct_baddie(baddie);

}




void remove_feature_from_map(Feature *feature, Map *const map)
{

    //unsigned short i;

    List_iterator *iterator;
    Baddie *baddie;

    #if 0
    /* remove the feature from the features array */
    for (i= 0; i < map->max_num_features && map->features[i] != feature; i++);
    if (i < map->max_num_features)
    {

        map->features[i]= 0;

        /* if the feature is a hive, disconnect the hive's baddies from the
           hive */
        if (feature->type == FEATURE_T_HIVE)
        {
    #endif

            #if 0
            for (i= 0; i < map->max_num_baddies; i++)
            {
                if (map->baddies[i] != 0 && map->baddies[i]->hive == feature)
                    map->baddies[i]->hive= 0;
            }
            #endif

    #if 0
            iterator= get_iterator(map->baddie_list);
            while (iterator != 0)
            {
                baddie= (type_baddie *) iterator->dat;
                if (baddie->hive == feature) baddie->hive= 0;
                iterator= next_iterator(iterator);
            }

        }

    }
    #endif

    /* if feature is a hive, null out the hives of any baddies who were spawned
       from it */
    if (feature->type == FEATURE_T_HIVE)
    {
        iterator= get_iterator(map->baddie_list);
        while (iterator != 0)
        {
            baddie= (Baddie *) iterator->dat;
            if (baddie->hive == feature) baddie->hive= 0;
            iterator= next_iterator(iterator);
        }
    }

    remove_from_list(map->feature_list, feature, 0);

    /* destroy feature */
    destruct_feature(feature);

} /* end remove_feature_from_map */




void remove_goodie_from_map(Goodie *goodie, Map *const map)
{

    #if 0
    unsigned short i;

    /* remove the goodie from the goodies array */
    for (i= 0; i < map->max_num_goodies && map->goodies[i] != goodie; i++);
    if (i < map->max_num_goodies) map->goodies[i]= 0;
    #endif

    remove_from_list(map->goodie_list, goodie, 0);

    /* destroy goodie */
    destruct_goodie(goodie);

} /* end remove_goodie_from_map */




void remove_player_from_map(Map *const map)
{
    destruct_player(map->player);
    map->player= 0;
}




void remove_projectile_from_map(Projectile *projectile, Map *const map)
{

    #if 0
    unsigned short i;

    /* remove the projectile from the projectiles array */
    for (i= 0; i < map->max_num_projectiles
               && map->projectiles[i] != projectile; i++);
    if (i < map->max_num_projectiles) map->projectiles[i]= 0;
    #endif

    remove_from_list(map->projectile_list, projectile, 0);

    /* destroy projectile */
    destruct_projectile(projectile);

} /* end remove_projectile_from_map */



#if 0
int set_map_tile_props(int msg, DIALOG *d, int c)
{

    int result= d_button_proc(msg, d, c);

    ((type_map_tile *) d->dp2)->type=
        (unsigned char) map_tile_props_dlg[MAP_TILE_TYPES_PULLDOWN].d1;

    return result;

}
#endif




void setup_tile(Tile *const tile, tile_type type, unsigned char anim_index)
{

    tile->type= type;
    tile->anim_index= anim_index;
    tile->anim_counter= 0;

    /* animate tile one frame */
    animate_tile(tile);

}




unsigned char write_map_to_file(const Map *const map, const char *filename,
                                unsigned char compress)
{

    unsigned char map_str_len;

    unsigned char row;
    unsigned char col;

    //unsigned short i;

    List_iterator *iterator;
    Baddie *baddie;
    Goodie *goodie;
    Feature *feature;

    write_to_log(manager.log, 0, "Writing map to file \"%s\".\n", filename);

    /* OPEN THE FILE */
    PACKFILE *file= pack_fopen(filename, ( compress ? "wp" : "w" ));

    if (file == 0)
    {
        write_to_log(manager.log, 1,
                     "Could not open file \"%s\" for writing.  %s.",
                     strerror(errno));
        return 1;
    }

    /* WRITE THE HEADER */

    /* file ID */
    pack_fwrite("KiCkLeMaPfIlE", 13, file);

    write_to_log(manager.log, 0, "Wrote header.\n");

    /* map and tile dimensions */
    pack_putc(map->m_width, file);
    pack_putc(map->m_height, file);
    pack_putc(map->t_width, file);
    pack_putc(map->t_height, file);

    /* map name */
    if (map->name == 0) pack_putc(0, file);
    else
    {
        map_str_len= strlen(map->name);
        pack_putc(map_str_len, file);
        pack_fwrite(map->name, sizeof(char) * map_str_len, file);
    }

    /* map author */
    if (map->author == 0) pack_putc(0, file);
    else
    {
        map_str_len= strlen(map->author);
        pack_putc(map_str_len, file);
        pack_fwrite(map->author, sizeof(char) * map_str_len, file);
    }

    /* time limit */
    pack_fwrite(&(map->time_limit), sizeof(unsigned short), file);

    /* WRITE THE TILE DATA */
    for (row= 0; row < map->m_height; row++)
    {
        for (col= 0; col < map->m_width; col++)
        {
            pack_putc(map->tiles[row][col].type, file);
            pack_putc(map->tiles[row][col].anim_index, file);
        }
    }

    /* WRITE THE PLAYER DATA */
    if (map->player == 0) pack_putc(UCHAR_MAX, file);
    else {
        pack_putc(map->player->entity->row, file);
        pack_putc(map->player->entity->col, file);
        pack_putc(map->player->entity->heading_x, file);
        pack_putc(map->player->entity->heading_y, file);
        pack_putc(map->player->status, file);
    }

    /* WRITE THE FEATURE DATA */
    #if 0
    for (i= 0; i < map->max_num_features; i++)
    {
        if (map->features[i] != 0)
        {
            pack_putc(map->features[i]->type, file);
            pack_putc(map->features[i]->entity->row, file);
            pack_putc(map->features[i]->entity->col, file);
            pack_putc(map->features[i]->entity->heading_x, file);
            pack_putc(map->features[i]->entity->heading_y, file);
            pack_putc(map->features[i]->entity->status, file);
        }
    }
    pack_putc(UCHAR_MAX, file);
    #endif

    iterator= get_iterator(map->feature_list);
    while (iterator != 0)
    {

        feature= (Feature *) iterator->dat;

        pack_putc(feature->type, file);
        pack_putc(feature->entity->row, file);
        pack_putc(feature->entity->col, file);
        pack_putc(feature->entity->heading_x, file);
        pack_putc(feature->entity->heading_y, file);
        pack_putc(feature->status, file);

        iterator= next_iterator(iterator);

    }
    pack_putc(UCHAR_MAX, file);

    /* WRITE THE BADDIE DATA */
    #if 0
    for (i= 0; i < map->max_num_baddies; i++)
    {
        if (map->baddies[i] != 0)
        {

            pack_putc(map->baddies[i]->type, file);
            pack_putc(map->baddies[i]->entity->row, file);
            pack_putc(map->baddies[i]->entity->col, file);
            pack_putc(map->baddies[i]->entity->heading_x, file);
            pack_putc(map->baddies[i]->entity->heading_y, file);
            pack_putc(map->baddies[i]->entity->status, file);

            if (map->baddies[i]->hive == 0)
            {
                pack_putc(UCHAR_MAX, file);
                pack_putc(UCHAR_MAX, file);
            }
            else
            {
                pack_putc(map->baddies[i]->hive->entity->row, file);
                pack_putc(map->baddies[i]->hive->entity->col, file);
            }

        }
    }
    pack_putc(UCHAR_MAX, file);
    #endif

    iterator= get_iterator(map->baddie_list);
    while (iterator != 0)
    {

        baddie= (Baddie *) iterator->dat;

        pack_putc(baddie->type, file);
        pack_putc(baddie->entity->row, file);
        pack_putc(baddie->entity->col, file);
        pack_putc(baddie->entity->heading_x, file);
        pack_putc(baddie->entity->heading_y, file);
        pack_putc(baddie->status, file);

        if (baddie->hive == 0)
        {
            pack_putc(UCHAR_MAX, file);
            pack_putc(UCHAR_MAX, file);
        }
        else
        {
            pack_putc(baddie->hive->entity->row, file);
            pack_putc(baddie->hive->entity->col, file);
        }

        iterator= next_iterator(iterator);

    }
    pack_putc(UCHAR_MAX, file);

    /* WRITE THE GOODIE DATA */
    #if 0
    for (i= 0; i < map->max_num_goodies; i++)
    {
        if (map->goodies[i] != 0)
        {
            pack_putc(map->goodies[i]->type, file);
            pack_putc(map->goodies[i]->entity->row, file);
            pack_putc(map->goodies[i]->entity->col, file);
            pack_putc(map->goodies[i]->entity->heading_x, file);
            pack_putc(map->goodies[i]->entity->heading_y, file);
            pack_putc(map->goodies[i]->entity->status, file);
        }
    }
    pack_putc(UCHAR_MAX, file);
    #endif

    iterator= get_iterator(map->goodie_list);
    while (iterator != 0)
    {

        goodie= (Goodie *) iterator->dat;

        pack_putc(goodie->type, file);
        pack_putc(goodie->entity->row, file);
        pack_putc(goodie->entity->col, file);
        pack_putc(goodie->entity->heading_x, file);
        pack_putc(goodie->entity->heading_y, file);
        pack_putc(goodie->status, file);

        iterator= next_iterator(iterator);

    }
    pack_putc(UCHAR_MAX, file);

    /* CLOSE THE FILE */
    pack_fclose(file);

    return 0;

} /* end write_map_to_file */
