/************************************
 *
 * Kickle 0.81
 * 2004 Drew Willcoxon
 * http://www.cs.uga.edu/~adw/
 * dripfeed@uga.edu
 *
 ************************************/


#include <stdlib.h>
#include <time.h>


#include <allegro.h>

#include "controls.h"
#include "common.h"
#include "config.h"
#include "gmenus.h"
#include "log.h"
#include "manager.h"
#include "map.h"
#include "menu.h"
#include "video.h"





/* log file filename */
#define LOG_FILENAME        "log.txt"



/* the game's manager */
Manager manager;




/* game statuses */
typedef enum
{
    GAME_S_RUNNING,
    GAME_S_DIED,
    GAME_S_WON,
    GAME_S_QUIT
} game_status;




void cleanup(void);


void do_game_over(const unsigned int player_score);


void do_help(void);


void do_won(const unsigned int player_score);


unsigned char init_kickle(void);


int main(int argc, char *argv[]);


char *os_type_str(void);


void run_maps(void);


void update_counters(void);




void cleanup(void)
{

    write_to_log(manager.log, 0, "\nPreparing to shut down.\n");

    uninit_video();

    write_to_log(manager.log, 0, "Unloading Allegro.\n");
    allegro_exit();

    write_to_log(manager.log, 0, "\nClosing log.\n");
    close_log(manager.log);

} /* end cleanup */




void do_game_over(const unsigned int player_score)
{
    show_game_over(player_score);
    clear_keybuf();
    readkey();
    do_box_in(5);
}




void do_help(void)
{
    show_help1();
    clear_keybuf();
    readkey();
    show_help2();
    clear_keybuf();
    readkey();
    show_help3();
    clear_keybuf();
    readkey();
    show_help4();
    clear_keybuf();
    readkey();
    do_box_in(5);
}




void do_won(const unsigned int player_score)
{
    show_won1();
    clear_keybuf();
    readkey();
    show_won2(player_score);
    clear_keybuf();
    readkey();
    do_box_in(5);
}




unsigned char init_kickle(void)
{

    time_t timet;

    manager.sprite_datafile= 0;

    /* open the log */
    manager.log= open_log(LOG_FILENAME);
    if (manager.log == 0)
    {
        fprintf(stderr, "%s %s:  log could not be created.\n\n", PROG_NAME,
                PROG_VER);
    }

    timet= time(0);

    write_to_log(manager.log, 0, "%s %s Log\n%s\nInitializing game.\n\n",
                 PROG_NAME, PROG_VER, asctime(localtime(&timet)));

    /* initialize Allegro */
    write_to_log(manager.log, 0, "Initializing Allegro.\n");
    allegro_init();

    if (manager.log == 0)
    {
        allegro_message("%s %s:  log could not be created.\n", PROG_NAME,
                        PROG_VER);
    }

    /* write some system info */
    write_to_log(manager.log, 0, "%s\n", allegro_id);

    write_to_log(manager.log, 0, "%s %i.%i\n\n", os_type_str(), os_version,
                 os_revision);

    /* load the configuration file */
    load_config(CONFIG_FILENAME, &manager);

    /* install the keyboard */
    write_to_log(manager.log, 0, "\nInstalling keyboard.\n");
    if (install_keyboard() < 0)
    {
        write_to_log(manager.log, 1, "Unable to install keyboard.\n");
        return 1;
    }

    /* install the timer */
    write_to_log(manager.log, 0, "Installing timer.\n");
    if (install_timer() < 0)
    {
        write_to_log(manager.log, 1, "Unable to install timer.\n");
        close_log(manager.log);
        return 1;
    }

    /* install the mouse */
    write_to_log(manager.log, 0, "Installing mouse.\n");
    if (install_mouse() < 0)
    {
        write_to_log(manager.log, 1,
                     "Unable to install mouse.  Map editor cannot be used.\n");
    }
    show_mouse(0);

    /* initialize video components */
    if (init_video(320, 240) > 0)
    {
        write_to_log(manager.log, 1, "Could not initialize video.\n");
        return 1;
    }

    /* initialize controls */
    init_controls();

    LOCK_VARIABLE(manager.logic_counter);
    LOCK_VARIABLE(manager.fps_aggregate);
    LOCK_VARIABLE(manager.fps_counter);
    LOCK_VARIABLE(manager.fps_average);
    LOCK_VARIABLE(manager.beats_counter);
    LOCK_VARIABLE(manager.bps);
    LOCK_FUNCTION(increment_logic_counter);

    /* timer interrupt */
    write_to_log(manager.log, 0, "Installing timer interrupt.\n");
    if (install_int_ex(update_counters, BPS_TO_TIMER(manager.bps)) < 0)
    {
        write_to_log(manager.log, 1, "Unable to install timer interrupt.\n");
        return 1;
    }

    /* map datafile */
    manager.map_datafile= load_datafile(MAP_DAT_FILENAME);
    if (manager.map_datafile == 0)
    {
        write_to_log(manager.log, 1,
                     "Unable to load map data.  Attempted to load file \"%s\".\n",
                     MAP_DAT_FILENAME);
        return 1;
    }

    srand(time(0));

    /* GUI */
    gui_fg_color= 0;
    gui_bg_color= 255;
    gui_mouse_focus= 1;

    manager.logic_counter= 0;
    manager.fps_aggregate= 0;
    manager.fps_counter= 0;
    manager.fps_average= 0;
    manager.beats_counter= 0;

    write_to_log(manager.log, 0, "\nInitialization complete.\n");

    return 0;

}




int main(int argc, char *argv[])
{

    unsigned char running;


    /* initialize the game */
    if (init_kickle() > 0)
    {
        write_to_log(manager.log, 1,
                     "Could not initialize game.\n\nTerminating game.\n");
        allegro_message(
            "%s %s:  There were errors initializing the game.\nAs a result, the game cannot be started.\nSee log for details.", PROG_NAME,
            PROG_VER);
        cleanup();
        return 1;
    }


    running= 1;
    while (running)
    {

        switch (do_title(&manager))
        {

            case TITLE_PLAY:
                run_maps();
                break;

            case TITLE_HELP:
                do_help();
                break;

            case TITLE_VIDEO:
            case TITLE_GAMEPLAY:
            case TITLE_KEYBOARD:
                if (apply_configs(CONFIG_FILENAME, &manager) > 0)
                {
                    write_to_log(manager.log, 1,
                        "Unable to set graphics mode.  Game cannot be run.\n");
                    allegro_message(
                        "%s %s:  Unable to set graphics mode.\nAs a result, the game cannot be started.\nSee log for details.\n",
                        PROG_NAME, PROG_VER);
                    running= 0;
                }
                break;

            case TITLE_EXIT:
                running= 0;
                break;

            case TITLE_ERROR:
                write_to_log(manager.log, 1,
                             "There was an error running the title menu.\n");
                allegro_message(
                    "%s %s:  There was an error running the title menu.\nAs a result, the game cannot be started.\nSee log for details.\n",
                    PROG_NAME, PROG_VER);
                running= 0;
                break;

            default:
                break;

        }

    } /* end while running game */


    write_to_log(manager.log, 0, "\nExiting game.\n");

    cleanup();

    return 0;

}
END_OF_MAIN();




/**
 * Straight from the Allegro docs.
 */
char *os_type_str(void)
{
    switch (os_type)
    {
        case OSTYPE_UNKNOWN:
            return "unknown, or regular MSDOS";
            break;
        case OSTYPE_WIN3:
            return "Windows 3.1 or earlier";
            break;
        case OSTYPE_WIN95:
            return "Windows 95";
            break;
        case OSTYPE_WIN98:
            return "Windows 98";
            break;
        case OSTYPE_WINME:
            return "Windows ME";
            break;
        case OSTYPE_WINNT:
            return "Windows NT";
            break;
        case OSTYPE_WIN2000:
            return "Windows 2000";
            break;
        case OSTYPE_WINXP:
            return "Windows XP";
            break;
        case OSTYPE_OS2:
            return "OS/2";
            break;
        case OSTYPE_WARP:
            return "OS/2 Warp 3";
            break;
        case OSTYPE_DOSEMU:
            return "Linux DOSEMU";
            break;
        case OSTYPE_OPENDOS:
            return "Caldera OpenDOS";
            break;
        case OSTYPE_LINUX:
            return "Linux";
            break;
        case OSTYPE_SUNOS:
            return "SunOS/Solaris";
            break;
        case OSTYPE_FREEBSD:
            return "FreeBSD";
            break;
        case OSTYPE_NETBSD:
            return "NetBSD";
            break;
        case OSTYPE_IRIX:
            return "IRIX";
            break;
        case OSTYPE_QNX:
            return "QNX";
            break;
        case OSTYPE_UNIX:
            return "Unknown Unix variant";
            break;
        case OSTYPE_BEOS:
            return "BeOS";
            break;
        case OSTYPE_MACOS:
            return "MacOS";
            break;
    }
    return "Unknown";
}




void run_maps(void)
{

    Map *map_orig;
    Map *map_play;
    map_status m_status;

    game_status g_status;
    unsigned short map_num;

    unsigned char running_map;

    unsigned int total_score;
    unsigned char num_lives;

    map_num= 0;
    total_score= 0;
    num_lives= 3;

    /* run all the maps of the game */
    g_status= GAME_S_RUNNING;
    while (g_status == GAME_S_RUNNING)
    {

        map_orig= get_map(map_num);

        /* run one map while player keeps dying and has lives left */
        do
        {

            map_play= clone_map(map_orig);

            /* set the player's score and number of lives to the totals */
            map_play->player->score= total_score;
            map_play->player->num_lives= num_lives;

            /* check for escape into the pause menu */
            running_map= 1;
            while (running_map)
            {

                m_status= play_map(map_play);

                /* if calling up the pause menu */
                if (m_status == MAP_S_QUIT)
                {

                    switch (do_pause_menu(&manager, map_play))
                    {
                        case TITLE_EXIT:
                            running_map= 0;
                            break;
                        case TITLE_ERROR:
                            running_map= 0;
                            m_status= MAP_S_QUIT;
                            break;
                        default:
                            break;
                    }

                } /* end if calling up the pause menu */

                else running_map= 0;

            } /* end loop for handling pause menu */

            /* update score and lives totals from the map just played */
            total_score= map_play->player->score;
            num_lives= map_play->player->num_lives;

            /* if player died, decrease number of lives */
            if (m_status == MAP_S_TIME_UP || m_status == MAP_S_KILLED)
                --num_lives;

            destruct_map(map_play);

            do_box_in(5);

        } while ((m_status == MAP_S_TIME_UP || m_status == MAP_S_KILLED)
                 && num_lives > 0);

        destruct_map(map_orig);

        /* decide the status of the game based on the map's status */
        switch (m_status)
        {

            case MAP_S_TIME_UP:
            case MAP_S_KILLED:
                g_status= GAME_S_DIED;
                break;

            case MAP_S_QUIT:
                g_status= GAME_S_QUIT;
                break;

            case MAP_S_WON:
                ++map_num;
                if (map_num >= NUM_MAPS) g_status= GAME_S_WON;
                break;

            default:
                break;

        } /* end switch on map status */

    } /* end while running game */

    /* here, we've run our entire set of maps, and the game's over */

    /* decide how to proceed based on the game's status */
    switch (g_status)
    {

        case GAME_S_WON:
            do_won(total_score);
            break;

        case GAME_S_DIED:
            do_game_over(total_score);
            break;

        default:
            break;

    }

} /* end run_maps */




void update_counters(void)
{

    ++manager.logic_counter;

    ++manager.beats_counter;
    if (manager.beats_counter >= manager.bps)
    {
        manager.beats_counter= 0;
        manager.fps_aggregate= manager.fps_counter;
        if (manager.fps_average == 0) manager.fps_average= manager.fps_counter;
        else
        {
            manager.fps_average=
                (manager.fps_average + manager.fps_counter) / 2;
        }
        manager.fps_counter= 0;
    }
    //else ++manager.beats_counter;

}
END_OF_FUNCTION(update_counters);
