/************************************
 *
 * Kickle 0.81
 * 2004 Drew Willcoxon
 * http://www.cs.uga.edu/~adw/
 * dripfeed@uga.edu
 *
 ************************************/


#include <stdio.h>
#include <stdlib.h>

#include "entity.h"
#include "goodie.h"
#include "log.h"
#include "manager.h"
#include "map.h"
#include "video.h"




/* the game's log file */
//extern type_log *manager.log;

extern Manager manager;


/* so that collect_goodie can center the goodie's floating text nicely */
extern FONT *font_ingame;




Goodie *clone_goodie(const Goodie *const goodie, Map *const map,
                     const unsigned char row, const unsigned char col)
{

    Goodie *clone=
        construct_goodie(goodie->type, row, col, goodie->entity->heading_x,
                         goodie->entity->heading_y, goodie->status,
                         map);

    if (clone == 0)
    {
        write_to_log(manager.log, 1, "Could not construct goodie clone.\n");
        return 0;
    }

    return clone;

} /* end clone_goodie */




Goodie *construct_goodie(const goodie_type type, const unsigned char row,
                         const unsigned char col, const signed char heading_x,
                         const signed char heading_y,
                         const goodie_status status, Map *const map)
{

    unsigned char i;

    /* allocate goodie and entity memory */
    Goodie *goodie= (Goodie *) malloc(sizeof(Goodie));

    if (goodie == 0)
    {
        write_to_log(manager.log, 1, "Could not allocate goodie memory.\n");
        return 0;
    }

    goodie->entity= construct_entity();
    if (goodie->entity == 0)
    {
        write_to_log(manager.log, 1, "Could not construct goodie's entity.\n");
        destruct_goodie(goodie);
        return 0;
    }

    /* set entity members */
    goodie->entity->heading_x= heading_x;
    goodie->entity->heading_y= heading_y;

    /* set goodie members */
    goodie->type= type;
    goodie->status= status;

    /* set goodie members dependent on type */
    switch (type)
    {

        case GOODIE_T_BAG:
            goodie->points= 1000;
            break;

        case GOODIE_T_POPSICLE:
            goodie->points= 3200;
            break;

        case GOODIE_T_1UP:
            goodie->points= 0;
            break;

    }

    /* animate goodie one frame */
    animate_goodie(goodie);

    /* set goodie's bounding box and add it to map */
    if (map != 0)
    {

        /* set normal speed */
        for (i= 4; (map->t_width % i > 0 || map->t_height % i > 0)
                   && i < map->t_width && i < map->t_height; i++);

        if (map->t_width % i > 0 || map->t_height % i > 0)
            goodie->entity->n_speed= 1;
        else
            goodie->entity->n_speed= i;

        /* bounding box */
        set_bb_size(goodie->entity, map);

        /* add to map */
        if (add_goodie_to_map(goodie, map, row, col) > 0)
        {
            write_to_log(manager.log, 1, "Could not add goodie to map.\n");
            destruct_goodie(goodie);
            return 0;
        }

    } /* end if map is nonnull */

    return goodie;

} /* end construct_goodie */




void collect_goodie(Goodie *const goodie, Player *const player,
                    const Map *const map, const unsigned char multiplier)
{

    Floating_text *points;

    char str[10];
    char *str_ptr;

    /* it's possible player will collide with a bag and the bad will collide
       with the player; make sure goodie is alive */
    if (goodie->status != GOODIE_S_KAPUT)
    {

        switch (goodie->type)
        {

            case GOODIE_T_BAG:

                str_ptr= "OK!";

                points=
                    construct_floating_text(str_ptr,
                                            goodie->entity->x
                                            + (map->t_width / 2)
                                            - (text_length(font_ingame, str_ptr)
                                               / 2),
                                            goodie->entity->y, 0, -1, 30);

                if (points != 0) append_to_list(map->point_list, points, 0);

                break;

            case GOODIE_T_1UP:

                ++player->num_lives;
                str_ptr= "1UP";

                points=
                    construct_floating_text(str_ptr,
                                            goodie->entity->x
                                            + (map->t_width / 2)
                                            - (text_length(font_ingame, str_ptr)
                                               / 2),
                                            goodie->entity->y, 0, -1, 30);

                if (points != 0) append_to_list(map->point_list, points, 0);

                break;

            default:

                if (goodie->points > 0 && multiplier > 0)
                {

                    if (multiplier == 1) sprintf(str, "%i", goodie->points);
                    else sprintf(str, "%ix%i", goodie->points, multiplier);

                    points=
                        construct_floating_text(str,
                                                goodie->entity->x
                                                + (map->t_width / 2)
                                                - (text_length(font_ingame, str)
                                                   / 2),
                                                goodie->entity->y, 0, -1, 30);

                    if (points != 0) append_to_list(map->point_list, points, 0);

                }

                break;

        } /* end switch on goodie type */

        player->score+= goodie->points * multiplier;
        set_goodie_status(goodie, GOODIE_S_KAPUT);

    } /* end if goodie is alive */

} /* end collect_goodie */




void destruct_goodie(Goodie *goodie)
{
    if (goodie->entity != 0) destruct_entity(goodie->entity);
    free(goodie);
    goodie= 0;
}




void do_goodie_logic(Goodie *const goodie, Map *const map)
{

    /* if goodie is square */
    if (goodie->entity->dx == 0 && goodie->entity->dy == 0)
    {

        /* if goodie is moving */
        if (goodie->status == GOODIE_S_MOVING)
        {

            //if (is_tile_empty_goodie(goodie->type, map, goodie->entity->row,
            //                       goodie->entity->col))
            //{
                set_goodie_status(goodie, BADDIE_S_STOPPED);
            //}
            //else
            //{
            //  move_goodie(goodie, map);ss
            //}

        }

    } /* end if goodie is square */

    else
    {

        find_goodie_to_player(goodie, map);
        step_entity(goodie->entity, goodie->entity->c_speed, map);

    }

} /* end do_goodie_logic */




void do_goodie_to_player(Goodie *const goodie, const Map *const map,
                         Player *const player)
{
    collect_goodie(goodie, player, map, 1);
}




void find_goodie_to_player(Goodie *const goodie, const Map *const map)
{

    if (map->player != 0
        && will_entities_collide(goodie->entity, map->player->entity, map, 1,
                                 1))
    {
        do_goodie_to_player(goodie, map, map->player);
    }

} /* end find_goodie_to_player */




void goodie_anim_done(Goodie *const goodie)
{


} /* end goodie_anim_done */




unsigned char goodie_baddie_coll(const goodie_type g_type,
                                 const baddie_type b_type,
                                 const baddie_status b_status)
{


    switch (b_status)
    {

        case BADDIE_S_KICKED:
        case BADDIE_S_KICKED_MELTING:
        case BADDIE_S_STEAMROLLED:
            return 0;
            break;

        default:
            return 1;
            break;

    }

    /* collision */
    return 1;
}




unsigned char goodie_feature_coll(const goodie_type g_type,
                                  const feature_type f_type,
                                  const feature_status f_status)
{

    if (f_type == FEATURE_T_BLACK_ICE) return 0;

    /* collision */
    return 1;

}




unsigned char goodie_goodie_coll(const goodie_type active_type,
                                 const goodie_type passive_type)
{
    /* collision */
    return 1;
}




unsigned char goodie_player_coll(const goodie_type g_type,
                                 const player_status p_status)
{
    /* collision */
    return 1;
}




unsigned char goodie_tile_coll(const goodie_type g_type,
                               const tile_type t_type)
{

    if (t_type == TILE_T_ICE) return 0;

    /* collision */
    return 1;

}




void move_goodie(Goodie *const goodie, const Map *const map,
                 unsigned char heading_x, unsigned char heading_y)
{

    if (goodie->status == GOODIE_S_STOPPED)
    {

        if (is_tile_empty_goodie(goodie->type, map,
                                 goodie->entity->row + heading_y,
                                 goodie->entity->col + heading_x))
        {
            set_heading(goodie->entity, heading_x, heading_y);
        }

        else if (is_tile_empty_goodie(goodie->type, map,
                                      goodie->entity->row - heading_y,
                                      goodie->entity->col + heading_x))
        {
            set_heading(goodie->entity, heading_x, -heading_y);
        }

        else if (is_tile_empty_goodie(goodie->type, map,
                                      goodie->entity->row + heading_y,
                                      goodie->entity->col - heading_x))
        {
            set_heading(goodie->entity, -heading_x, heading_y);
        }

        else set_heading(goodie->entity, -heading_x, -heading_y);

        set_goodie_status(goodie, GOODIE_S_MOVING);
        step_entity(goodie->entity, goodie->entity->c_speed, map);

    }

} /* end move_goodie */




void set_goodie_status(Goodie *const goodie, goodie_status status)
{

    switch (status)
    {

        case GOODIE_S_MOVING:
            goodie->status= status;
            goodie->entity->c_speed= goodie->entity->n_speed;
            break;

        case GOODIE_S_STOPPED:
            goodie->status= status;
            goodie->entity->c_speed= 0;
            break;

        default:
            goodie->status= status;

    }

} /* end set_goodie_status */
