/************************************
 *
 * Kickle 0.81
 * 2004 Drew Willcoxon
 * http://www.cs.uga.edu/~adw/
 * dripfeed@uga.edu
 *
 ************************************/


#include <allegro.h>
#include <stdio.h>

#include "common.h"
#include "config.h"
#include "gmenus.h"
#include "log.h"
#include "menu.h"
#include "video.h"




/* "on" if val is nonzero; "off" otherwise */
#define ON_OFF(val)         (val == 0 ? "off" : "on")




/**
 * Copies source's configuration variables to dest's.
 */
void copy_manager(Manager *const dest, const Manager *const source);


unsigned char do_gameplay_options(Manager *const manager, const Map *const map);


unsigned char do_keyboard_options(Manager *const manager, const Map *const map);


unsigned char do_options(Manager *const manager, const Map *const map);


unsigned char do_video_options(Manager *const manager, const Map *const map);


int get_scancode(void);




void copy_manager(Manager *const dest, const Manager *const source)
{

    dest->use_vsync= source->use_vsync;
    dest->draw_always= source->draw_always;
    dest->show_fps= source->show_fps;
    dest->fullscreen= source->fullscreen;
    dest->screen_w= source->screen_w;
    dest->screen_h= source->screen_h;

    dest->use_balk= source->use_balk;
    dest->balk_x= source->balk_x;
    dest->balk_y= source->balk_y;
    dest->bps= source->bps;

    dest->controls.key_up= source->controls.key_up;
    dest->controls.key_down= source->controls.key_down;
    dest->controls.key_left= source->controls.key_left;
    dest->controls.key_right= source->controls.key_right;
    dest->controls.key_breath= source->controls.key_breath;
    dest->controls.key_pillar= source->controls.key_pillar;
    dest->controls.key_spin= source->controls.key_spin;

    dest->ignore_time= source->ignore_time;
    dest->invincibility= source->invincibility;

}




unsigned char do_gameplay_options(Manager *const manager, const Map *const map)
{

    unsigned char status;
    int key_hit;

    Manager tmp_manager;

    Menu *menu;
    char str[50];

    enum
    {
        USE_BALK,
        SAVE,
        EXIT
    };

    char *labels[3];

    /* set up labels for menu creation */
    labels[USE_BALK]= "";
    labels[SAVE]= "save & return to main";
    labels[EXIT]= "no save & back";

    sprintf(str, "%s %s > options > gameplay", PROG_NAME, PROG_VER);

    /* construct menu */
    menu= construct_menu(str, 3, (char **) labels);

    if (menu == 0)
    {
        write_to_log(manager->log, 1,
                     "\nCould not construct gameplay options menu.\n");
        return TITLE_RUNNING;
    }

    write_to_log(manager->log, 0, "\nRunning gameplay options menu.\n");

    /* set up the labels properly */
    sprintf(str, "balk: %s", ON_OFF(manager->use_balk));
    set_item_label(menu, USE_BALK, str);

    /* copy manager to temp */
    copy_manager(&tmp_manager, manager);

    /* run the menu */
    key_hit= KEY_ENTER;
    status= TITLE_RUNNING;
    while (status == TITLE_RUNNING)
    {

        if (map != 0) draw_map(map);
        draw_menu(menu, (map == 0 ? 0 : 1));
        draw_to_screen(0);

        //while (key[key_hit]);
        clear_keybuf();

        key_hit= readkey() >> 8;

        //printf("key_hit == %i\n", key_hit);

        switch (key_hit)
        {

            case KEY_UP:
                dec_hilite_index(menu);
                break;

            case KEY_DOWN:
                inc_hilite_index(menu);
                break;

            case KEY_ESC:
                if (menu->hilite_index == EXIT) status= TITLE_EXIT;
                else menu->hilite_index= EXIT;
                break;

            case KEY_LEFT:
            case KEY_RIGHT:
            case KEY_ENTER:

                switch (menu->hilite_index)
                {

                    case USE_BALK:
                        tmp_manager.use_balk=
                            (tmp_manager.use_balk ? 0 : 1);
                        sprintf(str, "balk: %s",
                                ON_OFF(tmp_manager.use_balk));
                        set_item_label(menu, USE_BALK, str);
                        break;

                    case SAVE:
                        if (key_hit == KEY_ENTER)
                        {
                            write_to_log(manager->log, 0,
                                         "Save gameplay options selected.\n");
                            copy_manager(manager, &tmp_manager);
                            return TITLE_GAMEPLAY;
                        }
                        break;

                    case EXIT:
                        if (key_hit == KEY_ENTER)
                        {
                            write_to_log(manager->log, 0, "Back selected.\n");
                            status= TITLE_EXIT;
                        }
                        break;

                }

                break;

        } /* end switch on the key pressed */

    } /* end while running */

    destruct_menu(menu);

    return status;

} /* end do_gameplay_options */




unsigned char do_keyboard_options(Manager *const manager, const Map *const map)
{

    unsigned char status;
    int key_hit;

    Manager tmp_manager;

    Menu *menu;
    char str[50];

    enum
    {
        UP,
        DOWN,
        LEFT,
        RIGHT,
        BREATH,
        PILLAR,
        SPIN,
        SAVE,
        EXIT
    };

    char *labels[9];

    /* set up labels for menu creation */
    labels[UP]= "up";
    labels[DOWN]= "down";
    labels[LEFT]= "left";
    labels[RIGHT]= "right";
    labels[BREATH]= "ice breath";
    labels[PILLAR]= "ice pillar";
    labels[SPIN]= "spin";
    labels[SAVE]= "save & return to main";
    labels[EXIT]= "no save & back";

    sprintf(str, "%s %s > options > keyboard", PROG_NAME, PROG_VER);

    /* construct menu */
    menu= construct_menu(str, 9, (char **) labels);

    if (menu == 0)
    {
        write_to_log(manager->log, 1,
                     "\nCould not construct keyboard options menu.\n");
        return TITLE_RUNNING;
    }

    write_to_log(manager->log, 0, "\nRunning keyboard options menu.\n");

    /* copy manager to temp */
    copy_manager(&tmp_manager, manager);

    /* run the menu */
    key_hit= KEY_ENTER;
    status= TITLE_RUNNING;
    while (status == TITLE_RUNNING)
    {

        if (map != 0) draw_map(map);
        draw_menu(menu, (map == 0 ? 0 : 1));
        draw_to_screen(0);

        //while (key[key_hit]);
        clear_keybuf();

        key_hit= readkey() >> 8;

        //printf("key_hit == %i\n", key_hit);

        switch (key_hit)
        {

            case KEY_UP:
                dec_hilite_index(menu);
                break;

            case KEY_DOWN:
                inc_hilite_index(menu);
                break;

            case KEY_ESC:
                if (menu->hilite_index == EXIT) status= TITLE_EXIT;
                else menu->hilite_index= EXIT;
                break;

            case KEY_ENTER:

                switch (menu->hilite_index)
                {

                    case UP:
                        text_mode(22);
                        draw_text(160, 200, 32, 1, "Press the key you'd");
                        draw_text(160, 220, 32, 1, "like to use for UP.");
                        draw_to_screen(0);
                        tmp_manager.controls.key_up= get_scancode();
                        break;

                    case DOWN:
                        text_mode(22);
                        draw_text(160, 200, 32, 1, "Press the key you'd");
                        draw_text(160, 220, 32, 1, "like to use for DOWN.");
                        draw_to_screen(0);
                        tmp_manager.controls.key_down= get_scancode();
                        break;

                    case LEFT:
                        text_mode(22);
                        draw_text(160, 200, 32, 1, "Press the key you'd");
                        draw_text(160, 220, 32, 1, "like to use for LEFT.");
                        draw_to_screen(0);
                        tmp_manager.controls.key_left= get_scancode();
                        break;

                    case RIGHT:
                        text_mode(22);
                        draw_text(160, 200, 32, 1, "Press the key you'd");
                        draw_text(160, 220, 32, 1, "like to use for RIGHT.");
                        draw_to_screen(0);
                        tmp_manager.controls.key_right= get_scancode();
                        break;

                    case BREATH:
                        text_mode(22);
                        draw_text(160, 200, 32, 1, "Press the key you'd");
                        draw_text(160, 220, 32, 1,
                                  "like to use for ICE BREATH.");
                        draw_to_screen(0);
                        tmp_manager.controls.key_breath= get_scancode();
                        break;

                    case PILLAR:
                        text_mode(22);
                        draw_text(160, 200, 32, 1, "Press the key you'd");
                        draw_text(160, 220, 32, 1,
                                  "like to use for ICE PILLAR.");
                        draw_to_screen(0);
                        tmp_manager.controls.key_pillar= get_scancode();
                        break;

                    case SPIN:
                        text_mode(22);
                        draw_text(160, 200, 32, 1, "Press the key you'd");
                        draw_text(160, 220, 32, 1, "like to use for SPIN.");
                        draw_to_screen(0);
                        tmp_manager.controls.key_spin= get_scancode();
                        break;

                    case SAVE:
                        if (key_hit == KEY_ENTER)
                        {
                            write_to_log(manager->log, 0,
                                         "Save keyboard options selected.\n");
                            copy_manager(manager, &tmp_manager);
                            return TITLE_KEYBOARD;
                        }
                        break;

                    case EXIT:
                        if (key_hit == KEY_ENTER)
                        {
                            write_to_log(manager->log, 0, "Back selected.\n");
                            status= TITLE_EXIT;
                        }
                        break;

                }

                break;

        } /* end switch on the key pressed */

    } /* end while running */

    destruct_menu(menu);

    return status;

} /* end do_keyboard_options */




unsigned char do_options(Manager *const manager, const Map *const map)
{

    unsigned char status;
    int key_hit;

    Menu *menu;
    char title[50];

    enum
    {
        VIDEO,
        GAMEPLAY,
        KEYBOARD,
        EXIT
    };

    char *labels[4];
    labels[VIDEO]= "video";
    labels[GAMEPLAY]= "gameplay";
    labels[KEYBOARD]= "keyboard";
    labels[EXIT]= "back";

    sprintf(title, "%s %s > options", PROG_NAME, PROG_VER);

    menu= construct_menu(title, 4, labels);

    if (menu == 0)
    {
        write_to_log(manager->log, 1, "\nCould not construct options menu.\n");
        return TITLE_RUNNING;
    }

    write_to_log(manager->log, 0, "\nRunning options menu.\n");

    key_hit= KEY_ENTER;
    status= TITLE_RUNNING;
    while (status == TITLE_RUNNING)
    {

        if (map != 0) draw_map(map);
        draw_menu(menu, (map == 0 ? 0 : 1));
        draw_to_screen(0);

        //while (key[key_hit]);
        clear_keybuf();

        key_hit= readkey() >> 8;

        //printf("key_hit == %i\n", key_hit);

        switch (key_hit)
        {

            case KEY_UP:
                dec_hilite_index(menu);
                break;

            case KEY_DOWN:
                inc_hilite_index(menu);
                break;

            case KEY_ESC:
                if (menu->hilite_index == EXIT) status= TITLE_EXIT;
                else menu->hilite_index= EXIT;
                break;

            case KEY_ENTER:

                switch (menu->hilite_index)
                {

                    case VIDEO:
                        write_to_log(manager->log, 0,
                                     "Video options selected.\n");
                        status= do_video_options(manager, map);
                        if (status == TITLE_EXIT) status= TITLE_RUNNING;
                        break;

                    case GAMEPLAY:
                        write_to_log(manager->log, 0,
                                     "Gameplay options selected.\n");
                        status= do_gameplay_options(manager, map);
                        if (status == TITLE_EXIT) status= TITLE_RUNNING;
                        break;

                    case KEYBOARD:
                        write_to_log(manager->log, 0,
                                     "Keyboard options selected.\n");
                        status= do_keyboard_options(manager, map);
                        if (status == TITLE_EXIT) status= TITLE_RUNNING;
                        break;

                    case EXIT:
                        write_to_log(manager->log, 0, "Back selected.\n");
                        status= TITLE_EXIT;
                        break;

                }

                break;

        } /* end switch on the key pressed */

    } /* end while running */

    destruct_menu(menu);

    return status;

} /* end do_options */




menu_status do_pause_menu(Manager *const manager, const Map *const map)
{

    menu_status status;
    int key_hit;

    Menu *menu;

    enum
    {
        RESUME,
        OPTIONS,
        EXIT
    };

    char *labels[3];
    labels[RESUME]= "resume game";
    labels[OPTIONS]= "options";
    labels[EXIT]= "exit game";

    menu= construct_menu("PAUSED", 3, labels);

    if (menu == 0)
    {
        write_to_log(manager->log, 1, "\nCould not construct pause menu.\n");
        return TITLE_ERROR;
    }

    write_to_log(manager->log, 0, "\nRunning pause menu.\n");

    key_hit= KEY_ENTER;
    status= TITLE_RUNNING;
    while (status == TITLE_RUNNING)
    {

        draw_map(map);
        draw_menu(menu, 1);
        draw_to_screen(0);

        //while (key[key_hit]);
        clear_keybuf();

        key_hit= readkey() >> 8;

        //printf("key_hit == %i\n", key_hit);

        switch (key_hit)
        {

            case KEY_UP:
                dec_hilite_index(menu);
                break;

            case KEY_DOWN:
                inc_hilite_index(menu);
                break;

            case KEY_ESC:
                if (menu->hilite_index == RESUME)
                {
                    while (key[KEY_ESC]) poll_controls();
                    status= TITLE_PLAY;
                }
                else menu->hilite_index= RESUME;
                break;

            case KEY_ENTER:

                switch (menu->hilite_index)
                {

                    case RESUME:
                        write_to_log(manager->log, 0,
                                     "Resume game selected.\n");
                        status= TITLE_PLAY;
                        break;

                    case OPTIONS:

                        write_to_log(manager->log, 0, "Options selected.\n");

                        switch (do_options(manager, map))
                        {
                            case TITLE_VIDEO:
                            case TITLE_GAMEPLAY:
                            case TITLE_KEYBOARD:
                                if (apply_configs(CONFIG_FILENAME, manager) > 0)
                                {
                                    write_to_log(manager->log, 1,
                                        "Unable to set graphics mode.\n");
                                    allegro_message(
                                        "%s %s:  Unable to set graphics mode.\nSee log for details.\n",
                                        PROG_NAME, PROG_VER);
                                    status= TITLE_ERROR;
                                }
                                break;

                            default:
                                break;

                        }
                        break;

                    case EXIT:
                        write_to_log(manager->log, 0, "Exit selected.\n");
                        status= TITLE_EXIT;
                        break;

                }

                break;

        } /* end switch on the key pressed on title menu */

    } /* end while running */

    destruct_menu(menu);

    return status;

} /* end do_pause_menu */




menu_status do_title(Manager *const manager)
{

    unsigned char status;
    int key_hit;

    Menu *menu;
    char title[50];

    enum
    {
        PLAY,
        OPTIONS,
        HELP,
        EXIT
    };

    char *labels[4];
    labels[PLAY]= "play";
    labels[OPTIONS]= "options";
    labels[HELP]= "help";
    labels[EXIT]= "exit";

    sprintf(title, "%s %s", PROG_NAME, PROG_VER);

    menu= construct_menu(title, 4, labels);

    if (menu == 0)
    {
        write_to_log(manager->log, 1, "\nCould not construct title menu.\n");
        return TITLE_ERROR;
    }

    write_to_log(manager->log, 0, "\nRunning title menu.\n");

    key_hit= KEY_ENTER;
    status= TITLE_RUNNING;
    while (status == TITLE_RUNNING)
    {

        draw_menu(menu, 0);
        draw_to_screen(0);

        //while (key[key_hit]);
        clear_keybuf();

        key_hit= readkey() >> 8;

        //printf("key_hit == %i\n", key_hit);

        switch (key_hit)
        {

            case KEY_UP:
                dec_hilite_index(menu);
                break;

            case KEY_DOWN:
                inc_hilite_index(menu);
                break;

            case KEY_ESC:
                if (menu->hilite_index == EXIT) status= TITLE_EXIT;
                else menu->hilite_index= EXIT;
                break;

            case KEY_ENTER:

                switch (menu->hilite_index)
                {

                    case PLAY:
                        write_to_log(manager->log, 0, "Play selected.\n");
                        status= TITLE_PLAY;
                        break;

                    case OPTIONS:
                        write_to_log(manager->log, 0, "Options selected.\n");
                        status= do_options(manager, 0);
                        if (status == TITLE_EXIT) status= TITLE_RUNNING;
                        break;

                    case HELP:
                        write_to_log(manager->log, 0, "Help selected.\n");
                        status= TITLE_HELP;
                        break;

                    case EXIT:
                        write_to_log(manager->log, 0, "Exit selected.\n");
                        status= TITLE_EXIT;
                        break;

                }

                break;

        } /* end switch on the key pressed on title menu */

    } /* end while running */

    destruct_menu(menu);

    return status;

} /* end do_title */




unsigned char do_video_options(Manager *const manager, const Map *const map)
{

    unsigned char status;
    int key_hit;

    Manager tmp_manager;

    Menu *menu;
    char str[50];

    enum
    {
        FULLSCREEN,
        RES,
        VSYNC,
        DRAW_ALWAYS,
        SAVE,
        EXIT
    };

    signed char res_index;
    unsigned char num_res= 7;
    unsigned short res[][2]=
        {{320, 200}, {320, 240}, {640, 480}, {1024, 768}, {1152, 864},
         {1280, 1024}, {1600, 1200}};

    char *labels[6];

    /* set up labels for menu creation */
    labels[FULLSCREEN]= "";
    labels[RES]= "";
    labels[VSYNC]= "";
    labels[DRAW_ALWAYS]= "";
    labels[SAVE]= "save & return to main";
    labels[EXIT]= "no save & back";

    sprintf(str, "%s %s > options > video", PROG_NAME, PROG_VER);

    /* construct menu */
    menu= construct_menu(str, 6, (char **) labels);

    if (menu == 0)
    {
        write_to_log(manager->log, 1,
                     "\nCould not construct video options menu.\n");
        return TITLE_RUNNING;
    }

    write_to_log(manager->log, 0, "\nRunning video options menu.\n");

    /* set up the labels properly */
    sprintf(str, "fullscreen: %s", ON_OFF(manager->fullscreen));
    set_item_label(menu, FULLSCREEN, str);

    sprintf(str, "resolution: %ix%i", manager->screen_w, manager->screen_h);
    set_item_label(menu, RES, str);

    sprintf(str, "vsync: %s", ON_OFF(manager->use_vsync));
    set_item_label(menu, VSYNC, str);

    sprintf(str, "draw always: %s", ON_OFF(manager->draw_always));
    set_item_label(menu, DRAW_ALWAYS, str);

    /* get index in res array */
    if (manager->screen_w == 320 && manager->screen_h == 240) res_index= 1;

    else
    {
        for (res_index= 0; res[res_index][0] < manager->screen_w; res_index++);
    }

    /* copy manager to temp */
    copy_manager(&tmp_manager, manager);

    /* run the menu */
    key_hit= KEY_ENTER;
    status= TITLE_RUNNING;
    while (status == TITLE_RUNNING)
    {

        if (map != 0) draw_map(map);
        draw_menu(menu, (map == 0 ? 0 : 1));
        draw_to_screen(0);

        //while (key[key_hit]);
        clear_keybuf();

        key_hit= readkey() >> 8;

        //printf("key_hit == %i\n", key_hit);

        switch (key_hit)
        {

            case KEY_UP:
                dec_hilite_index(menu);
                break;

            case KEY_DOWN:
                inc_hilite_index(menu);
                break;

            case KEY_ESC:
                if (menu->hilite_index == EXIT) status= TITLE_EXIT;
                else menu->hilite_index= EXIT;
                break;

            case KEY_LEFT:
            case KEY_RIGHT:
            case KEY_ENTER:

                switch (menu->hilite_index)
                {

                    case FULLSCREEN:
                        tmp_manager.fullscreen=
                            (tmp_manager.fullscreen ? 0 : 1);
                        sprintf(str, "fullscreen: %s",
                                ON_OFF(tmp_manager.fullscreen));
                        set_item_label(menu, FULLSCREEN, str);
                        break;

                    case RES:
                        if (key_hit == KEY_LEFT) --res_index;
                        else ++res_index;
                        if (res_index >= num_res) res_index= 0;
                        else if (res_index < 0) res_index= num_res - 1;
                        tmp_manager.screen_w= res[res_index][0];
                        tmp_manager.screen_h= res[res_index][1];
                        sprintf(str, "resolution: %ix%i", tmp_manager.screen_w,
                                tmp_manager.screen_h);
                        set_item_label(menu, RES, str);
                        break;

                    case VSYNC:
                        tmp_manager.use_vsync= (tmp_manager.use_vsync ? 0 : 1);
                        sprintf(str, "vsync: %s",
                                ON_OFF(tmp_manager.use_vsync));
                        set_item_label(menu, VSYNC, str);
                        break;

                    case DRAW_ALWAYS:
                        tmp_manager.draw_always=
                            (tmp_manager.draw_always ? 0 : 1);
                        sprintf(str, "draw always: %s",
                                ON_OFF(tmp_manager.draw_always));
                        set_item_label(menu, DRAW_ALWAYS, str);
                        break;

                    case SAVE:
                        if (key_hit == KEY_ENTER)
                        {
                            write_to_log(manager->log, 0,
                                         "Save video options selected.\n");
                            manager->old_screen_w= manager->screen_w;

                            manager->old_screen_h= manager->screen_h;
                            manager->old_fullscreen= manager->fullscreen;
                            copy_manager(manager, &tmp_manager);
                            return TITLE_VIDEO;
                        }
                        break;

                    case EXIT:
                        if (key_hit == KEY_ENTER)
                        {
                            write_to_log(manager->log, 0, "Back selected.\n");
                            status= TITLE_EXIT;
                        }
                        break;

                }

                break;

        } /* end switch on the key pressed */

    } /* end while running */

    destruct_menu(menu);

    return status;

} /* end do_video_options */




int get_scancode(void)
{

    int scancode;
    unsigned char running;

    clear_keybuf();

    /* loop until we get a keypress or key shift */
    running= 1;
    while (running)
    {

        /* if there's a key in the buffer, get it */
        if (keypressed())
        {
            scancode= readkey() >> 8;
            running= 0;
        }

        /* otherwise, look for keys that aren't placed in the buffer */
        else
        {

            poll_keyboard();

            if (key_shifts)
            {

                if (key[KEY_LSHIFT]) scancode= KEY_LSHIFT;
                else if (key[KEY_RSHIFT]) scancode= KEY_RSHIFT;
                else if (key[KEY_LCONTROL]) scancode= KEY_LCONTROL;
                else if (key[KEY_RCONTROL]) scancode= KEY_RCONTROL;
                else if (key[KEY_ALT]) scancode= KEY_ALT;
                else if (key[KEY_ALTGR]) scancode= KEY_ALTGR;

                running= 0;

            }

        }

    }

    return scancode;

} /* end get_scancode */
