/************************************
 *
 * Kickle 0.81
 * 2004 Drew Willcoxon
 * http://www.cs.uga.edu/~adw/
 * dripfeed@uga.edu
 *
 ************************************/


#ifndef _ENTITY_H_
#define _ENTITY_H_


#include "tentity.h"
#include "tmap.h"


/**
 * Returns the column adjacent to entity.  This is defined as the closest column
 * to entity which is entity facing but which entity does not overlap.  If
 * entity is not facing a column (i.e., entity's heading is vertical), entity's
 * current column is returned.
 */
unsigned char adj_col(const Entity *entity);


/**
 * Returns the row adjacent to entity.  This is defined as the closest row
 * to entity which is entity facing but which entity does not overlap.  If
 * entity is not facing a row (i.e., entity's heading is horizontal), entity's
 * current row is returned.
 */
unsigned char adj_row(const Entity *entity);


/**
 * If entity's current speed and change in position is such that entity will
 * not land squarely on a tile after its move, corrects entity's position so
 * that it will land squarely.  This situation could occur if entity changes
 * speed mid-move.
 */
void adjust_pos(Entity *const entity, const Map *const map);


/**
 * Allocates and returns an entity.  This function does NOT set the entity's
 * size or its bounding box sizes.  set_bb_size should be called on the returned
 * entity, and only after the entity's size is set.
 */
Entity *construct_entity(void);


/**
 * Allocates and returns a clone of entity.
 */
Entity *clone_entity(const Entity *const entity);


/**
 * Deallocates entity.
 */
void destruct_entity(Entity *entity);


/**
 * Changes entity's heading so that it faces the specified target location.
 */
void face_target(Entity *const entity, const unsigned char target_row,
                 const unsigned char target_col);


/**
 * Returns a positive number if entity is not facing the specified target.
 * Returns 0 otherwise.
 */
unsigned char has_back_to(const Entity *const entity,
                          const unsigned char target_row,
                          const unsigned char target_col);


/**
 * Returns a positive number if entity occupies the specified tile on map.
 * fudge_x and fudge_y indicate how many pixel columns and how may pixel rows,
 * respectively, of entity's bounding box must overlap the tile for the
 * entity to be considered as occupying the tile.
 */
unsigned char is_entity_on_tile(const Entity *const entity,
                                const Map *const map,
                                const unsigned char row,
                                const unsigned char col,
                                const unsigned short fudge_x,
                                const unsigned short fudge_y);


/**
 * Resets entity's animation variables so that entity's animation will restart
 * from the beginning on the next animation update.
 */
void reset_anim(Entity *const entity);


/**
 * Reverses entity's heading.
 */
void reverse_heading(Entity *const entity);


/**
 * Sets the sizes of entity's bounding box, based on entity's pixel size and
 * map's tile size.
 */
void set_bb_size(Entity *const entity, const Map *const map);


/**
 * If active is to the left or right of passive, sets passive flush with the
 * leftmost or rightmost column that passive overlaps, respectively.
 * If active is above or below passive, sets passive flush with the
 * topmost or bottommost row that passive overlaps, respectively.
 */
void set_flush(Entity *const active, Entity *const passive,
               const Map *const map);


/**
 * Sets entity's heading to the specified heading.
 */
void set_heading(Entity *const entity, signed char x, signed char y);


/**
 * Sits entity squarely on the specified tile of map, updating entity's map
 * position, screen position, change in position, and bounding box screen
 * position.
 */
void square_entity(Entity *const entity, unsigned char row,
                   unsigned char col, const Map *const map);


/**
 * Makes entity step once on map.  The size of the step is determined by
 * num_pixels.  Map can be null, in which case entity map location is not
 * updated.
 */
void step_entity(Entity *const entity, const unsigned char num_pixels,
                 const Map *const map);


/**
 * Returns a positive number if active and passive will collide on map after
 * active takes one more step.  fudge_x and fudge_y indicate how many pixel
 * columns and how may pixel rows, respectively, of active's and passive's
 * bounding boxes must overlap for a collision to occur between them.
 */
unsigned char will_entities_collide(const Entity *const active,
                                    const Entity *const passive,
                                    const Map *const map,
                                    const unsigned short fudge_x,
                                    const unsigned short fudge_y);


/**
 * Returns a positive number if after stepping once more, entity will lie
 * outside map's bounds.  Returns 0 otherwise.
 */
unsigned char will_step_oob(const Entity *const entity, const Map *const map);


#endif
