/************************************
 *
 * Kickle 0.81
 * 2004 Drew Willcoxon
 * http://www.cs.uga.edu/~adw/
 * dripfeed@uga.edu
 *
 ************************************/


#include <stdlib.h>

#include "entity.h"
#include "log.h"
#include "manager.h"
#include "map.h"




/* the x and y offset (in pixels) of entity's bounding box from its sprite */
#define BB_X_OFFSET(entity, map)    -(((entity)->p_width % (map)->t_width) / 2)
#define BB_Y_OFFSET(entity, map)    -((entity)->p_height % (map)->t_height)




extern Manager manager;


#if 0
// how would this make sense if an entity can span multiple tiles?
type_map_tile *adj_tile(const type_entity *const entity,
                        const type_map *const map)
{

    /* up */
    if (entity->heading_y < 0)
    {
        if (entity->row == 0) return 0;
        return &(map->tiles[entity->row - 1][entity->col]);
    }

    /* down */
    if (entity->heading_y > 0)
    {
        if (entity->row >= map->m_height - 1) return 0;
        return &(map->tiles[entity->row + 1][entity->col]);
    }

    /* left */
    if (entity->heading_x < 0)
    {
        if (entity->col == 0) return 0;
        return &(map->tiles[entity->row][entity->col - 1]);
    }

    /* right */
    if (entity->heading_x > 0)
    {
        if (entity->col >= map->m_width - 1) return 0;
        return &(map->tiles[entity->row][entity->col + 1]);
    }

    return 0;

} /* end adj_tile */
#endif




unsigned char adj_col(const Entity *entity)
{
    if (entity->heading_x < 0) return entity->col - 1;
    else if (entity->heading_x > 0)
    {
        if (entity->dx == 0) return entity->col + entity->bb_t_width;
        return entity->col + entity->bb_t_width + 1;
    }
    return entity->col;
}




unsigned char adj_row(const Entity *entity)
{
    if (entity->heading_y < 0) return entity->row - 1;
    else if (entity->heading_y > 0)
    {
        if (entity->dy == 0) return entity->row + entity->bb_t_height;
        return entity->row + entity->bb_t_height + 1;
    }
    return entity->row;
}




void adjust_pos(Entity *const entity, const Map *const map)
{

    unsigned char num_pixels_leftover;

    /* heading up */
    if (entity->heading_y < 0)
        num_pixels_leftover= entity->dy % entity->c_speed;

    /* heading down */
    else if (entity->heading_y > 0)
        num_pixels_leftover= (map->t_height - entity->dy) % entity->c_speed;

    /* heading left */
    else if (entity->heading_x < 0)
        num_pixels_leftover= entity->dx % entity->c_speed;

    /* heading right */
    else if (entity->heading_x > 0)
        num_pixels_leftover= (map->t_width - entity->dx) % entity->c_speed;

    /* step entity if necessary */
    if (num_pixels_leftover > 0)
        step_entity(entity, num_pixels_leftover, map);

} /* end adjust_pos */




Entity *construct_entity(void)
{

    /* allocate entity memory */
    Entity *entity= (Entity *) malloc(sizeof(Entity));
    if (entity == 0)
    {
        write_to_log(manager.log, 1, "Could not allocate entity memory.\n");
        return 0;
    }

    /* set members */
    entity->dx= 0;
    entity->dy= 0;

    entity->heading_x= 0;
    entity->heading_y= 1;

    entity->c_speed= 0;

    entity->logic_counter= 0;

    entity->anim_counter= 0;

    entity->anim_frame_counter= 0;
    entity->anim_play_counter= 0;

    return entity;

} /* end construct_entity */




Entity *clone_entity(const Entity *const entity)
{

    Entity *clone= construct_entity();
    if (clone == 0)
    {
        write_to_log(manager.log, 1,
                     "Could not allocate entity clone memory.\n");
        return 0;
    }

    //clone->status= entity->status;
    clone->p_width= entity->p_width;
    clone->p_height= entity->p_height;
    clone->bb_x= entity->bb_x;
    clone->bb_y= entity->bb_y;
    clone->bb_p_width= entity->bb_p_width;
    clone->bb_p_height= entity->bb_p_height;
    clone->bb_t_width= entity->bb_t_width;
    clone->bb_t_height= entity->bb_t_height;
    clone->row= entity->row;
    clone->col= entity->col;
    clone->x= entity->x;
    clone->y= entity->y;
    clone->dx= entity->dx;
    clone->dy= entity->dy;
    clone->heading_x= entity->heading_x;
    clone->heading_y= entity->heading_y;
    clone->c_speed= entity->c_speed;
    clone->n_speed= entity->n_speed;
    clone->logic_counter= entity->logic_counter;
    clone->anim_counter= entity->anim_counter;
    clone->anim_frame= entity->anim_frame;

    return clone;

} /* end clone_entity */




void destruct_entity(Entity *entity)
{
    free(entity);
    entity= 0;
}




void face_target(Entity *const entity, const unsigned char target_row,
                 const unsigned char target_col)
{

    /* entity on same row as target */
    if (entity->row == target_row)
    {
        if (entity->col > target_col) set_heading(entity, -1, 0);
        else set_heading(entity, 1, 0);
    }
    /* entity on same column as target */
    else if (entity->col == target_col)
    {
        if (entity->row > target_row) set_heading(entity, 0, -1);
        else set_heading(entity, 0, 1);
    }
    /* entity on neither same row nor same column as target */
    else
    {
        if (has_back_to(entity, target_row, target_col))
            reverse_heading(entity);
    }

} /* end face_target */




unsigned char has_back_to(const Entity *const entity,
                          const unsigned char target_row,
                          const unsigned char target_col)
{

    /* same row as target */
    if (entity->row == target_row)
    {
        if ((entity->col > target_col && entity->heading_x >= 0)
            || (entity->col < target_col && entity->heading_x <= 0))
        {
            return 1;
        }
    }
    /* same column */
    else if (entity->col == target_col)
    {
        if ((entity->row > target_row && entity->heading_y >= 0)
            || (entity->row < target_row && entity->heading_y <= 0))
        {
            return 1;
        }
    }
    /* below and to the right of target */
    else if ((entity->row > target_row && entity->col > target_col)
             && (entity->heading_y > 0 || entity->heading_x > 0))
    {
        return 1;
    }
    /* above and to the right of target */
    else if ((entity->row < target_row && entity->col > target_col)
             && (entity->heading_y < 0 || entity->heading_x > 0))
    {
        return 1;
    }
    /* above and to the left of target */
    else if ((entity->row < target_row && entity->col < target_col)
             && (entity->heading_y < 0 || entity->heading_x < 0))
    {
        return 1;
    }
    /* below and to the left of target */
    else if ((entity->row > target_row && entity->col < target_col)
             && (entity->heading_y > 0 || entity->heading_x < 0))
    {
        return 1;
    }

    /* facing target */
    return 0;

} /* end has_back_to */



#if 0
unsigned char hyp_dx(const type_entity *const entity,
                     const type_map *const map)
{
    return hyp_x(entity) + entity_x_offset(entity, map)
           - map->tiles[0][hyp_col(entity, map)].x;
}




unsigned char hyp_dy(const type_entity *const entity,
                     const type_map *const map)
{
    return hyp_y(entity) + entity_y_offset(entity, map)
           - map->tiles[hyp_row(entity, map)][0].y;
}
#endif


#if 0
unsigned char hyp_col(const type_entity *const entity,
                      const type_map *const map)
{
    return x_to_col(hyp_x(entity) + entity_x_offset(entity, map), map);
}




unsigned char hyp_row(const type_entity *const entity,
                      const type_map *const map)
{
    return y_to_row(hyp_y(entity) + entity_y_offset(entity, map), map);
}




signed short hyp_x(const type_entity *const entity)
{
    return entity->x + (entity->c_speed * entity->heading_x);
}




signed short hyp_y(const type_entity *const entity)
{
    return entity->y + (entity->c_speed * entity->heading_y);
}
#endif



unsigned char is_entity_on_tile(const Entity *const entity,
                                const Map *const map,
                                const unsigned char row,
                                const unsigned char col,
                                const unsigned short fudge_x,
                                const unsigned short fudge_y)
{

    signed short x_overlap;
    signed short y_overlap;

    if (entity->bb_x < map->tiles[row][col].x)
        x_overlap= entity->bb_x + entity->bb_p_width - map->tiles[row][col].x;
    else
        x_overlap= map->tiles[row][col].x + map->t_width - entity->bb_x;


    if (entity->bb_y < map->tiles[row][col].y)
        y_overlap= entity->bb_y + entity->bb_p_height - map->tiles[row][col].y;
    else
        y_overlap= map->tiles[row][col].y + map->t_height - entity->bb_y;

    if (x_overlap > 0 && y_overlap > 0
        && x_overlap >= fudge_x && y_overlap >= fudge_y)
    {
        return 1;
    }

    return 0;

} /* end is_entity_on_tile */




void reset_anim(Entity *const entity)
{
    entity->anim_counter= 0;
    entity->anim_frame_counter= 0;
    entity->anim_play_counter= 0;
}




void reverse_heading(Entity *const entity)
{
    entity->heading_x= -entity->heading_x;
    entity->heading_y= -entity->heading_y;
}




void set_bb_size(Entity *const entity, const Map *const map)
{

    entity->bb_t_width= entity->p_width / map->t_width;
    entity->bb_t_height= entity->p_height / map->t_height;

    entity->bb_p_width= entity->bb_t_width * map->t_width;
    entity->bb_p_height= entity->bb_t_height * map->t_height;

} /* end set_bb_size */




void set_flush(Entity *const active, Entity *const passive,
               const Map *const map)
{

    if (active->heading_y != 0)
    {
        if (passive->dx > 0)
        {
            if (active->x <= passive->x)
                square_entity(passive, passive->row, passive->col, map);
            else
                square_entity(passive, passive->row, passive->col + 1, map);
        }
    }

    if (active->heading_x != 0)
    {
        if (passive->dy > 0)
        {
            if (active->y <= passive->y)
                square_entity(passive, passive->row, passive->col, map);
            else
                square_entity(passive, passive->row + 1, passive->col, map);
        }
    }

} /* end set_flush */




void set_heading(Entity *const entity, signed char x, signed char y)
{

    if (entity->heading_x != x && entity->heading_y != y)
    {
        entity->anim_counter= 0;
        entity->anim_frame_counter= 0;
        entity->anim_play_counter= 0;
    }

    entity->heading_x= x;
    entity->heading_y= y;

}




void square_entity(Entity *const entity, unsigned char row,
                   unsigned char col, const Map *const map)
{
    entity->row= row;
    entity->col= col;

    entity->bb_x= map->tiles[0][col].x;
    entity->bb_y= map->tiles[row][0].y;

    entity->x= map->tiles[0][col].x + BB_X_OFFSET(entity, map);
    entity->y= map->tiles[row][0].y + BB_Y_OFFSET(entity, map);

    entity->dx= 0;
    entity->dy= 0;

} /* end square_entity */




void step_entity(Entity *const entity, const unsigned char num_pixels,
                 const Map *const map)
{

    signed short num_x_pixels= num_pixels * entity->heading_x;
    signed short num_y_pixels= num_pixels * entity->heading_y;

    entity->x+= num_x_pixels;
    entity->y+= num_y_pixels;

    entity->l_heading_x= entity->heading_x;
    entity->l_heading_y= entity->heading_y;

    if (map != 0)
    {

        entity->bb_x+= num_x_pixels;
        entity->bb_y+= num_y_pixels;

        entity->row= Y_TO_ROW(entity->bb_y, map);
        entity->col= X_TO_COL(entity->bb_x, map);

        /* check if entity moved in a negative direction when d* = 0 */
        if (entity->dx == 0 && num_x_pixels < 0)
            entity->dx= (map->t_width + num_x_pixels) % map->t_width;
        else
            entity->dx= (entity->dx + num_x_pixels) % map->t_width;

        if (entity->dy == 0 && num_y_pixels < 0)
            entity->dy= (map->t_height + num_y_pixels) % map->t_height;
        else
            entity->dy= (entity->dy + num_y_pixels) % map->t_height;

    }

} /* end step_entity */




unsigned char will_entities_collide(const Entity *const active,
                                    const Entity *const passive,
                                    const Map *const map,
                                    const unsigned short fudge_x,
                                    const unsigned short fudge_y)
{

    Entity *act_clone;

    signed short x_overlap;
    signed short y_overlap;

    /* if active is not moving, it won't collide */
    if (active->c_speed == 0) return 0;

    /* an entity shouldn't collide with itself */
    if (active == passive) return 0;

    /* compute active's new locations if it were to step once more */
    act_clone= clone_entity(active);
    if (act_clone == 0)
    {
        write_to_log(manager.log, 1,
                     "Could not test if entities collide because entity clone could not be constructed.\n");
        return 0;
    }

    step_entity(act_clone, act_clone->c_speed, map);

    x_overlap= 0;
    y_overlap= 0;

    if (act_clone->bb_x == passive->bb_x)
        x_overlap= act_clone->bb_x + act_clone->bb_p_width - passive->bb_x;
    else if (act_clone->bb_x < passive->bb_x && act_clone->heading_x >= 0)
        x_overlap= act_clone->bb_x + act_clone->bb_p_width - passive->bb_x;
    else if (act_clone->bb_x > passive->bb_x && act_clone->heading_x <= 0)
        x_overlap= passive->bb_x + passive->bb_p_width - act_clone->bb_x;

    if (act_clone->bb_y == passive->bb_y)
        y_overlap= act_clone->bb_y + active->bb_p_height - passive->bb_y;
    else if (act_clone->bb_y < passive->bb_y && act_clone->heading_y >= 0)
        y_overlap= act_clone->bb_y + active->bb_p_height - passive->bb_y;
    else if (act_clone->bb_y > passive->bb_y && act_clone->heading_y <= 0)
        y_overlap= passive->bb_y + passive->bb_p_height - act_clone->bb_y;

    /* no longer need active's clone */
    destruct_entity(act_clone);

    /* if the bounding boxes overlap at all */
    if (x_overlap > 0 && y_overlap > 0)
    {

        /* if passive is not moving, unconditionally collide */
        //if (passive->c_speed == 0) return 1;

        /* compare overlap to fudge */
        if (x_overlap >= fudge_x && y_overlap >= fudge_y)
        {
            if (active->heading_y != 0 && x_overlap >= y_overlap) return 1;
            if (active->heading_x != 0 && y_overlap >= x_overlap) return 1;
        }

    }

    return 0;

} /* end will_entities_collide */




unsigned char will_step_oob(const Entity *const entity, const Map *const map)
{

    /* if entity is currently moving, he must be in bounds */
    if (entity->dx > 0 || entity->dy > 0) return 0;

    /* if entity isn't moving, he can't step out of bounds */
    if (entity->c_speed == 0) return 0;

    /* otherwise, check if he'll step out of bounds */

    /* heading up */
    if (entity->heading_y < 0 && entity->row == 0)
        return 1;

    /* heading down */
    else if (entity->heading_y > 0
             && entity->row + entity->bb_t_height >= map->m_height)
    {
        return 1;
    }

    /* heading left */

    if (entity->heading_x < 0 && entity->col == 0)
        return 1;

    /* heading right */
    else if (entity->heading_x > 0
             && entity->col + entity->bb_t_width >= map->m_width)
    {
        return 1;
    }

    /* not out of bounds */
    return 0;

} /* end will_step_oob */
