/************************************
 *
 * Kickle 0.81
 * 2004 Drew Willcoxon
 * http://www.cs.uga.edu/~adw/
 * dripfeed@uga.edu
 *
 ************************************/


#include <allegro.h>

#include "common.h"
#include "config.h"
#include "spritdat.h"
#include "video.h"




/* configuration keys */
#define VIDEO_SECTION           "Video"
#define VSYNC_KEY               "vsync"
#define DRAW_ALWAYS_KEY         "draw_always"
#define FPS_KEY                 "show_fps"
#define FULLSCREEN_KEY          "fullscreen"
#define SCREEN_W_KEY            "screen_width"
#define SCREEN_H_KEY            "screen_height"

#define GAMEPLAY_SECTION        "Gameplay"
#define USE_BALK_KEY            "balk"
#define BALK_X_KEY              "balk_x"
#define BALK_Y_KEY              "balk_y"
#define BPS_KEY                 "fps"

#define KEYBOARD_SECTION        "Keyboard"
#define KB_UP_KEY               "up"
#define KB_DOWN_KEY             "down"
#define KB_LEFT_KEY             "left"
#define KB_RIGHT_KEY            "right"
#define KB_BREATH_KEY           "breath"
#define KB_PILLAR_KEY           "pillar"
#define KB_SPIN_KEY             "spin"

/* configuration default values */
#define VSYNC_DEF               1
#define DRAW_ALWAYS_DEF         0
#define FPS_DEF                 0
#define FULLSCREEN_DEF          0
#define SCREEN_W_DEF            640
#define SCREEN_H_DEF            480

#define USE_BALK_DEF            1
#define BALK_X_DEF              6
#define BALK_Y_DEF              6
#define BPS_DEF                 40

#define KB_UP_DEF               KEY_UP
#define KB_DOWN_DEF             KEY_DOWN
#define KB_LEFT_DEF             KEY_LEFT
#define KB_RIGHT_DEF            KEY_RIGHT
#define KB_BREATH_DEF           KEY_ALT
#define KB_PILLAR_DEF           KEY_LCONTROL
#define KB_SPIN_DEF             KEY_Z



/* cheats */
#define CHEATS_SECTION          "Cheats"
#define IGNORE_TIME_KEY         "ignore_time"
#define IGNORE_TIME_DEF         0
#define INVINCIBILITY_KEY       "invincibility"
#define INVINCIBILITY_DEF       0



/* "on" if val is nonzero; "off" otherwise */
#define ON_OFF(val)         (val == 0 ? "off" : "on")




unsigned char apply_configs(const char *filename, Manager *const manager)
{

    //PALETTE *kickle_pal;

    write_to_log(manager->log, 0, "\nUpdating configurations.\n");

    /* apply graphics mode change if needed */
    if (manager->old_fullscreen != manager->fullscreen
        || manager->old_screen_w != manager->screen_w
        || manager->old_screen_h != manager->screen_h)
    {

        attempt_gfx_mode();

        #if 0
        write_to_log(manager->log, 0,
                     "\nChanging graphics mode to %i x %i x 256 %s.\n",
                     manager->screen_w, manager->screen_h,
                     (manager->fullscreen ? "fullscreen" : "windowed"));

        /* try to change into requested mode */
        if (set_gfx_mode((manager->fullscreen ? GFX_AUTODETECT_FULLSCREEN
                         : GFX_AUTODETECT_WINDOWED), manager->screen_w,
                         manager->screen_h, 0, 0) != 0)
        {

            write_to_log(manager->log, 1,
                         "Unable to change graphics mode:  %s\nAttempting to change back into previous mode, %i x %i x 256 %s.\n",
                         allegro_error, manager->old_screen_w,
                         manager->old_screen_h,
                         (manager->old_fullscreen ? "fullscreen" : "windowed"));
            allegro_message(
                "%s %s:  unable to change graphics mode to %i x %i x 256 %s.\n%s\nAttempting to change back into previous mode, %i x %i x 256 %s.\n",
                PROG_NAME, PROG_VER, manager->screen_w, manager->screen_h,
                (manager->fullscreen ? "fullscreen" : "windowed"),
                allegro_error, manager->old_screen_w, manager->old_screen_h,
                (manager->old_fullscreen ? "fullscreen" : "windowed"));

            manager->fullscreen= manager->old_fullscreen;
            manager->screen_w= manager->old_screen_w;
            manager->screen_h= manager->old_screen_h;

            /* try to change into previous mode */
            if (set_gfx_mode((manager->old_fullscreen
                             ? GFX_AUTODETECT_FULLSCREEN
                             : GFX_AUTODETECT_WINDOWED), manager->old_screen_w,
                             manager->old_screen_h, 0, 0) != 0)
            {

                write_to_log(manager->log, 1,
                             "Unable to change into previous graphics mode:  %s\nAttempting to change into safe mode.\n",
                             allegro_error);
                allegro_message(
                    "%s %s:  unable to change into previous graphics mode.\n%s\nAttempting to change into safe mode.\n",
                    PROG_NAME, PROG_VER, allegro_error);

                /* try to change into safe mode */
                if (set_gfx_mode(GFX_SAFE, manager->old_screen_w,
                                 manager->old_screen_h, 0, 0) != 0)
                {
                    write_to_log(manager->log, 1,
                                 "Unable to change into safe mode: %s\n",
                                 allegro_error);
                    allegro_message(
                        "%s %s:  unable to change into safe mode.\n%s\n",
                        PROG_NAME, PROG_VER, allegro_error);
                    return 1;
                }

                manager->screen_w= SCREEN_W;
                manager->screen_h= SCREEN_H;

            } /* end change into previous mode */

        } /* end change into requested mode */

        kickle_pal= (PALETTE *) manager->sprite_datafile[SPRITE_DATA_PAL].dat;
        set_palette(*kickle_pal);
        #endif

    } /* end apply mode change if needed */

    /* write to config file */
    write_to_log(manager->log, 0, "Writing configurations to file.\n");

    /* video configurations */
    set_config_int(VIDEO_SECTION, VSYNC_KEY, manager->use_vsync);
    set_config_int(VIDEO_SECTION, DRAW_ALWAYS_KEY, manager->draw_always);
    set_config_int(VIDEO_SECTION, FPS_KEY, manager->show_fps);
    set_config_int(VIDEO_SECTION, FULLSCREEN_KEY, manager->fullscreen);
    set_config_int(VIDEO_SECTION, SCREEN_W_KEY, manager->screen_w);
    set_config_int(VIDEO_SECTION, SCREEN_H_KEY, manager->screen_h);

    /* gameplay configurations */
    set_config_int(GAMEPLAY_SECTION, USE_BALK_KEY, manager->use_balk);
    set_config_int(GAMEPLAY_SECTION, BALK_X_KEY, manager->balk_x);
    set_config_int(GAMEPLAY_SECTION, BALK_Y_KEY, manager->balk_y);
    set_config_int(GAMEPLAY_SECTION, BPS_KEY, manager->bps);

    /* keyboard configurations */
    set_config_int(KEYBOARD_SECTION, KB_UP_KEY, manager->controls.key_up);
    set_config_int(KEYBOARD_SECTION, KB_DOWN_KEY, manager->controls.key_down);
    set_config_int(KEYBOARD_SECTION, KB_LEFT_KEY, manager->controls.key_left);
    set_config_int(KEYBOARD_SECTION, KB_RIGHT_KEY, manager->controls.key_right);
    set_config_int(KEYBOARD_SECTION, KB_BREATH_KEY,
                   manager->controls.key_breath);
    set_config_int(KEYBOARD_SECTION, KB_PILLAR_KEY,
                   manager->controls.key_pillar);
    set_config_int(KEYBOARD_SECTION, KB_SPIN_KEY, manager->controls.key_spin);

    flush_config_file();

    return 0;

} /* end apply_configs */




void load_config(const char *filename, Manager *const manager)
{

    /* check if the config file exists */
    if (exists(filename))
    {
        write_to_log(manager->log, 0, "Reading configuration file.\n");
    }
    else
    {
        write_to_log(manager->log, 0,
                     "\nConfiguration file not found.  Using defaults.\n");
    }

    set_config_file(filename);

    /* read configurations */

    /* video configurations */
    manager->use_vsync=
        (unsigned char) get_config_int(VIDEO_SECTION, VSYNC_KEY, VSYNC_DEF);
    manager->draw_always=
        (unsigned char)
        get_config_int(VIDEO_SECTION, DRAW_ALWAYS_KEY, DRAW_ALWAYS_DEF);
    manager->show_fps=
        (unsigned char) get_config_int(VIDEO_SECTION, FPS_KEY, FPS_DEF);
    manager->fullscreen=
        (unsigned char)
        get_config_int(VIDEO_SECTION, FULLSCREEN_KEY, FULLSCREEN_DEF);
    manager->screen_w=
        (unsigned short)
        get_config_int(VIDEO_SECTION, SCREEN_W_KEY, SCREEN_W_DEF);
    manager->screen_h=
        (unsigned short)
        get_config_int(VIDEO_SECTION, SCREEN_H_KEY, SCREEN_H_DEF);

    manager->old_fullscreen= manager->fullscreen;
    manager->old_screen_w= manager->screen_w;
    manager->old_screen_h= manager->screen_h;

    /* gameplay configurations */
    manager->use_balk=
        (unsigned char)
        get_config_int(GAMEPLAY_SECTION, USE_BALK_KEY, USE_BALK_DEF);
    manager->balk_x=
        (unsigned short)
        get_config_int(GAMEPLAY_SECTION, BALK_X_KEY, BALK_X_DEF);
    manager->balk_y=
        (unsigned short)
        get_config_int(GAMEPLAY_SECTION, BALK_Y_KEY, BALK_Y_DEF);
    manager->bps=
        (unsigned short) get_config_int(GAMEPLAY_SECTION, BPS_KEY, BPS_DEF);
    if (manager->bps == 0) manager->bps= BPS_DEF;

    /* keyboard configurations */
    manager->controls.breath_flag= 0;
    manager->controls.pillar_flag= 0;
    manager->controls.key_up=
        (unsigned char)
        get_config_int(KEYBOARD_SECTION, KB_UP_KEY, KB_UP_DEF);
    manager->controls.key_down=
        (unsigned char)
        get_config_int(KEYBOARD_SECTION, KB_DOWN_KEY, KB_DOWN_DEF);
    manager->controls.key_left=
        (unsigned char)
        get_config_int(KEYBOARD_SECTION, KB_LEFT_KEY, KB_LEFT_DEF);
    manager->controls.key_right=
        (unsigned char)
        get_config_int(KEYBOARD_SECTION, KB_RIGHT_KEY, KB_RIGHT_DEF);
    manager->controls.key_breath=
        (unsigned char)
        get_config_int(KEYBOARD_SECTION, KB_BREATH_KEY, KB_BREATH_DEF);
    manager->controls.key_pillar=
        (unsigned char)
        get_config_int(KEYBOARD_SECTION, KB_PILLAR_KEY, KB_PILLAR_DEF);
    manager->controls.key_spin=
        (unsigned char)
        get_config_int(KEYBOARD_SECTION, KB_SPIN_KEY, KB_SPIN_DEF);

    /* development/cheats */
    manager->ignore_time=
        (unsigned char)
        get_config_int(CHEATS_SECTION, IGNORE_TIME_KEY, IGNORE_TIME_DEF);
    manager->invincibility=
        (unsigned char)
        get_config_int(CHEATS_SECTION, INVINCIBILITY_KEY, INVINCIBILITY_DEF);

    /* write configurations to log */

    /* video */
    write_to_log(manager->log, 0, "Vsync %s.\n",
                 ON_OFF(manager->use_vsync));
    write_to_log(manager->log, 0, "Draw always %s.\n",
                 ON_OFF(manager->draw_always));
    write_to_log(manager->log, 0, "Show FPS %s.\n",
                 ON_OFF(manager->show_fps));
    write_to_log(manager->log, 0, "Fullscreen %s.\n",
                 ON_OFF(manager->fullscreen));
    write_to_log(manager->log, 0, "Screen width = %u.\n", manager->screen_w);
    write_to_log(manager->log, 0, "Screen height = %u.\n", manager->screen_h);

    /* gameplay */
    write_to_log(manager->log, 0, "Player balking %s.\n",
                 ON_OFF(manager->use_balk));
    if (manager->use_balk)
    {
        write_to_log(manager->log, 0, "Balk x = %u.\n", manager->balk_x);
        write_to_log(manager->log, 0, "Balk y = %u.\n", manager->balk_y);
    }
    write_to_log(manager->log, 0, "BPS = %u.\n", manager->bps);

    /* keyboard */
    write_to_log(manager->log, 0, "Keyboard up = %i.\n",
                 manager->controls.key_up);
    write_to_log(manager->log, 0, "Keyboard down = %i.\n",
                 manager->controls.key_down);
    write_to_log(manager->log, 0, "Keyboard left = %i.\n",
                 manager->controls.key_left);
    write_to_log(manager->log, 0, "Keyboard right = %i.\n",
                 manager->controls.key_right);
    write_to_log(manager->log, 0, "Keyboard breath = %i.\n",
                 manager->controls.key_breath);
    write_to_log(manager->log, 0, "Keyboard pillar = %i.\n",
                 manager->controls.key_pillar);
    write_to_log(manager->log, 0, "Keyboard spin = %i.\n",
                 manager->controls.key_spin);

} /* end load_config */
