/************************************
 *
 * Kickle 0.81
 * 2004 Drew Willcoxon
 * http://www.cs.uga.edu/~adw/
 * dripfeed@uga.edu
 *
 ************************************/


#include <string.h>
#include <stdlib.h>

#include "baddie.h"
#include "entity.h"
#include "feature.h"
#include "goodie.h"
#include "list.h"
#include "log.h"
#include "manager.h"
#include "map.h"
#include "player.h"
#include "video.h"




/* the maximum value of a and b */
#define MAX_VAL(a, b)   (a > b ? a : b)


/* the absolute value of a */
#define ABS_VAL(a)      (a < 0 ? -a : a)




/* the number of pixels per step that an ice block moves when kicked by a
   baddie */
#define BADDIE_KICK_SPEED       4




/* the game's log file */
//extern type_log *manager.log;

extern Manager manager;


/* so that baddies' points displays' text can be centered around the baddie
   nicely */
extern FONT *font_ingame;


void baddie_anim_done(Baddie *const baddie, Map *const map)
{

    //printf("%s enter anim_status = %i\n", __FUNCTION__, baddie->entity->status);
    //printf("editing = %i\n", manager.editing);

    if (manager.editing) return;

    switch (baddie->status)
    {

        //case BADDIE_S_STOPPED:
        //  set_baddie_status(baddie, BADDIE_S_MOVING, 0);
        //  break;

        //case BADDIE_S_FROZEN:
        //  set_baddie_status(baddie, BADDIE_S_MELTING, 0);
        //  break;

        //case BADDIE_S_MELTING:
        //  if (baddie->type == BADDIE_T_BADDIE1)
        //      set_baddie_status(baddie, BADDIE_S_KAPUT, 0);
        //  else
        //      set_baddie_status(baddie, BADDIE_S_STOPPED, 0);
        //  break;

        case BADDIE_S_SINKING:
            set_baddie_status(baddie, BADDIE_S_KAPUT, map);
            break;

        case BADDIE_S_SHATTERING:
            set_baddie_status(baddie, BADDIE_S_KAPUT, map);
            break;

        case BADDIE_S_NEW_ICE:
            make_new_ice(map, baddie->entity->row, baddie->entity->col);
            set_baddie_status(baddie, BADDIE_S_KAPUT, map);
            break;

        case BADDIE_S_SPAWNING:
            set_baddie_status(baddie, BADDIE_S_STOPPED, map);
            break;

        case BADDIE_S_KICKING:
            set_baddie_status(baddie, BADDIE_S_STOPPED, map);
            break;

        default:
            break;

    }

    //printf("%s exit anim_status = %i\n", __FUNCTION__, baddie->entity->status);

} /* end baddie_anim_done */




unsigned char baddie_baddie_coll(const baddie_type active_type,
                                 const baddie_status active_status,
                                 const baddie_type passive_type,
                                 const baddie_status passive_status)
{

    if (active_type == BADDIE_T_BADDIE5)
    {
        if (passive_type == BADDIE_T_BADDIE5) return 1;
        else return 0;
    }

    if (passive_type == BADDIE_T_BADDIE5) return 0;

    /* collision */
    return 1;

}




unsigned char baddie_feature_coll(const baddie_type b_type,
                                  const baddie_status b_status,
                                  const feature_type f_type,
                                  const feature_status f_status)
{

    switch (b_status)
    {

        case BADDIE_S_KICKED:
        case BADDIE_S_KICKED_MELTING:
        case BADDIE_S_STEAMROLLED:

            if (f_type == FEATURE_T_HOLE || f_type == FEATURE_T_BLACK_ICE)
            {
                return 0;
            }

            break;

        case BADDIE_S_SPAWNING:
            if (f_type == FEATURE_T_HIVE) return 0;
            break;

        default:

            if (f_type == FEATURE_T_BLACK_ICE)
            {
                if (b_type != BADDIE_T_BADDIE1) return 0;
            }

            break;

    }

    /* collision */
    return 1;

} /* end baddie_feature_col */




unsigned char baddie_goodie_coll(const baddie_type b_type,
                                 const baddie_status b_status,
                                 const goodie_type g_type,
                                 const goodie_status g_status)
{
    /* collision */
    return 1;
}




unsigned char baddie_player_coll(const baddie_type b_type,
                                 const baddie_status b_status,
                                 const player_status p_status)
{
    //printf("%s pstatus == %i\n", __FUNCTION__, p_status);
    switch (p_status)
    {
        case PLAYER_S_DYING:
        case PLAYER_S_WON:
        case PLAYER_S_KAPUT:
            return 0;
            break;
        default:
            return 1;
            break;
    }
    return 1;
}




#if 0
unsigned char baddie_tile_coll(const baddie_type b_type,
                               const baddie_status b_status,
                               const tile_type t_type)
{

    //if (b_type == BADDIE_T_BADDIE5)
    //{



    //}

    //else
    //{

        if (t_type == TILE_T_ICE) return 0;

        switch (b_status)
        {

            /* no collision */
            case BADDIE_S_KICKED:
            case BADDIE_S_KICKED_MELTING:
            case BADDIE_S_STEAMROLLED:
            case BADDIE_S_SINKING:
            case BADDIE_S_NEW_ICE:
                return 0;
                break;

            /* collision */
            default:
                return 1;
                break;

        }

    //}

    /* collision */
    return 1;

} /* end baddie_tile_coll */
#endif



unsigned char baddie_tile_coll(const baddie_type b_type,
                               const baddie_status b_status,
                               const Map *const map,
                               const unsigned char row,
                               const unsigned char col)
{

    Feature *feature;
    Goodie *goodie;

    unsigned char irow;
    unsigned char icol;

    /* check map bounds */
    if (row >= map->m_height || col >= map->m_width) return 1;

    /* if baddie5 */
    if (b_type == BADDIE_T_BADDIE5)
    {

        /* collision may not exist only if the specified tile is ice */
        if (map->tiles[row][col].type == TILE_T_ICE)
        {

            /* check map bounds */
            if (row == 0 || col == 0
                || row == map->m_height - 1 || col == map->m_width - 1)
            {
                return 0;
            }

            /* check the eight tiles around the specified tile */
            for (irow= row - 1; irow <= row + 1; irow++)
            {
                for (icol= col - 1; icol <= col + 1; icol++)
                {
                    /* don't check the specified tile */
                    if (irow != row || icol != col)
                    {

                        /* check the tile */
                        if (map->tiles[irow][icol].type == TILE_T_WATER)
                        {
                            //printf("water %s\n", __FUNCTION__);
                            return 0;
                        }

                        /* check for features */
                        feature= feature_on_tile(map, irow, icol);
                        if (feature != 0
                            && baddie_feature_coll(b_type, b_status,
                                                   feature->type,
                                                   feature->status))
                        {
                            return 0;
                        }

                        /* check for goodies */
                        goodie= goodie_on_tile(map, irow, icol);
                        if (goodie != 0
                            && baddie_goodie_coll(b_type, b_status,
                                                  goodie->type,
                                                  goodie->status))
                        {
                            return 0;
                        }

                    }
                }
            }

        } /* end if specified tile is ice */

    } /* end if baddie5 */

    /* else, if not baddie5 */
    else
    {

        if (map->tiles[row][col].type == TILE_T_ICE) return 0;

        switch (b_status)
        {

            /* no collision */
            case BADDIE_S_KICKED:
            case BADDIE_S_KICKED_MELTING:
            case BADDIE_S_STEAMROLLED:
            case BADDIE_S_SINKING:
            case BADDIE_S_NEW_ICE:
                return 0;
                break;

            /* collision */
            default:
                return 1;
                break;

        }

    }

    /* collision */
    return 1;

} /* end baddie_tile_coll */




Baddie *clone_baddie(const Baddie *const baddie, Map *const map,
                     const unsigned char row, const unsigned char col,
                     Feature *const hive)
{

    Baddie *clone=
        construct_baddie(baddie->type, row, col, baddie->entity->heading_x,
                         baddie->entity->heading_y, baddie->status,
                         hive, map);

    if (clone == 0)
        write_to_log(manager.log, 1, "Could not construct baddie clone.\n");

    return clone;

} /* end clone_baddie */




Baddie *construct_baddie(const baddie_type type, const unsigned char row,
                         const unsigned char col, const signed char heading_x,
                         const signed char heading_y,
                         const baddie_status status, Feature *const hive,
                         Map *const map)
{

    /* allocate baddie and entity memory */
    Baddie *baddie= (Baddie *) malloc(sizeof(Baddie));
    if (baddie == 0)
    {
        write_to_log(manager.log, 1, "Could not allocate baddie memory.\n");
        return 0;
    }

    baddie->entity= construct_entity();
    if (baddie->entity == 0)
    {
        write_to_log(manager.log, 1, "Could not construct baddie's entity.\n");
        destruct_baddie(baddie);
        return 0;
    }

    /* set entity members */
    baddie->entity->heading_x= heading_x;
    baddie->entity->heading_y= heading_y;

    /* set baddie members */
    baddie->type= type;
    baddie->status= status;
    baddie->hive= hive;
    baddie->num_goodies_hit= 0;
    baddie->frozen_counter= 0;
    baddie->kicked_by_player= 0;

    /* set members dependent on baddie type */
    switch (type)
    {

        case BADDIE_T_BADDIE1:
            baddie->entity->n_speed= 2;
            baddie->frozen_time= 1500;
            baddie->steamroll_points= 1000;
            baddie->shatter_points= 0;
            break;

        case BADDIE_T_BADDIE2:
            baddie->entity->n_speed= 1;
            baddie->frozen_time= 250;
            baddie->steamroll_points= 1500;
            baddie->shatter_points= 1000;
            break;

        case BADDIE_T_BADDIE3:
            baddie->entity->n_speed= 1;
            baddie->frozen_time= 250;
            baddie->steamroll_points= 1500;
            baddie->shatter_points= 1000;
            break;

        case BADDIE_T_BADDIE4:
            baddie->entity->n_speed= 1;
            baddie->frozen_time= 400;
            baddie->steamroll_points= 1500;
            baddie->shatter_points= 1000;
            break;

        case BADDIE_T_BADDIE5:
            baddie->entity->n_speed= 2;
            break;

    }

    /* animate him one frame */
    animate_baddie(baddie, map);

    /* finally, set his bounding box size and add him to map */
    if (map != 0)
    {
        set_bb_size(baddie->entity, map);
        if (add_baddie_to_map(baddie, map, row, col) > 0)
        {
            write_to_log(manager.log, 1, "Could not add baddie to map.\n");
            destruct_baddie(baddie);
            return 0;
        }
        //baddie->entity->status_row= row;
        //baddie->entity->status_col= col;
    }

    return baddie;

} /* end construct_baddie */




void destruct_baddie(Baddie *baddie)
{
    if (baddie->entity != 0) destruct_entity(baddie->entity);
    free(baddie);
    baddie= 0;
}




void do_baddie_logic(Baddie *const baddie, Map *const map,
                     const signed short target_row,
                     const signed short target_col)
{

    //write_to_log(manager.log, 0, "%p %i\n", baddie, baddie->entity->c_speed);

    #if 0
    if (baddie->entity->logic_status == BADDIE_LS_STEAMROLLED)
    {
        printf("%p steamrolled: screen x %i y %i\n",
            baddie,
            baddie->entity->screen_x,
            baddie->entity->screen_y);
    }
    if (baddie->entity->logic_status == BADDIE_LS_NORMAL)
    {
        printf("%p normal: screen x %i y %i\n",
            baddie,
            baddie->entity->screen_x,
            baddie->entity->screen_y);
    }
    #endif

    //Baddie *coll_baddie;
    //type_feature *coll_feature;
    //type_tile *coll_tile;
    //unsigned short i;



    /* if baddie sits squarely, update his square logic */
    if (baddie->entity->dx == 0 && baddie->entity->dy == 0)
        do_baddie_square(baddie, map, target_row, target_col);

    /* if baddie wants to move, do collision detection */
    if (baddie->entity->c_speed > 0) find_baddie_to_tile(baddie, map);
    if (baddie->entity->c_speed > 0) find_baddie_to_feature(baddie, map);
    if (baddie->entity->c_speed > 0) find_baddie_to_goodie(baddie, map);
    if (baddie->entity->c_speed > 0) find_baddie_to_baddie(baddie, map);
    if (baddie->entity->c_speed > 0 && manager.invincibility == 0)
        find_baddie_to_player(baddie, map);

    /* check out of bounds */
    if (baddie->status == BADDIE_S_MOVING
        && will_step_oob(baddie->entity, map))
    {
        set_baddie_status(baddie, BADDIE_S_STOPPED, map);
        turn_back_baddie(baddie, map);
    }

    #if 0
    {

        /* find baddie-to-baddie collision */
        coll_baddie= 0;
        for (i= 0; i < map->max_num_baddies && coll_baddie == 0; i++)
        {
            if (map->baddies[i] != 0
                && entity_to_entity_collision(baddie->entity,
                                              map->baddies[i]->entity, map,
                                              COLLISION_FUDGE_X,
                                              COLLISION_FUDGE_Y))
            {
                coll_baddie= map->baddies[i];
            }
        }

        /* find baddie-to-feature collision */
        coll_feature= 0;
        for (i= 0; i < map->max_num_features && coll_feature == 0; i++)
        {
            if (map->features[i] != 0
                && entity_to_entity_collision(baddie->entity,
                                              map->features[i]->entity, map,
                                              1, 1))
            {
                coll_feature= map->features[i];
                //printf("baddie-feature collision\n");
            }
        }

        /* find baddie-to-tile collision */
        coll_tile= baddie_to_tile_coll_tile(baddie, map);

        /* collision response logic */
        if (coll_baddie != 0)
            do_baddie_to_baddie_coll_logic(baddie, map, coll_baddie);
        if (coll_feature != 0)
            do_baddie_to_map_feature_coll_logic(baddie, map, coll_feature);
        if (coll_tile != 0)
            do_baddie_to_tile_coll_logic(baddie, map, coll_tile);

        /* check out of bounds */
        if (baddie->entity->logic_status == BADDIE_S_MOVING
            && entity_out_of_bounds(baddie->entity, map))
        {
            set_baddie_status(baddie, BADDIE_S_STOPPED, map);
            turn_back_baddie(baddie, map);
        }

    } /* end if baddie wants to move after his square logic */
    #endif

    /* if baddie is allowed to step after collision detection, step him */
    if (baddie->entity->c_speed > 0)
    {
        adjust_pos(baddie->entity, map);
        step_entity(baddie->entity, baddie->entity->c_speed, map);
    }

    /* if baddie is kicked, update frozen counter */
    if (baddie->status == BADDIE_S_KICKED)
    {
        ++baddie->frozen_counter;
        if (baddie->frozen_counter >= (baddie->frozen_time / 2))
            set_baddie_status(baddie, BADDIE_S_KICKED_MELTING, map);
    }
    else if (baddie->status == BADDIE_S_KICKED_MELTING)
    {
        ++baddie->frozen_counter;
        if (baddie->frozen_counter >= baddie->frozen_time)
            set_baddie_status(baddie, BADDIE_S_KAPUT, map);
    }

} /* end do_baddie_logic */




void do_baddie_square(Baddie *const baddie, Map *const map,
                      const signed short target_row,
                      const signed short target_col)
{

    /* update baddie logic based on his status */
    switch (baddie->status)
    {

        /* if baddie is stopped */
        case BADDIE_S_STOPPED:
            do_baddie_square_stopped(baddie, map, target_row, target_col);
            break;

        /* if baddie is frozen */
        case BADDIE_S_FROZEN:

            /* if baddie is a tanuki, he doesn't melt */
            if (baddie->type == BADDIE_T_BADDIE4)
            {

                /* see if tanuki has melted yet */
                if (baddie->frozen_counter >= baddie->frozen_time)
                    set_baddie_status(baddie, BADDIE_S_STOPPED, map);

                /* if not, increment his logic counter */
                else ++baddie->frozen_counter;

            }

            /* else, if baddie is not a tanuki */
            else
            {

                /* see if baddie is melting yet */
                if (baddie->frozen_counter >= (baddie->frozen_time / 2))
                    set_baddie_status(baddie, BADDIE_S_MELTING, map);

                /* if not, increment his logic counter */
                else ++baddie->frozen_counter;

            }

            break;

        /* if baddie is melting */
        case BADDIE_S_MELTING:

            /* see if baddie has melted yet */
            if (baddie->frozen_counter > baddie->frozen_time)
            {

                /* if baddie1, kill him */
                if (baddie->type == BADDIE_T_BADDIE1)
                    set_baddie_status(baddie, BADDIE_S_KAPUT, map);

                /* otherwise, unfreeze him */
                else set_baddie_status(baddie, BADDIE_S_STOPPED, map);

            }

            /* if not, increment his logic counter */
            else ++baddie->frozen_counter;

            break;

        /* if baddie is kicked (he's necessarily a baddie1) */
        case BADDIE_S_KICKED:
        case BADDIE_S_KICKED_MELTING:

            //printf("baddie %p square and KICKED\n", baddie);

            /* increase the amount of tiles this kicked ice block has moved */
            ++baddie->entity->logic_counter;

            /* give him speed */
            baddie->entity->c_speed= baddie->entity->n_speed;

            /* if baddie is overboard, make him new ice */
            if (map->tiles[baddie->entity->row]
                          [baddie->entity->col].type == TILE_T_WATER
                && baddie->entity->dx == 0 && baddie->entity->dy == 0)
            {
                set_baddie_status(baddie, BADDIE_S_NEW_ICE, map);
            }

            break; /* end if baddie is kicked */

        /* if baddie is moving normally */
        case BADDIE_S_MOVING:
            do_baddie_square_moving(baddie, map, target_row, target_col);
            break;

        /* if baddie is being steamrolled */
        case BADDIE_S_STEAMROLLED:

            /* if baddie is overboard, sink him */
            if (map->tiles[baddie->entity->row]
                          [baddie->entity->col].type == TILE_T_WATER)
            {
                set_baddie_status(baddie, BADDIE_S_SINKING, map);
            }

            break;

        /* if baddie is becoming new ice (he's a baddie1) */
        case BADDIE_S_NEW_ICE:

            //set_baddie_status(baddie, BADDIE_S_KAPUT, map);

            /* make new ice */
            //make_new_ice(map, baddie->entity->row, baddie->entity->col);

            break;

        default:
            break;


    } /* end baddie status switch */

} /* end do_baddie_square */




void do_baddie_square_moving(Baddie *const baddie, Map *const map,
                             const signed short target_row,
                             const signed short target_col)
{

    unsigned char diff_row;
    unsigned char diff_col;


    /* update logic based on baddie type */
    switch (baddie->type)
    {

        /* baddie1 */
        case BADDIE_T_BADDIE1:

            /* if no target, move randomly */
            if (target_row < 0 || target_col < 0)
            {
                set_random_heading(baddie, 20);
                set_baddie_status(baddie, BADDIE_S_STOPPED, map);
            }

            /* else, if there is a target */
            else
            {

                /* if within n tiles of target, track it */
                if (ABS_VAL((signed) (baddie->entity->row - target_row)) <= 3
                    && ABS_VAL((signed) (baddie->entity->col - target_col))
                       <= 3)
                {
                    if (has_back_to(baddie->entity, target_row, target_col))
                        face_target(baddie->entity, target_row, target_col);
                }

                else set_random_heading(baddie, 5);

                set_baddie_status(baddie, BADDIE_S_STOPPED, map);

            } /* end if there's a target */

            break;

        case BADDIE_T_BADDIE2:

            /* if no target, move randomly */
            if (target_row < 0 || target_col < 0)
                set_random_heading(baddie, 40);

            /* else, if there is a target */
            else
            {

                diff_row= ABS_VAL((signed) (baddie->entity->row - target_row));
                diff_col= ABS_VAL((signed) (baddie->entity->col - target_col));

                //printf("%s diff_row %i col %i\n", __FUNCTION__, diff_row,
                //     diff_col);

                /* if within n tiles of target, track it */
                if (diff_row <= 3 && diff_col <= 3)
                {
                    if (has_back_to(baddie->entity, target_row, target_col))
                    {
                        if ((diff_row <= 1 && diff_col <= 1) || rand() % 2 == 0)
                            face_target(baddie->entity, target_row, target_col);
                    }
                }

                else set_random_heading(baddie, 20);

                set_baddie_status(baddie, BADDIE_S_STOPPED, map);

            } /* end if there's a target */

            break;

        case BADDIE_T_BADDIE3:

            /* if no target, move randomly */
            if (target_row < 0 || target_col < 0)
                set_random_heading(baddie, 40);

            /* else, if there is a target */
            else
            {

                /* if within n tiles of target, track it */
                if (ABS_VAL((signed) (baddie->entity->row - target_row)) <= 3
                    && ABS_VAL((signed) (baddie->entity->col - target_col))
                       <= 3)
                {
                    if (has_back_to(baddie->entity, target_row, target_col)
                        && rand() % 2 == 0)
                    {
                        face_target(baddie->entity, target_row, target_col);
                    }
                }

                else set_random_heading(baddie, 20);

            } /* end if there's a target */

            break;

        case BADDIE_T_BADDIE4:

            /* if no target, move randomly */
            if (target_row < 0 || target_col < 0)
                set_random_heading(baddie, 40);

            /* else, if there is a target */
            else
            {

                diff_row= ABS_VAL((signed) (baddie->entity->row - target_row));
                diff_col= ABS_VAL((signed) (baddie->entity->col - target_col));

                /* if within n tiles of target, track it */
                if (diff_row <= 5 && diff_col <= 5)
                {
                    if (has_back_to(baddie->entity, target_row, target_col))
                    {
                        if ((diff_row <= 2 && diff_col <= 2) || rand() % 2 == 0)
                            face_target(baddie->entity, target_row, target_col);
                    }
                }

                else set_random_heading(baddie, 25);

            } /* end if there's a target */

            break;

        case BADDIE_T_BADDIE5:

            #if 0
            if (is_tile_empty_baddie(baddie->type, baddie->status, map,
                                     baddie->entity->row
                                     + baddie->entity->heading_x,
                                     baddie->entity->col
                                     + baddie->entity->heading_y) == 0)
            {
                set_heading(baddie->entity, baddie->entity->heading_y,
                            baddie->entity->heading_x);
            }

            else if (is_tile_empty_baddie(baddie->type, baddie->status, map,
                                          baddie->entity->row
                                          - baddie->entity->heading_x,
                                          baddie->entity->col
                                          - baddie->entity->heading_y) == 0)
            {
                set_heading(baddie->entity, -baddie->entity->heading_y,
                            -baddie->entity->heading_x);
            }
            #endif

            break;

        default:
            break;

    } /* end switch on baddie type */

} /* end do_baddie_square_moving */




void do_baddie_square_stopped(Baddie *const baddie, Map *const map,
                              const signed short target_row,
                              const signed short target_col)
{

    unsigned char b_adj_row;
    unsigned char b_adj_col;

    Baddie *adj_baddie;

    /* update logic based on baddie type */
    switch (baddie->type)
    {

        /* baddie1 */
        case BADDIE_T_BADDIE1:

            //#if 0
            /* update logic counter ... */
            if (baddie->entity->logic_counter > 20)
            {
                baddie->entity->logic_counter= 0;
                set_baddie_status(baddie, BADDIE_S_MOVING, map);
            }
            else ++baddie->entity->logic_counter;
            //#endif

            break;

        /* baddie2 */
        case BADDIE_T_BADDIE2:

            /* see if there's an ice block in front of the baddie; if so, kick
               it */

            b_adj_row= adj_row(baddie->entity);
            b_adj_col= adj_col(baddie->entity);

            if (b_adj_row < map->m_height && b_adj_col < map->m_width)
            {

                adj_baddie=
                    baddie_on_tile(map, b_adj_row, b_adj_col, map->t_width,
                                   map->t_height);

                if (adj_baddie != 0 && adj_baddie->type == BADDIE_T_BADDIE1
                    && (adj_baddie->status == BADDIE_S_FROZEN
                        || adj_baddie->status == BADDIE_S_MELTING))
                {

                    adj_baddie->kicked_by_player= 0;
                    set_baddie_status(adj_baddie,
                        (adj_baddie->status == BADDIE_S_FROZEN
                         ? BADDIE_S_KICKED : BADDIE_S_KICKED_MELTING), map);
                    adj_baddie->entity->n_speed= BADDIE_KICK_SPEED;
                    set_heading(adj_baddie->entity, baddie->entity->heading_x,
                                baddie->entity->heading_y);
                    set_baddie_status(baddie, BADDIE_S_KICKING, map);

                }

                else set_baddie_status(baddie, BADDIE_S_MOVING, map);

            }

            else set_baddie_status(baddie, BADDIE_S_MOVING, map);

            break;

        case BADDIE_T_BADDIE4:

            if (baddie->entity->logic_counter > 20)
            {
                baddie->entity->logic_counter= 0;
                set_baddie_status(baddie, BADDIE_S_MOVING, map);
            }
            else ++baddie->entity->logic_counter;

            break;

        default:
            set_baddie_status(baddie, BADDIE_S_MOVING, map);
            break;

    } /* end switch on baddie type */

} /* end do_baddie_square_stopped */




/*

make sure to check for collisions between ice blocks in which ice blocks
are moving in same direction and speed -- these should be ignored


ice blocks:

when set_baddie_status(baddie, BADDIE_S_KICKED) is called,
baddie->entity->logic is set to 0; on square logic, logic_counter is inc'ed.

if KICKED ice block collides with a nonmoving (FROZEN) ice block,
set the FROZEN block and all the ones in front of it to KICKED

if KICKED ice block collides with an obstacle (feature, etc) and its
logic_counter <= 1, set it to SHATTERING, and check behind it.  If behind
it there are KICKED ice blocks with the same direction whose logic_counters
are <= 1, set them to shattered too

OR?

if KICKED ice block collides with a nonmoving (FROZEN/MELTING) ice block
1. if the KICKED block has logic_counter <= 1
    1. if at the end of the ice block chain there is an obstacle, set everything
       to SHATTERING
    2. if there is nothing at the end of the chain, set the rest of the chain
       to KICKED
2. else
    1. stop the KICKED ice block -- set it to FROZEN (or MELTING)
    2. if there is nothing at the end of the ice block chain, set the last
       block to KICKED

*/
void do_baddie_to_baddie(Baddie *const active, const Map *const map,
                         Baddie *const passive)
{

    unsigned char row;
    unsigned char col;

    unsigned char row2;
    unsigned char col2;

    unsigned char found;
    unsigned char action;

    Baddie *baddie;
    //type_feature *feature;

    /* update active's and passive's logic based on active's status */
    switch (active->status)
    {

        /* if active is kicked (baddie1) */
        case BADDIE_S_KICKED:
        case BADDIE_S_KICKED_MELTING:

            /* update active's and passive's logic based on passive's status */
            switch (passive->status)
            {

                /* if passive is moving normally, set him flush, then continue
                   down to the case where passive is stopped */
                case BADDIE_S_MOVING:
                    set_flush(active->entity, passive->entity, map);

                    /* NO BREAK: continue as if passive were stopped */

                /* if passive is stopped (he's also necessarily square) */
                case BADDIE_S_STOPPED:

                    //printf("active c_speed %i\n", active->entity->c_speed);

                    /* set passive's status, and speed and heading based on
                       active's */
                    passive->kicked_by_player= active->kicked_by_player;
                    set_baddie_status(passive, BADDIE_S_STEAMROLLED, map);
                    passive->entity->c_speed= active->entity->c_speed;
                    set_heading(passive->entity, active->entity->heading_x,
                                active->entity->heading_y);

                    break;

                /* if passive is frozen or melting (he's also necessarily
                   square) */
                case BADDIE_S_FROZEN:
                case BADDIE_S_MELTING:

                    /* if passive is also an ice block */
                    if (passive->type == BADDIE_T_BADDIE1)
                    {

                        /* find the end of the ice block chain */
                        found= 0;
                        row= passive->entity->row;
                        col= passive->entity->col;
                        while (!found
                               && row < map->m_height && col < map->m_width)
                        {
                            row += active->entity->heading_y;
                            col += active->entity->heading_x;
                            baddie= baddie_on_tile(map, row, col, map->t_width,
                                                   map->t_height);
                            if (baddie == 0 || baddie->type != BADDIE_T_BADDIE1
                                || (baddie->status != BADDIE_S_FROZEN
                                    && baddie->status
                                    != BADDIE_S_MELTING))
                            {
                                found= 1;
                            }
                        }

                        printf("end of chain at row %i col %i\n", row, col);
                        printf("active's logic counter == %i\n",
                            active->entity->logic_counter);

                        /* action:
                           0 = shatter all blocks in the chain
                           1 = kick all the blocks in the chain
                           2 = stop active and kick only the last block in the
                               chain
                           3 = stop active */

                        action= 0;

                        /* if the end of the chain is outside map boundaries,
                           we should kick the blocks */
                        if (row >= map->m_height || col >= map->m_width)
                            action= 1;

                        /* determine appropriate action */
                        if (action != 1)
                        {

                            //feature= feature_on_tile(map, row, col);

                            /* if the tile at end of the chain is walkable by a
                               kicked ice block */
                            if (is_tile_empty_baddie(BADDIE_T_BADDIE1,
                                                     BADDIE_S_KICKED, map, row,
                                                     col))
                            {
                                /* if active was kicked adjacent to passive */
                                if (active->entity->logic_counter <= 1)
                                    action= 1;
                                else
                                    action= 2;
                            }

                            /* else if the tile is not walkable */
                            else
                            {
                                /* if active was kicked adjacent to passive */
                                if (active->entity->logic_counter <= 1)
                                    action= 0;
                                else
                                    action= 3;
                            }

                        } /* end if we need to determine action */

                        printf("action %i\n", action);

                        /* act on all the blocks in the chain if necessary */
                        if (action == 0 || action == 1)
                        {

                            //
                            row2= passive->entity->row;
                            col2= passive->entity->col;
                            printf("row2 %i col2 %i row %i col %i\n",
                                row2, col2, row, col);
                            //

                            for (row2= passive->entity->row,
                                 col2= passive->entity->col;
                                 row2 != row || col2 != col;
                                 row2 += active->entity->heading_y,
                                 col2 += active->entity->heading_x)
                            {

                                printf("iter\n");

                                baddie= baddie_on_tile(map, row2, col2,
                                                       map->t_width,
                                                       map->t_height);

                                /* if kicking the chain */
                                if (action == 1)
                                {

                                    printf("kicking block in chain at row %i col %i\n", row2, col2);

                                    set_baddie_status(baddie,
                                        (baddie->status
                                         == BADDIE_S_FROZEN
                                         ? BADDIE_S_KICKED
                                         : BADDIE_S_KICKED_MELTING),
                                        map);

                                    baddie->entity->n_speed=
                                        active->entity->c_speed;
                                    set_heading(baddie->entity,
                                                active->entity->heading_x,
                                                active->entity->heading_y);

                                }

                                /* if shattering the chain */
                                else
                                {
                                    set_baddie_status(baddie,
                                                      BADDIE_S_SHATTERING,
                                                      map);
                                }

                            } /* end for loop on ice block chain */

                        } /* end if we need to act on all the whole chain */

                        /* if we need to kick only the last block */
                        if (action == 2)
                        {

                            baddie=
                                baddie_on_tile(map,
                                    row - active->entity->heading_y,
                                    col - active->entity->heading_x,
                                    map->t_width, map->t_height);

                            set_baddie_status(baddie,
                                (baddie->status == BADDIE_S_FROZEN
                                 ? BADDIE_S_KICKED : BADDIE_S_KICKED_MELTING),
                                 map);

                            baddie->entity->n_speed= active->entity->n_speed;
                            set_heading(baddie->entity,
                                        active->entity->heading_x,
                                        active->entity->heading_y);

                        }

                        /* if we need to shatter active */
                        if (action == 0)
                            set_baddie_status(active, BADDIE_S_SHATTERING, map);

                        /* if we need to stop active */
                        if (action == 2 || action == 3)
                        {
                            set_baddie_status(active,
                                (active->status == BADDIE_S_KICKED
                                 ? BADDIE_S_FROZEN : BADDIE_S_MELTING), map);
                            //active->entity->c_speed= 0;
                            //active->entity->n_speed= 0;
                        }

                        /* if active remains kicked, set his speed to 0 so he
                           won't get ahead of passive */
                        if (action == 1)
                            active->entity->c_speed= 0;

                    } /* end if passive is also an ice block */

                    /* else, if passive is not an ice block, shatter passive */
                    else set_baddie_status(passive, BADDIE_S_SHATTERING, map);

                    break; /* end if passive is frozen or melting */

                /* if passive is kicked or kicked melting (baddie1) */
                case BADDIE_S_KICKED:
                case BADDIE_S_KICKED_MELTING:

                    /* if active and passive are colliding going the same
                       direction */
                    if (active->entity->heading_y == passive->entity->heading_y
                        && active->entity->heading_x
                           == passive->entity->heading_x)
                    {
                        /* zero active's speed -- if it's proper he'll pick up
                           his speed again on his square logic */
                        active->entity->c_speed= 0;
                    }

                    /* else, if active and passive are colliding head-on,
                       shatter both */
                    else if (active->entity->heading_y
                             == -passive->entity->heading_y
                             && active->entity->heading_x
                                == -passive->entity->heading_x)
                    {
                        set_baddie_status(active, BADDIE_S_SHATTERING, map);
                        set_baddie_status(passive, BADDIE_S_SHATTERING, map);
                    }

                    /* else, if they're not colliding head-on nor in the same
                       direction */
                    else
                    {

                        /* set passive flush with active */
                        set_flush(active->entity, passive->entity, map);

                        /* set passive's heading to active's */
                        set_heading(passive->entity, active->entity->heading_x,
                                    active->entity->heading_y);

                        /* stop active */
                        set_baddie_status(active,
                                          (active->status
                                           == BADDIE_S_KICKED ? BADDIE_S_FROZEN
                                           : BADDIE_S_MELTING), map);

                    }

                    break; /* end if passive kicked or kicked melting */

                default:
                    break;

            } /* end switch on passive's status */

            break; /* end if active is kicked or kicked melting */

        /* if active is moving normally, try a new way to go or turn back */
        case BADDIE_S_MOVING:
            //if (passive->entity->status == BADDIE_S_MOVING
            //  && passive->entity->heading_x == -active->entity->heading_x
            //  && passive->entity->heading_y == -active->entity->heading_y
            //  && (passive->entity->dx > 0 || passive->entity->dy > 0))
            //{
            //  turn_back_baddie(passive, map);
            //}
            turn_back_baddie(active, map);
            break;

        /* if active is steamrolled */
        case BADDIE_S_STEAMROLLED:

            /* update active's and passive's logic based on passive's status */
            switch (passive->status)
            {

                /* if passive is frozen, kill active */
                case BADDIE_S_FROZEN:
                case BADDIE_S_MELTING:
                    set_baddie_status(active, BADDIE_S_KAPUT, map);
                    break;

                /* if passive is moving normally, steamroll passive as well */
                case BADDIE_S_MOVING:

                    /* set passive flush with the row/column which active is
                       on */
                    set_flush(active->entity, passive->entity, map);

                    /* NO BREAK: continue into next case as if passive were
                       stopped */

                /* if passive is stopped, steamroll passive as well */
                case BADDIE_S_STOPPED:

                    //printf("active c_speed %i\n", active->entity->c_speed);

                    /* set passive's status, and speed and heading based on
                       active's */
                    set_baddie_status(passive, BADDIE_S_STEAMROLLED, map);
                    passive->entity->c_speed= active->entity->c_speed;
                    passive->entity->n_speed= active->entity->c_speed;
                    set_heading(passive->entity, active->entity->heading_x,
                                active->entity->heading_y);

                    break;

                default:
                    break;

            } /* end switch on passive's status */

            break; /* end if active is steamrolled */

        default:
            break;

    } /* end switch on active's status */

} /* end do_baddie_to_baddie */




#if 0
void do_baddie_to_feature(type_baddie *const baddie, const type_map *const map,
                          type_feature *const feature)
{

    /* update baddie logic based on his status */
    switch (baddie->entity->status)
    {

        /* if baddie is kicked, set him back to frozen (baddie1) */
        case BADDIE_S_KICKED:
        case BADDIE_S_KICKED_MELTING:

            /* update logic based on feature type */
            switch (feature->type)
            {

                case FEATURE_T_HOLE:
                case FEATURE_T_BLACK_ICE:
                    /* do nothing */
                    break;

                case FEATURE_T_SPRING:

                    /* the spring only works if it's hit on the front or back */
                    if ((feature->entity->heading_x == 0
                         && baddie->entity->heading_x == 0)
                        || (feature->entity->heading_y == 0
                            && baddie->entity->heading_y == 0))
                    {

                        /* make spring face the ice block */
                        set_heading(feature->entity, -baddie->entity->heading_x,
                                    -baddie->entity->heading_y);

                        set_feature_status(feature, FEATURE_S_SPRINGING);

                        /* set ice block back to frozen; it will be kicked when
                           the spring's animation is complete */
                        set_baddie_status(baddie,
                                          (baddie->entity->status
                                           == BADDIE_S_KICKED
                                           ? BADDIE_S_FROZEN
                                           : BADDIE_S_MELTING),
                                          map);

                        break;

                    }

                    /* otherwise, NO BREAK; drop through and handle collision
                       like collision with a wall */

                default:

                    if (baddie->entity->logic_counter <= 1)
                        set_baddie_status(baddie, BADDIE_S_SHATTERING, map);
                    else
                    {
                        set_baddie_status(baddie,
                                          (baddie->entity->status
                                           == BADDIE_S_KICKED
                                           ? BADDIE_S_FROZEN
                                           : BADDIE_S_MELTING),
                                          map);
                    }

                    break;

            } /* end switch on feature type */

            break; /* end if baddie is kicked */

        /* if baddie is steamrolled, crush him */
        case BADDIE_S_STEAMROLLED:

            /* update logic based on feature type */
            switch (feature->type)
            {

                case FEATURE_T_HOLE:
                case FEATURE_T_BLACK_ICE:
                    /* do nothing */
                    break;

                default:
                    set_baddie_status(baddie, BADDIE_S_KAPUT, map);
                    break;

            }

            break;

        /* if baddie is moving normally, try to find a new way to go */
        case BADDIE_S_MOVING:
            if (feature->type == FEATURE_T_BLACK_ICE)
            {
                if (baddie->type == BADDIE_T_BADDIE1)
                    turn_back_baddie(baddie, map);
            }
            else turn_back_baddie(baddie, map);
            break;

    } /* end switch on baddie status */

} /* end do_baddie_to_feature */
#endif



void do_baddie_to_feature(Baddie *const baddie, const Map *const map,
                          Feature *const feature)
{

    /* update baddie logic based on his status */
    switch (baddie->status)
    {

        /* if baddie is kicked, set him back to frozen (baddie1) */
        case BADDIE_S_KICKED:
        case BADDIE_S_KICKED_MELTING:

            /* update logic based on feature type */
            switch (feature->type)
            {

                case FEATURE_T_SPRING:

                    /* the spring only works if it's hit on the front or back */
                    if ((feature->entity->heading_x == 0
                         && baddie->entity->heading_x == 0)
                        || (feature->entity->heading_y == 0
                            && baddie->entity->heading_y == 0))
                    {

                        /* make spring face the ice block */
                        set_heading(feature->entity, -baddie->entity->heading_x,
                                    -baddie->entity->heading_y);

                        set_feature_status(feature, FEATURE_S_SPRINGING);

                        /* set ice block back to frozen; it will be kicked when
                           the spring's animation is complete */
                        set_baddie_status(baddie,
                                          (baddie->status
                                           == BADDIE_S_KICKED
                                           ? BADDIE_S_FROZEN
                                           : BADDIE_S_MELTING),
                                          map);

                        break;

                    }

                    /* otherwise, NO BREAK; drop through and handle collision
                       like collision with a wall */

                default:

                    if (baddie->entity->logic_counter <= 1)
                        set_baddie_status(baddie, BADDIE_S_SHATTERING, map);
                    else
                    {
                        set_baddie_status(baddie,
                                          (baddie->status
                                           == BADDIE_S_KICKED
                                           ? BADDIE_S_FROZEN
                                           : BADDIE_S_MELTING),
                                          map);
                    }

                    break;

            } /* end switch on feature type */

            break; /* end if baddie is kicked */

        /* if baddie is steamrolled, crush him */
        case BADDIE_S_STEAMROLLED:
            set_baddie_status(baddie, BADDIE_S_KAPUT, map);
            break;

        /* if baddie is moving normally, try to find a new way to go */
        case BADDIE_S_MOVING:
            turn_back_baddie(baddie, map);
            break;

        default:
            break;

    } /* end switch on baddie status */

} /* end do_baddie_to_feature */




void do_baddie_to_goodie(Baddie *const baddie, const Map *const map,
                         Goodie *const goodie)
{

    /* update baddie logic based on his status */
    switch (baddie->status)
    {

        /* if baddie is steamrolled or kicked, move the goodie */
        case BADDIE_S_KICKED:
        case BADDIE_S_KICKED_MELTING:
        case BADDIE_S_STEAMROLLED:

            /* if the goodie is a bag, move it; else, collect it */
            if (goodie->type == GOODIE_T_BAG)
            {
                move_goodie(goodie, map, -baddie->entity->heading_x,
                            -baddie->entity->heading_y);
            }

            /* else if the goodie is not a bag */
            else
            {
                ++baddie->num_goodies_hit;
                collect_goodie(goodie, map->player, map,
                               baddie->num_goodies_hit + 1);
            }

            break;

        /* if baddie is moving normally, try to find a new way to go */
        case BADDIE_S_MOVING:
            turn_back_baddie(baddie, map);
            break;

        default:
            break;

    } /* end switch on baddie status */

} /* end do_baddie_to_goodie */




void do_baddie_to_player(Baddie *const baddie, const Map *const map,
                         Player *const player)
{

    set_player_status(player, PLAYER_S_DYING, 0);
}




void do_baddie_to_tile(Baddie *const baddie, const Map *const map,
                       const unsigned char row, const unsigned char col)
{
    turn_back_baddie(baddie, map);
} /* end do_baddie_to_tile */




void find_baddie_to_baddie(Baddie *const baddie, const Map *const map)
{

    Baddie *coll_baddie;
    //unsigned short i;

    unsigned char coll_fudge_x;
    unsigned char coll_fudge_y;

    List_iterator *iterator;

    /* loop while there's more baddies and baddie wants to move */
    coll_baddie= 0;
    #if 0
    for (i= 0; i < map->max_num_baddies && coll_baddie == 0
         && baddie->entity->c_speed > 0; i++)
    {

        if (map->baddies[i] != 0)
        {

            /* use collision fudge only if map->baddies[i] is moving */
            if (map->baddies[i]->entity->status == BADDIE_S_MOVING)
            {
                coll_fudge_x= COLL_FUDGE_X;
                coll_fudge_y= COLL_FUDGE_Y;
            }

            /* otherwise, use no fudge */
            else
            {
                coll_fudge_x= 1;
                coll_fudge_y= 1;
            }

            if (will_entities_collide(baddie->entity, map->baddies[i]->entity,
                                      map, coll_fudge_x, coll_fudge_y))
            {
                coll_baddie= map->baddies[i];
                //do_baddie_to_baddie(baddie, map, map->baddies[i]);
            }

        }

    } /* end for loop */
    #endif

    iterator= get_iterator(map->baddie_list);
    while (iterator != 0 && coll_baddie == 0 && baddie->entity->c_speed > 0)
    {

        coll_baddie= (Baddie *) iterator->dat;

        /* use collision fudge only if map->baddies[i] is moving */
        if (coll_baddie->status == BADDIE_S_MOVING)
        {
            coll_fudge_x= COLL_FUDGE_X;
            coll_fudge_y= COLL_FUDGE_Y;
        }

        /* otherwise, use no fudge */
        else
        {
            coll_fudge_x= 1;
            coll_fudge_y= 1;
        }

        if (will_entities_collide(baddie->entity, coll_baddie->entity,
                                  map, coll_fudge_x, coll_fudge_y) == 0
            || baddie_baddie_coll(baddie->type, baddie->status,
                                  coll_baddie->type,
                                  coll_baddie->status) == 0)
        {
            coll_baddie= 0;
            //do_baddie_to_baddie(baddie, map, map->baddies[i]);
        }

        iterator= next_iterator(iterator);

    }

    if (coll_baddie != 0)
        do_baddie_to_baddie(baddie, map, coll_baddie);

} /* end find_baddie_to_baddie */




void find_baddie_to_feature(Baddie *const baddie, const Map *const map)
{

    #if 0
    unsigned short i;

    /* loop while there's more features and baddie wants to move */
    for (i= 0; i < map->max_num_features && baddie->entity->c_speed > 0; i++)
    {
        if (map->features[i] != 0
            && will_entities_collide(baddie->entity, map->features[i]->entity,
                                     map, 1, 1))
        {
            do_baddie_to_feature(baddie, map, map->features[i]);
        }
    }
    #endif

    Feature *feature;

    List_iterator *iterator= get_iterator(map->feature_list);

    while (iterator != 0 && baddie->entity->c_speed > 0)
    {

        feature= (Feature *) iterator->dat;

        if (will_entities_collide(baddie->entity, feature->entity, map, 1, 1)
            && baddie_feature_coll(baddie->type, baddie->status,
                                   feature->type, feature->status))
        {
            do_baddie_to_feature(baddie, map, feature);
        }

        iterator= next_iterator(iterator);

    }

} /* end find_baddie_to_feature */




void find_baddie_to_goodie(Baddie *const baddie, const Map *const map)
{

    #if 0
    unsigned short i;

    /* loop while there's more goodies and baddie wants to move */
    for (i= 0; i < map->max_num_goodies && baddie->entity->c_speed > 0; i++)
    {
        if (map->goodies[i] != 0
            && will_entities_collide(baddie->entity, map->goodies[i]->entity,
                                     map, 1, 1))
        {
            do_baddie_to_goodie(baddie, map, map->goodies[i]);
        }
    }
    #endif

    Goodie *goodie;

    List_iterator *iterator= get_iterator(map->goodie_list);

    while (iterator != 0 && baddie->entity->c_speed > 0)
    {

        goodie= (Goodie *) iterator->dat;

        if (will_entities_collide(baddie->entity, goodie->entity, map, 1, 1)
            && baddie_goodie_coll(baddie->type, baddie->status,
                                  goodie->type, goodie->status))
        {
            do_baddie_to_goodie(baddie, map, goodie);
        }

        iterator= next_iterator(iterator);

    }

} /* end find_baddie_to_goodie */




void find_baddie_to_player(Baddie *const baddie, const Map *const map)
{

    unsigned char coll_fudge_x;
    unsigned char coll_fudge_y;

    if (baddie->entity->c_speed > 0 && map->player != 0)
    {

        if (baddie->status == BADDIE_S_MOVING)
        {
            coll_fudge_x= COLL_FUDGE_X;
            coll_fudge_y= COLL_FUDGE_Y;
        }
        else
        {
            coll_fudge_x= 1;
            coll_fudge_y= 1;
        }

        if (will_entities_collide(baddie->entity, map->player->entity,
                                  map, coll_fudge_x, coll_fudge_y)
            && baddie_player_coll(baddie->type, baddie->status,
                                  map->player->status))
        {
            do_baddie_to_player(baddie, map, map->player);
        }

    }

} /* end find_baddie_to_player */




void find_baddie_to_tile(Baddie *const baddie, const Map *const map)
{

    unsigned char row;
    unsigned char col;

    if (baddie->entity->dx > 0 || baddie->entity->dy > 0) return;

    if (baddie->entity->heading_y != 0)
    {

        row= adj_row(baddie->entity);

        for (col= baddie->entity->col;
             col < baddie->entity->col + baddie->entity->bb_t_width
             && baddie->entity->c_speed > 0; col++)
        {
            if (row < map->m_height && col < map->m_width
                && baddie_tile_coll(baddie->type, baddie->status,
                                    map, row, col))
            {
                do_baddie_to_tile(baddie, map, row, col);
            }
        }

    }

    if (baddie->entity->heading_x != 0)
    {

        col= adj_col(baddie->entity);

        for (row= baddie->entity->row;
             row < baddie->entity->row + baddie->entity->bb_t_height
             && baddie->entity->c_speed > 0; row++)
        {
            if (row < map->m_height && col < map->m_width
                && baddie_tile_coll(baddie->type, baddie->status,
                                    map, row, col))
            {
                do_baddie_to_tile(baddie, map, row, col);
            }
        }

    }

} /* end find_baddie_to_tile */




void set_baddie_status(Baddie *const baddie, baddie_status status,
                       const Map *const map)
{

    Floating_text *points;
    char str[6];

    switch (status)
    {

        /* if stopping baddie normally */
        case BADDIE_S_STOPPED:

            baddie->status= BADDIE_S_STOPPED;
            baddie->entity->c_speed= 0;

            //baddie->entity->anim_counter= 0;
            //baddie->entity->anim_frame_counter= 0;
            //baddie->entity->anim_play_counter= 0;
            reset_anim(baddie->entity);

            //baddie->entity->logic_counter= 0;

            //baddie->status_row= baddie->entity->row;
            //baddie->status_col= baddie->entity->col;

            break;

        /* if freezing baddie */
        case BADDIE_S_FROZEN:
        case BADDIE_S_MELTING:

            /* if freezing, reset his logic counter */
            if (status == BADDIE_S_FROZEN
                && baddie->status != BADDIE_S_KICKED
                && baddie->status != BADDIE_S_KICKED_MELTING
                && (baddie->type != BADDIE_T_BADDIE4
                    || (baddie->type == BADDIE_T_BADDIE4
                        && baddie->status != BADDIE_S_FROZEN)))
            {
                baddie->frozen_counter= 0;
            }

            //baddie->entity->logic_counter= 0;

            //if (baddie->entity->status == BADDIE_S_KICKED
            //  || baddie->entity->status== BADDIE_S_KICKED_MELTING)
            //{


            //}

            //else
            //{


            //#if 0
            //if (baddie->entity->logic_status != BADDIE_S_FROZEN)
            //if (baddie->entity->status == BADDIE_S_MOVING)
            //{

                //baddie->entity->logic_status= BADDIE_S_FROZEN;
                //baddie->entity->status= status;

            if (map != 0)
            {

                if (baddie->entity->dx >= map->t_width / 2
                    && is_tile_empty_baddie(baddie->type,
                                            baddie->status, map,
                                            baddie->entity->row,
                                            baddie->entity->col + 1))
                {
                    square_entity(baddie->entity, baddie->entity->row,
                                  baddie->entity->col + 1, map);
                }
                else if (baddie->entity->dx > 0)
                {
                    square_entity(baddie->entity, baddie->entity->row,
                                  baddie->entity->col, map);
                }

                if (baddie->entity->dy >= map->t_height / 2
                    && is_tile_empty_baddie(baddie->type,
                                            baddie->status, map,
                                            baddie->entity->row + 1,
                                            baddie->entity->col))
                {
                    square_entity(baddie->entity, baddie->entity->row + 1,
                                  baddie->entity->col, map);
                }
                else if (baddie->entity->dy > 0)
                {
                    square_entity(baddie->entity, baddie->entity->row,
                                  baddie->entity->col, map);
                }

            }

                baddie->entity->c_speed= 0;
                //baddie->entity->n_speed= 0;

            //}
            //#endif

            //}


            reset_anim(baddie->entity);
            baddie->status= status;

            //baddie->entity->anim_counter= 0;
            //baddie->entity->anim_frame_counter= 0;
            //baddie->entity->anim_play_counter= 0;


            //baddie->status_row= baddie->entity->row;
            //baddie->status_col= baddie->entity->col;

            #if 0
            /* handle freezing depending on baddie status */
            switch (baddie->entity->logic_status)
            {

                /* set him to frozen if ... */
                case BADDIE_S_STOPPED:
                case BADDIE_S_MELTING:
                    baddie->entity->logic_status= BADDIE_S_FROZEN;
                    break;

                /* if kicked_melting, set him to kicked_frozen */
                //case BADDIE_S_KICKED_MELTING:
                //  baddie->entity->logic_status= BADDIE_S_KICKED;
                //  break;

                /* set him to frozen, square him, and stop him if ... */
                case BADDIE_S_MOVING:
                case BADDIE_S_KICKED: /* needed when we stop a kicked block */
                case BADDIE_S_KICKED_MELTING: /* likewise */

                    baddie->entity->logic_status=
                        (baddie->entity->logic_status ==
                         BADDIE_S_KICKED_MELTING ? BADDIE_S_MELTING
                         : BADDIE_S_FROZEN);

                    if (baddie->entity->d_screen_x >= map->tile_width / 2)
                        set_entity_flush_col_right(baddie->entity, map);
                    else if (baddie->entity->d_screen_x > 0)
                        set_entity_flush_col_left(baddie->entity, map);
                    else if (baddie->entity->d_screen_y >= map->tile_height / 2)
                        set_entity_flush_row_down(baddie->entity, map);
                    else if (baddie->entity->d_screen_y > 0)
                        set_entity_flush_row_up(baddie->entity, map);

                    baddie->entity->current_speed= 0;

                    break;

            } /* end switch on baddie status if freezing baddie */
            #endif

            break; /* end if freezing baddie */

        #if 0
        /* if melting baddie */
        case BADDIE_S_MELTING:

            if (baddie->entity->logic_status != BADDIE_S_MELTING)
            {
                baddie->entity->logic_status= BADDIE_S_MELTING;
                baddie->entity->current_speed= 0;
                baddie->entity->logic_counter= 0;
            }

            break;
        #endif

        /* if kicking baddie */
        case BADDIE_S_KICKED:
        case BADDIE_S_KICKED_MELTING:

            if (baddie->status != status)
            {

                baddie->status= status;
                baddie->entity->logic_counter= 0;
                baddie->num_goodies_hit= 0;

                //baddie->entity->anim_counter= 0;
                //baddie->entity->anim_frame_counter= 0;
                //baddie->entity->anim_play_counter= 0;
                reset_anim(baddie->entity);

                //baddie->entity->status_row= baddie->entity->row;
                //baddie->entity->status_col= baddie->entity->col;

            }

            break;

        /* if moving baddie normally */
        case BADDIE_S_MOVING:

            baddie->status= BADDIE_S_MOVING;
            baddie->entity->c_speed= baddie->entity->n_speed;

            //baddie->entity->anim_counter= 0;
            //baddie->entity->anim_frame_counter= 0;
            //baddie->entity->anim_play_counter= 0;
            reset_anim(baddie->entity);

            //baddie->entity->status_row= baddie->entity->row;
            //baddie->entity->status_col= baddie->entity->col;

            break;

        /* if steamrolling baddie */
        case BADDIE_S_STEAMROLLED:

            baddie->status= status;
            baddie->entity->c_speed= 0;
            reset_anim(baddie->entity);

            if (map->player != 0 && baddie->steamroll_points > 0
                && baddie->kicked_by_player)
            {
                map->player->score+= baddie->steamroll_points;
                sprintf(str, "%i", baddie->steamroll_points);
                points=
                    construct_floating_text(str,
                                            baddie->entity->x
                                            + (map->t_width / 2)
                                            - (text_length(font_ingame, str)
                                               / 2),
                                            baddie->entity->y, 0, -1, 30);
                if (points != 0) append_to_list(map->point_list, points, 0);
            }

            break;

        /* if shattering baddie */
        case BADDIE_S_SHATTERING:

            baddie->status= status;
            baddie->entity->c_speed= 0;
            reset_anim(baddie->entity);

            if (map->player != 0 && baddie->shatter_points > 0
                && baddie->kicked_by_player)
            {
                map->player->score+= baddie->shatter_points;
                sprintf(str, "%i", baddie->shatter_points);
                points=
                    construct_floating_text(str,
                                            baddie->entity->x
                                            + (map->t_width / 2)
                                            - (text_length(font_ingame, str)
                                               / 2),
                                            baddie->entity->y, 0, -1, 30);
                if (points != 0) append_to_list(map->point_list, points, 0);
            }

            break;

        case BADDIE_S_SINKING:
        case BADDIE_S_NEW_ICE:
        case BADDIE_S_KAPUT:

            baddie->status= status;
            baddie->entity->c_speed= 0;
            reset_anim(baddie->entity);

            break;

        /* and the rest */
        default:

            baddie->status= status;

            //baddie->entity->anim_counter= 0;
            //baddie->entity->anim_frame_counter= 0;
            //baddie->entity->anim_play_counter= 0;
            reset_anim(baddie->entity);

            //baddie->entity->status_row= baddie->entity->row;
            //baddie->entity->status_col= baddie->entity->col;

            break;

    } /* end switch on status */

} /* end set_baddie_status */




void set_random_heading(Baddie *const baddie, const unsigned char probability)
{

    signed char new_x;
    signed char new_y;

    if (rand() % 100 < probability)
    {

        switch (rand() % 4)
        {

            /* head up */
            case 0:
                new_x= 0;
                new_y= -1;
                break;

            /* head down */
            case 1:
                new_x= 0;
                new_y= 1;
                break;

            /* head left */
            case 2:
                new_x= -1;
                new_y= 0;
                break;

            /* head right */
            case 3:
                new_x= 1;
                new_y= 0;
                break;

        }

        if (new_x == -baddie->entity->heading_x
            && new_y == -baddie->entity->heading_y)
        {
            if (rand() % 4 == 1) set_heading(baddie->entity, new_x, new_y);
        }
        else set_heading(baddie->entity, new_x, new_y);

    }

} /* end set_random_heading */




void turn_back_baddie(Baddie *const baddie, const Map *const map)
{

    unsigned char heading_choice;

    unsigned char go_reverse;

    signed char new_x;
    signed char new_y;

    //set_baddie_status(baddie, BADDIE_S_STOPPED, map);

    /* if squarely on tile, select a random heading */
    if (baddie->entity->dx == 0 && baddie->entity->dy == 0)
    {

        heading_choice= rand() % 4;
        switch (heading_choice)
        {
            case 0:
                new_x= 0;
                new_y= -1;
                break;
            case 1:
                new_x= 0;
                new_y= 1;
                break;
            case 2:
                new_x= -1;
                new_y= 0;
                break;
            case 3:
                new_x= 1;
                new_y= 0;
                break;
        }

        if (baddie->type == BADDIE_T_BADDIE5) go_reverse= 0;
        else go_reverse= 1;

        //if (go_reverse
        //  || (new_x != -baddie->entity->l_heading_x
        //      || new_y != -baddie->entity->l_heading_y))
        //{
            set_heading(baddie->entity, new_x, new_y);
        //}

        set_baddie_status(baddie, BADDIE_S_STOPPED, map);

    }

    /* else if not squarely on tile, reverse course */
    else reverse_heading(baddie->entity);

} /* end turn_back_baddie */
