// Moon Defense ver 1.6 Jakub Debski '2004

#include "objects.h"
#include "data.h"
#include <math.h>

#include <vector>
#include <string>
#include <iostream>
#include <fstream>
#include <algorithm>

using namespace std;

volatile int timercnt=0, fps=0, framecnt=0;

//#define MOUSE_REQUIRES_TIME

DATAFILE *data;
BITMAP *backdrop, *framebuf;
CSatellite *Satellite;
CSight *Sight;
CBlocking *blocking;
background *Back;

sprite_list sprites;

int bomb_steps;
int score;
bool fire_pressed;
int satellite_destroyed;
int satellite_immortal;
int number_of_satellites;
int number_of_ufo;
bool game_over=false;
bool game_plays=false;
int number_of_bombs=0;

bool run_without_sound=false;
bool run_without_music=false;
bool show_fps=false;
bool inverse_sound=false;

BITMAP *explosion_anim[EXPLOSION_TYPES][150];
BITMAP *ufo_frame[UFO_FRAMES];
BITMAP *friendly_missile_frame;
BITMAP *enemy_missile_frame;
BITMAP *satellite_frame;
BITMAP *sight_frame;
BITMAP *satellite_immortal_frame;
BITMAP *bomb_frame[360];


char *mask_explosion_anim[EXPLOSION_TYPES][150];
char *mask_explosion_clear[EXPLOSION_TYPES];
char *mask_ufo_frame;
char *mask_missile;
char *mask_bomb[360];


#define MENU_NEW_GAME 0
#define MENU_HIGH_SCORES 1
#define MENU_CONTROL 2
#define MENU_FULL_SCREEN 3
#define MENU_QUIT 4
#define MENU_TOTAL 5

int menu_selected=0;

bool menu_key_down_pressed=false;
bool menu_key_up_pressed=false;

bool full_screen=true;

bool control_mouse=false;

#ifdef MOUSE_REQUIRES_TIME
long mouse_without_move;
int last_mouse_x;
int last_mouse_y;
bool mouse_can_fire=false;
#endif

bool show_high_score=false;
bool check_high_score=false;

struct score_entry {
	string name;
	unsigned int score_value;
};

struct scorelt { bool operator()( const score_entry& p1, const score_entry& p2 )
{ return p1.score_value > p2.score_value; } };

class CHighScores {
private:
	vector < score_entry > with_standard_control;
	vector < score_entry > with_mouse_control;
	string last_name;
	void generate();
	
public:
	CHighScores();
	void load();
	void save();
	bool add_new(BITMAP *bmp);
	void show(BITMAP *bmp);
};

bool CHighScores::add_new(BITMAP *bmp)
{
	// check if player got a high score

	vector <score_entry> *vec;

	if (control_mouse)
		vec = &with_mouse_control;
	else
		vec = &with_standard_control;

	if ((*vec)[9].score_value<score)
	{
		// got a high score
		// ask for name
		textprintf(bmp, font, 300, 250, 15, "You've got a high score!");	   
		textprintf(bmp, font, 290, 300, 13, "Enter your name:");
		textprintf(bmp, font, 430, 300, 78, string(last_name+"_").c_str());
	}
	else
	{
		fire_pressed=true;
		return true;		
	}

	if (key[KEY_BACKSPACE])
	{
		last_name="";
	}

	unsigned char c;
	if (keypressed())
	{
		if (last_name.size()<20)
		{
			c = readkey()&0xff;
			if ((c>='a' && c<='z') || (c>='A' && c<='Z') || c=='_')
			  last_name+=c;
		}
	}

	if (key[KEY_ENTER])
	{
		score_entry t;
		t.name = last_name;
		t.score_value = score;
		(*vec).push_back(t);
		sort((*vec).begin(),(*vec).end(),scorelt());
		(*vec).pop_back();
		save();
		game_plays=false;		
		show_high_score=true;
		fire_pressed=true;
		return true;
	}
	return false;
}

CHighScores::CHighScores()
{
	with_standard_control.resize(10);
	with_mouse_control.resize(10);
}

void CHighScores::show(BITMAP *bmp)
{
    textprintf(bmp, font, 338, 15, 15, "[ HIGH SCORES ]");	   
    textprintf(bmp, font, 95, 35, 78, "Standard control");	   
    textprintf(bmp, font, 575, 35, 78, "Mouse control");	   
	
    for (int a=0;a<10;a++)
	{
		textprintf(bmp, font, 50, 60+a*12, 15-a, "%s",with_standard_control[a].name.c_str());	   		
		textprintf(bmp, font, 200, 60+a*12, 15-a, "%8d",with_standard_control[a].score_value);	   		

		textprintf(bmp, font, 520, 60+a*12, 15-a, "%s",with_mouse_control[a].name.c_str());	   		
		textprintf(bmp, font, 670, 60+a*12, 15-a, "%8d",with_mouse_control[a].score_value);	   		
		
	}
}

void CHighScores::generate()
{
	for (int a=0;a<10;a++)
	{
		with_standard_control[a].score_value=(10-a)*1000;
		with_standard_control[a].name="Jakub";

		with_mouse_control[a].score_value=(10-a)*1000;
		with_mouse_control[a].name="Jakub";
	}
	last_name="Unnamed";	
}

void CHighScores::save()
{
	ofstream out("scores.dat");
	if (out.is_open())
	{
		for (int a=0;a<10;a++)
		{
			out << with_standard_control[a].name << '\n';
			out << with_standard_control[a].score_value << '\n';
			out << with_mouse_control[a].name << '\n';
			out << with_mouse_control[a].score_value << '\n';
		}
		out << last_name << '\n';
		out << control_mouse << '\n';
		out << full_screen << '\n';
	}
}

void CHighScores::load()
{
	ifstream in("scores.dat");
	if (in.is_open())
	{
		for (int a=0;a<10;a++)
		{
			in >> with_standard_control[a].name;
			in >> with_standard_control[a].score_value;
			in >> with_mouse_control[a].name;
			in >> with_mouse_control[a].score_value;
		}
		in >> last_name;
		in >> control_mouse;
		in >> full_screen;
	}
	else
	{
		generate();
		save();
	}
}

int ShowMenu()
{
	   string menu_txt[MENU_TOTAL] = {
		   "New Game", "High Scores", "Control:", "Run in:", "Quit"
	   };
	   textprintf(framebuf, font, 30, 45, 78, "Moon Defense v1.6 by Jakub Debski");

	   if (!fire_pressed && (key[KEY_ENTER]||joy_b1||key_shifts&KB_CTRL_FLAG||key[KEY_SPACE]||mouse_b&1))
	   {
		   fire_pressed = true;		   
		   return menu_selected;
	   }

	   if (key[KEY_UP]||joy_up)
	   {
		   if (!menu_key_up_pressed)
		   {
			   menu_key_up_pressed=true;
			   menu_selected--;
			   if (menu_selected<0)
				   menu_selected=MENU_TOTAL-1;
		   }
	   }
	   else
		   menu_key_up_pressed=false;
		   
	   if (key[KEY_DOWN]||joy_down)
	   {
		   if (!menu_key_down_pressed)
		   {
			   menu_key_down_pressed=true;
			   menu_selected++;
			   if (menu_selected>MENU_TOTAL-1)
				   menu_selected=0;
		   }
	   }
	   else
		   menu_key_down_pressed=false;
	   
	   int color;
	   for (int a=0;a<MENU_TOTAL;a++)
	   {
		   if (menu_selected==a)
			   color=15;
		   else
			   color=10;
		   
		   textprintf(framebuf, font, 30, 70+a*12, color, menu_txt[a].c_str());		   
		   if (a==MENU_CONTROL)
		   {
			   if (control_mouse)
				  textprintf(framebuf, font, 100, 70+a*12, color, "Mouse");		   
			   else
				  textprintf(framebuf, font, 100, 70+a*12, color, "Standard");		   
		   }
		   else if (a==MENU_FULL_SCREEN)
		   {
			   if (full_screen)
				   textprintf(framebuf, font, 100, 70+a*12, color, "Full Screen");
			   else
				   textprintf(framebuf, font, 100, 70+a*12, color, "Window");
		   }			   		   
	   }
	   
	   return MENU_TOTAL;
}

void frames()
{
	fps = framecnt; framecnt=0;
}

void timer()
{
	timercnt++;
}


void generate_explosion(int type)
{
   BITMAP *temp;
   int a,frame;
   for (a=0;a<150;a++)
   {
     explosion_anim[type][a] = create_bitmap(EXPLOSION_SIZE,EXPLOSION_SIZE);
     clear_bitmap(explosion_anim[type][a]);
	 mask_explosion_anim[type][a] = new char [EXPLOSION_SIZE*EXPLOSION_SIZE+1];
	 memset(mask_explosion_anim[type][a],0,EXPLOSION_SIZE*EXPLOSION_SIZE);
   }
     
   mask_explosion_clear[type] = new char [EXPLOSION_SIZE*EXPLOSION_SIZE+1];
   memset(mask_explosion_clear[type],0,EXPLOSION_SIZE*EXPLOSION_SIZE);
   
   temp = create_bitmap(EXPLOSION_SIZE,EXPLOSION_SIZE);
   clear_bitmap(temp);
   
   
 for (frame=0;frame<150;frame++)
 {
    int ax,ay;
    
    // add some bright points
    if (frame<50)
    {    
       ax=rand()%20-10;
       ay=rand()%20-10;
       circle(temp,EXPLOSION_SIZE/2 + ax, EXPLOSION_SIZE/2 + ay,4,15);
    }
    // remove points
    if (frame>50)
    {    
       ax=rand()%20-10;
       ay=rand()%20-10;
       circlefill(temp,EXPLOSION_SIZE/2 + ax, EXPLOSION_SIZE/2 + ay,4,0);       
    }
        
    int sum;
    for (ax=0;ax<EXPLOSION_SIZE;ax++)
      for (ay=0;ay<EXPLOSION_SIZE;ay++)
      {
         sum = 0;
         sum += getpixel(temp,ax,ay);
         sum += getpixel(temp,ax-1,ay);
         sum += getpixel(temp,ax+1,ay);
         sum += getpixel(temp,ax,ay-1);
         sum += getpixel(temp,ax,ay+1);
         sum += getpixel(temp,ax-1,ay-1);
         sum += getpixel(temp,ax+1,ay-1);
         sum += getpixel(temp,ax-1,ay+1);
         sum += getpixel(temp,ax+1,ay+1);         

        if ( frame<100)         
           sum = sum/8;         
        else
           sum = sum/9;         
         
         if (sum<0)
           sum=0;
           
         if (sum>15)
           sum=15;

         putpixel(explosion_anim[type][frame],ax,ay,sum);
         
         if (sum!=0)
		 {
			 mask_explosion_anim[type][frame][EXPLOSION_SIZE*ay+ax]=1;
             mask_explosion_clear[type][EXPLOSION_SIZE*ay+ax]=1;
		 }
      }      
      
      blit(explosion_anim[type][frame],temp,0,0,0,0,EXPLOSION_SIZE,EXPLOSION_SIZE);
  }     
  destroy_bitmap(temp);
}

void get_frames()
{
	int ax,ay;

	RLE_SPRITE *temp;

	temp = (RLE_SPRITE *)data[OBJ_MISSILE].dat;
	friendly_missile_frame = create_bitmap(temp->w,temp->h);
	clear_bitmap(friendly_missile_frame);
	draw_rle_sprite(friendly_missile_frame,temp,0,0);
	
	temp = (RLE_SPRITE *)data[OBJ_EN_MISSILE].dat;
	enemy_missile_frame = create_bitmap(temp->w,temp->h);
	clear_bitmap(enemy_missile_frame);
	draw_rle_sprite(enemy_missile_frame,temp,0,0);

	mask_missile = new char [temp->w * temp->h + 1];		
	memset(mask_missile,0,temp->w*temp->h);

	for (ax=0;ax<temp->w;ax++)
		for (ay=0;ay<temp->h;ay++)
		{
			if (getpixel(friendly_missile_frame,ax,ay)!=0)
				mask_missile[temp->w*ay+ax]=1;
		}
			
	temp = (RLE_SPRITE *)data[OBJ_SATELLITE].dat;
	satellite_frame = create_bitmap(temp->w,temp->h);
	clear_bitmap(satellite_frame);
	draw_rle_sprite(satellite_frame,temp,0,0);

	satellite_immortal_frame = create_bitmap(temp->w,temp->h);
	clear_bitmap(satellite_immortal_frame);
	
	for (ax=0;ax<temp->w;ax++)
		for (ay=0;ay<temp->h;ay++)
		{
			if (getpixel(satellite_frame,ax,ay)!=0)
				putpixel(satellite_immortal_frame,ax,ay,62);
		}		

	temp = (RLE_SPRITE *)data[OBJ_SIGHT].dat;
	sight_frame = create_bitmap(temp->w,temp->h);
	clear_bitmap(sight_frame);
	draw_rle_sprite(sight_frame,temp,0,0);


   for (int a=0;a<UFO_FRAMES;a++)
   {
     ufo_frame[a] = create_bitmap(UFO_SIZE_X,UFO_SIZE_Y);
     clear_bitmap(ufo_frame[a]);
   }

   BITMAP *t;   
   t = create_bitmap(BOMB_SIZE, BOMB_SIZE);        
   clear_bitmap(t);
   draw_rle_sprite(t,(RLE_SPRITE *)data[OBJ_BOMB].dat,0,0);    
   
   for (int an=0;an<360;an++)
   {
	  bomb_frame[an]=create_bitmap(BOMB_SIZE, BOMB_SIZE);
	  clear_bitmap(bomb_frame[an]);
      rotate_sprite(bomb_frame[an], t, 0, 0, itofix((int) ( ( (double) an)/1.40625) )); // convert 360 to 256

	  mask_bomb[an] = new char [BOMB_SIZE*BOMB_SIZE+1];
	  memset(mask_bomb[an],0,BOMB_SIZE*BOMB_SIZE);
	  
	  for (ax=0;ax<BOMB_SIZE;ax++)
	  {
		  for (ay=0;ay<BOMB_SIZE;ay++)
		  {
			  if (getpixel(bomb_frame[an],ax,ay)!=0)
				  mask_bomb[an][BOMB_SIZE*ay+ax] = 1;
			  else
				  mask_bomb[an][BOMB_SIZE*ay+ax] = 0;
		  }
	  }	  
   }
   destroy_bitmap(t);
   
   
   draw_rle_sprite(ufo_frame[0],((RLE_SPRITE *)data[OBJ_UFO00].dat),0,0);
   draw_rle_sprite(ufo_frame[1],((RLE_SPRITE *)data[OBJ_UFO01].dat),0,0);
   draw_rle_sprite(ufo_frame[2],((RLE_SPRITE *)data[OBJ_UFO02].dat),0,0);
   draw_rle_sprite(ufo_frame[3],((RLE_SPRITE *)data[OBJ_UFO03].dat),0,0);
   draw_rle_sprite(ufo_frame[4],((RLE_SPRITE *)data[OBJ_UFO04].dat),0,0);
   draw_rle_sprite(ufo_frame[5],((RLE_SPRITE *)data[OBJ_UFO05].dat),0,0);
   draw_rle_sprite(ufo_frame[6],((RLE_SPRITE *)data[OBJ_UFO06].dat),0,0);
   draw_rle_sprite(ufo_frame[7],((RLE_SPRITE *)data[OBJ_UFO07].dat),0,0);
   draw_rle_sprite(ufo_frame[8],((RLE_SPRITE *)data[OBJ_UFO08].dat),0,0);
   draw_rle_sprite(ufo_frame[9],((RLE_SPRITE *)data[OBJ_UFO09].dat),0,0);
   draw_rle_sprite(ufo_frame[10],((RLE_SPRITE *)data[OBJ_UFO10].dat),0,0);
   draw_rle_sprite(ufo_frame[11],((RLE_SPRITE *)data[OBJ_UFO11].dat),0,0);
   draw_rle_sprite(ufo_frame[12],((RLE_SPRITE *)data[OBJ_UFO12].dat),0,0);
   draw_rle_sprite(ufo_frame[13],((RLE_SPRITE *)data[OBJ_UFO13].dat),0,0);
   draw_rle_sprite(ufo_frame[14],((RLE_SPRITE *)data[OBJ_UFO14].dat),0,0);
   draw_rle_sprite(ufo_frame[15],((RLE_SPRITE *)data[OBJ_UFO15].dat),0,0);
   draw_rle_sprite(ufo_frame[16],((RLE_SPRITE *)data[OBJ_UFO16].dat),0,0);
   draw_rle_sprite(ufo_frame[17],((RLE_SPRITE *)data[OBJ_UFO17].dat),0,0);
   draw_rle_sprite(ufo_frame[18],((RLE_SPRITE *)data[OBJ_UFO18].dat),0,0);
   draw_rle_sprite(ufo_frame[19],((RLE_SPRITE *)data[OBJ_UFO19].dat),0,0);
   draw_rle_sprite(ufo_frame[20],((RLE_SPRITE *)data[OBJ_UFO20].dat),0,0);
   draw_rle_sprite(ufo_frame[21],((RLE_SPRITE *)data[OBJ_UFO21].dat),0,0);
   draw_rle_sprite(ufo_frame[22],((RLE_SPRITE *)data[OBJ_UFO22].dat),0,0);
   draw_rle_sprite(ufo_frame[23],((RLE_SPRITE *)data[OBJ_UFO23].dat),0,0);

   mask_ufo_frame = new char [UFO_SIZE_X * UFO_SIZE_Y + 1];
   for (ax=0;ax<temp->w;ax++)
	   for (ay=0;ay<temp->h;ay++)
	   {
		   if (getpixel(ufo_frame[21],ax,ay)!=0)
			   mask_ufo_frame[temp->w*ay+ax]=1;
	   }
}

void NewGame(void)
{
	clear_bitmap(backdrop);
	blocking->clearblocking();
	
	// add some stars

	int index,ax,ay;
	for (index=0;index<100;index++)
	{
		putpixel (backdrop, rand()%SCREEN_SIZE_X, rand()%SCREEN_SIZE_Y, index);
	}
	
	// draw moon
	
	blit((BITMAP *)data[OBJ_MOON].dat, backdrop, 0, 0, SCREEN_SIZE_X/2-MOON_SIZE_X/2, SCREEN_SIZE_Y/2-MOON_SIZE_Y/2, MOON_SIZE_X, MOON_SIZE_Y);
	draw_rle_sprite(backdrop,((RLE_SPRITE *)data[OBJ_BASE].dat),SCREEN_SIZE_X/2 - ((RLE_SPRITE *)data[OBJ_BASE].dat)->w/2,SCREEN_SIZE_Y/2 - ((RLE_SPRITE *)data[OBJ_BASE].dat)->h/2);
	
	// moon is blocking
	
	for (ax=0;ax<MOON_SIZE_X;ax++)
		for (ay=0;ay<MOON_SIZE_Y;ay++)
		{
			int col = getpixel((BITMAP *)data[OBJ_MOON].dat,ax,ay);
			if (col!=0)
			{     
				blocking->setblocking(SCREEN_SIZE_X/2-MOON_SIZE_X/2 + ax,SCREEN_SIZE_Y/2-MOON_SIZE_Y/2 + ay, BLK_PLANET);
			}
		} 

	bomb_steps = 4000;
	score=0;

	satellite_destroyed=0;
	satellite_immortal=0;
	number_of_satellites=5;
	number_of_ufo=0;
	game_over=false;
#ifdef MOUSE_REQUIRES_TIME
	mouse_without_move=0;
#endif
	check_high_score=false;
	
	// delete list of sprites

	sprite *temp;
	while(sprites.size()>0)
	{
		temp=*sprites.begin();
		delete temp;
		sprites.pop_front();
	}

	number_of_bombs=0;

	Back = new background; 
	sprites.push_back(Back);	

	Satellite = new CSatellite(50,100);
	Sight = new CSight(50,100);
	
}

bool SetGFXMode()
{
	int gfx_type;

	if (full_screen)
		gfx_type = GFX_AUTODETECT_FULLSCREEN;
	else
		gfx_type = GFX_AUTODETECT_WINDOWED;
	
	if (set_gfx_mode(gfx_type, 800, 600, 0, 0) != 0)
	{
		set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
		allegro_message("Unable to set graphic mode\n%s\n", allegro_error);
		return false;
	}	 

	set_palette((RGB*)data[OBJ_PALETTE].dat);	
	return true;
}

int main(int argc, char *argv[])
{
 int a;
 
 srand( (unsigned)time( NULL ) );
 
 allegro_init();
 install_keyboard();
 install_mouse();
 install_timer();
 initialise_joystick();

 // protect timers from move to virtual memory
 LOCK_VARIABLE(framecnt);
 LOCK_VARIABLE(fps);
 LOCK_VARIABLE(timercnt);
 LOCK_VARIABLE(timer);
 LOCK_VARIABLE(frames); 
 
 bool run_in_window = false;
 
 if (argc>1)
 {
   for (a=1;a<argc;a++)
   {
     if (strcmp(argv[a],"-nosound")==0)
       run_without_sound=true;
     if (strcmp(argv[a],"-nomusic")==0)
       run_without_music=true;
     if (strcmp(argv[a],"-window")==0)
       run_in_window=true;
     if (strcmp(argv[a],"-inversesound")==0)
		 inverse_sound=true;
     if (strcmp(argv[a],"-fps")==0)
       show_fps=true;
   }
 }

 data=load_datafile("game.dat");
 if (data==NULL)
 {
	 set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
	 allegro_message("Unable to load game data (game.dat file)\n");
	 return 1;
 }

 int digi_type, midi_type;
 
 if (run_without_sound)
   digi_type = DIGI_NONE;
 else
   digi_type = DIGI_AUTODETECT;
   
 if (run_without_music)
   midi_type = MIDI_NONE;
 else
   midi_type = MIDI_AUTODETECT;

 CHighScores HighScores; 
 HighScores.load(); 

 if (run_in_window)
	 full_screen=false;

   /* install digital and midi sound drivers */
   if (!(run_without_sound && run_without_music))
   {
      if (install_sound(digi_type, midi_type, NULL) != 0)
      {
        allegro_message("Error initialising sound system\n%s\n", allegro_error);
        return 1;
      } 
   }

   if (!SetGFXMode())
	   return 1;
   
 set_window_title("Moon Defense 1.6  Jakub Debski '2004");

 // timer that set speed of game

 install_int_ex(timer,BPS_TO_TIMER(225));
 
 // timer that count frames per second

 if (show_fps)
	install_int_ex(frames,SECS_TO_TIMER(1)); 

 backdrop=create_bitmap(SCREEN_SIZE_X,SCREEN_SIZE_Y);
 
 blocking=new CBlocking;

 // get bitmap frames
 
 get_frames();
 
 // generate explosions
 for (int type=0;type<EXPLOSION_TYPES;type++)
	 generate_explosion(type);

 // create double buffer
 framebuf=create_bitmap(800,600);
 clear(framebuf);

 NewGame();

 if (!run_without_music)     
   play_midi((MIDI*)data[OBJ_MIDI_GAME].dat,TRUE);

 
 while (1) {

	if (key[KEY_ESC])
		game_plays=false;

  // draw sprites
  
   sprites.remove(Satellite);
   sprites.remove(Sight);
   
   // is base destroyed or no satellites left?
   
   if (!game_over)
   {
	   if (blocking->getblocking(SCREEN_SIZE_X/2,SCREEN_SIZE_Y/2)!=BLK_PLANET || number_of_satellites==-1)
	   {
		   game_over=true;
		   check_high_score=true;
	   }
   }
      
   if (!game_over && game_plays)
   {
     if (!satellite_destroyed)
          sprites.push_back(Satellite);
          
       sprites.push_back(Sight);   
   }

   if (game_over)
   {      
	   bomb_steps=200;      
   }
  
      
  if (!(key[KEY_ENTER]||joy_b1||key_shifts&KB_CTRL_FLAG||key[KEY_SPACE]||mouse_b&1))
  {
    fire_pressed = false;
  }
   
  // draw sprites
  
   sprite_list::const_iterator spr=sprites.begin();
   while (spr!=sprites.end()) {
    (*spr)->draw(framebuf);
    spr++;
   }

   if (show_fps)
	   textprintf(framebuf, font, 30, 25, 15, "FPS: %d", fps);
   
   if (game_plays)
   {
	   // print score
	   textprintf(framebuf, font, 30, 10, 15, "Score: %08d", score);

	   // draw satellites left
	   for (int s=0;s<number_of_satellites;s++)
	   {
		   stretch_sprite(framebuf, satellite_frame, 650+s*20, 10, 15, 15); 
	   }	   
   }

	   if (check_high_score)
	   {
		   if (HighScores.add_new(framebuf)==true)
		   {
			   check_high_score=false;
		   }
	   }
      else if (!game_plays)
	  {
		  if (show_high_score)
		  {
			  HighScores.show(framebuf);
			  if (key[KEY_ESC])
				  show_high_score=false;
		  }
		  else
		  {
			  // menu
			  switch(ShowMenu()) {
			  case MENU_NEW_GAME:
				  NewGame();
				  game_plays=true;			  
				  break;
			  case MENU_CONTROL:
				  control_mouse=!control_mouse;
				  break;
			  case MENU_HIGH_SCORES:
				  show_high_score=true;
				  break;
			  case MENU_FULL_SCREEN:
				  full_screen=!full_screen;
				  if (!SetGFXMode())
					  return 1;				  
				  break;
			  case MENU_QUIT:
				  HighScores.save();
				  return 0;
			  }
		  }
	  }

  
  // display frame
//  vsync();
  acquire_screen();
  blit(framebuf,screen,0,0,0,0,800,600);
  release_screen();
  framecnt++;
  
  if (keyboard_needs_poll())
    poll_keyboard();

  if (mouse_needs_poll())
	  poll_mouse();
    
  poll_joystick();

  // animate sprites

  while(timercnt>0)
  {
#ifdef MOUSE_REQUIRES_TIME	  
	if (control_mouse)
	{
		if (abs(last_mouse_x-mouse_x)<3 && abs(last_mouse_y-mouse_y)<3)
		{
			mouse_without_move++;
		}
		else
		{
			mouse_without_move=0;
			last_mouse_x=mouse_x;
			last_mouse_y=mouse_y;
		}
		long dist;
		dist = (Sight->getX()-Satellite->getX())*(Sight->getX()-Satellite->getX());
		dist += (Sight->getY()-Satellite->getY())*(Sight->getY()-Satellite->getY());
		dist = sqrt(dist);

		if (mouse_without_move>(dist/5)-50)
			mouse_can_fire=true;
		else
			mouse_can_fire=false;
			
	}
#endif
  
	  if (satellite_destroyed>0)
	  {
		  satellite_destroyed--;
		  if (satellite_destroyed==0)
			  satellite_immortal=600;
	  }
	  if (satellite_immortal>0)
		  satellite_immortal--;
	  
	  // add new bomb 
	  if (rand()%(bomb_steps/3)==0)
		  sprites.push_back(new CBomb);
	  if (number_of_bombs==0 && rand()%(control_mouse?100:500)==0)
		  sprites.push_back(new CBomb);
	  
	  // add ufo
	  if (!game_over && score>2000)
	  {
		  if (number_of_ufo<2 && rand()%bomb_steps/2==0)
			  sprites.push_back(new CUfo);
	  }
	  
	  sprite_list::iterator spr=sprites.begin();
	  while (spr!=sprites.end())
	  {
		  sprite *itm=*spr;
		  if (itm->animate())
		  {
			  sprites.erase(spr++);
			  delete itm;
		  } else spr++;
	  }
	  
	  timercnt--;
  } // end of timercnt

 }
 return 0;
}

END_OF_MAIN();

