//  **************************************************************************
//
//  The Ramachandran-plot Explorer(C) 2003 Bosco K. Ho
//
//  rama.cpp
//
//  This is a molecule viewer that reads the standard protein text
//  format for the 3 dimensional coordinates of a protein
//  molecule.
//
//  The emphasis of this program is on interactivity with the protein.
//
//  Uses the allegro games library for graphics and input
//
//  **************************************************************************
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU Lesser General Public License as published
//  by the Free Software Foundation; either version 2.1 of the License, or(at
//  your option) any later version.
//
//  This program is distributed in the hope that it will be useful,  but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
//  **************************************************************************
//
//  Development log
//
//  To do list:
//      - extend to multiple chains and hetereo atoms
//
//  25/12/2003:
//      - added comments and license text
//  27/12/2003:
//      - fixed the copy residue bug
//      - added pop-camera_up text when mouse is over atom
//      - stream-lined vector library, including the rotation and center
//        transform
//      - accurate resNo accounting in insert/delete operations
//      - simplified the pdbReader code
//  29/12/2003:
//      - modularised the 3d graphics class into a separate file
//      - added a full atom name method
//  30/12/2003:
//      - create WidgetList class
//  7/3/2004:

//      - found out-of-bounds array read in the drawing of
//        the peptide C-N bond. the for list counted one too many
//        and the protein::hasAtom procedure did not check
//        residue index bounds
//  11/3/2004:
//      - fixed load and save dialogs so that they don't crash
//        with bad input
//      - removed the ESC exit, added a quit button
//      - on loading a new protein, will find the closest CA atom
//        to the center of mass and do a zoom to showState the whole protein
//      - will evaluate sidechain-mainchain h-bonds within a residue
//      - prevent H atom projects on non-amino acid residues
//      - changes the window title to new save file name in window
//        title
//   4/4/2004:
//      - self h-bond evaluator
//      - remembers to update resKey and resName on mutation
//      - avoids projection of H and drawing of peptide bond 
//        for non-standard amino acids
//      - now changes name to generic in title when file is saved
//      - fix bug: exits if loaded file is empty
//      - added argv to open the file if argument is supplied
//      - major refactor to separate protein display from widget from
//        general protein manupilation routines
//      - added a highlight text to describe atom when mouse is over
//        atom
//      - added a routine to quickly center and zoom protein on load
//        and also adjusts scroll-bar correctly
//      - added half-bar for green and pick in scrollbar
//      - when dragging active residues in sequence box, can go off limits
//        and scrolls
//      - added the H2, H3, OXT, and HXT atoms
//      - now allows 3 stereo modes
//      - now does cannonical h-bonds (orange), weak h-bonds (pink)
//        and disulfide bonds (yellow)
//
//
//  **************************************************************************


#include <string>
#include <allegro.h>
#include "protdisp.h"
#include "display.h"
#include "control.h"


using namespace std;


bool isProgramEnd = false;
bool isAnimate = false;
ProteinDisplay protein;
DisplayBox* displayBox;
int BUTTON_W = 140; 


void handle_close(void)
{
	isProgramEnd = true;
}



void animate_protein(void)
{
  const double PHI1 = -150.0;
  const double PSI1 =  120.0;
  const double PHI2 =  -60.0;
  const double PSI2 =  -40.0;
	const double d    = sqrt((PHI1-PHI2)* (PHI1-PHI2)+ (PSI1-PSI2)* (PSI1-PSI2));
	const double dphi = (PHI2-PHI1)/d;
	const double dpsi = (PSI2-PSI1)/d;

	for (int c=1; c <= protein.nRes(); c+=1)
	{
		double phi = protein.res(c).phi + dphi;
		double psi = protein.res(c).psi + dpsi;
		normalise(phi);
		normalise(psi);
		protein.res(c).phi = phi;
		protein.res(c).psi = psi;
		for (int d=1; d<= getNChi(protein.res(c).resKey); d+=1)

		{
			double chi = protein.res(c).chi(d) + dphi*2;
			normalise(chi);
			protein.res(c).chi(d) = chi;
		}
	}
}




void change_graphics_option(void)
{
  displayBox->setFocusWithSameOrientation();
}



void build_denovo_protein()
{
  const double PHI1 = -150.0;
  const double PSI1 =  120.0;
  protein.clear();
  for (int j=1; j<= 20; j+=1)
  {
    protein.appendRes(j);
    if (j==PRO)
      protein.res(j).phi = -60;
    else
      protein.res(j).phi = PHI1;
    protein.res(j).psi = PSI1;
  }
  protein.rebuild();
  set_window_title("The Ramachandran Plot Explorer");
}



void load_protein(char *name, Protein& protein)
{
  PdbReader pdb(name);
  if (!pdb.isFileLoaded()) return;
  pdb.loadChain(protein);

  if (protein.nRes()==0) return;

	for (int i=1; i <= protein.nRes(); i++)
	  protein.sproutHydrogen(i);

  std::string s1 = protein.getName();
  while ( (s1.length() > 0) && (s1[s1.length()-1] == ' ') )
    s1.erase(s1.length()-1, 1);
  std::string s = "[" + protein.getPdbCode() + "]  " +  s1;
  if (s == "[]  ") s = name;
  s += " - The Ramachandran Plot Explorer";
  set_window_title(s.c_str());
}



void open_file()
{
  static char save_str[600] = "";
  if (file_select_ex("Open", save_str, "pdb", 500, 600, 200) != 0)
  {
    load_protein(save_str, protein);
    if (protein.nRes() == 0)
      build_denovo_protein();
    displayBox->initializeFocus();
  }
}



void save_file()
{
	static char save_str[600] = "";
	if (save_str[0] == 0) strcpy(save_str, "peptide.pdb");

	if (file_select_ex("Save pdb file", save_str, "pdb", 500, 600, 200) != 0)
	{
	  std::ofstream out(save_str);
    if (!out.fail())
      out << protein << std::endl;

    std::string s = save_str;
    s += " - The Ramachandran Plot Explorer";
    set_window_title(s.c_str());
  }

}



void set_colors()
{
  PALETTE pal;
  int c;
  for (c=1; c<256; c++)
  {
     pal[c].r = default_palette[c].r;
     pal[c].b = default_palette[c].b;
     pal[c].g = default_palette[c].g;
  }
  pal[BLACK       ].r =       pal[BLACK       ].g =       pal[BLACK       ].b = 0;
  pal[RED         ].r = 255;  pal[RED         ].g =       pal[RED         ].b = 0;
  pal[BLUE        ].b = 255;  pal[BLUE        ].r =       pal[BLUE        ].g = 20;
  pal[YELLOW      ].r =       pal[YELLOW      ].g = 245;  pal[YELLOW      ].b = 0;
  pal[GREEN       ].g = 255;  pal[GREEN       ].r =       pal[GREEN       ].b = 0;
  pal[DARK_GREEN  ].g = 20;   pal[DARK_GREEN  ].r =       pal[DARK_GREEN  ].b = 0;
  pal[LIGHT_BLUE  ].b = 255;  pal[LIGHT_BLUE  ].r =       pal[LIGHT_BLUE  ].g = 240;
  pal[PURPLE      ].g = 0;    pal[PURPLE      ].r =       pal[PURPLE      ].b = 255;
  pal[ORANGE      ].r = 255;  pal[ORANGE      ].b = 0;    pal[ORANGE      ].g = 150;
  pal[LIGHT_ORANGE].r = 255;  pal[LIGHT_ORANGE].b = 150;  pal[LIGHT_ORANGE].g = 180;

  // set up a grey-scale in the top half of the palette
  for (c=128; c<256; c++)
     pal[c].r = pal[c].g = pal[c].b = (c-128)*2;

  set_palette(pal);
}



int main(int argc, char **argv)
{
  allegro_init();
  install_keyboard();
  install_mouse();
  install_timer();

  if (set_gfx_mode(GFX_AUTODETECT_WINDOWED, 800, 600, 0, 0) != 0)
  	if (set_gfx_mode(GFX_AUTODETECT_WINDOWED, 640, 480, 0, 0) != 0)
  	if (set_gfx_mode(GFX_AUTODETECT, 1024, 768, 0, 0) != 0)
	  {
	    set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
		  allegro_message("Error setting graphics mode\n%s\n", allegro_error);
		  exit(1);
	  }

  set_colors();

  set_window_close_hook(handle_close);
 
  if (argc >= 2)
    load_protein(argv[1], protein);

  if (protein.nRes() == 0)
    build_denovo_protein();

  displayBox = new DisplayBox(&protein, 0, 0, SCREEN_W, SCREEN_H);


  WidgetList widgetList;

  widgetList.push_back(displayBox);
  widgetList.push_back(new RamaBox    (displayBox, 15, SCREEN_H - 200,        170, 170) );
  widgetList.push_back(new ChiBox     (displayBox, 15, SCREEN_H - 210 - 4*16, 170,  16) );
  widgetList.push_back(new SeqFocusBox(displayBox, 15, 10 + 10 + 14,           14,   6) );
  widgetList.push_back(new SequenceBox(displayBox, 15, 10 + 10,                14,  14) );
  widgetList.push_back(new ScrollBar  (displayBox, 15, 10,       14*VIEW_SIZE + 1,  10) );

  widgetList.push_back(new ToggleButton(SCREEN_W - BUTTON_W - 5, 15,        BUTTON_W, 17, &isAnimate,               "animate",       change_graphics_option) );
  
  widgetList.push_back(new ToggleButton(SCREEN_W - BUTTON_W - 5, 50 + 0*20, BUTTON_W, 17, &displayBox->isCentered,  "center",        change_graphics_option) );
  widgetList.push_back(new ToggleButton(SCREEN_W - BUTTON_W - 5, 50 + 1*20, BUTTON_W, 17, &displayBox->isHbond,     "H-bond",        change_graphics_option) );
  widgetList.push_back(new ToggleButton(SCREEN_W - BUTTON_W - 5, 50 + 2*20, BUTTON_W, 17, &displayBox->isSolid,     "stick & ball",  change_graphics_option) );
  widgetList.push_back(new ToggleButton(SCREEN_W - BUTTON_W - 5, 50 + 3*20, BUTTON_W, 17, &displayBox->isSidechain, "sidechain",     change_graphics_option) );
  widgetList.push_back(new ToggleButton(SCREEN_W - BUTTON_W - 5, 50 + 4*20, BUTTON_W, 17, &displayBox->isPeptide,   "peptide plane", change_graphics_option) );
  widgetList.push_back(new ToggleButton(SCREEN_W - BUTTON_W - 5, 50 + 5*20, BUTTON_W, 17, &displayBox->isHydrogen,  "hydrogen",      change_graphics_option) );

  widgetList.push_back(new Button(SCREEN_W - BUTTON_W - 5, 190,        BUTTON_W, 17, "save pdb",  save_file) );
  widgetList.push_back(new Button(SCREEN_W - BUTTON_W - 5, 190 + 1*20, BUTTON_W, 17, "open pdb",  open_file) );

  widgetList.push_back(new StateSelector(SCREEN_W - BUTTON_W - 5, 250 + 0*16, BUTTON_W, 17, &displayBox->pickState, PICK_RESIDUE,  "pick res",        change_graphics_option) );
  widgetList.push_back(new StateSelector(SCREEN_W - BUTTON_W - 5, 250 + 1*16, BUTTON_W, 17, &displayBox->pickState, PICK_DISTANCE, "pick dist",       change_graphics_option) );

  widgetList.push_back(new StateSelector(SCREEN_W - BUTTON_W - 5, 305 + 0*16, BUTTON_W, 17, &displayBox->showState, SHOW_ALL,      "show all",        change_graphics_option) );
  widgetList.push_back(new StateSelector(SCREEN_W - BUTTON_W - 5, 305 + 1*16, BUTTON_W, 17, &displayBox->showState, SHOW_SEGMENT,  "show scrollbar",  change_graphics_option) );
  widgetList.push_back(new StateSelector(SCREEN_W - BUTTON_W - 5, 305 + 2*16, BUTTON_W, 17, &displayBox->showState, SHOW_SPHERE,   "show sphere",     change_graphics_option) );

  widgetList.push_back(new StateSelector(SCREEN_W - BUTTON_W - 5, 375 + 0*16, BUTTON_W, 17, &displayBox->stereoState, STEREO_NONE,         "no stereo",    change_graphics_option) );
  widgetList.push_back(new StateSelector(SCREEN_W - BUTTON_W - 5, 375 + 1*16, BUTTON_W, 17, &displayBox->stereoState, STEREO_CROSSEYE,     "cross-eye",    change_graphics_option) );
  widgetList.push_back(new StateSelector(SCREEN_W - BUTTON_W - 5, 375 + 2*16, BUTTON_W, 17, &displayBox->stereoState, STEREO_WALL_TO_WALL, "wall-to-wall", change_graphics_option) );

  widgetList.push_back(new Button(SCREEN_W - BUTTON_W - 5, SCREEN_H - 40, BUTTON_W, 17, "exit",  handle_close) );

  
  BITMAP* buffer = create_bitmap(SCREEN_W, SCREEN_H);
  int last_retrace_count = retrace_count;

  while (true)
  {
    // animate the protein
    while (last_retrace_count < retrace_count)
    {
      if (isAnimate) animate_protein();
      last_retrace_count++;
    }
    if (isAnimate)
    {
      protein.rebuild();
      change_graphics_option();
    }

    // draw the screen
    widgetList.draw(buffer);
    drawMouse(buffer);
    vsync();
    blit(buffer, screen, 0, 0, 0, 0, SCREEN_W, SCREEN_H);

    if (isProgramEnd) break;

    widgetList.handle_mouse();

    yield_timeslice();
  }

  // clean camera_up memory
  destroy_bitmap(buffer);

  return 0;
}

END_OF_MAIN();

