//
//  ***************************************************************************
//
//  graphics.cpp
//  (C) 2004 Bosco K. Ho 
//
//  GraphicsView handles Vector3d inputs, sorts by depth in 3d, and converts 
//  it into a 2d allegro bitmap. Requires camera information.
//
//  ***************************************************************************
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU Lesser General Public License as published
//  by the Free Software Foundation; either version 2.1 of the License, or (at
//  your option) any later version.
//  
//  This program is distributed in the hope that it will be useful,  but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details. 
//  
//  You should have received a copy of the GNU Lesser General Public License 
//  along with this program; if not, write to the Free Software Foundation, 
//  Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
//  ****************************************************************************
//

#include <math.h>
#include <allegro.h>
#include "graphic3d.h"


static const int OUTLINE_COLOR = 197;



//******** drawing routines ********************



void wireframe (BITMAP *bmp, int vertices, int v[], int color) 
{
	for (int i=0; i<= (vertices-2); i+=1) 
		line (bmp, v[i*2], v[i*2 + 1], v[i*2 + 2], v[i*2 + 3], color);
	line (bmp, v[(vertices-1)*2], v[(vertices-1)*2 + 1], v[0], v[1], color);
}



//******** GfxList ********************



void GfxList::calculateMatrix(Vector3d& pos, Vector3d& front, Vector3d& up)
{
	get_camera_matrix(
	  &projectionMatrix, 
		ftofix(pos.x()),   ftofix(pos.y()),   ftofix(pos.z()), 
	  ftofix(front.x()), ftofix(front.y()), ftofix(front.z()), 
		ftofix(up.x()),    ftofix(up.y()),    ftofix(up.z()), 
		itofix(38), 
		itofix(1));
  cameraPos   = pos;
	cameraFront = front;
	cameraUp    = up.perpendicular(front);
}



void GfxList::getTransform(Vector3d& v, double r, 
                    int *xout, int *yout, int *zout, int *radius)
{ 
	fixed x, y, z;
	projectPoint (v, &x, &y, &z);
	if (z <= itofix (0)) return;

	*xout = fixtoi(x);
	*zout = fixtoi(z);
	*yout = fixtoi(y);

 	fixed x_r, y_r, z_r;
  projectPoint ( v + cameraUp.scaled(r), &x_r, &y_r, &z_r);
  if (z_r <= itofix (0)) return;
	*radius = fixtoi(fixhypot(x - x_r, y - y_r));
}



void GfxList::projectPoint (Vector3d p, fixed *xout, fixed *yout, fixed *zout)
{
	fixed x, y, z;

  apply_matrix(&projectionMatrix, 
	  ftofix (p.x ()), ftofix (p.y ()), ftofix (p.z ()), 	&x, &y, &z); 
	persp_project(x, y, z, xout, yout);
	*zout = z;
  *xout += ftofix (xOffset);
}



void GfxList::push(GfxObject& g) 
{
  _gList.push_back(g);
}



void GfxList::clear() 
{
  _gList.clear();
}



void GfxList::sort () 
{ 
 _gList.sort();
}



void GfxList::draw(BITMAP *bmp) 
{
	for (GList::iterator i = _gList.begin(); i != _gList.end(); i++) 
	{
		switch (i->id) 
		{
			case GFX_CIRCLEFILL:
				circlefill (bmp, i->v[0], i->v[1], i->v[2], i->color);
				break;
			case GFX_LINE:
				line (bmp, i->v[0], i->v[1], i->v[2], i->v[3], i->color);
				break;
			case GFX_CIRCLE:
				circle (bmp, i->v[0], i->v[1], i->v[2], i->color);
				break;
			case GFX_CIRCLE_OUTLINE :
				circlefill (bmp, i->v[0], i->v[1], i->v[2], i->color);
				circle (bmp, i->v[0], i->v[1], i->v[2]+1, OUTLINE_COLOR);
				break;
			case GFX_QUADFILL:
				polygon (bmp, 4, i->v, i->color);
				break;
			case GFX_QUAD:
				wireframe (bmp, 4, i->v, i->color);
				break;
			case GFX_QUAD_OUTLINE:
				polygon (bmp, 4, i->v, i->color);
				wireframe (bmp, 4, i->v, OUTLINE_COLOR);
				break;
			case GFX_NUMBER:
        textprintf (bmp, font, i->v[0]-1, i->v[1]-1, OUTLINE_COLOR, "%.2f", i->val);
        textprintf (bmp, font, i->v[0]+1, i->v[1]-1, OUTLINE_COLOR, "%.2f", i->val);
        textprintf (bmp, font, i->v[0]-1, i->v[1]+1, OUTLINE_COLOR, "%.2f", i->val);
        textprintf (bmp, font, i->v[0]+1, i->v[1]+1, OUTLINE_COLOR, "%.2f", i->val);
        textprintf (bmp, font, i->v[0], i->v[1], i->color, "%.2f", i->val);
				break;
		}
  }
}



//  Projecting 3d primitives into 2d primitives



void GfxList::project_thickline(Vector3d& p1, Vector3d& p2, int color, double width) 
{
	fixed x1, y1, z1;
	projectPoint(p1, &x1, &y1, &z1);
	if (z1 <= itofix (0)) return;

	fixed x1_r, y1_r, z1_r;
	projectPoint(p1 + cameraUp.scaled(width), &x1_r, &y1_r, &z1_r);
	if (z1_r <= itofix (0)) return;

  fixed r1 = fixhypot(x1 - x1_r, y1 - y1_r);

	fixed x2, y2, z2;
	projectPoint(p2, &x2, &y2, &z2);
	if (z2 <= itofix (0)) return;

	fixed x2_r, y2_r, z2_r;
	projectPoint(p2 + cameraUp.scaled(width), &x2_r, &y2_r, &z2_r);
	if (z2_r <= itofix (0)) return;

  fixed r2 = fixhypot(x2 - x2_r, y2 - y2_r);
	
	fixed dx = - (y2 - y1);
	fixed dy = x2 - x1;
	fixed d  = fixhypot(dx, dy);

	dx = fixdiv(dx, d);
	dy = fixdiv(dy, d);

  GfxObject gg;
	gg.id    = GFX_QUADFILL;
	gg.v[0]  = fixtoi(x1 + fixmul(dx, r1));
	gg.v[1]  = fixtoi(y1 + fixmul(dy, r1));
	gg.v[2]  = fixtoi(x2 + fixmul(dx, r2));
	gg.v[3]  = fixtoi(y2 + fixmul(dy, r2));
	gg.v[4]  = fixtoi(x2 - fixmul(dx, r2));
	gg.v[5]  = fixtoi(y2 - fixmul(dy, r2));
	gg.v[6]  = fixtoi(x1 - fixmul(dx, r1));
	gg.v[7]  = fixtoi(y1 - fixmul(dy, r1));
	gg.color = color;
	if (z1>z2) 
		gg.z   = fixtoi(z1); 
	else
		gg.z   = fixtoi(z2); 
  push(gg);

	r1 += itofix (1);
	r2 += itofix (1);

	gg.id    = GFX_QUAD;
	gg.v[0]  = fixtoi(x1 + fixmul(dx, r1));
	gg.v[1]  = fixtoi(y1 + fixmul(dy, r1));
	gg.v[2]  = fixtoi(x2 + fixmul(dx, r2));
	gg.v[3]  = fixtoi(y2 + fixmul(dy, r2));
	gg.v[4]  = fixtoi(x2 - fixmul(dx, r2));
	gg.v[5]  = fixtoi(y2 - fixmul(dy, r2));
	gg.v[6]  = fixtoi(x1 - fixmul(dx, r1));
	gg.v[7]  = fixtoi(y1 - fixmul(dy, r1));
	gg.color = OUTLINE_COLOR;
  push(gg);
}


void GfxList::project_thinline(Vector3d& p1, Vector3d& p2, int color) 
{
  fixed x1, y1, z1;
	projectPoint(p1, &x1, &y1, &z1);
	if (z1 <= itofix (0)) return;

  fixed x2, y2, z2;
	projectPoint(p2, &x2, &y2, &z2);
  if (z2 <= itofix (0)) return;

  GfxObject gg;
	gg.id    = GFX_LINE;
	gg.v[0]  = fixtoi(x1);
	gg.v[1]  = fixtoi(y1);
	gg.v[2]  = fixtoi(x2);
	gg.v[3]  = fixtoi(y2);
	gg.color = color;
	if (z1>z2) 
		gg.z   = fixtoi (z1); 
	else
		gg.z   = fixtoi (z2); 
  push(gg);
}



void GfxList::project_number(Vector3d& p1, int color, double val) 
{
  fixed x1, y1, z1;
	projectPoint (p1, &x1, &y1, &z1);
	if (z1 <= itofix (0)) return;

  GfxObject gg;
	gg.id    = GFX_NUMBER;
	gg.v[0]  = fixtoi (x1) + 3;
	gg.v[1]  = fixtoi (y1) - 7;
	gg.color = color;
	gg.z     = fixtoi (0); 
  gg.val   = val;
  push(gg);
}



void GfxList::project_quad (Vector3d v[], int color)
{
  GfxObject gg;

	gg.color = color;
	gg.id    = GFX_QUAD_OUTLINE;
	gg.z     = 0;
	for (int i=0; i<=3; i+=1) 
	{
  	fixed x, y, z;
		projectPoint (v[i], &x, &y, &z);
		if (z <= itofix (0)) return;
		gg.v[i*2]     = fixtoi (x);
		gg.v[i*2 + 1] = fixtoi (y);
		if (fixtoi (z) > gg.z) 
			gg.z = fixtoi (z);
	}
  push(gg);
}



void GfxList::project_circlefill (Vector3d& p, double r, int color)
{
	fixed x, y, z;
	projectPoint (p, &x, &y, &z);
	if (z <= itofix (0)) return;

 	fixed x_r, y_r, z_r;
  projectPoint ( p + cameraUp.scaled(r), &x_r, &y_r, &z_r);
  if (z_r <= itofix (0)) return;

	fixed radius = fixhypot (x - x_r, y - y_r);

  GfxObject gg;

	gg.id    = GFX_CIRCLE_OUTLINE;
  gg.v[0]  = fixtoi(x);
	gg.v[1]  = fixtoi(y);
	gg.v[2]  = fixtoi(radius);
	gg.color = color;
	gg.z     = fixtoi(z);
  push(gg);
}



void GfxList::project_enlarged_circle (Vector3d& p, double r, int enlarged_r, int color)
{
	fixed x, y, z;
  projectPoint (p, &x, &y, &z);
	if (z <= itofix (0)) return;

  fixed x_r, y_r, z_r;
  projectPoint (p + cameraUp.scaled(r), &x_r, &y_r, &z_r);
  if (z_r <= itofix (0)) return;

  fixed radius = fixhypot (x-x_r, y-y_r);

  GfxObject gg;

	gg.id=GFX_CIRCLE;
	gg.v[0]=fixtoi (x);
	gg.v[1]=fixtoi (y);
	gg.v[2]=fixtoi (radius) + enlarged_r + 1;
	gg.color=color;
	gg.z=fixtoi (z);
  push(gg);

	gg.id=GFX_CIRCLE;
	gg.v[2]=fixtoi (radius) + enlarged_r;
  push(gg);

}



//  GraphicsView



void GraphicsView::project_thickline (Vector3d& p1, Vector3d& p2, int color, double width)
{
  grList.project_thickline(p1, p2, color, width);
  if (isStereo)
    grListR.project_thickline(p1, p2, color, width);
}



void GraphicsView::project_thinline (Vector3d& p1, Vector3d& p2, int color)
{
  grList.project_thinline(p1, p2, color);
  if (isStereo)
    grListR.project_thinline(p1, p2, color);
}


void GraphicsView::project_number (Vector3d& p1, int color, double val)
{
  grList.project_number(p1, color, val);
  if (isStereo)
    grListR.project_number(p1, color, val);
}


void GraphicsView::project_quad (Vector3d v[], int color)
{
  grList.project_quad(v, color);
  if (isStereo)
    grListR.project_quad(v, color);
}


void GraphicsView::project_circlefill (Vector3d& p, double r, int color)
{
  grList.project_circlefill(p, r, color);
  if (isStereo)
 	  grListR.project_circlefill(p, r, color);
}



void GraphicsView::project_enlarged_circle (Vector3d& p, double r, int enlarged_r, int color)
{
  grList.project_enlarged_circle(p, r, enlarged_r, color);
  if (isStereo)
    grListR.project_enlarged_circle(p, r, enlarged_r, color);
}



bool GraphicsView::isInCircle(Vector3d& v, double r_out, int testX, int testY, int *zout)
{
  int x, y, r;
  grList.getTransform(v, r_out, &x, &y, zout, &r);

	if ((testX >= x-r-1) && (testX <= x+r+1)
    && (testY >= y-r-1) && (testY <= y+r+1))
	{
	  return true;
  }

  if (! isStereo) return false;

  grListR.getTransform(v, r_out, &x, &y, zout, &r);
	if ((testX >= x-r-1) && (testX <= x+r+1)
    && (testY >= y-r-1) && (testY <= y+r+1))
	{
	  return true;
  }

  return false;
}


void GraphicsView::draw(BITMAP *bmp)
{
	/* draw left eye p.o.v. if stereo */
	if (isStereo)
    set_clip(bmp, bmp->w/2 - stereoSeparation - 10, 0, bmp->w/2-10, bmp->h);

  grList.sort();
  grList.draw(bmp);

	if (isStereo)
	{
  	/* draw right eye p.o.v. */
	  set_clip(bmp, bmp->w/2+10, 0, bmp->w/2 + stereoSeparation+10, bmp->h);
    grListR.sort();
    grListR.draw(bmp);

	  set_clip(bmp, 0, 0, bmp->w, bmp->h);
  }
}


void GraphicsView::calculateMatrix(Vector3d& pos, Vector3d& front, Vector3d& up)
{
  set_projection_viewport(0, 0, SCREEN_W, SCREEN_H);

  grList.calculateMatrix(pos, front, up);
  grList.setXOffset(0);

	if (!isStereo) return;

  grList.setXOffset(-stereoSeparation/2 - 11);

	Vector3d front_r = front;

	front_r.rotate(-stereoViewAngle, up);

	Vector3d pos_r = pos + front - front_r;

  grListR.calculateMatrix(pos_r, front_r, up);
  grListR.setXOffset(stereoSeparation/2 + 10);

}

