//
//  **************************************************************************
//
//  residue.h
//
//  (C) 2003 Bosco K. Ho 
//
//  **************************************************************************
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU Lesser General Public License as published
//  by the Free Software Foundation; either version 2.1 of the License, or (at
//  your option) any later version.
//  
//  This program is distributed in the hope that it will be useful,  but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details. 
//  
//  You should have received a copy of the GNU Lesser General Public License 
//  along with this program; if not, write to the Free Software Foundation, 
//  Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
//  **************************************************************************
//

#ifndef residue_h_
#define residue_h_

#include <cstdio>
#include <vector>
#include <set>
#include <map>
#include <iostream>

#include "atoms.h"


/// enumerated type for amino acids
/// order follows alphabetical order of 1 letter code
const int nRes = 20; 
typedef enum 
{
	XXX ,
	ALA , /* A */
	CYS , /* C */
	ASP , /* D */
	GLU , /* E */
	PHE , /* F */
	GLY , /* G */
	HIS , /* H */
	ILE , /* I */
	LYS , /* K */
	LEU , /* L */
	MET , /* M */
	ASN , /* N */
	PRO , /* P */
	GLN , /* Q */
	ARG , /* R */
	SER , /* S */
	THR , /* T */
	VAL , /* V */
	TRP , /* W */
	TYR   /* Y */
};

int getResKey(const char c);
int getResKey(const std::string& s);
const char getResChar(int key);
const std::string& getResName3(int key);
const std::string& getResName(int key);


int getAtomKey(std::string s);
std::string getAtomName(int resKey, int atomKey);
/// enumerated type for atom types the order goes from the CA up
/// so that varying an atom in a sidechain implies that he position of atoms with
/// a higher value also needs to be varied
const int nAtoms = 66 ;  
typedef enum 
{
	XX  ,
	CA  ,
	C   ,
	HA1 ,
	HA2 ,
	N   ,
	O   ,
	H   ,
	H2  ,
	H3  ,
	OXT ,
	HXT ,
	CB  ,
	HB1 ,
	HB2 ,
	HB3 ,
	CG2 ,
	HG21,
	HG22,
	HG23,
	SG1 ,
	CG1 ,
	OG1 ,
	HG11,
	HG12,
	HG13,
	SD1 ,
	CD1 ,
	CD2 ,
	ND1 ,
	ND2 ,
	OD1 ,
	OD2 ,
	HD11,
	HD12,
	HD13,
	HD21,
	HD22,
	HD23,
	CE1 ,
	CE2 ,
	CE3 ,
	OE1 ,
	OE2 ,
	NE1 ,
	NE2 ,
	HE11,
	HE12,
	HE13,
	HE21,
	HE22,
	HE23,
	HE31,
	CZ1 ,
	CZ2 ,
	CZ3 ,
	NZ1 ,
	HZ11,
	HZ12,
	HZ13,
	HZ21,
	HZ31,
	CH2 ,
	OH1 ,
	NH1 ,
	NH2 ,
	HH11,
	HH12,
	HH21,
	HH22   
};



// look-up table for blocks of atom iterators in a residue
class ResBlk 
{
  public:
    ResBlk() {}
    ~ResBlk() {}
    int            resKey;             
	  std::string    resName;
	  int            resNo;
	  double         phi;
	  double         psi;
	  double         omega;

	  Atom&          atom(int atomKey);
	  int            nAtoms()          { return _atomIterList.size(); }
    AtomIter&      atomBegin()       { return *(_atomIterList.begin()); };
    AtomIter&      atomEnd()         { return _atomEnd; };
    double&        chi(int n)        { return _chi[n-1]; };
    void           pushAtomIter(AtomIter& a);

	  double         _chi[5];
	  AtomIterList   _atomIterList;
    AtomIter       _atomEnd;
};



int getNChi(int resKey);
const int resChiTetrad[nRes+1][5][4] = 
{
  // XXX
  { 
    {  XX   }
  },
  // ALA
  { 
    {  XX    }
  },
  // CYS
  { 
    {  N, CA, CB, SG1 },
    { XX }
  },
  // ASP
  {  
    {  N, CA,  CB, CG1 },
    { CA, CB, CG1, OD1 },
    { XX }
  },
  // GLU
  {  
    {   N, CA, CB, CG1 },
    {  CA, CB, CG1, CD1 },
    {  CB, CG1, CD1, OE1 },
    {  XX }
  },
  // PHE
  {  
    {  N, CA,  CB, CG1 },
    { CA, CB, CG1, CD1 },
    { XX }
  },
  // GLY
  {  
    { XX }
  },
  // HIS
  {  
    {  N, CA,  CB, CG1 },
    { CA, CB, CG1, ND1 },
    { XX }
  },
  // ILE
  {  
    {  N, CA,  CB, CG1 },
    { CA, CB, CG1, CD1 },
    { XX }
  },
  // LYS
  {  
    {   N,  CA,  CB, CG1 },
    {  CA,  CB, CG1, CD1 },
    {  CB, CG1, CD1, CE1 },
    { CG1, CD1, CE1, NZ1 },
    {  XX }
  },
  // LEU
  {  
    {   N, CA,  CB, CG1 },
    {  CA, CB, CG1, CD1 },
    {  XX }
  },
  // MET
  {  
    {   N, CA,   CB, CG1 },
    {  CA, CB,  CG1, SD1 },
    {  CB, CG1, SD1, CE1 },
    {  XX }
  },
  // ASN
  {  
    {  N, CA,  CB, CG1 },
    { CA, CB, CG1, OD1 },
    { XX }
  },
  // PRO
  {  
    { XX }
  },
  // GLN
  {  
    {   N,  CA,  CB, CG1 },
    {  CA,  CB, CG1, CD1 },
    {  CB, CG1, CD1, OE1 },
    {  XX }
  },
  // ARG
  {  
    {   N,   CA,  CB, CG1 },
    {  CA,   CB, CG1, CD1 },
    {  CB,  CG1, CD1, NE1 },
    {  CG1, CD1, NE1, CZ1 },
    {  XX }
  },
  // SER
  {  
    {  N, CA, CB, OG1 },
    { XX }
  },
  // THR
  {  
    {  N, CA, CB, OG1 },
    { XX }
  },
  // VAL
  {  
    {  N, CA, CB, CG1 },
    { XX }
  },
  // TRP
  {  
    {  N, CA,  CB, CG1 },
    { CA, CB, CG1, CD1 },
    { XX }
  },
  // TYR
  {  
    {  N, CA,  CB, CG1 },
    { CA, CB, CG1, CD1 },
    { XX }
  },
};



bool isAllowedAtom(int resKey, int atomKey);
const int resAllowedAtoms[nRes+1][nAtoms] = 
{
  // XXX
  { XX
  },
  // ALA
  { N  ,  CA ,  C  ,  O  ,  CB ,  H  , 
    HA1,  HB1, HB2 , HB3 ,  XX
  },
  // CYS
  {  N  ,  CA ,  C  ,  O  ,  CB ,  SG1, 
     H  ,  HA1,  HB1,  HB2, XX
  },
  // ASP
  {  N  ,  CA ,  C  ,  O  ,  CB ,  CG1, 
     OD1,  OD2,  H  , HA1, 
     HB1,  HB2,  XX
  },
  // GLU
  {  N  ,  CA ,  C  ,  O  ,  CB ,  CG1, 
     CD1,  OE1,  OE2,  H  , 
     HA1,  HB1,  HB2, HG11, HG12,  XX
  },
  // PHE
  {  N  ,  CA ,  C  ,  O  ,  CB ,  CG1, 
     CD1,  CD2,  CE1,  CE2,  CZ1,  H  , 
     HA1,  HB1,  HB2, HD11, HE11, HZ11, 
    HE21, HD21,  XX
  },
  // GLY
  {  N  ,  CA ,  C  ,  O  ,  H  ,  HA1, 
     HA2,  XX
  },
  // HIS
  {  N  ,  CA ,  C  ,  O  ,  CB ,  CG1, 
     ND1,  CD2,  CE1,  NE2,  H  , 
     HA1,  HB1,  HB2, HE11, HD21,  
     XX
  },
  // ILE
  {  N  ,  CA ,  C  ,  O  ,  CB ,  CG1, 
     CG2,  CD1,  H  ,  HA1,  HB1, HG21, 
    HG22, HG23, HG11, HG12, HD11, HD12, 
    HD13,  XX
  },
  // LYS
  {  N  ,  CA ,  C  ,  O  ,  CB ,  CG1, 
     CD1,  CE1,  NZ1,  H  , HZ11, HZ12, 
    HZ13,  HA1,  HB1,  HB2, HG11, HG12, 
    HD11, HD12, HE11, HE12, HZ13,   XX
  },
  // LEU
  {  N  ,  CA ,  C  ,  O  ,  CB ,  CG1, 
     CD1,  CD2,  H  ,  HA1,  HB1,  HB2, 
    HG11, HD11, HD12, HD13, HD21, HD22, 
    HD23,
    XX
  },
  // MET
  {  N  ,  CA ,  C  ,  O  ,  CB ,  CG1, 
     SD1,  CE1,  H  ,  HA1,  HB1,  HB2, 
    HG11, HG12, HE11, HE12, HE13,  XX
  },
  // ASN
  {  N  ,  CA ,  C  ,  O  ,  CB ,  CG1, 
     OD1,  ND2,  H  , HD21, HD22,  HA1, 
     HB1,  HB2,  XX
  },
  // PRO
  {  N  ,  CA ,  C  ,  O  ,  CB ,  CG1, 
     CD1, HD12, HD11, HG12, HG11,  HB1,
     HB2, HA1,  XX
  },
  // GLN
  {  N  ,  CA ,  C  ,  O  ,  CB ,  CG1, 
     CD1,  OE1,  NE2,  H  , HE21, HE22, 
     HA1,  HB1,  HB2, HG11, HG12,  XX
  },
  // ARG
  {  N  ,  CA ,  C  ,  O  ,  CB ,  CG1, 
     CD1,  NE1,  CZ1,  NH1,  NH2,  H  , 
    HH11, HH21, HE11,  HA1, 
     HB1,  HB2, HG11, HG12, HD11, HD12,  
     HH12, HH22,  XX
  },
  // SER
  {  N  ,  CA ,  C  ,  O  ,  CB ,  OG1, 
     H  , HG11,  HA1,  HB1,  HB2,  XX
  },
  // THR
  {  N  ,  CA ,  C  ,  O  ,  CB ,  OG1, 
     CG2,  H  , HG11,  HA1,  HB1, HG21, 
    HG22, HG23,  XX
  },
  // VAL
  {  N  ,  CA ,  C  ,  O  ,  CB ,  CG1, 
     CG2,  H  ,  HA1,  HB1, HG11, HG12, 
    HG13, HG21, HG22, HG23,  XX
  },
  // TRP
  {  N  ,  CA ,  C  ,  O  ,  CB ,  CG1, 
     CD1,  CD2,  NE1,  CE2,  CE3,  CZ2, 
     CZ3,  CH2,  H  , HE11,  HA1,  HB1, 
     HB2, HD11, HZ21, HH21, HZ31, HE31,
     XX
  },
  // TYR
  {  N  ,  CA ,  C  ,  O  ,  CB ,  CG1, 
     CD1,  CD2,  CE1,  CE2,  CZ1,  OH1, 
     H  , HH11,  HA1,  HB1,  HB2, HD11, 
    HE11, HE21, HD21,  XX
  }
};
 


bool isTitrableAtom(int resKey, int atomKey);
// add H2, H3 and OXT ???
const int resTitrableAtoms[nRes+1][nAtoms] = 
{
  {  XX  },               // XXX
  {  XX  },               // ALA
  {  HG11,  XX  },        // CYS
  {  HD21, HD11,  XX },   // ASP
  {  HE11, HE21,  XX },   // GLU
  {  XX  },               // PHE
  {  XX  },               // GLY
  {  HD11, HE21,  XX  },  // HIS
  {  XX  },               // ILE
  {  XX  },               // LYS
  {  XX  },               // LEU
  {  XX  },               // MET
  {  XX  },               // ASN
  {  XX  },               // PRO
  {  XX  },               // GLN
  {  XX  },               // ARG
  {  XX  },               // SER
  {  XX  },               // THR
  {  XX  },               // VAL
  {  XX  },               // TRP
  {  XX  }                // TYR
};


#endif

