//
//  **************************************************************************
//
//  residue.cpp
//
//  (C) 2003 Bosco K. Ho 
//
//  All the data structures and access functions that stores 
//  information about the different
//  amino acid types that form the residues in a protein. 
//
//  Includes ResBlk structure that is a look-up table, which collects all
//	the relevant information for each group of Atoms that corresponds
//	to a given residue. This includes an extensive list AtomIter
//	that points to the relevant atoms. In order not to disturb the
//	allowing structure, it is imperative that a single-linked or
//	double-linked list is used to represent the collection of
//	atoms. Although accessing the atom list is slow, this penalty
//	is avoided by having a look-up table in ResBlk, 
//
//	Both residue type and atom type are converted from the relevant
//	string to an enumerated type. This makes easier-to-read code and
//	cleaner data structures (no reliance on strings for comparison).
//	Furthermore, one can separate the atom recognition as a interface
//	layer in order to handle ambiguous naming of atoms.
//  
//  **************************************************************************
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU Lesser General Public License as published
//  by the Free Software Foundation; either version 2.1 of the License, or (at
//  your option) any later version.
//  
//  This program is distributed in the hope that it will be useful,  but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details. 
//  
//  You should have received a copy of the GNU Lesser General Public License 
//  along with this program; if not, write to the Free Software Foundation, 
//  Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
//  **************************************************************************
//

#include "residue.h"

using namespace std;



// data structure to store information for access functions
class ResTypeInfoBlk 
{
  public:
  	ResTypeInfoBlk(const char _c, std::string _ch3, std::string _name) :
	  	name1(_c), name3(_ch3), name(_name) {}
    char         name1;
	  std::string  name3;
	  std::string  name;
};



static const ResTypeInfoBlk resTypeInfo[] = 
{
	ResTypeInfoBlk('X', "XXX", "unknown"          ),
	ResTypeInfoBlk('A', "ALA", "alanine"		      ),
	ResTypeInfoBlk('C', "CYS", "cystine"	      	),
	ResTypeInfoBlk('D', "ASP", "aspartic acid"		),
	ResTypeInfoBlk('E', "GLU", "glutamatic acid"	),
	ResTypeInfoBlk('F', "PHE", "phenlyalanine"		),
	ResTypeInfoBlk('G', "GLY", "glycine"	        ),
	ResTypeInfoBlk('H', "HIS", "histidine"	    	),
	ResTypeInfoBlk('I', "ILE", "isoleucine"	      ),  
	ResTypeInfoBlk('K', "LYS", "lysine"	      	  ), 
	ResTypeInfoBlk('L', "LEU", "leucine"	      	),
	ResTypeInfoBlk('M', "MET", "methionine"	      ),   
	ResTypeInfoBlk('N', "ASN", "asparagine"	      ),  
	ResTypeInfoBlk('P', "PRO", "proline"   	      ),
	ResTypeInfoBlk('Q', "GLN", "glutamine"		    ),
	ResTypeInfoBlk('R', "ARG", "arginine"	  	    ),
	ResTypeInfoBlk('S', "SER", "serine"	       	  ),  
	ResTypeInfoBlk('T', "THR", "threonine"  	    ),
	ResTypeInfoBlk('V', "VAL", "valine"	    	    ), 
	ResTypeInfoBlk('W', "TRP", "tryptophan"	   	  ),
	ResTypeInfoBlk('Y', "TYR", "tyrosine"         )  
};  



int getResKey(const char c)
{
  for (int i=1; i<=nRes; i++)
		if (c == resTypeInfo[i].name1) return i;
	return XXX;
}



int getResKey(const std::string& s) 
{
  if (s.length() == 1) 
	{
		for (int i=1; i<=nRes; i++)
			if (s[0] == resTypeInfo[i].name1)  return i;
	}
  else if (s.length() == 3) 
	{
		for (int i=1; i<=nRes; i++)
			if (s == resTypeInfo[i].name3)  return i;
	}
	else 
	{
		for (int i=1; i<=nRes; i++)
			if (s == resTypeInfo[i].name)  return i;
	}
  cerr << "Warning: residue not parsed: " << s << endl;
	return XXX;
}



const std::string& getResName3(int key) 
{
  return resTypeInfo[key].name3;
}



const char getResChar(int key) 
{
  return resTypeInfo[key].name1;
}



const std::string& getResName(int key) 
{
  return resTypeInfo[key].name;
}



bool isAllowedAtom(int resKey, int atomKey) 
{
  for (int j=0; resAllowedAtoms[resKey][j] != XX; j++)
    if (atomKey == resAllowedAtoms[resKey][j]) return true;
  return false;
}



bool isTitrableAtom(int resKey, int atomKey) {
  int j;
  for (j=0; resTitrableAtoms[resKey][j] != XX; j++)
    if (atomKey == resTitrableAtoms[resKey][j]) return true;
  return false;
}



int getNChi(int resKey) {
  for (int nChi=0; nChi<5; nChi++)
    if (resChiTetrad[resKey][nChi][0] == XX) return nChi;
  return 0;  
}



static const std::string atomName[nAtoms+1] = 
{
	"    " ,
	" CA1" ,
	" C 1" ,
	"1HA1" ,
	"2HA1" ,
	" N 1" ,
	" O 1" ,
	"1H 1" ,
	" CB1" ,
	"1HB1" ,
	"2HB1" ,
	"3HB1" ,
	" CG2" ,
	"1HG2" ,
	"2HG2" ,
	"3HG2" ,
	" SG1" ,
	" CG1" ,
	" OG1" ,
	"1HG1" ,
	"2HG1" ,
	"3HG1" ,
	" SD1" ,
	" CD1" ,
	" CD2" ,
	" ND1" ,
	" ND2" ,
	" OD1" ,
	" OD2" ,
	"1HD1" ,
	"2HD1" ,
	"3HD1" ,
	"1HD2" ,
	"2HD2" ,
	"3HD2" ,
	" CE1" ,
	" CE2" ,
	" CE3" ,
	" OE1" ,
	" OE2" ,
	" NE1" ,
	" NE2" ,
	"1HE1" ,
	"2HE1" ,
	"3HE1" ,
	"1HE2" ,
	"2HE2" ,
	"3HE2" ,
	"1HE3" ,
	" CZ1" ,
	" CZ2" ,
	" CZ3" ,
	" NZ1" ,
	"1HZ1" ,
	"2HZ1" ,
	"3HZ1" ,
	"1HZ2" ,
	"1HZ3" ,
	" CH2" ,
	" OH1" ,
	" NH1" ,
	" NH2" ,
	"1HH1" ,
	"2HH1" ,
	"1HH2" ,
	"2HH2"   
};


// parses an atom name string into the atom key
int getAtomKey(string s)
{
	if (s.length() < 4) return XX;
	if (s[3]==' ') 
		s[3]='1';

	if ((s[1]=='H') && (s[0]==' ')) 
		s[0]='1';

/*  for (int i=1; i<=nAtoms; i+=1) 
	{
		if (s == atomName[i]) 
		{
			return i;
		}
  }
*/
  
	if (s == " CA1") return CA;
	if (s == " C 1") return C ;
	if (s == "1HA1") return HA1;
	if (s == "2HA1") return HA2;
	if (s == " N 1") return N ;
	if (s == " O 1") return O ;
  if (s == "1H 1") return H ;
  if (s == "2H 1") return H2;
  if (s == "3H 1") return H3;
  if (s == " OXT") return OXT ;
  if (s == "1HXT") return HXT ;
  if (s == " CB1") return CB ;
  if (s == "1HB1") return HB1;
  if (s == "2HB1") return HB2;
  if (s == "3HB1") return HB3;
  if (s == " CG2") return CG2 ;
  if (s == "1HG2") return HG21;
  if (s == "2HG2") return HG22;
  if (s == "3HG2") return HG23;
  if (s == " SG1") return SG1 ;
  if (s == " CG1") return CG1 ;
  if (s == " OG1") return OG1 ;
  if (s == "1HG1") return HG11;
  if (s == "2HG1") return HG12;
  if (s == "3HG1") return HG13;
  if (s == " SD1") return SD1 ;
  if (s == " CD1") return CD1 ;
  if (s == " CD2") return CD2 ;
  if (s == " ND1") return ND1 ;
  if (s == " ND2") return ND2 ;
  if (s == " OD1") return OD1 ;
  if (s == " OD2") return OD2 ;
  if (s == "1HD1") return HD11;
  if (s == "2HD1") return HD12;
  if (s == "3HD1") return HD13;
  if (s == "1HD2") return HD21;
  if (s == "2HD2") return HD22;
  if (s == "3HD2") return HD23;
  if (s == " CE1") return CE1 ;
  if (s == " CE2") return CE2 ;
  if (s == " CE3") return CE3 ;
  if (s == " OE1") return OE1 ;
  if (s == " OE2") return OE2 ;
  if (s == " NE1") return NE1 ;
  if (s == " NE2") return NE2 ;
  if (s == "1HE1") return HE11;
  if (s == "2HE1") return HE12;
  if (s == "3HE1") return HE13;
  if (s == "1HE2") return HE21;
  if (s == "2HE2") return HE22;
  if (s == "3HE2") return HE23;
  if (s == "1HE3") return HE31;
  if (s == " CZ1") return CZ1 ;
  if (s == " CZ2") return CZ2 ;
  if (s == " CZ3") return CZ3 ;
  if (s == " NZ1") return NZ1 ;
  if (s == "1HZ1") return HZ11;
  if (s == "2HZ1") return HZ12;
  if (s == "3HZ1") return HZ13;
  if (s == "1HZ2") return HZ21;
  if (s == "1HZ3") return HZ31;
  if (s == " CH2") return CH2 ;
  if (s == " OH1") return OH1 ;
  if (s == " NH1") return NH1 ;
  if (s == " NH2") return NH2 ;
  if (s == "1HH1") return HH11;
  if (s == "2HH1") return HH12;
  if (s == "1HH2") return HH21;
  if (s == "2HH2") return HH22;
  
	cerr << "Warning: atom not parsed: " << s << endl;
	return XX;
}



string getAtomName(int resKey, int atomKey)
{
	string s;
	switch (atomKey) 
  {
		case CA  : s = " CA "; break;
		case CB  : s = " CB "; break;
		case HA1 :	
          if (resKey == GLY) 
						   s = "1HA "; 
					else 
						   s = " HA "; 
					break;
		case HA2 : s = "2HA "; break;
		case H   : s = " H  "; break;
		case H2  : s = "2H  "; break;
		case H3  : s = "3H  "; break;
		case OXT : s = " OXT"; break;
		case HXT : s = " HXT"; break;
		case O   : s = " O  "; break;
		case N   : s = " N  "; break;
		case C   : s = " C  "; break;
    case HB1 : s = "1HB1"; break;
    case HB2 : s = "2HB1"; break;
    case HB3 : s = "3HB1"; break;
    case CG2 : s = " CG2"; break;
    case HG21: s = "1HG2"; break;
    case HG22: s = "2HG2"; break;
    case HG23: s = "3HG2"; break;
    case SG1 : s = " SG1"; break;
    case CG1 : s = " CG1"; break;
    case OG1 : s = " OG1"; break;
    case HG11: s = "1HG1"; break;
    case HG12: s = "2HG1"; break;
    case HG13: s = "3HG1"; break;
    case SD1 : s = " SD1"; break;
    case CD1 : s = " CD1"; break;
    case CD2 : s = " CD2"; break;
    case ND1 : s = " ND1"; break;
    case ND2 : s = " ND2"; break;
    case OD1 : s = " OD1"; break;
    case OD2 : s = " OD2"; break;
    case HD11: s = "1HD1"; break;
    case HD12: s = "2HD1"; break;
    case HD13: s = "3HD1"; break;
    case HD21: s = "1HD2"; break;
    case HD22: s = "2HD2"; break;
    case HD23: s = "3HD2"; break;
    case CE1 : s = " CE1"; break;
    case CE2 : s = " CE2"; break;
    case CE3 : s = " CE3"; break;
    case OE1 : s = " OE1"; break;
    case OE2 : s = " OE2"; break;
    case NE1 : s = " NE1"; break;
    case NE2 : s = " NE2"; break;
    case HE11: s = "1HE1"; break;
    case HE12: s = "2HE1"; break;
    case HE13: s = "3HE1"; break;
    case HE21: s = "1HE2"; break;
    case HE22: s = "2HE2"; break;
    case HE23: s = "3HE2"; break;
    case HE31: s = "1HE3"; break;
    case CZ1 : s = " CZ1"; break;
    case CZ2 : s = " CZ2"; break;
    case CZ3 : s = " CZ3"; break;
    case NZ1 : s = " NZ1"; break;
    case HZ11: s = "1HZ1"; break;
    case HZ12: s = "2HZ1"; break;
    case HZ13: s = "3HZ1"; break;
    case HZ21: s = "1HZ2"; break;
    case HZ31: s = "1HZ3"; break;
    case CH2 : s = " CH2"; break;
    case OH1 : s = " OH1"; break;
    case NH1 : s = " NH1"; break;
    case NH2 : s = " NH2"; break;
    case HH11: s = "1HH1"; break;
    case HH12: s = "2HH1"; break;
    case HH21: s = "1HH2"; break;
    case HH22: s = "2HH2"; break;

    //		default:	s = atomName[atomKey];
	}

	return s;
}



//// ResBlk class routines



Atom& ResBlk::atom(int atomKey) 
{
  AtomIterList::iterator iter = _atomIterList.begin(); 
  for (iter; iter != _atomIterList.end(); iter++) 
		if ((*iter)->atomKey == atomKey) return *(*iter);
	return nullAtom;
}




void ResBlk::pushAtomIter(AtomIter& a) 
{
	_atomIterList.push_back(a);
}




