//
//  **************************************************************************
//
//  protein.h
//
//  (C) 2003 Bosco K. Ho 
//
//  Adapted from the The Biomolecule Toolkit (C) 2001, Eric Alm 
//  
//  **************************************************************************
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU Lesser General Public License as published
//  by the Free Software Foundation; either version 2.1 of the License, or (at
//  your option) any later version.
//  
//  This program is distributed in the hope that it will be useful,  but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details. 
//  
//  You should have received a copy of the GNU Lesser General Public License 
//  along with this program; if not, write to the Free Software Foundation, 
//  Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
//  **************************************************************************
//

#ifndef protein_h_
#define protein_h_

#include <cstdio>
#include <vector>
#include <set>
#include <map>
#include <iostream>

#include "molecule.h"
#include "atoms.h"
#include "residue.h"


class Protein : public Molecule 
{
  public:
    Protein(char chain_id = ' ', const std::string name = "untitled") :
            Molecule(chain_id, name), _resList() {};
    Protein(const Protein & rhs);
    ~Protein() {}

    void    scanResBlk();
    void    checkMissingAtoms();
    void    extractAngles();
    void    setPdbCode(std::string pdbCode) { _pdbCode = pdbCode; } 
    std::string    getPdbCode()     { return _pdbCode; } 

    Atom&   operator () (int resIndex, int atomKey);
    Atom&   atom(int resIndex, int atomKey);
    bool    hasAtom(int resIndex, int atomKey);

    double  getMaxLength();

    void    insertAtom(int resIndex, Atom a);
    void    insertAtom(int resIndex, int atomKey);
    void    eraseAtom(int resIndex, int atomKey);

    int     nRes() { return _resList.size(); } ;
    ResBlk& res(int resIndex);

    void    initializeAtomsInRes(int resIndex);
    void    eraseExcessAtomsInRes(int resIndex);

    void    insertRes(int resIndex, int resKey);
    void    eraseRes(int resIndex);
    void    appendRes(int resKey);
    void    mutateRes(int resIndex, int resKey);

    void    deleteRes(int leftRes, int rightRes);
    void    copyResToRight(int leftRes, int rightRes);
    void    spliceSegment(int target, int seg1, int seg2);
  
    void    rotatePhi(int i, double phi);
    void    rotatePsi(int i, double psi);
    void    rotateChi(int i, int n, double chi);

    void    sproutBackboneFromPrev(int i);
    void    sproutSidechain(int i, int n);
    void    sproutWholeSidechain(int i);
    void    rebuildBackboneFromScratch(int i);
 
    void    projectAtom(int i, int a1, double length, double angle, double dihed,
                        int a2, int a3, int a4);
    void    projectHydrogen(int i, int a1, double length, double angle, double dihed,
                            int a2, int a3, int a4);
    void    sproutHydrogen(int i);

    std::string  fullAtomName(int i, int a);

    friend  std::ostream& operator << (std::ostream & os, Protein& p);

  private:
    std::vector<ResBlk> _resList;
    std::string         _pdbCode;
};


#endif

