//
//  **************************************************************************
//
//  pdbreader.cpp
//  (C) 2003 Bosco K. Ho 
//
//  Adapted from the The Biomolecule Toolkit (C) 2001, Eric Alm 
//	Keeps the PDB interface separate from the rest of the code, allowing 
//  easier code maintainence 
//
//  **************************************************************************
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU Lesser General Public License as published
//  by the Free Software Foundation; either version 2.1 of the License, or (at
//  your option) any later version.
//  
//  This program is distributed in the hope that it will be useful,  but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details. 
//  
//  You should have received a copy of the GNU Lesser General Public License 
//  along with this program; if not, write to the Free Software Foundation, 
//  Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
//  **************************************************************************
//
 

#include <fstream>
#include <algorithm>
#include "pdbreader.h"


using namespace std;


PdbReader::~PdbReader() 
{
  if ((_filename != "") && (_stream != NULL))  // input was via file
    delete _stream;
}



// Open _stream for input, check for errors
void PdbReader::loadStream()
{
  // open the stream or file and make sure it is good
  if (_hasLoadedStream) return;

  // input via file
  if (_filename != "") 
  { 
    _stream = new ifstream(_filename.c_str());
    if (_stream->fail()) 
      return;
  }
  else 
  { // input via passed stream
    if (_stream->fail()) 
      return;
  }
  _hasLoadedStream = true;
}



// load one model from the stream into _buffer
void PdbReader::loadModel()
{
  if (_hasLoadedModel) return;

  loadStream();
  string current_line;
  getline(*_stream, current_line);

  while (!_stream->eof() && current_line.substr(0,6) != "ENDMDL") 
  {
    _buffer.push_back(current_line);
    getline(*_stream,current_line);
  }

  _hasLoadedModel = true;
}



// return the Pdb header
string PdbReader::getHeader()
{
  loadModel();

  BufferList::iterator lineIter;
  for (lineIter = _buffer.begin(); lineIter < _buffer.end(); lineIter++) 
  {
    if (lineIter->substr(0,6) == "HEADER") 
      return lineIter->substr(10,40);
  }
  return "";
}



// return the Pdb header
string PdbReader::getPdbCode() 
{
  loadModel();

  BufferList::iterator lineIter;
  for (lineIter = _buffer.begin(); lineIter < _buffer.end(); lineIter++) 
  {
    if (lineIter->substr(0,6) == "HEADER") 
      return lineIter->substr(62,4);
  }
  return "";
}



// determine the chains in the current model
vector<char> PdbReader::getChainLabels()
{
  loadModel();

  vector<char> _chains;
  BufferList::iterator lineIter;
  for (lineIter = _buffer.begin(); lineIter < _buffer.end(); lineIter++) 
  {
    if (lineIter->substr(0,4) != "ATOM") continue;  // only search ATOM records
    char ch = (*lineIter)[21]; // get the chain
    if (find(_chains.begin(), _chains.end(), ch) == _chains.end()) 
      _chains.push_back(ch); 
  }
  return _chains;
}



void PdbReader::loadChain(Protein& p, const char chainLabel) 
{
  char chainId = chainLabel;
  if (chainId == '?') {
    std::vector<char> pdb_chains = getChainLabels();
    chainId = pdb_chains[0];
  }

  loadModel();

  p.clear();

  // parse the file
  BufferList::iterator lineIter;
  BufferList::iterator end = _buffer.end();
  for (lineIter = _buffer.begin(); lineIter < end; lineIter++) 
  {
    Atom atom;

    if (lineIter->substr(0,6) != "ATOM  ") continue;


    atom.chainId     = (*lineIter)[21]; 
    if (atom.chainId != chainId) continue;
    
    //load lineIter
    atom.atomNo      = atoi( lineIter->substr(6,5).c_str() );
    atom.atomName    = lineIter->substr(12,4); 
    atom.atomKey     = getAtomKey(atom.atomName);  
    atom.altLocNo    = (*lineIter)[16];
    atom.resName     = lineIter->substr(17,3);  
    atom.resNo       = atoi( lineIter->substr(22,4).c_str() );
    atom.resKey      = getResKey(atom.resName); 
    atom.insertCode  = (*lineIter)[26];
    atom.pos         = Vector3d( atof( lineIter->substr(30,8).c_str() ),
                                 atof( lineIter->substr(38,8).c_str() ),
                                 atof( lineIter->substr(46,8).c_str() ) );
    if (lineIter->size() >= 60)
      atom.occup     = atof( lineIter->substr(54,6).c_str() );
    else
      atom.occup     = defaultOccupancy;

    if (lineIter->size() >= 66) 
      atom.bFactor   = atof( lineIter->substr(60,6).c_str() );
    else 
      atom.bFactor   = defaultBFactor;
    
    atom.isOn        = true;     

    // Skip altLoc positions:
    if ( atom.altLocNo != ' ' && atom.altLocNo != 'A' &&
         atom.altLocNo != '1')
      continue;

    p.pushAtom(atom);
  }

  p.scanResBlk();
  p.extractAngles();
  p.setName(getHeader());
  p.setPdbCode(getPdbCode());

}
