//
//  ***************************************************************************
//
//  display.cpp
//  (C) 2004 Bosco K. Ho 
//
//  DisplayBox is a widget that displays a protein in an allegro
//  bitmap. DisplayBox uses GraphicsView to convert 3d to 2d
//  drawing primitives; handles mouse input to rotate, zoom, and
//  to choose different active residues by clicking; and 
//  provides various options to display the protein.
//
//  ***************************************************************************
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU Lesser General Public License as published
//  by the Free Software Foundation; either version 2.1 of the License, or (at
//  your option) any later version.
//  
//  This program is distributed in the hope that it will be useful,  but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details. 
//  
//  You should have received a copy of the GNU Lesser General Public License 
//  along with this program; if not, write to the Free Software Foundation, 
//  Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
//  ****************************************************************************
//


#include "display.h"


bool isInSphere(Vector3d& v1, Vector3d& v2)
{
  return(distanceSquared(v1, v2) < CA_DIST_SQ);
}




DisplayBox::DisplayBox(ProteinDisplay* p, int x, int y, int w, int h)
   : Widget(x, y, w, h), protein(p),
   isCentered(false), stereoState(STEREO_NONE), isHbond(true), isSidechain(true),
   isSolid(true), isPeptide(true), isHydrogen(true), pickState(PICK_RESIDUE),
   showState(SHOW_ALL)
{
  protein->setGraphicsView(&graphicsView);
  graphicsView.setStereoSeparation(SCREEN_W / 3);
  initializeFocus();
}



void DisplayBox::countBonds()
{
  hbondList.clear();
  weakHbondList.clear();
  disulfideBondList.clear();

  if (!isHbond) return;

  if (showState == SHOW_SPHERE)
  {
    for (int i=1; i<=protein->nRes(); i+=1)
    {
      if (isInSphere(protein->atom(i, CA).pos, focus_pos))
      {
        if (isSidechain)
          hbondList.countHbondsWithinRes(protein, i);
        for (int j=i+1; j<=protein->nRes(); j+=1)
        {
          if (isInSphere(protein->atom(j, CA).pos, focus_pos))
          {
            if (j > i+1)
            {
              if (isSidechain)
              {
                hbondList.countHbondsBetweenResidues(protein, i, j);
                weakHbondList.countHbondsBetweenResidues(protein, i, j);
                disulfideBondList.countBondsBetweenResidues(protein, i, j);
             }
              else
              {
                hbondList.countBackboneHbondsBetweenResidues(protein, i, j);
                weakHbondList.countBackboneHbondsBetweenResidues(protein, i, j);
              }
            }
            if (j > i)
            {
              if (isSidechain)
              {
                hbondList.countHbondsBetweenResidues(protein, j, i);
                weakHbondList.countHbondsBetweenResidues(protein, j, i);
                disulfideBondList.countBondsBetweenResidues(protein, j, i);
             }
              else
              {
                hbondList.countBackboneHbondsBetweenResidues(protein, j, i);
                weakHbondList.countBackboneHbondsBetweenResidues(protein, j, i);
              }
            }

          }
        }
      }
    }
  }
  else
  {
    int show_left_res  = 1;
    int show_right_res = protein->nRes();

    if (showState == SHOW_SEGMENT)
    {
      show_left_res = view_left_res;
      show_right_res = view_right_res;
    }

    for (int i=show_left_res; i<=show_right_res; i+=1)
    {
      if (isSidechain)
        hbondList.countHbondsWithinRes(protein, i);
      for (int j=i+1; j<=show_right_res; j+=1)
      {
        if (j > i+1)
        {
          if (isSidechain)
          {
            hbondList.countHbondsBetweenResidues(protein, i, j);
            weakHbondList.countHbondsBetweenResidues(protein, i, j);
            disulfideBondList.countBondsBetweenResidues(protein, i, j);
          }
          else
          {
            hbondList.countBackboneHbondsBetweenResidues(protein, i, j);
            weakHbondList.countBackboneHbondsBetweenResidues(protein, i, j);
          }
        }
        if (j > i)
        {
          if (isSidechain)
          {
            hbondList.countHbondsBetweenResidues(protein, j, i);
            weakHbondList.countHbondsBetweenResidues(protein, j, i);
            disulfideBondList.countBondsBetweenResidues(protein, j, i);
         }
          else
          {
            hbondList.countBackboneHbondsBetweenResidues(protein, j, i);
            weakHbondList.countBackboneHbondsBetweenResidues(protein, j, i);
          }
        }
      }
    }
  }
}



// sets the focus to the residue closest to the center of mass
void DisplayBox::initializeFocus(void)
{
  if (protein->nRes()==0) return;
  protein->computeCenterOfMass();

  active_left_res = 1;

  Vector3d centerOfMass = protein->getCenterOfMass();
  if (protein->nRes() > 0)
  {
    float least = distanceSquared(centerOfMass, protein->atom(1, CA).pos);
    for (int i=1; i<=protein->nRes(); i+=1)
    {

      float new_least = distanceSquared(centerOfMass, protein->atom(i, CA).pos);
      if (new_least < least)
      {
        active_left_res = i;
        least = new_least;
      }
    }
  }

  active_right_res = active_left_res;
  focus_res        = active_left_res;
  focus_atom       = CA;

  view_left_res = active_left_res - VIEW_SIZE/2;
  if (view_left_res < 1) view_left_res = 1;
  view_right_res = view_left_res + VIEW_SIZE - 1;

  if (view_right_res > protein->nRes()) 
    view_right_res = protein->nRes();
  if (view_right_res < 1) 
    view_right_res = 1;

  if (VIEW_SIZE < protein->nRes())
  {
    while ( (view_right_res - view_left_res + 1 < VIEW_SIZE) && (view_left_res > 1) )
      view_left_res -= 1;
  }

  setFocus();

  camera_look = focus_axis.scaled(-protein->getMaxLength()*2.0);
	camera_up = focus_up.scaled(1.0);

	computeCamera();
}



// computes a new transformation matrix based on the given
// focus position and camera-look and camera-up vectors
void DisplayBox::computeCamera(void)
{
	Vector3d camera_pos = focus_pos - camera_look;
  graphicsView.isStereo = stereoState != STEREO_NONE;
  if (stereoState == STEREO_CROSSEYE) 
    graphicsView.setStereoViewAngle(8);
  if (stereoState == STEREO_WALL_TO_WALL) 
    graphicsView.setStereoViewAngle(-8);
  graphicsView.calculateMatrix(camera_pos, camera_look, camera_up);
}



// given the focus_res and focus_atom values, changes the focus position
// to that focus_atom. but if the option is to center then calculates
// the current center of mass based on user selected options
// once the focus is calculated, does house-keeping, calculates the
// new camera position and counts H-bonds
void DisplayBox::setFocus(void)
{
  if (protein->nRes() == 0) return;

  Vector3d ca_n = protein->atom(focus_res, N).pos - protein->atom(focus_res, CA).pos;
	Vector3d ca_c = protein->atom(focus_res, C).pos - protein->atom(focus_res, CA).pos;
	focus_up      = -(ca_n + ca_c);
	focus_axis    = cross(ca_n, ca_c);

  if (isCentered)
	{
    if (showState == SHOW_SPHERE)
		{
      Vector3d _centerOfMass(0.0, 0.0, 0.0);
      double nAtoms = 0;


      for (int i=1; i<=protein->nRes(); i++)
        if (isInSphere(protein->atom(i, CA).pos, focus_pos))
				{
         	AtomIter iter;
          for (iter = protein->res(i).atomBegin(); iter != protein->res(i).atomEnd(); iter++)
					{
            _centerOfMass += iter->pos;
            nAtoms += 1.0;
          }

        }
      _centerOfMass /= nAtoms;

			focus_pos = _centerOfMass;
    }
    else
		{
      int show_left_res  = 1;
      int show_right_res = protein->nRes();

      if (showState == SHOW_SEGMENT)
			{

        show_left_res = view_left_res;
        show_right_res = view_right_res;
      }

      Vector3d _centerOfMass(0.0, 0.0, 0.0);
      double nAtoms = 0;

      for (int i=show_left_res; i<=show_right_res; i++)
			{
       	AtomIter iter;
        for (iter = protein->res(i).atomBegin(); iter != protein->res(i).atomEnd(); iter++)
				{
          _centerOfMass += iter->pos;
          nAtoms += 1.0;
        }
      }
      _centerOfMass /= (nAtoms);
			focus_pos = _centerOfMass;
    }

  }
	else
  	focus_pos = protein->atom(focus_res, focus_atom).pos;

  computeCamera();
  countBonds();
}



// since we have a way of defining a standard reference frame w.r.t to each amino acid
// backbone, this function jumps to another focus_position but tries to
// maintain the relative orientation of the camera to the standard reference frame
void DisplayBox::setFocusWithSameOrientation(void)
{
	/* get relative angles of old focus */
	double camera_look_a = angle(focus_up, camera_look);
	double camera_look_b = dihedral(camera_look, focus_up, focus_axis);

	double zoom   = camera_look.length();
	double up_a   = dihedral(camera_up, camera_look, focus_up);

	setFocus();

	/* reconstruct camera_look and camera_up using the relative angles of old focus */
	camera_look = focus_axis.rotated(camera_look_b, focus_up);

	Vector3d perp = cross(focus_up, camera_look);
	camera_look.rotate(-(90 - camera_look_a), perp);
	camera_look.scaleTo(zoom);

	perp = cross(focus_up, camera_look);
	camera_up = camera_look.rotated(-90, perp);
	camera_up.rotate(up_a, camera_look);
  computeCamera();
}



int DisplayBox::handle(int event)
{
  switch (event)
	{

	  case WGT_MOUSE_B1_DOWN:

		{
			if (highlight_res != 0)
			{
				if (pickState == PICK_DISTANCE)
				{
					if (line_res == 0)
					{
						line_res  = highlight_res;
						line_atom = highlight_atom;
						line_res2 = 0;
					}
					else if (line_res2 == 0)
					{
						if ((highlight_res == line_res) && (highlight_atom == line_atom))
		  				line_res = 0;
						else
						{
							line_res2  = highlight_res;
							line_atom2 = highlight_atom;
						}
					}

					else
					{
						line_res  = highlight_res;
						line_atom = highlight_atom;
						line_res2 = 0;
					}
				}
				else
				{
					active_left_res  = highlight_res;
					active_right_res = highlight_res;
				}
  	  }
			_saveMouseY = mouse_y;
	  	_saveMouseX = mouse_x;
		}
		break;

	  case WGT_MOUSE_B1_DRAG:
		{
			double rotate = (mouse_x - _saveMouseX) / 3.0;
			camera_look.rotate(-rotate, camera_up);

		  Vector3d norm = cross(camera_look, camera_up);

			double pitch  = (mouse_y - _saveMouseY) / 3.0;
			camera_look.rotate(-pitch, norm);
			camera_up.rotate(-pitch, norm);


			computeCamera();


			_saveMouseY=mouse_y;
			_saveMouseX=mouse_x;
		}
    break;

	  case WGT_MOUSE_B2_DOWN:
		{
			if (highlight_res != 0)
      {
				focus_res  = highlight_res;
				focus_atom = highlight_atom;
				setFocus();
			}
			_saveMouseX = mouse_x;
			_saveMouseY = mouse_y;
		}
    break;

	  case WGT_MOUSE_B2_DRAG:
		{
			int new_zoom = (int)(camera_look.length() + ((mouse_y - _saveMouseY)/3.0));
			if (new_zoom > 0) camera_look.scaleTo(new_zoom);
			camera_up.rotate((mouse_x - _saveMouseX) / 3.0, camera_look);
			computeCamera();

			_saveMouseY = mouse_y;
			_saveMouseX = mouse_x;
    }
		break;

    default:
      return 0;
	}
	return 1;
}



void DisplayBox::draw(BITMAP *bmp)
{
  graphicsView.clear();

  protein->isHydrogen  = isHydrogen;
  protein->isSolid     = isSolid;
  protein->isSidechain = isSidechain;

  // clear highlight check;
  protein->highlight_res = 0;

  int i;
  // draw the h-bonds
  for (i=1; i<=hbondList.list.size(); i+=1)
  {
    if (isHydrogen)
      protein->draw_line(hbondList.list[i-1]._i, hbondList.list[i-1]._o,
                    hbondList.list[i-1]._j, hbondList.list[i-1]._h, ORANGE);
    else
      protein->draw_line(hbondList.list[i-1]._i, hbondList.list[i-1]._o,
                    hbondList.list[i-1]._j, hbondList.list[i-1]._n, ORANGE);
  }


  // draw the weakh-bonds
  for (i=1; i<=weakHbondList.list.size(); i+=1)
  {
    if (isHydrogen)
      protein->draw_line(weakHbondList.list[i-1]._i, weakHbondList.list[i-1]._o,
                    weakHbondList.list[i-1]._j, weakHbondList.list[i-1]._h, LIGHT_ORANGE);
    else
      protein->draw_line(weakHbondList.list[i-1]._i, weakHbondList.list[i-1]._o,
                    weakHbondList.list[i-1]._j, weakHbondList.list[i-1]._n, LIGHT_ORANGE);
  }

  
  // draw the disulfide bonds
  for (i=1; i<=disulfideBondList.list.size(); i+=1)
  {
    protein->draw_line(disulfideBondList.list[i-1]._i, disulfideBondList.list[i-1]._o,
                    disulfideBondList.list[i-1]._j, disulfideBondList.list[i-1]._h, YELLOW);
  }

  
  // draw protein
	if (showState == SHOW_SPHERE)
	{
    int i;
    if (isPeptide)
      for (i=2; i <= protein->nRes(); i++)
        if (isInSphere(protein->atom(i, CA).pos, focus_pos))
				{
	      	protein->draw_peptide_plane(i);
          if ( (!isInSphere(protein->atom(i+1, CA).pos, focus_pos)) && (i+1<=protein->nRes()) )
	      	  protein->draw_peptide_plane(i+1);
        }

    for (i=2; i <= protein->nRes(); i++)
      if (isInSphere(protein->atom(i, CA).pos, focus_pos))
			{
  	  	protein->draw_bond(i-1, C, i, N);
        if ((!isInSphere(protein->atom(i+1, CA).pos, focus_pos)) && (i+1<=protein->nRes()))
	     	  protein->draw_bond(i, C, i+1, N);
      }

    for (i=1; i <= protein->nRes(); i++)
      if (isInSphere(protein->atom(i, CA).pos, focus_pos))
     		protein->draw_residue(i);
  }
  else
  {
    int show_left_res  = 1;
    int show_right_res = protein->nRes();

    int i;
    if (showState == SHOW_SEGMENT)
    {
      show_left_res  = view_left_res;
      show_right_res = view_right_res;
    }

    if (isPeptide)
    	for (i=show_left_res; i<=show_right_res+1; i+=1)
	    	if ((i>1) && (i<=protein->nRes())) protein->draw_peptide_plane(i);

  	for (i=show_left_res-1; i<=show_right_res; i+=1)
	  	if ( (i >= 1) && (i < protein->nRes()) )
			  protein->draw_bond(i, C, i+1, N);

    for (i=show_left_res; i <= show_right_res; i++)
      protein->draw_residue(i);
  }


  // draw highlights for active residues
  for (i = active_left_res; i <= active_right_res; i+=1)
	  protein->draw_highlight_atom(i, CA, GREEN, 2);


  // draw highlights for rotation center
  protein->draw_highlight_atom(focus_res, focus_atom, PURPLE, 3);


  // draw highlights for line tool or the cursor highlighted atom
  highlight_res  = protein->highlight_res;
  highlight_atom = protein->highlight_atom;
  highlight_z    = protein->highlight_z;
  if (highlight_res == 0)
    setMouseText("");
  else
    setMouseText(protein->fullAtomName(highlight_res, highlight_atom));
  if (pickState == PICK_DISTANCE)
	{
		if ((line_res!=0) && (line_res2!=0) )
      protein->draw_distance_measure(line_res, line_atom, line_res2, line_atom2);

		else if ((line_res!=0) && (line_res2==0)&& (highlight_res!=0) )
      protein->draw_distance_measure(line_res, line_atom, highlight_res, highlight_atom);
		else if ((line_res!=0) && (line_res2==0))
			protein->draw_highlight_atom(line_res, line_atom, BLUE, 4);
		if (highlight_res!=0)
			if (!((highlight_res == line_res) && (highlight_atom == line_atom)) )
				if (!((highlight_res == line_res2) && (highlight_atom == line_atom2)) )
			 		protein->draw_highlight_atom(highlight_res, highlight_atom, RED, 4);
	}
	else
		protein->draw_highlight_atom(highlight_res, highlight_atom, PURPLE, 4);


  graphicsView.sort();
  graphicsView.draw(bmp);

  text_mode(-1);
  textout(bmp, font,
    "The Ramachandran Plot Explorer V0.5(c) 2003 Bosco K Ho : DSSP H-bonds : Allegro game engine", 15, SCREEN_H - 12, WHITE);

  textprintf(bmp, font, 15, 45,WHITE, "%d", protein->res(view_left_res).resNo);
 
  int view_size;
  if (VIEW_SIZE > protein->nRes()) 
    view_size = protein->nRes();
  else 
    view_size = VIEW_SIZE;
  textprintf_right(bmp, font, 15 + 14*view_size+1, 45,WHITE, "%d",
      protein->res(view_right_res).resNo);
  
  textprintf(bmp, font, 15, 57, PURPLE, "orientated on");
  textout(bmp, font, protein->fullAtomName(focus_res, focus_atom).c_str(), 125, 57, PURPLE);
}




