/* Author: Tobi Vollebregt */

/*  TankZone: My second Allegro game.
 *  Copyright (C) 2003  Tobi Vollebregt
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  See `License.txt', which contains a verbatim copy of the
 *  GNU General Public License, for details.
 *
 *  Please send your reaction to: tobivollebregt@hotmail.com
 */

/* 27/05/03: Added partial support for masked sprites. */

#include <math.h>       /* for sqrt() function in get_sprite4() */
#include "sprite4.h"

int _multiply_sprite_gamma=255;          // Gamma value for lighting

/*
    TODO: Fix the quirk that colors near the mask color get corrupted,
    because no palette entry is generated for those.
    (this is caused by the behaviour of generate_optimized_palette(),
    there is no way to mark a color as "exact match only", which would be
    sufficient.)

    Hmm, I might change generate_optimized_palette().

    TODO: Add masked sprite support to draw_sprite4().
*/

SPRITE4 *get_sprite4(BITMAP *bmp)
{
    SPRITE4 *sprite;
    PALETTE pal;
    signed char rsvd[256];
    int dist,bestdist,bestmatch=0;
    int c,r,g,b,dr,dg,db,pixel,i,x,y;
    unsigned char *ptr;
    int masked = 0;
    int maskcolor = bitmap_mask_color (bmp);

    /* width must be even */
    ASSERT(bmp && (bmp->w&1)==0);

    /* autodetect if bitmap has masked pixels */
    for (y = 0; y < bmp->h; ++y)
        for (x = 0; x < bmp->w; ++x)
            if (getpixel (bmp, x, y) == maskcolor) { masked = 1; goto break2; }
    break2:

    /* allocate memory for sprite structure */
    if(!(sprite=(SPRITE4 *)malloc(sizeof(SPRITE4)))) return NULL;

    /* set width and height */
    sprite->w=bmp->w;
    sprite->h=bmp->h;
    sprite->masked = masked;

    /* fill rsvd table */
    for(i=0; i<16; i++) rsvd[i]=0;
    for(i=16; i<256; i++) rsvd[i]=-1;
    if (masked)
    {
        pal[0].r = 63;
        pal[0].g = 0;
        pal[0].b = 63;
        rsvd[0] = 1;
    }

    /* generate optimized 16 color palette */
    if(generate_optimized_palette(bmp,pal,rsvd)<=0)
    {   free(sprite);
        return NULL;
    }

    /* copy first 16 colors of allegro palette to sprite palette */
    for(i=0; i<16; i++)
    {   sprite->palette_color_16[i]=
            makecol(pal[i].r<<2,pal[i].g<<2,pal[i].b<<2);
    }

    /* allocate memory for bitmap data */
    if(!(sprite->data=(unsigned char *)malloc(bmp->w/2*bmp->h)))
    {   free(sprite);
        return NULL;
    }

    /* some helper defines */
    #define _getpixel8  _getpixel
    #define GET(depth) \
        c=_getpixel##depth(bmp,x,y);    \
        r=getr##depth(c);   \
        g=getg##depth(c);   \
        b=getb##depth(c);

    /* start loop */
    for(y=0; y<bmp->h; y++)
    {
        ptr=&sprite->data[bmp->w/2*y];

        for(x=0; x<bmp->w; x++,ptr++)
        {
            #define READPIXEL   \
                switch(bitmap_color_depth(bmp)) \
                {   case 8: GET(8); break;  \
                    case 15: GET(15); break;    \
                    case 16: GET(16); break;    \
                    case 24: GET(24); break;    \
                    case 32: GET(32); break;    \
                    default:    \
                        free(sprite->data); \
                        free(sprite);   \
                        return NULL;    \
                }

            #define BESTMATCH   \
                if (masked && c == maskcolor) bestmatch = 0;  \
                else    \
                {   bestdist=65536; \
                    for(i=masked; i<16; i++) \
                    {   dr=(pal[i].r<<2)-r; \
                        dg=(pal[i].g<<2)-g; \
                        db=(pal[i].b<<2)-b; \
                        dist=(int)sqrt((double)(dr*dr+dg*dg+db*db));    \
                        if(dist<bestdist)   \
                        {   bestdist=dist;  \
                            bestmatch=i;    \
                        }   \
                    }   \
                }

            READPIXEL
            BESTMATCH

            pixel=bestmatch;
            x++;

            READPIXEL
            BESTMATCH

            pixel|=bestmatch<<4;

            *ptr=pixel;

            #undef READPIXEL
            #undef BESTMATCH
        }
    }

    /* undef defines */
    #undef GET
    #undef _getpixel8

    /* return pointer */
    return sprite;
}

void destroy_sprite4(SPRITE4 *sprite)
{
    /* free up memory */
    if(sprite)
    {   free(sprite->data);
        free(sprite);
    }
}

void draw_sprite4(BITMAP *dst, SPRITE4 *sprite, int dx, int dy)
{
    unsigned long xpal[256];
    unsigned short *pal;
    int i, y, w, h, mult;
    int dxbeg, dybeg;
    int sxbeg, sybeg;
    unsigned char *s,*sxend;
    unsigned long *d;

    ASSERT(dst && sprite && (dx&1)==0);
    ASSERT(bitmap_color_depth(dst)==15 || bitmap_color_depth(dst)==16);
    ASSERT(!dst->clip || ((dst->cl&1)|(dst->cr&1))==0);

    /* got clipping code from allegro */
    if(dst->clip)
    {
        int tmp;

        tmp = dst->cl - dx;
        sxbeg = ((tmp < 0) ? 0 : tmp);
        dxbeg = sxbeg + dx;

        tmp = dst->cr - dx;
        w = ((tmp > sprite->w) ? sprite->w : tmp) - sxbeg;
        if(w <= 0) return;

        tmp = dst->ct - dy;
        sybeg = ((tmp < 0) ? 0 : tmp);
        dybeg = sybeg + dy;

        tmp = dst->cb - dy;
        h = ((tmp > sprite->h) ? sprite->h : tmp) - sybeg;
        if(h <= 0) return;
    }
    else
    {   w = sprite->w;
        h = sprite->h;
        sxbeg = 0;
        sybeg = 0;
        dxbeg = dx;
        dybeg = dy;
    }

    pal=sprite->palette_color_16;

    /* expand to 256 colors */
    for(i=0; i<256; i++) xpal[i]=(pal[i>>4]<<16)|pal[i&15];

    mult=sprite->w/2;
    sxbeg/=2;
    w/=2;

    /* start loop through scanlines */
    for(y=0; y<h; y++)
    {
        /* initialize pointers */
        s=&sprite->data[mult*(y+sybeg)+sxbeg];
        d=(unsigned long *)(((unsigned short *)dst->line[y+dybeg])+dxbeg);
        sxend=s+w;

        /* start per two pixels loop */
        for(; s<sxend; s++,d++) *d=xpal[*s];
    }
}

#ifndef USE_ASM

void draw_multiply_sprite4(BITMAP *dst, SPRITE4 *sprite, int dx, int dy, int r, int g, int b)
{
    unsigned long pal[16],xpal[256];
    int i,y,w,h,mult,c,cr,cg,cb;
    int dxbeg, dybeg;
    int sxbeg, sybeg;
    unsigned char *s,*sxend;

    ASSERT(dst && sprite && (dx&1)==0);
    ASSERT(bitmap_color_depth(dst)==15 || bitmap_color_depth(dst)==16);
    ASSERT(!dst->clip || ((dst->cl&1)|(dst->cr&1))==0);

    /* got clipping code from allegro */
    if(dst->clip)
    {
        int tmp;

        tmp = dst->cl - dx;
        sxbeg = ((tmp < 0) ? 0 : tmp);
        dxbeg = sxbeg + dx;

        tmp = dst->cr - dx;
        w = ((tmp > sprite->w) ? sprite->w : tmp) - sxbeg;
        if(w <= 0) return;

        tmp = dst->ct - dy;
        sybeg = ((tmp < 0) ? 0 : tmp);
        dybeg = sybeg + dy;

        tmp = dst->cb - dy;
        h = ((tmp > sprite->h) ? sprite->h : tmp) - sybeg;
        if(h <= 0) return;
    }
    else
    {   w = sprite->w;
        h = sprite->h;
        sxbeg = 0;
        sybeg = 0;
        dxbeg = dx;
        dybeg = dy;
    }

    /* make 16 colors palette with multiply blender */
    if(bitmap_color_depth(dst)==16)
    {   for(i=sprite->masked; i<16; i++)
        {   c=sprite->palette_color_16[i];
            cr=getr16(c)*r/_multiply_sprite_gamma;
            cg=getg16(c)*g/_multiply_sprite_gamma;
            cb=getb16(c)*b/_multiply_sprite_gamma;
            pal[i]=makecol16(MIN(cr,255),MIN(cg,255),MIN(cb,255));
        }
    }else
    {   for(i=sprite->masked; i<16; i++)
        {   c=sprite->palette_color_16[i];
            cr=getr15(c)*r/_multiply_sprite_gamma;
            cg=getg15(c)*g/_multiply_sprite_gamma;
            cb=getb15(c)*b/_multiply_sprite_gamma;
            pal[i]=makecol15(MIN(cr,255),MIN(cg,255),MIN(cb,255));
        }
    }

    mult=sprite->w/2;
    sxbeg/=2;
    w/=2;

    /* we need different code for masked sprites */
    if (sprite->masked)
    {
        unsigned short *d;

        /* start loop through scanlines */
        for(y=0; y<h; y++)
        {
            /* initialize pointers */
            s=&sprite->data[mult*(y+sybeg)+sxbeg];
            d=((unsigned short *)dst->line[y+dybeg])+dxbeg;
            sxend=s+w;

            /* start per two pixels loop */
            for(; s<sxend; s++,d++)
            {
                if (*s & 15) *d = pal[*s & 15];
                d++;
                if (*s >> 4) *d = pal[*s >> 4];
            }
        }
    }
    else
    {
        unsigned long *d;

        /* expand to 256 colors */
        for(i=0; i<256; i++) xpal[i]=(pal[i>>4]<<16)|pal[i&15];

        /* start loop through scanlines */
        for(y=0; y<h; y++)
        {
            /* initialize pointers */
            s=&sprite->data[mult*(y+sybeg)+sxbeg];
            d=(unsigned long *)(((unsigned short *)dst->line[y+dybeg])+dxbeg);
            sxend=s+w;

            /* start per two pixels loop */
            for(; s<sxend; s++,d++) *d=xpal[*s];
        }
    }
}

#endif
