/* Author: Tobi Vollebregt */

/*  TankZone: My second Allegro game.
 *  Copyright (C) 2003  Tobi Vollebregt
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  See `License.txt', which contains a verbatim copy of the
 *  GNU General Public License, for details.
 *
 *  Please send your reaction to: tobivollebregt@hotmail.com
 */

#include <allegro.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "fade16.h"

typedef signed char bool;

#define EPSILON         .001    // How accurate should plane generation be?
#define MAX_LIGHTS      5       // The maximum number of lights we will use
#define LENSFLARE_R     16      // Radius of the lensflares

typedef struct VERTEX
{   struct VERTEX *link;        // Pointer to next vertex in linked list
    float x,y,z;                // Object space coordinates
    float tx,ty,tz;             // Camera space coordinates
    float px,py;                // Projected coordinates
    unsigned char r,g,b;        // Vertex color
    unsigned char lr,lg,lb;     // Lighted vertex color
    float light_d[MAX_LIGHTS];  // To prevent innessesary calculations
    float light_vx[MAX_LIGHTS]; // d: 1-(distance_to_light/intensity)
    float light_vy[MAX_LIGHTS]; // vx,vy,vz: Normal from vertex to light
    float light_vz[MAX_LIGHTS]; // used for normal-lighting later on
    int processed1;             // Has it been transformed?
    int processed2;             // Has it been lighted and projected?
} VERTEX;

typedef struct PLANE
{   struct PLANE *link;         // Pointer to next plane in linked list
    float nx,ny,nz;             // Plane normal in object space
    float dist;                 // Distance to origin in object space
    VERTEX *v1,*v2,*v3;         // Three vertices on this infinite plane
    float tnx,tny,tnz;          // Plane normal in camera space
    float tdist;                // Distance to origin in camera space
    bool visible;               // Is this plane visible this frame?
    int processed;              // Has it been transformed and bf-culled?
} PLANE;

typedef struct POLYGON
{   struct POLYGON *link;       // Pointer to next polygon in linked list
    PLANE *plane;               // It is on this infinite plane
    int vc;                     // Vertex count
    VERTEX **vtx;               // Pointers to vertices
    V3D_f *v3d,**pv3d;          // Helper arrays for Allegro's rasterization
} POLYGON;

typedef struct OBJECT
{   struct OBJECT *link;        // Pointer to next object in linked list
    VERTEX *vertices;           // Pointer to head of vertex list
    PLANE *planes;              // Pointer to head of plane list
    POLYGON *polygons;          // Pointer to head of polygon list
    int vertexcount;            // Number of vertices
    int planecount;             // Number of planes
    int polygoncount;           // Number of polygons
    MATRIX_f matrix;            // Object matrix
} OBJECT;

typedef struct LIGHT
{   float x,y,z;                // World/object space light coordinates
    float tx,ty,tz;             // Camera space light coordinates
    float px,py;                // Projected coordinates
    unsigned char r,g,b;        // Lighting color
    float intensity;            // Intensity (maximum distance to light)
    float oo_intensity;         // 1.0/(intensity^2)
} LIGHT;

typedef struct SCENE
{   MATRIX_f matrix;            // Camera matrix
    OBJECT *objects;            // Pointer to head of object list
    int numlights;              // Number of lights in this scene
    LIGHT light[MAX_LIGHTS];    // The real lights
    int maxpolygons;            // Maximum number of polygons we will render
} SCENE;

#define assert(x) {if(!(x)) {set_gfx_mode(GFX_TEXT,0,0,0,0);printf("Assertion failed at line %d of file %s.\n",__LINE__,__FILE__);exit(1);}}
//#define assert(x)

bool solidmode=1;
bool bfculling=1;
bool lighting=1;

static int frameid=0,timer=0;

static volatile int speed_counter,fps,frame;

static void increment_speed_counter(void)
{   speed_counter++;
}END_OF_STATIC_FUNCTION(increment_speed_counter);

static void fps_proc(void)
{   fps=frame; frame=0;
}END_OF_STATIC_FUNCTION(fps_proc);



////////////////////////////// VERTEX code

VERTEX *vertex_create(void)
{
    VERTEX *vertex;

    if(!(vertex=(VERTEX *)malloc(sizeof(VERTEX)))) return NULL;
    memset(vertex,0,sizeof(VERTEX));

    return vertex;
}

void vertex_destroy(VERTEX *vertex)
{
    if(vertex) free(vertex);
}

void vertex_setposition(VERTEX *vertex, float x, float y, float z)
{
    assert(vertex);

    vertex->x=x;
    vertex->y=y;
    vertex->z=z;
}

void vertex_setcolor(VERTEX *vertex, int r, int g, int b)
{
    assert(vertex);
    assert(r>=0 && r<256 && g>=0 && g<256 && b>=0 && b<256);

    vertex->r=r;
    vertex->g=g;
    vertex->b=b;
}



////////////////////////////// POLYGON code

POLYGON *polygon_create(void)
{
    POLYGON *polygon;

    if(!(polygon=(POLYGON *)malloc(sizeof(POLYGON)))) return NULL;
    memset(polygon,0,sizeof(POLYGON));

    return polygon;
}

void polygon_destroy(POLYGON *polygon)
{
    if(polygon)
    {   if(polygon->vtx) free(polygon->vtx);
        if(polygon->v3d) free(polygon->v3d);
        if(polygon->pv3d) free(polygon->pv3d);
    }
}

int polygon_setvertices(POLYGON *polygon, int vc, ...)
{
    va_list va;
    int i;

    if(vc!=polygon->vc)
    {   if(polygon->vtx) {free(polygon->vtx); polygon->vtx=NULL;}
        if(polygon->v3d) {free(polygon->v3d); polygon->v3d=NULL;}
        if(polygon->pv3d) {free(polygon->pv3d); polygon->pv3d=NULL;}
        polygon->vc=vc;
        if(!(polygon->vtx=(VERTEX **)malloc(vc*sizeof(VERTEX *)))) return 1;
        if(!(polygon->v3d=(V3D_f *)malloc(vc*sizeof(V3D_f)))) return 1;
        if(!(polygon->pv3d=(V3D_f **)malloc(vc*sizeof(V3D_f *)))) return 1;
        for(i=0; i<vc; i++) polygon->pv3d[i]=polygon->v3d+i;
    }

    va_start(va,vc);

    for(i=0; i<vc; i++)
    {   polygon->vtx[i]=va_arg(va,VERTEX *);
        assert(polygon->vtx[i]);
    }

    va_end(va);

    return 0;
}



////////////////////////////// PLANE code

PLANE *plane_create(void)
{
    PLANE *plane;

    if(!(plane=(PLANE *)malloc(sizeof(PLANE)))) return NULL;
    memset(plane,0,sizeof(PLANE));

    return plane;
}

void plane_destroy(PLANE *plane)
{
    if(plane) free(plane);
}

void plane_setabcd(PLANE *plane, float a, float b, float c, float d)
{
    assert(plane);

    plane->nx=a;
    plane->ny=b;
    plane->nz=c;
    plane->dist=d;
}

void plane_setvertices(PLANE *plane, VERTEX *v1, VERTEX *v2, VERTEX *v3)
{
    assert(plane);
    assert(v1 && v2 && v3);

    plane->v1=v1;
    plane->v2=v2;
    plane->v3=v3;
}



////////////////////////////// OBJECT code

OBJECT *object_create(void)
{
    OBJECT *object;

    if(!(object=(OBJECT *)malloc(sizeof(OBJECT)))) return NULL;
    memset(object,0,sizeof(OBJECT));

    object->matrix=identity_matrix_f;

    return object;
}

void object_destroy(OBJECT *object)
{
    VERTEX *v;
    PLANE *plane;
    POLYGON *poly;

    if(object)
    {   while(object->polygons)
        {   poly=object->polygons;
            object->polygons=object->polygons->link;
            polygon_destroy(poly);
        }
        while(object->planes)
        {   plane=object->planes;
            object->planes=object->planes->link;
            plane_destroy(plane);
        }
        while(object->vertices)
        {   v=object->vertices;
            object->vertices=object->vertices->link;
            vertex_destroy(v);
        }
        free(object);
    }
}

void object_addvertex(OBJECT *object, VERTEX *vertex)
{
    assert(object);
    assert(vertex);

    vertex->link=object->vertices;
    object->vertices=vertex;
    object->vertexcount++;
}

int object_removevertex(OBJECT *object, VERTEX *vertex)
{
    VERTEX *loop,*prev=NULL;

    assert(object);

    loop=object->vertices;
    while(loop)
    {   if(loop==vertex)
        {   if(prev) prev->link=loop->link;
            else object->vertices=loop->link;
            vertex->link=NULL;
            object->vertexcount--;
            return 0;
        }
        prev=loop;
        loop=loop->link;
    }

    return 1;
}

void object_addpolygon(OBJECT *object, POLYGON *polygon)
{
    assert(object);
    assert(polygon);

    polygon->link=object->polygons;
    object->polygons=polygon;
    object->polygoncount++;
}

int object_removepolygon(OBJECT *object, POLYGON *polygon)
{
    POLYGON *loop,*prev=NULL;

    assert(object);

    loop=object->polygons;
    while(loop)
    {   if(loop==polygon)
        {   if(prev) prev->link=loop->link;
            else object->polygons=loop->link;
            polygon->link=NULL;
            object->polygoncount--;
            return 0;
        }
        prev=loop;
        loop=loop->link;
    }

    return 1;
}

void object_addplane(OBJECT *object, PLANE *plane)
{
    assert(object);
    assert(plane);

    plane->link=object->planes;
    object->planes=plane;
    object->planecount++;
}

int object_removeplane(OBJECT *object, PLANE *plane)
{
    PLANE *loop,*prev=NULL;

    assert(object);

    loop=object->planes;
    while(loop)
    {   if(loop==plane)
        {   if(prev) prev->link=loop->link;
            else object->planes=loop->link;
            plane->link=NULL;
            object->planecount--;
            return 0;
        }
        prev=loop;
        loop=loop->link;
    }

    return 1;
}

OBJECT *bitmap2object(BITMAP *bmp, MATRIX_f *matrix)
{
    OBJECT *obj=NULL;
    VERTEX *v,**vtx=NULL;
    POLYGON *poly;
    PLANE *plane;
    int (*get_r)(int),(*get_g)(int),(*get_b)(int);
    int x,y,z,w,h,i,j,r,g,b,color[4],l,t;
    float cx,cy,cz,nx,ny,nz,dist;

    assert(bmp);
    assert(matrix);



    /* create object */

    if(!(obj=object_create())) goto memerr;



    /* create object vertices (only the ones that are actually needed) */

    w=bmp->w+1; h=bmp->h+1;
    if(!(vtx=(VERTEX **)malloc(2*w*h*sizeof(VERTEX *)))) goto memerr;
    memset(vtx,0,2*w*h*sizeof(VERTEX *));

    switch(bitmap_color_depth(bmp))
    {   case 8: get_r=getr8; get_g=getg8; get_b=getb8; break;
        case 15: get_r=getr15; get_g=getg15; get_b=getb15; break;
        case 16: get_r=getr16; get_g=getg16; get_b=getb16; break;
        case 24: get_r=getr24; get_g=getg24; get_b=getb24; break;
        case 32: get_r=getr32; get_g=getg32; get_b=getb32; break;
        default: object_destroy(obj); free(vtx); return NULL;
    }

    for(y=0; y<h; y++)
    {   for(x=0; x<w; x++)
        {   for(z=0; z<=1; z++)
            {   color[0]=getpixel(bmp,x-1,y);
                color[1]=getpixel(bmp,x,y);
                color[2]=getpixel(bmp,x,y-1);
                color[3]=getpixel(bmp,x-1,y-1);
                if(color[0]>0 || color[1]>0 || color[2]>0 || color[3]>0)
                {   apply_matrix_f(matrix,(float)x,(float)y,z?(-1.):(1.),&cx,&cy,&cz);
                    for(i=j=r=g=b=0; i<4; i++)
                    {   if(color[i]>0)
                        {   r+=get_r(color[i]);
                            g+=get_g(color[i]);
                            b+=get_b(color[i]);
                            j++;
                        }
                    }
                    if(!(v=vertex_create())) goto memerr;
                    vertex_setposition(v,cx,cy,cz);
                    vertex_setcolor(v,r/j,g/j,b/j);
                    vtx[(y*w+x)*2+z]=v;
                    object_addvertex(obj,v);
                }
            }
        }
    }



    /* create object polygons */

    #define poly3(v1,v2,v3) \
    {   if(!(poly=polygon_create())) goto memerr;   \
        if(polygon_setvertices(poly,3,v3,v2,v1)) {polygon_destroy(poly); goto memerr;}  \
        object_addpolygon(obj,poly);    \
    }

    #define poly4(v1,v2,v3,v4)  \
    {   if(!(poly=polygon_create())) goto memerr;   \
        if(polygon_setvertices(poly,4,v4,v3,v2,v1)) {polygon_destroy(poly); goto memerr;}   \
        object_addpolygon(obj,poly);    \
    }

    #define P0  vtx[(y*w+x)*2]
    #define P1  vtx[((y+1)*w+x)*2]
    #define P2  vtx[((y+1)*w+x+1)*2]
    #define P3  vtx[(y*w+x+1)*2]
    #define P0a vtx[(y*w+x)*2+1]
    #define P1a vtx[((y+1)*w+x)*2+1]
    #define P2a vtx[((y+1)*w+x+1)*2+1]
    #define P3a vtx[(y*w+x+1)*2+1]

    for(y=-1; y<=bmp->h; y++)
    {   for(x=-1; x<=bmp->w; x++)
        {   if(getpixel(bmp,x,y)>0)
            {   poly4(P0,P1,P2,P3);             // front face
                poly4(P3a,P2a,P1a,P0a);         // back face
            }else
            {   l=(getpixel(bmp,x-1,y)<=0)?0:1;
                r=(getpixel(bmp,x+1,y)<=0)?0:1;
                t=(getpixel(bmp,x,y-1)<=0)?0:1;
                b=(getpixel(bmp,x,y+1)<=0)?0:1;

                if(!l && r && !t && b)
                {   poly3(P3,P1,P2);            // front face
                    poly3(P3a,P2a,P1a);         // back face
                    poly4(P1,P3,P3a,P1a);       // side
                }else if(!l && r && t && !b)
                {   poly3(P0,P2,P3);            // front face
                    poly3(P0a,P3a,P2a);         // back face
                    poly4(P0,P0a,P2a,P2);       // side
                }else if(l && !r && !t && b)
                {   poly3(P0,P1,P2);            // front face
                    poly3(P0a,P2a,P1a);         // back face
                    poly4(P0,P2,P2a,P0a);       // side
                }else if(l && !r && t && !b)
                {   poly3(P0,P1,P3);            // front face
                    poly3(P0a,P3a,P1a);         // back face
                    poly4(P1,P1a,P3a,P3);       // side
                }else
                {   if(l) poly4(P0,P1,P1a,P0a); // left
                    if(r) poly4(P2,P3,P3a,P2a); // right
                    if(t) poly4(P0,P0a,P3a,P3); // top
                    if(b) poly4(P1,P2,P2a,P1a); // bottom
                }
            }
        }
    }

    #undef P3a
    #undef P2a
    #undef P1a
    #undef P0a
    #undef P3
    #undef P2
    #undef P1
    #undef P0
    #undef poly4
    #undef poly3

    if(vtx)
    {   free(vtx);
        vtx=NULL;
    }



    /* create planes (not more than the absolute minimum) */

    poly=obj->polygons;
    while(poly)
    {
        cross_product_f(poly->vtx[1]->x-poly->vtx[0]->x,poly->vtx[1]->y-poly->vtx[0]->y,poly->vtx[1]->z-poly->vtx[0]->z,
            poly->vtx[2]->x-poly->vtx[0]->x,poly->vtx[2]->y-poly->vtx[0]->y,poly->vtx[2]->z-poly->vtx[0]->z,&nx,&ny,&nz);
        normalize_vector_f(&nx,&ny,&nz);
        dist=poly->vtx[0]->x*nx+poly->vtx[0]->y*ny+poly->vtx[0]->z*nz;

        plane=obj->planes;
        while(plane)
        {   if(fabs(nx-plane->nx)<EPSILON && fabs(ny-plane->ny)<EPSILON && fabs(nz-plane->nz)<EPSILON && fabs(dist-plane->dist)<EPSILON)
            {   poly->plane=plane;
                break;
            }
            plane=plane->link;
        }
        if(!plane)
        {   if(!(plane=plane_create())) goto memerr;
            plane_setabcd(plane,nx,ny,nz,dist);
            plane_setvertices(plane,poly->vtx[0],poly->vtx[1],poly->vtx[2]);
            object_addplane(obj,plane);
            poly->plane=plane;
        }

        poly=poly->link;
    }

    return obj;

    memerr:

    if(vtx) free(vtx);
    if(obj) object_destroy(obj);

    return NULL;
}

OBJECT *text2object(char *text, MATRIX_f *matrix, int col)
{
    OBJECT *obj;
    BITMAP *bmp;

    assert(text);
    assert(matrix);

    if(!(bmp=create_bitmap(text_length(font,text),text_height(font)))) return NULL;

    text_mode(0);
    textout(bmp,font,text,0,0,col);

    obj=bitmap2object(bmp,matrix);

    destroy_bitmap(bmp);

    return obj;
}



////////////////////////////// LensFlare initialization code
/*
BITMAP *create_lensflare(int r, int g, int b)
{
    BITMAP *bmp=create_bitmap_ex(32,2*LENSFLARE_R,2*LENSFLARE_R);
    int x,y,dx,dy;
    float dist;

    if(!bmp) return NULL;

    for(y=0; y<bmp->h; y++)
    {   for(x=0; x<bmp->w; x++)
        {   dx=x-LENSFLARE_R;
            dy=y-LENSFLARE_R;
            dist=1.0-((float)(dx*dx+dy*dy)/(LENSFLARE_R*LENSFLARE_R));
            _putpixel32(bmp,x,y,makeacol32((int)(r*dist),
                (int)(g*dist),(int)(b*dist),(int)(255*dist)));
        }
    }

    return bmp;
}*/



////////////////////////////// SCENE code

SCENE *scene_create(void)
{
    SCENE *scene;

    if(!(scene=(SCENE *)malloc(sizeof(SCENE)))) return NULL;
    memset(scene,0,sizeof(SCENE));

    scene->matrix=identity_matrix_f;

    return scene;
}

void scene_destroy(SCENE *scene)
{
    OBJECT *object;

    if(scene)
    {   while(scene->objects)
        {   object=scene->objects;
            scene->objects=scene->objects->link;
            object_destroy(object);
        }
        free(scene);
    }
}

void scene_addobject(SCENE *scene, OBJECT *object)
{
    assert(scene);
    assert(object);

    object->link=scene->objects;
    scene->objects=object;
    scene->maxpolygons+=object->polygoncount;
}

int scene_removeobject(SCENE *scene, OBJECT *object)
{
    OBJECT *loop,*prev=NULL;

    assert(scene);

    loop=scene->objects;
    while(loop)
    {   if(loop==object)
        {   if(prev) prev->link=loop->link;
            else scene->objects=loop->link;
            object->link=NULL;
            scene->maxpolygons-=object->polygoncount;
            if(scene->maxpolygons<0) scene->maxpolygons=0;
            return 0;
        }
        prev=loop;
        loop=loop->link;
    }

    return 1;
}

int scene_setlight(SCENE *scene, int index, float x, float y, float z, int r, int g, int b, float intensity)
{
    assert(scene);
    assert(r>=0 && r<256 && g>=0 && g<256 && b>=0 && b<256);
    assert(intensity>EPSILON);

    if(index<0 || index>=MAX_LIGHTS) return 1;
    if(index>=scene->numlights) scene->numlights=index+1;

    scene->light[index].x=x;
    scene->light[index].y=y;
    scene->light[index].z=z;
    scene->light[index].r=r;
    scene->light[index].g=g;
    scene->light[index].b=b;
    scene->light[index].intensity=intensity;
    scene->light[index].oo_intensity=1./(intensity*intensity);

    return 0;
}

#define SWAP(a,b)   {temp=a;a=b;b=temp;}

typedef struct SORTITEM
{   int z;                      // Average z-coordinate of this polygon
    POLYGON *poly;              // Pointer to the polygon
} SORTITEM;

static void quick_sort(SORTITEM *sortitem, int low, int high)
{
    int i,j;
    SORTITEM partition,temp;

    if(low<high)
    {   if((high-low)==1)
        {   if(sortitem[low].z>sortitem[high].z)
                SWAP(sortitem[low],sortitem[high]);
        }else
        {
            partition=sortitem[high];
            i=low;
            j=high;
            do
            {   while((i<j) && (sortitem[i].z<=partition.z)) i++;
                while((j>i) && (sortitem[j].z>=partition.z)) j--;
                if(i<j) SWAP(sortitem[i],sortitem[j]);
            } while(i<j);
            SWAP(sortitem[i],sortitem[high]);
            quick_sort(sortitem,low,(i-1));
            quick_sort(sortitem,(i+1),high);
        }
    }
}

int scene_render(BITMAP *scrn, SCENE *scene)
{
    static SORTITEM *sortitem=NULL;
    static int numsortitems=0;

    OBJECT *obj;
    POLYGON *poly;
    PLANE *plane;
    VERTEX *v;
    LIGHT *light;
    MATRIX_f matrix;
    int i,j,c,c2,r,g,b,vc=0,pc=0;
    float dx,dy,dz,d,oo_d,avz;

    if(!scrn || !scene)
    {   if(sortitem)
        {   free(sortitem);
            sortitem=NULL;
        }
        numsortitems=0;

        return 0;
    }

    if(numsortitems!=scene->maxpolygons)
    {   if(sortitem)
        {   free(sortitem);
            sortitem=NULL;
        }
        if(!(sortitem=(SORTITEM *)malloc(scene->maxpolygons*sizeof(SORTITEM))))
            return -1;
        numsortitems=scene->maxpolygons;
    }

    /* tranformate lights to camera space */
    if(lighting)
    {   for(i=0; i<scene->numlights; i++)
        {   light=&scene->light[i];
            apply_matrix_f(&scene->matrix,light->x,light->y,light->z,&light->tx,&light->ty,&light->tz);
            persp_project_f(light->tx,light->ty,light->tz,&light->px,&light->py);
        }
    }

    /* walk through all objects */
    for(obj=scene->objects; obj; obj=obj->link)
    {
        /* calculate object to camera space matrix */
        matrix_mul_f(&obj->matrix,&scene->matrix,&matrix);

        /* walk through all polygons */
        for(poly=obj->polygons; poly; poly=poly->link)
        {
            /* perform backface culling on plane if not done earlier */
            plane=poly->plane;
            if(plane->processed!=frameid)
            {   plane->processed=frameid;
                if(plane->v1->processed1!=frameid)
                {   v=plane->v1;
                    apply_matrix_f(&matrix,v->x,v->y,v->z,&v->tx,&v->ty,&v->tz);
                    v->processed1=frameid; vc++;
                }
                if(plane->v2->processed1!=frameid)
                {   v=plane->v2;
                    apply_matrix_f(&matrix,v->x,v->y,v->z,&v->tx,&v->ty,&v->tz);
                    v->processed1=frameid; vc++;
                }
                if(plane->v3->processed1!=frameid)
                {   v=plane->v3;
                    apply_matrix_f(&matrix,v->x,v->y,v->z,&v->tx,&v->ty,&v->tz);
                    v->processed1=frameid; vc++;
                }
                cross_product_f(plane->v2->tx-plane->v1->tx,plane->v2->ty-plane->v1->ty,plane->v2->tz-plane->v1->tz,
                    plane->v3->tx-plane->v1->tx,plane->v3->ty-plane->v1->ty,plane->v3->tz-plane->v1->tz,&plane->tnx,&plane->tny,&plane->tnz);
                normalize_vector_f(&plane->tnx,&plane->tny,&plane->tnz);
                plane->tdist=plane->v1->tx*plane->tnx+plane->v1->ty*plane->tny+plane->v1->tz*plane->tnz;
                if(bfculling)
                {   if(plane->tdist>0.)
                    {   plane->visible=FALSE;
                        goto cullpolygon;
                    }else plane->visible=TRUE;
                }else
                    plane->visible=TRUE;
            }
            else if(bfculling && !plane->visible) goto cullpolygon;

            /* walk through polygon vertices */
            avz=0.;
            for(i=0; i<poly->vc; i++)
            {
                v=poly->vtx[i];
                if(v->processed1!=frameid)
                {   /* transformate point to camera space and mark as processed */
                    apply_matrix_f(&matrix,v->x,v->y,v->z,&v->tx,&v->ty,&v->tz);

                    v->processed1=frameid; vc++;
                }
                /* cull when polygon is (partially) behind the camera */
                if(v->tz<=0.) goto cullpolygon;

                if(v->processed2!=frameid)
                {   if(lighting)
                    {   /* light vertex (diffuse lighting) */
                        r=g=b=0;
                        for(j=0; j<scene->numlights; j++)
                        {   light=&scene->light[j];
                            dx=light->tx-v->tx; dy=light->ty-v->ty; dz=light->tz-v->tz;
                            oo_d=1./(d=sqrt(dx*dx+dy*dy+dz*dz));
                            d=1.-(d*d*light->oo_intensity);
                            if(d>0.)
                            {   r+=(int)((float)light->r*d);
                                g+=(int)((float)light->g*d);
                                b+=(int)((float)light->b*d);
                            }
                            v->light_d[j]=d;
                            v->light_vx[j]=dx*oo_d;
                            v->light_vy[j]=dy*oo_d;
                            v->light_vz[j]=dz*oo_d;
                        }
                        if(r>255) r=255;
                        if(g>255) g=255;
                        if(b>255) b=255;
                        v->lr=v->r*r/255;
                        v->lg=v->g*g/255;
                        v->lb=v->b*b/255;
                    }

                    /* project vertex and mark as processed */
                    persp_project_f(v->tx,v->ty,v->tz,&v->px,&v->py);

                    v->processed2=frameid;
                }

                /* calculate average z */
                avz+=v->tz;

                /* copy coordinates to Allegro structure */
                poly->v3d[i].x=v->px;
                poly->v3d[i].y=v->py;

                /* light polygon (specular lighting) */
                if(lighting)
                {   r=v->lr; g=v->lg; b=v->lb;
                    for(j=0; j<scene->numlights; j++)
                    {   light=&scene->light[j];
                        d=plane->tnx*v->light_vx[j]+plane->tny*v->light_vy[j]+plane->tnz*v->light_vz[j];
                        if(d>0. && v->light_d[j]>0.)
                        {   d*=v->light_d[j];
                            r+=(int)((float)light->r*d);//*(float)v->r*.00390625);
                            g+=(int)((float)light->g*d);//*(float)v->g*.00390625);
                            b+=(int)((float)light->b*d);//*(float)v->b*.00390625);
                        }
                    }
                    if(r>255) r=255;
                    if(g>255) g=255;
                    if(b>255) b=255;

                    /* make color and copy to Allegro structure */
                    poly->v3d[i].c=(r<<16)|(g<<8)|b;
                }else
                    poly->v3d[i].c=(v->r<<16)|(v->g<<8)|v->b;
            }

            /* add to zsort list */
            sortitem[pc].poly=poly;
            sortitem[pc].z=(int)avz/poly->vc;
            if(++pc>=numsortitems) goto perfpaintersalgo;

            cullpolygon: ;
        }
    }

    perfpaintersalgo:

    /* perform painter's algorithm: z-sorting */
    quick_sort(sortitem,0,pc-1);

    /* draw all polygons */
    for(i=pc-1; i>=0; i--)
    {
        poly=sortitem[i].poly;
        if(solidmode)
            polygon3d_f(scrn,POLYTYPE_GRGB,NULL,poly->vc,poly->pv3d);
        else
        {   for(c=0,c2=poly->vc-1; c<poly->vc; c2=c,c++)
            {   r=((poly->v3d[c].c>>16)+(poly->v3d[c2].c>>16))/2;
                g=(((poly->v3d[c].c>>8)&255)+((poly->v3d[c2].c>>8)&255))/2;
                b=((poly->v3d[c].c&255)+(poly->v3d[c2].c&255))/2;
                line(scrn,poly->v3d[c].x,poly->v3d[c].y,poly->v3d[c2].x,poly->v3d[c2].y,makecol(r,g,b));
            }
        }
    }

    /* draw lights (maybe to implement lensflares later...) */
    /*if(lighting)
    {   drawing_mode(DRAW_MODE_TRANS,NULL,0,0);
        set_add_blender(0,0,0,255);
        for(i=0; i<scene->numlights; i++)
        {   light=&scene->light[i];
            if(light->tz>=1. && getpixel(scrn,(int)light->px,(int)light->py)==0)
            {   dx=(255/LENSFLARE_R)/sqrt((float)light->r*light->r+(float)light->g*light->g+(float)light->b*light->b);
                c=makecol((int)(light->r*dx),(int)(light->g*dx),(int)(light->b*dx));
                for(j=LENSFLARE_R; j; j--)
                    circlefill(scrn,(int)light->px,(int)light->py,j,c);
            }
        }
        solid_mode();
    }*/

    return (pc<<16)|vc;
}



////////////////////////////// MAIN part of intro

#define STATE_STANDBY           0
#define STATE_IDLE              1
#define STATE_FINISHED          2
#define STATE_DISPLAYING        3
#define STATE_HIDING            4

typedef struct EFFECT
{   int begin,end;
    float xfrom,yfrom,zfrom;
    float xto,yto,zto;
} EFFECT;

typedef struct TEXT3D
{   char *text;
    unsigned char r,g,b;
    float xscale,yscale,zscale;
    int rotatespeed;
    EFFECT in,out;
    int state,angle;
    OBJECT *object;
} TEXT3D;

int animate(BITMAP *dbuffer, BITMAP *dbuffer2, TEXT3D *data)
{
    static bool virgin=1;

    SCENE *scene=NULL;

    bool f2pressed=0;   // wireframe mode
    bool f3pressed=0;   // backface culling
    bool f4pressed=0;   // lighting

    int i,c,numf,finishing=0;
    float t,x,y,z;

    int lxangle=0,lyangle=0,lzangle=0;

    bool done=0;

    if(virgin)
    {   LOCK_VARIABLE(fps);
        LOCK_VARIABLE(frame);
        LOCK_VARIABLE(speed_counter);
        LOCK_FUNCTION(fps_proc);
        LOCK_FUNCTION(increment_speed_counter);
        virgin=0;
    }

    fps=frame=0;
    speed_counter=1;

    /*allegro_init();
    install_timer();
    install_keyboard();

    set_color_depth(16);
    set_gfx_mode(GFX_AUTODETECT,640,480,0,0);*/

    // here starts the real intro

    if(bitmap_color_depth(screen)==15 || bitmap_color_depth(screen)==16)
    {
        blit(screen,dbuffer,0,0,0,0,SCREEN_W,SCREEN_H);

        for(i=31; i>=0; i--)
        {   fade(dbuffer,dbuffer2,i);
            blit(dbuffer2,screen,0,0,0,0,SCREEN_W,SCREEN_H);
        }
    }

    //rest(1000);
    //textout_centre(screen,font,"PRESS A KEY TO START...",SCREEN_W/2,SCREEN_H/2,makecol(255,255,255));
    //readkey();

    set_projection_viewport(0,0,SCREEN_W,SCREEN_H);

    /* create scene and set lights */

    if(!(scene=scene_create()))
    {   destroy_bitmap(dbuffer);
        return 1;
    }

    get_camera_matrix_f(&scene->matrix,0.,0.,-256.,0.,0.,1.,0.,-1.,0.,64.,.75);

    /*scene_setlight(scene,0,-256.,-256.,-256., 128,0,0, 512.);
    scene_setlight(scene,1,256.,-256.,-256., 0,128,0, 512.);
    scene_setlight(scene,2,256.,256.,-256., 0,0,128, 512.);
    scene_setlight(scene,3,-256.,256.,-256., 128,128,0, 512.);
    scene_setlight(scene,4,0.,0.,-256, 128,128,128, 512.);*/
    scene_setlight(scene,4,0.,0.,-256, 128,128,128, 512.);

    install_int(fps_proc,1000);
    install_int_ex(increment_speed_counter,BPS_TO_TIMER(60));

    while(!done)
    {
        frame++;
        frameid++;

        while(speed_counter>0 && !done)
        {
            speed_counter--;
            timer++;

            if(finishing)
            {
                if(++finishing>=63) done=1;

                //scene_setlight(scene,4,0.,0.,-256, 128,128,128, 512+16*finishing);
            }

            {   /* rotate coloured lights */
                MATRIX_f m1,m2,matrix;
                float ll;

                lxangle=(lxangle+1)&255;
                lyangle=(lyangle+2)&255;

                get_rotation_matrix_f(&m1,(float)lxangle,(float)lyangle,(float)lzangle);
                get_scaling_matrix_f(&m2,1.,1,.5);
                matrix_mul_f(&m1,&m2,&matrix);

                ll=(float)(513-(8*finishing));
                apply_matrix_f(&matrix,-256.,0.,-256.,&x,&y,&z);
                scene_setlight(scene,0,x,y,z, 128,0,0, ll);
                apply_matrix_f(&matrix,256.,0.,-256.,&x,&y,&z);
                scene_setlight(scene,1,x,y,z, 0,128,0, ll);
                apply_matrix_f(&matrix,256.,0.,256.,&x,&y,&z);
                scene_setlight(scene,2,x,y,z, 0,0,128, ll);
                apply_matrix_f(&matrix,-256.,0.,256.,&x,&y,&z);
                scene_setlight(scene,3,x,y,z, 128,128,0, ll);
            }

            for(i=numf=0; data[i].text; i++)
            {
                switch(data[i].state)
                {
                    case STATE_STANDBY:

                        if(timer>=data[i].in.begin)
                        {
                            MATRIX_f m1,m2,matrix;

                            get_translation_matrix_f(&m1,(float)(-text_length(font,data[i].text)/2),(float)(-text_height(font)/2),0.);
                            get_scaling_matrix_f(&m2,data[i].xscale,data[i].yscale,data[i].zscale);
                            matrix_mul_f(&m1,&m2,&matrix);

                            data[i].object=text2object(data[i].text,&matrix,makecol(data[i].r,data[i].g,data[i].b));
                            scene_addobject(scene,data[i].object);

                            get_translation_matrix_f(&data[i].object->matrix,data[i].in.xfrom,data[i].in.yfrom,data[i].in.zfrom);

                            data[i].state=STATE_DISPLAYING;
                        }

                        break;

                    case STATE_IDLE:

                        if(data[i].out.begin<0)
                        {
                            get_translation_matrix_f(&data[i].object->matrix,data[i].out.xfrom,data[i].out.yfrom,data[i].out.zfrom);

                            data[i].state=STATE_FINISHED;
                        }
                        else
                        {
                            if(timer>=data[i].out.begin)
                            {
                                get_translation_matrix_f(&data[i].object->matrix,data[i].out.xfrom,data[i].out.yfrom,data[i].out.zfrom);

                                data[i].state=STATE_HIDING;
                            }
                            else
                            {
                                get_translation_matrix_f(&data[i].object->matrix,data[i].in.xto,data[i].in.yto,data[i].in.zto);
                            }
                        }

                        break;

                    case STATE_FINISHED:

                        numf++;

                        break;

                    case STATE_DISPLAYING:

                        if(timer>data[i].in.end)
                        {
                            get_translation_matrix_f(&data[i].object->matrix,data[i].in.xto,data[i].in.yto,data[i].in.zto);

                            data[i].state=STATE_IDLE;
                        }
                        else
                        {
                            t=(float)(timer-data[i].in.begin)/
                                (float)(data[i].in.end-data[i].in.begin);
                            x=(1.-t)*data[i].in.xfrom+t*data[i].in.xto;
                            y=(1.-t)*data[i].in.yfrom+t*data[i].in.yto;
                            z=(1.-t)*data[i].in.zfrom+t*data[i].in.zto;

                            get_translation_matrix_f(&data[i].object->matrix,x,y,z);
                        }

                        break;

                    case STATE_HIDING:

                        if(timer>data[i].out.end)
                        {
                            scene_removeobject(scene,data[i].object);
                            object_destroy(data[i].object);
                            data[i].object=NULL;

                            data[i].state=STATE_FINISHED;
                        }
                        else
                        {
                            t=(float)(timer-data[i].out.begin)/
                                (float)(data[i].out.end-data[i].out.begin);
                            x=(1.-t)*data[i].out.xfrom+t*data[i].out.xto;
                            y=(1.-t)*data[i].out.yfrom+t*data[i].out.yto;
                            z=(1.-t)*data[i].out.zfrom+t*data[i].out.zto;

                            get_translation_matrix_f(&data[i].object->matrix,x,y,z);
                        }

                        break;
                }

                if(data[i].object)
                {
                    MATRIX_f m1,m2;

                    data[i].angle+=data[i].rotatespeed;
                    data[i].angle&=255;

                    get_y_rotate_matrix_f(&m1,(float)data[i].angle);
                    m2=data[i].object->matrix;

                    matrix_mul_f(&m1,&m2,&data[i].object->matrix);
                }
            }

            if(i==numf && !finishing) finishing=1;
        }

        clear_to_color(dbuffer,makecol(finishing*4,finishing*4,finishing*4));
        c=scene_render(dbuffer,scene);

        if(key[KEY_F1])
        {
            text_mode(-1);
            textout_right(dbuffer,font,"Solid Drawing",SCREEN_W-1,10,solidmode?makecol(255,255,255):makecol(96,96,96));
            textout_right(dbuffer,font,"Backface Culling",SCREEN_W-1,20,bfculling?makecol(255,255,255):makecol(96,96,96));
            textout_right(dbuffer,font,"Lighting",SCREEN_W-1,30,lighting?makecol(255,255,255):makecol(96,96,96));
            textprintf(dbuffer,font,0,0,makecol(255,255,255),"%d fps;  VC: %d; PC: %d;",fps,c&0xFFFF,c>>16);

            for(i=c=0; data[i].text; i++)
                if(data[i].object)
                    textprintf(dbuffer,font,0,20+10*(c++),makecol(0,255,0),"%s:  VC: %d; PC: %d; Planes: %d;",data[i].text,data[i].object->vertexcount,data[i].object->polygoncount,data[i].object->planecount);
        }

        blit(dbuffer,screen,0,0,0,0,SCREEN_W,SCREEN_H); 

        if(key[KEY_F2])                 // Wireframe mode
        {   if(!f2pressed)
            {   solidmode=!solidmode;
                f2pressed=1;
            }
        }else f2pressed=0;
        if(key[KEY_F3])                 // Backface culling
        {   if(!f3pressed)
            {   bfculling=!bfculling;
                f3pressed=1;
            }
        }else f3pressed=0;
        if(key[KEY_F4])                 // Lighting
        {   if(!f4pressed)
            {   lighting=!lighting;
                f4pressed=1;
            }
        }else f4pressed=0;

        if(key[KEY_ESC]) done=1;
    }

    remove_int(increment_speed_counter);
    remove_int(fps_proc);

    /* free up temp render mem and scene */

    while(scene->objects)
    {
        OBJECT *object=scene->objects;

        scene_removeobject(scene,object);
        object_destroy(object);
    }

    scene_render(NULL,scene);
    scene_destroy(scene);

    if(bitmap_color_depth(screen)==15 || bitmap_color_depth(screen)==16)
    {
        for(i=31; i>=0; i--)
        {   fade(dbuffer,dbuffer2,i);
            blit(dbuffer2,screen,0,0,0,0,SCREEN_W,SCREEN_H);
        }
    }

    while(key[KEY_ESC]) ;

    return 0;
}

TEXT3D introdata[]=
{   {   "Welcome to",
                128,128,128,    6,12,10,    2,
            {   0,      100,    0,0,512,    0,0,0},
            {   101,    200,    0,0,0,      0,0,512}, 0, 0, 0
    },
    {   "a game",
                128,128,128,    8,8,10,     -2,
            {   101,    200,    0,0,512,    0,-48,0},
            {   300,    350,    0,-48,0,    0,-384,0}, 0, 0, 0
    },
    {   "by",
                128,128,128,    8,8,10,     2,
            {   151,    250,    0,0,512,    0,48,0},
            {   300,    350,    0,48,0,     0,384,0}, 0, 0, 0
    },
    {   "Tobi",
                128,128,128,    16,8,10,    -2,
            {   251,    350,    0,0,512,    0,-48,0},
            {   451,    500,    0,-48,0,    0,0,512}, 0, 0, 0
    },
    {   "Vollebregt",
                128,128,128,    6,8,10,     -2,
            {   301,    400,    0,0,512,    0,48,0},
            {   401,    500,    0,48,0,     0,0,512}, 0, 0, 0
    },
    {   "made",
                128,128,128,    6,12,10,    2,
            {   401,    500,    0,-48,512,  0,-48,0},
            {   501,    600,    0,-48,0,    0,-384,0}, 0, 0, 0
    },
    {   "with",
                128,128,128,    6,12,10,    -2,
            {   401,    500,    0,48,512,   0,48,0},
            {   501,    600,    0,48,0,     0,384,0}, 0, 0, 0
    },
    {   "DJGPP",
                128,0,0,        8,8,12,     1,
            {   501,    600,    0,0,512,    0,-48,0},
            {   701,    800,    0,-48,0,     -512,0,0}, 0, 0, 0
    },
    {   "and",
                128,128,128,    4,4,8,      3,
            {   551,    651,    0,0,512,    0,0,0},
            {   751,    850,    0,0,0,      0,0,512}, 0, 0, 0
    },
    {   "Allegro",
                0,0,128,        8,8,10,     -1,
            {   601,    700,    0,0,512,    0,48,0},
            {   801,    900,    0,48,0,     512,0,0}, 0, 0, 0
    },
    {   "get",
                128,0,0,        8,8,16,     3,
            {   901,    1000,   0,0,512,    0,0,0},
            {   1001,   1100,   0,0,0,      384,-384,0}, 0, 0, 0
    },
    {   "ready",
                0,128,0,        8,8,16,     -2,
            {   1001,   1100,   0,0,512,    0,0,0},
            {   1101,   1200,   0,0,0,      -384,384,0}, 0, 0, 0
    },
    {   "to",
                128,128,128,    8,8,16,     -3,
            {   1101,   1200,   0,0,512,    0,0,0},
            {   1201,   1300,   0,0,0,      -128,-128,-128}, 0, 0, 0
    },
    {   "enter",
                128,128,128,    9,9,20,     2,
            {   1201,   1300,   0,0,512,    0,0,0},
            {   1301,   1400,   0,0,0,      0,0,512}, 0, 0, 0
    },
    {   "TankZone",
                128,128,128,    7,24,24,    0,
            {   1301,   1400,   0,0,512,    0,0,0},
            {   -1,     0,      0,0,0,      0,0,0}, 0, 0, 0
    },
    {   NULL,
                0, 0, 0,        0, 0, 0,    0,
            {   0,      0,      0, 0, 0,    0, 0, 0 },
            {   0,      0,      0, 0, 0,    0, 0, 0 }, 0, 0, 0
    }
};

int intro(BITMAP *dbuffer, BITMAP *dbuffer2)
{
    int ret;

    ret=animate(dbuffer,dbuffer2,introdata);

    return ret;
}
