/* Author: Tobi Vollebregt */

/*  TankZone: My second Allegro game.
 *  Copyright (C) 2003  Tobi Vollebregt
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  See `License.txt', which contains a verbatim copy of the
 *  GNU General Public License, for details.
 *
 *  Please send your reaction to: tobivollebregt@hotmail.com
 */

/* 28/04/2003: Removed "#include <math.h>", nothing from math.h was used.
    Removed "#include <allegro.h>", allegro.h is already included in engine.h. */

/* 17/01/2003: Fixed memory leak when loading a new tileset (ie. loading a
    tileset while some tiles where already set.) */

/* 18/01/2003: Changed the behaviour of the engine concerning the copying of
    tile bitmaps. It now copies the whole bitmap instead of the pointer,
    because some memory leaks existed in save_tileset/load_tileset. */

/* 18/05/2003: Changed the bits per boolean from 8 to 1 in load_tileset() and
    save_tileset(). Obviously, 8 bits for a boolean value is quite a waste. */

/* 27/05/2003: Added support for up to 3 overlay tilemaps. */

/* 16/06/2003: Fixed a bug with masked sprites in load_tileset() and
    save_tileset(). Made draw_scene() check for empty tiles on overlay maps. */

#ifdef DEBUGMODE
#define DEBUGLEVEL  3
#else
#define DEBUGLEVEL  0   /* 0=none, 1=low, 2=medium, 3=high */
#endif

#include <string.h>     /* for the memset() function in several functions */
#include "engine.h"
#include "sprite4.h"    /* for the 4->16 bit sprite expanding */

#if DEBUGLEVEL>1
/* ASSERT1 only compiles if DEBUGLEVEL is medium or high */
#define ASSERT1(x)  ASSERT(x)
#else
#define ASSERT1(x)
#endif

#if DEBUGLEVEL>2
/* ASSERT2 only compiles if DEBUGLEVEL is high */
#define ASSERT2(x)  ASSERT(x)
#else
#define ASSERT2(x)
#endif

typedef struct TILE {
    SPRITE4 *sprite;
    BITMAP *bitmap;
    ENGINE_DRAWER drawer;
} TILE;

typedef struct NODE {
    struct NODE **parent;   // Pointer to ( pointer to this node )
    struct NODE *child[2];  // Children (0=lower, 1=higher)
    int color;              // Color in which it was lighted
    BITMAP *bitmap;         // Bitmap with lighted tile
    int hits;               // Hits with this node
 #if DEBUGLEVEL>2
    int tile;               // for debugging (should not be needed)
 #endif
} NODE;

static int engine_cache_hits=0,                 // To optimize the cache size
    engine_cache_misses=0;  // (can be displayed somewhere in main program.)
static int lighting_mask=~0;// Mask value to reduce the amount of tiles in cache
static int tilesize,tileshift,tilemask;         // Size of tiles
static TILE tiles[1024];                        // Tile data
static BITMAP *tilemap;                         // Pointer to tilemap
static int tilemap_xoffset,tilemap_yoffset;     // Offset of tilemap
static BITMAP *lightmap;                        // Pointer to lightmap
static int lightmap_xoffset,lightmap_yoffset;   // Offset of lightmap
static BITMAP *destination;                     // Destination bitmap
static BITMAP *tilebuffer;                      // Buffer for lighting one tile
static int ringbuf_size;                        // Size of the ringbuffer
static int ringbuf_pos;                         // Position in the ringbuffer
static NODE *ringbuf;                           // Ringbuffer (array nodes)
static NODE *root[1024];                        // Root of B searching tree
static BITMAP* overlay[3]; // Overlay tilemaps, drawn on top of tilemap.

static BITMAP *get_bitmap_from_cache(int tile, int color);



int engine_init(int size)
{
    /* clear memory */
    /* a normal compiler should initialized all globals to zero, and then
        this will work fine */
    engine_exit();

    tilesize=size;

    /* calculate tileshift */
    tileshift=0;
    while((1<<tileshift)!=tilesize && (1<<tileshift)<tilesize) tileshift++;

    /* tilesize must be a power of two */
    ASSERT((1<<tileshift)==tilesize);

    tilemask=tilesize-1;

    /* create buffer */
    if(!(tilebuffer=create_bitmap(tilesize,tilesize))) return 1;

    return 0;
}

void engine_exit(void)
{
    int i;

    /* free up tiles */
    for(i=0; i<(int) (sizeof (tiles)/sizeof (tiles[0])); i++)
    {
        if(tiles[i].sprite) destroy_sprite4(tiles[i].sprite);
        if (tiles[i].bitmap) destroy_bitmap (tiles[i].bitmap);
    }
    memset(tiles,0,sizeof(tiles));

    tilemap = lightmap = destination = 0;
    memset (overlay, 0, sizeof (overlay));

    /* destroy buffer */
    if(tilebuffer)
    {   destroy_bitmap(tilebuffer);
        tilebuffer=NULL;
    }

    /* destroy ringbuffer */
    engine_enable_cache(0);
}

int engine_set_tile(int index, BITMAP *bmp)
{
    SPRITE4 *tile;
    BITMAP* b;

    ASSERT(index>=0 && index<256);
    ASSERT(bmp && bmp->w==tilesize && bmp->h==tilesize);

    /* create 4 bit tile */
    if(!(tile=get_sprite4(bmp))) return 1;

    if (!(b=create_bitmap (tilesize, tilesize)))
    {   destroy_sprite4 (tile);
        return 1;
    }
    blit (bmp, b, 0, 0, 0, 0, tilesize, tilesize);

    /* destroy when this tile did already exists */
    if(tiles[index].sprite) destroy_sprite4(tiles[index].sprite);
    if (tiles[index].bitmap) destroy_bitmap (tiles[index].bitmap);

    /* copy to array */
    tiles[index].sprite=tile;
    tiles[index].bitmap=b;
    tiles[index].drawer=NULL;

    return 0;
}

int engine_set_tile_ex(int index, BITMAP *bmp, ENGINE_DRAWER drawer)
{
    int ret=engine_set_tile(index,bmp);

    if(!ret) tiles[index].drawer=drawer;

    return ret;
}

void engine_set_tile_drawer(int index, ENGINE_DRAWER drawer)
{
    ASSERT(index>=0 && index<256);

    tiles[index].drawer=drawer;
}

BITMAP *engine_get_tile(int index)
{
    ASSERT(index>=0 && index<256);

    return tiles[index].bitmap;
}

BITMAP *engine_get_tile_ex(int index, int r, int g, int b)
{
    ASSERT(index>=0 && index<256);
    ASSERT(tiles[index].sprite);
    ASSERT(r>=0 && r<256 && g>=0 && g<256 && b>=0 && b<256);

    /* if cache is installed, get bitmap from cache */
    if(ringbuf) return get_bitmap_from_cache(index,makecol24(r,g,b));

    /* else light bitmap to static buffer */
    draw_multiply_sprite4(tilebuffer,tiles[index].sprite,0,0,r,g,b);

    return tilebuffer;
}

void engine_draw_tile_ex(BITMAP *dst, int index, int x, int y, int r, int g, int b)
{
    ASSERT(dst && ((dst->cl&1)|(dst->cr&1))==0);
    ASSERT(bitmap_color_depth(dst)==15 || bitmap_color_depth(dst)==16);
    ASSERT(index>=0 && index<256);
    ASSERT(tiles[index].sprite);
    ASSERT(r>=0 && r<256 && g>=0 && g<256 && b>=0 && b<256);

    if(ringbuf)
    {   BITMAP *bm=get_bitmap_from_cache(index,makecol24(r,g,b));
        blit(bm,dst,0,0,x,y,tilesize,tilesize);
    }
    else
    {   x&=~1;  /* Make sure x is even */
        draw_multiply_sprite4(dst,tiles[index].sprite,x,y,r,g,b);
    }
}

void engine_set_tilemap(BITMAP *map, int xoffset, int yoffset)
{
    ASSERT(map && bitmap_color_depth(map)==8);

    tilemap=map;
    tilemap_xoffset=xoffset;
    tilemap_yoffset=yoffset;
}

void engine_set_lightmap(BITMAP *map, int xoffset, int yoffset)
{
    ASSERT(map && bitmap_color_depth(map)==24);

    lightmap=map;
    lightmap_xoffset=xoffset;
    lightmap_yoffset=yoffset;
}

void engine_set_destination(BITMAP *scrn)
{
    ASSERT(((scrn->cl&1)|(scrn->cr&1))==0);
    ASSERT(bitmap_color_depth(scrn)==15 || bitmap_color_depth(scrn)==16);

    destination=scrn;
}

BITMAP *engine_get_destination(void)
{
    return destination;
}

void engine_set_gamma(int gamma)
{
    ASSERT(gamma>0);

    _multiply_sprite_gamma=gamma;
}

int engine_get_gamma(void)
{
    return _multiply_sprite_gamma;
}

void engine_draw_scene(int x, int y)
{
    int tx,ty,txbeg,tybeg,ly,lxbeg,lybeg;
    int dx,dy,dxbeg,dybeg,dxend,dyend;
    int w,h,c;
    unsigned o,ot;
    unsigned char *tptr, *optr[sizeof (overlay)/sizeof (overlay[0])];
    unsigned long laddr;
    BITMAP *b;
    ENGINE_DRAWER_INFO info;

    ASSERT(tilemap && lightmap && destination);

    memset (optr, 0, sizeof (optr));

    /* make sure x is even */
    x&=~1;

    /* calculate begin and end values for loop */
    w=destination->cr-destination->cl;
    h=destination->cb-destination->ct;
    txbeg=(x>>tileshift)+tilemap_xoffset;
    tybeg=(y>>tileshift)+tilemap_yoffset;
    lxbeg=((x>>tileshift)+lightmap_xoffset)*3;  // cuz its 24 bit
    lybeg=(y>>tileshift)+lightmap_yoffset;
    dxbeg=destination->cl-(x&tilemask);
    dybeg=destination->ct-(y&tilemask);
    dxend=destination->cr;
    dyend=destination->cb;

    /* select lightmap for 24 bit reading */
    bmp_select(lightmap);

    for(ty=tybeg,ly=lybeg,dy=dybeg; dy<dyend; ty++,ly++,dy+=tilesize)
    {
        /* calculate tilemap and lightmap adresses */
        tptr=tilemap->line[ty]+txbeg;
        laddr=((unsigned long)lightmap->line[ly])+lxbeg;
        for (o = 0; o < sizeof (overlay) / sizeof (overlay[0]); ++o)
            if (overlay[o]) optr[o] = overlay[o]->line[ty] + txbeg;

        for(tx=txbeg,dx=dxbeg; dx<dxend; tx++,tptr++,laddr+=3,dx+=tilesize)
        {
            /* read 24 bit color */
            c=bmp_read24(laddr);

            /* call overridden drawer when possible */
            if(tiles[*tptr].drawer)
            {   info.mapx=tx;
                info.mapy=ty;
                info.scrx=dx;
                info.scry=dy;
                info.tile=*tptr;
                info.r=getr24(c);
                info.g=getg24(c);
                info.b=getb24(c);
                tiles[*tptr].drawer(&info);
            }
            else
            {   /* find it in the cache */
                if(ringbuf)
                {   b=get_bitmap_from_cache(*tptr,c);
                    blit(b,destination,0,0,dx,dy,tilesize,tilesize);
                }
                else /* or draw it at the classic way */
                    draw_multiply_sprite4(destination,tiles[*tptr].sprite,dx,dy,getr24(c),getg24(c),getb24(c));
            }
            /* Draw overlay tiles on top of the main tilemap. */
            for (o = 0; o < sizeof (overlay) / sizeof (overlay[0]); ++o)
                if (optr[o])
                {
                    ot = *(optr[o]) + ((o + 1) << 8);
                    /* call overridden drawer when possible */
                    if(tiles[ot].drawer)
                    {   info.mapx=tx;
                        info.mapy=ty;
                        info.scrx=dx;
                        info.scry=dy;
                        info.tile=ot;
                        info.r=getr24(c);
                        info.g=getg24(c);
                        info.b=getb24(c);
                        tiles[ot].drawer(&info);
                    }
                    else if (tiles[ot].bitmap)
                    {   /* find it in the cache */
                        if(ringbuf)
                        {   b=get_bitmap_from_cache(ot,c);
                            masked_blit(b,destination,0,0,dx,dy,tilesize,tilesize);
                        }
                        else /* or draw it at the classic? way */
                            draw_multiply_sprite4(destination,tiles[ot].sprite,dx,dy,getr24(c),getg24(c),getb24(c));
                    }
                }
        }
    }
}



////////// NEW FUNCS IMPLEMENTING A CACHE (like the classic system in tankzone)
////////// a ringbuffer with binary searching trees
////////// and a simple algo to find the best node to recycle

int engine_enable_cache(int size)
{
    int i,j;

    ASSERT(size>=0);

    /* nothing has to be done if the size hasn't changed */
    if(size==ringbuf_size) return 0;

    /* destroy previous ringbuffer */
    if(ringbuf)
    {   for(i=0; i<ringbuf_size; i++)
            destroy_bitmap(ringbuf[i].bitmap);
        free(ringbuf);
        ringbuf=NULL;
    }

    ringbuf_size=size;

    memset(root,0,sizeof(root));

    if(size)
    {
        /* allocate new */
        ringbuf=(NODE *)malloc(sizeof(NODE)*size);
        if(!ringbuf) {ringbuf_size=0; return 1;}
        memset(ringbuf,0,sizeof(NODE)*size);
        for(i=0; i<size; i++)
        {   ringbuf[i].bitmap=create_bitmap(tilesize,tilesize);
            if(!ringbuf[i].bitmap)
            {   for(j=0; j<i; j++)
                    destroy_bitmap(ringbuf[j].bitmap);
                free(ringbuf);
                ringbuf=NULL;
                ringbuf_size=0;
                return 1;
            }
            ringbuf[i].bitmap->clip=FALSE;
            clear_bitmap(ringbuf[i].bitmap);
        }
    }

    return 0;
}

int engine_enable_cache_ex(int size, int acc)
{
    int mask=((1<<acc)-1)<<(8-acc);

    ASSERT(acc>0 && acc<=8);

    lighting_mask=(mask<<24)|(mask<<16)|(mask<<8)|mask;

    return engine_enable_cache(size);
}

static void insert(NODE **root, NODE *node);

static BITMAP *get_bitmap_from_cache(int tile, int color)
{
    NODE **ptr=&root[tile];
    NODE *node=root[tile];
 #if DEBUGLEVEL>1
    int debug=0;
 #endif

    color&=lighting_mask;

    if(!node) goto cr_new;

    while(node->color!=color)
    {   if(color<node->color)
        {   if(node->child[0]) node=node->child[0];
            else {ptr=&node->child[0]; goto cr_new;}
        }
        else
        {   if(node->child[1]) node=node->child[1];
            else {ptr=&node->child[1]; goto cr_new;}
        }
        ASSERT1(++debug<2000);
    }

    ASSERT2(node->tile==tile);

    engine_cache_hits++;
    node->hits++;

    return node->bitmap;

 cr_new:

    engine_cache_misses++;

    /* We want to find a suitable node to recycle */
    do
    {   node=&ringbuf[ringbuf_pos];
        if(++ringbuf_pos>=ringbuf_size) ringbuf_pos=0;
    }
    while(ringbuf[ringbuf_pos].hits<node->hits);

    /* 1) remove */
    if(node->parent)
    {   *node->parent=node->child[0];
        if(node->child[0]) node->child[0]->parent=node->parent;
        if(node->child[1]) insert(node->parent,node->child[1]);
    }

    /* 2) light bitmap */
    node->color=color;
    if (tiles[tile].sprite->masked)
        clear_to_color (node->bitmap, bitmap_mask_color (node->bitmap));
    draw_multiply_sprite4(node->bitmap,tiles[tile].sprite,0,0,getr24(color),getg24(color),getb24(color));

 #if DEBUGLEVEL>2
    node->tile=tile;
 #endif

    /* 3) link to tree */
    node->parent=ptr;
    *ptr=node;
    node->child[0]=node->child[1]=NULL;

    node->hits=0;

    return node->bitmap;
}

static void insert(NODE **root, NODE *node)
{
 #if DEBUGLEVEL>1
    int debug=0;
 #endif

    while(*root)
    {   if(node->color<(*root)->color)
            root=&((*root)->child[0]);
        else root=&((*root)->child[1]);
        ASSERT1(++debug<2000);
    }
    node->parent=root;
    *root=node;
}

double engine_get_cache_hit_percent(void)
{
    int total=engine_cache_hits+engine_cache_misses;
    double perc=100.0*engine_cache_hits/total;

    //engine_cache_hits=engine_cache_misses=0;

    /*engine_cache_hits=100*engine_cache_hits/total;
    engine_cache_misses=100*engine_cache_misses/total;*/

    /*int hi=MIN(engine_cache_hits,engine_cache_misses);

    engine_cache_hits-=hi;
    engine_cache_misses-=hi;*/

    return perc;
}



////////// NEW FUNCS IMPLEMENTING ANOTHER `CACHE'
////////// engine_set_tile() function was way too slow (256 tiles,800Mhz: 6,7s)
////////// so I decided to write funcs to load/save the binary sprite data.

typedef struct TILESET_HDR {
    char desc[16];  // =`TJENGINETILESET' (15 chars + null = 16)
    int version;    // =5
    int colordepth; // Colordepth must equal the selected colordepth
    int tilesize;   // Tilesize must equal the global tilesize
    unsigned char tileinuse[sizeof(tiles)/sizeof(tiles[0])/8]; // Which tiles are in use? (1 bit per tile)
    unsigned char masked[sizeof(tiles)/sizeof(tiles[0])/8];
} TILESET_HDR;

const char tileset_filedesc[16]="TJENGINETILESET";

// Save tileset may be called after uploading all the tiles
// with engine_set_tile(_ex). It saves generated palettes and
// sprite data to file.

int engine_save_tileset(const char *filename, int pack)
{
    const long ntile=tilesize/2*tilesize;
    const long npal=sizeof(short)*16;
    const long nbitmap=sizeof(short)*tilesize*tilesize;

    PACKFILE *fp;
    TILESET_HDR hdr;
    int i;

    memcpy(hdr.desc,tileset_filedesc,16);
    hdr.version=5;
    hdr.colordepth=makecol(145,82,19); /* a color that differs in all depths */
    hdr.tilesize=tilesize;

    /* It was a waste to use 8 bits to store a boolean value, so I changed it to 1 bit. */
    memset (hdr.tileinuse, 0, sizeof (hdr.tileinuse));
    for(i=0; i<(int) (sizeof(tiles)/sizeof(tiles[0])); i++)
        hdr.tileinuse[i >> 3] |= (tiles[i].sprite ? 1 : 0) << (i & 7);
    memset (hdr.masked, 0, sizeof (hdr.tileinuse));
    for (i = 0; i < (int) (sizeof (tiles) / sizeof (tiles[0])); ++i)
        if (hdr.tileinuse[i >> 3])
            hdr.masked[i >> 3] |= (tiles[i].sprite->masked ? 1 : 0) << (i & 7);

    fp=pack_fopen(filename,pack?F_WRITE_PACKED:F_WRITE_NOPACK);
    if(!fp) return 1;

    pack_fwrite(&hdr,sizeof(hdr),fp);

    for(i=0; i<(int) (sizeof(tiles)/sizeof(tiles[0])); i++)
    {   if(hdr.tileinuse[i >> 3] & (1 << (i & 7)))
        {   pack_fwrite(tiles[i].sprite->palette_color_16,npal,fp);
            pack_fwrite(tiles[i].sprite->data,ntile,fp);
            pack_fwrite(tiles[i].bitmap->dat,nbitmap,fp);
        }
    }
    if(pack_fclose(fp)) return 999;

    return 0;
}

// Load tileset must be called instead of a number of calls to engine_set_tile

int engine_load_tileset(const char *filename)
{
    const long ntile=tilesize/2*tilesize;
    const long npal=sizeof(short)*16;
    const long nbitmap=sizeof(short)*tilesize*tilesize;

    PACKFILE *fp;
    TILESET_HDR hdr;
    int i;

    /* (bugfix 17/01/2003, copied from engine_exit()) */
    /* free up tiles */
    for(i=0; i<(int) (sizeof(tiles)/sizeof(tiles[0])); i++)
    {
        if(tiles[i].sprite) destroy_sprite4(tiles[i].sprite);
        if (tiles[i].bitmap) destroy_bitmap (tiles[i].bitmap);
    }
    memset(tiles,0,sizeof (tiles));

    fp=pack_fopen(filename,F_READ_PACKED);
    if(!fp) return 1;

    pack_fread(&hdr,sizeof(hdr),fp);

    /* We check our file description */
    if(memcmp(hdr.desc,tileset_filedesc,16)!=0
        || hdr.version!=5 || hdr.tilesize!=tilesize
        || hdr.colordepth!=makecol(145,82,19))
    {   pack_fclose(fp);
        return 2;
    }

    /* and load our tiles */
    for(i=0; i<(int) (sizeof(tiles)/sizeof(tiles[0])); i++)
    {   if(hdr.tileinuse[i >> 3] & (1 << (i & 7)))
        {   if(!(tiles[i].sprite=(SPRITE4*)malloc(sizeof(SPRITE4)))
                || !(tiles[i].sprite->data=(unsigned char*)malloc(ntile))
                || !(tiles[i].bitmap=create_bitmap(tilesize,tilesize)))
            {   pack_fclose(fp);
                return 3;
            }
            tiles[i].sprite->w=tilesize;
            tiles[i].sprite->h=tilesize;
            tiles[i].sprite->masked = (hdr.masked[i >> 3] & (1 << (i & 7))) ? 1 : 0;
            pack_fread(tiles[i].sprite->palette_color_16,npal,fp);
            pack_fread(tiles[i].sprite->data,ntile,fp);
            pack_fread(tiles[i].bitmap->dat,nbitmap,fp);
        }
    }
    if(pack_fclose(fp)) return 999;

    return 0;
}



///////////////////////////////////////////////////////////////////////
///////////////// EXTENSION 3: Support for overlay maps (currently 3)

// Note that most changes for this extension are in the original code.
// Only the new API function is here.

// The (x,y) offset for the overlay maps is the same as for the tilemap.
void engine_set_overlay (BITMAP* map, int index)
{
    ASSERT (index >= 0 && index < (int) (sizeof (overlay) / sizeof (overlay[0])));
    ASSERT (map && bitmap_color_depth (map) == 8);
    ASSERT (!tilemap || (map->w == tilemap->w && map->h == tilemap->h));

    overlay[index] = map;
}
