#ifndef SHELLSHOCK_VISUALS_C
#define SHELLSHOCK_VISUALS_C

#include <alleggl.h>
#include <GL/glu.h>
#include "visuals.h"
#include "system.h"
#include "common.h"
#include "sshock.h"

void ss_anim_init(void) {
  // What a demanding task
  ss_anim_sys.list=NULL;
  system_event(0, "Initialized: Animation system");
}

void ss_anim_exit(void) {
  ss_anim_flush();
  system_event(0, "Uninitialized: Animation system");
}

int ss_visuals_init(void) {
  set_color_depth(16);
  if (install_allegro_gl()) {
    system_event(2, "Visuals: Could not initialize AllegroGL");
    return 0;
  }
  allegro_gl_clear_settings();
  allegro_gl_set(AGL_DOUBLEBUFFER, 1);
  allegro_gl_set(AGL_RENDERMETHOD, 1);
  allegro_gl_set(AGL_COLOR_DEPTH, 16);
  allegro_gl_set(AGL_Z_DEPTH, 16);
  allegro_gl_set(AGL_WINDOWED, 1);
  allegro_gl_set(AGL_SUGGEST, AGL_DOUBLEBUFFER | AGL_COLOR_DEPTH | AGL_RENDERMETHOD | AGL_Z_DEPTH | AGL_WINDOWED);  
  if (set_gfx_mode(GFX_OPENGL_WINDOWED, 640, 480, 0, 0)) { 
    system_event(2, "Visuals: Could not set video mode");
    remove_allegro_gl(); return 0; 
  }
  allegro_gl_begin();
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glOrtho(-320.0, 320.0, -240.0, 240.0, 1.0, 600.0);
  glMatrixMode(GL_MODELVIEW);
  glEnable(GL_DEPTH_TEST);
  glDepthFunc(GL_LEQUAL);
  glClearDepth(640.0);
  glClearColor(0.0,0.0,0.0,0.0);
  glHint(GL_FOG_HINT, GL_NICEST);
  glEnable(GL_TEXTURE);
  glEnable(GL_TEXTURE_2D);
  glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR);
  glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR);
  allegro_gl_end();
  set_window_title("Duckie Dash for Speedhack '04 by Inphernic");
  system_event(0, "Initialized: Visuals");
  return 1;
}

void ss_visuals_exit(void) {
  remove_allegro_gl();
  system_event(0, "Uninitialized: Visuals");
}

void ss_blit(void) {
  allegro_gl_flip();
}

void ss_cycle_start(void) {
  allegro_gl_begin();
  glClear(GL_DEPTH_BUFFER_BIT);
  glLoadIdentity();
  ss_sea_draw();
  glRotatef((ss_game.ca-(M_PI))/M_PI*180.0, 0, 0, -1);
  glTranslatef(0-ss_game.cx, 0, 0);
  glTranslatef(0, 0-ss_game.cy, 0);
}

void ss_normcycle_start(void) {
  allegro_gl_begin();
  glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);
  glLoadIdentity();
  glRotatef((ss_game.ca-(M_PI))/M_PI*180.0, 0, 0, -1);
  glTranslatef(0-ss_game.cx, 0, 0);
  glTranslatef(0, 0-ss_game.cy, 0);
}

void ss_cycle_end(void) {
  ss_blit();
  allegro_gl_end();
//  rest(1);
  fpsc++; 
}

void ss_anim_flush(void) {
  while(ss_anim_sys.list) {
    ss_anim_destroy(ss_ll_access(ss_anim_sys.list));
    ss_ll_npop(ss_anim_sys.list,ss_anim_sys.list);
  }
  ss_anim_sys.list=NULL;
  system_event(0, "Visuals: Animation pool flushed");
}

ss_anim_props *ss_anim_props_create(void) {
  ss_anim_props *t=NULL;
  t=malloc(sizeof(ss_anim_props));
  if (t) ss_anim_props_reset(NULL, t);
  return t;
}

void ss_anim_props_reset(ss_anim *a, ss_anim_props *t) {
  if (t) {
    // if the animation has been given, "inherit" properties from it
    if (a) {
      t->cur_frame=0;
      t->cur_ttl=a->frame[t->cur_frame]->ttl;
      t->dir=a->dir;
      t->loop=a->loop;
    }
    else {
      t->cur_ttl=0;
      t->cur_frame=0;
      t->dir=0;
      t->loop=0;
    }
  }
}

void ss_anim_props_destroy(ss_anim_props *t) {
  if (t) free(t);
}

ss_anim *ss_anim_create(unsigned char frames) {
  ss_anim *t=NULL;
  if (!frames) return NULL;
  t=malloc(sizeof(ss_anim));
  if (t) {
    t->loop=0;
    t->dir=0;
    t->frames=frames;
    t->name=NULL;
    t->frame=ss_anim_frames_create(frames);
    if (!t->frame) {
      ss_anim_destroy(t);
      t=NULL;
    }
    else ss_ll_opush(ss_anim_sys.list, t);
  }
  return t;
}

void ss_anim_destroy(ss_anim *anim) {
  if (anim) {
    if (anim->name) free(anim->name);
    anim->name=NULL;
    ss_anim_frames_destroy(anim->frames, anim->frame);
    free(anim);
  }
}

ss_anim_frame *ss_anim_frame_create(void) {
  ss_anim_frame *t=NULL;
  t=malloc(sizeof(ss_anim_frame));
  if (t) {
    t->frame=0;
  }
  return t;
}

void ss_anim_frame_destroy(ss_anim_frame *t) {
  if (t) {
    if (t->frame) glDeleteTextures(1, &t->frame);
    t->frame=0;
    free(t);
  }
}

ss_anim_frame **ss_anim_frames_create(unsigned char frames) {
  unsigned char i; ss_anim_frame **t=NULL;
  if (!frames) return NULL;
  t=malloc(sizeof(ss_anim_frame *)*frames);
  if (t) {
    for(i=0; i<frames; i++) t[i]=NULL;
    for(i=0; i<frames; i++) {
      t[i]=ss_anim_frame_create();
      if (!t[i]) {
        ss_anim_frames_destroy(frames, t);
        t=NULL; i=frames;
      }
    }
  } 
  return t;
}

void ss_anim_frames_destroy(unsigned char frames, ss_anim_frame **f) {
  unsigned char i;
  if (!frames || !f) {
    for(i=0; i<frames; i++) if (f[i]) { ss_anim_frame_destroy(f[i]); f[i]=NULL; }
    free(f);
  }
}

ss_anim *ss_anim_load(char *name) {
  unsigned char *dbuf=NULL, err=0, buf[256]; long i; DATAFILE *dat=NULL; ss_anim *x=NULL;
  if (!name) return NULL;
  snprintf(buf, 255, "./data/%s.dat", name);
  dat=load_datafile(buf);
  if (dat) {
    if (dat[0].size>0) dbuf=malloc(sizeof(unsigned char)*dat[0].size);
    if (!dbuf) return NULL;
    for(i=0; i<dat[0].size; i++) {
      dbuf[i]=((char *)dat[0].dat)[i];
    }
    i=dat[0].size-1;
    // Header size check
    if (i<7) err=1;
    // Header check
    if (!err) if (dbuf[0]!='S' || dbuf[1]!='S' || dbuf[2]!='A' || dbuf[3]!='M') err=1;
    // Parameter check
    if (!err) {
      if (dat[0].size<7+(dbuf[4])) err=1;
      if (dbuf[5]>2) err=1;
      if (dbuf[6]>1) err=1;
    }
    // Animation object creation
    if (!err) { x=ss_anim_create(dbuf[4]); if (!x) err=1; }
    // Parameter placement
    if (!err) {
      x->loop=dbuf[5]; x->dir=dbuf[6];
      for(i=7; i<dat[0].size; i++) { x->frame[i-7]->ttl=dbuf[i]; }
    }
    // Texture (frame) placement
    if (!err) {
      for(i=0; i<dbuf[4]; i++) {
        x->frame[i]->frame=allegro_gl_make_masked_texture(dat[i+1].dat);
        if (!x->frame[i]) { err=1; i=dbuf[4]; }
      }
    }
    // Name placement
    if (!err) {
      x->name=ss_stringcopy(name);
      if (!x->name) err=1;
    }
    if (err) { if (x) { ss_anim_destroy(x); x=NULL; } }
    free(dbuf);
    unload_datafile(dat);
  }
  else system_event(1, "Visuals: Unable to locate animation '%s'", name);
  if (x) system_event(0, "Visuals: Animation '%s' loaded", name);
  else system_event(1, "Visuals: Unable to load animation '%s'", name);
  return x;
}

ss_anim *ss_anim_fetch(char *name) {
  ss_linkage *c=ss_anim_sys.list;
  if (!name) return NULL;
  while(c) {
    // Search for an animation with this name from the animation list
    if (!strcmp(((ss_anim *)ss_ll_access(c))->name, name)) {
      system_event(0, "Visuals: Animation '%s' located in cache", name);
      return ss_ll_access(c);
    }
    c=c->n;
  }
  system_event(0, "Visuals: Animation '%s' not found from cache", name);
  return ss_anim_load(name);
}

#endif
