/* Author: Tobi Vollebregt */

/*  Blobland 2, a 2D space shooter made for the SpeedHack 2002-B competition.
 *  Copyright (C) 2002,2003  Tobi Vollebregt
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  See `License.txt', which contains a verbatim copy of the
 *  GNU General Public License, for details.
 *
 *  Please send your reaction to: tobivollebregt@hotmail.com
 */

/*
    Changes by Tobi Vollebregt:
  8th of May, 2003: Suppressed "unused parameter" warning for CEnemy::Zero().
                    Added missing initializers to CHiscore::dialog
                    Removed the parameters to blob_killall(): no arguments were passed from demo.c.
*/

/* modified for exporting functions blob_drawall and blob_update. */
/* added Dirty rectangle support */

#include <allegro.h>
#include <math.h>
#include <string.h>
#include <time.h>

extern "C" void Dirty(int x, int y, int w, int h);

#define NUM_ENEMIES 8
#define ENEMY_SIZE  64
#define ENEMY_BLOB_SIZE 10
#define HORIZ_ACCEL 0x10000
#define VERT_ACCEL  0x8000
#define HORIZ_SLOWDOWN  0x4000
#define VERT_SLOWDOWN   0x4000
#define UNUSED  0x7fffffff
inline int M(int x) { return (x*x); }

#ifdef DJGPP
#define rand    random
#define srand   srandom
#endif

#ifndef PI
#define PI  3.14159265358979323846
#endif



class CHiscore;
class CStar;
class CParticle;
class CEnemy;
class CPlayer;

CHiscore *hiscore;
CStar *stars;
CParticle *parts;
CEnemy *enemies;
CPlayer *player;



class CHiscore
{
 public:

    class Item
    {
     public:

        char name[64];
        int score;
    };

    bool Load();
    bool Save();
    void Add(const char *name, int score);

    void InitDialog(int score);
    bool UpdateDialog(BITMAP *buf);

    void Draw(BITMAP *buf);

    void Reset() { memset(items,0,sizeof(items)); }

 private:

    static const char *const password;
    static const char *const filename;
    static DIALOG dialog[];

    static int qsort_cmp(const void *e1, const void *e2)
        { return ((Item *)e2)->score-((Item *)e1)->score; }

    Item items[10];

    DIALOG_PLAYER *player;
    char name[64];
    int score;
};

const char *const CHiscore::password="we make it not that easy to hack the hiscore data :)";
const char *const CHiscore::filename="hiscore.dat";

DIALOG CHiscore::dialog[]=
{
    {   d_edit_proc, 0,0, 64*8,12, 0,0, 0,D_EXIT, 63,0, NULL,NULL,NULL},
    {   d_text_proc, 0,0, 0,0,     0,0, 0,0,      0,0,  (void*)"Enter your name for hiscore list:",NULL,NULL},
    {   NULL, 0,0, 0,0, 0,0, 0,0, 0,0, NULL, NULL, NULL }
};

bool CHiscore::Load()
{
    packfile_password(password);
    PACKFILE *fp=pack_fopen(filename,"rp");
    if(!fp) return false;
    pack_fread(items,sizeof(items),fp);
    bool ret=(pack_ferror(fp)?false:true);
    pack_fclose(fp);
    packfile_password(NULL);
    return ret;
}

bool CHiscore::Save()
{
    packfile_password(password);
    PACKFILE *fp=pack_fopen(filename,"wp");
    if(!fp) return false;
    pack_fwrite(items,sizeof(items),fp);
    bool ret=(pack_ferror(fp)?false:true);
    pack_fclose(fp);
    packfile_password(NULL);
    return ret;
}

void CHiscore::Add(const char *name, int score)
{
    if(score>items[9].score)
    {
        ustrzcpy(items[9].name,sizeof(items[9].name),name);
        items[9].score=score;
        qsort(items,10,sizeof(Item),qsort_cmp);
    }
}

void CHiscore::InitDialog(int s)
{
    if(s<items[9].score) {score=-1; return; }

    memset(name,0,sizeof(name));
    dialog[0].x=dialog[1].x=SCREEN_W/2-dialog[0].w/2;
    dialog[0].y=SCREEN_H/2-dialog[0].h/2;
    dialog[1].y=dialog[0].y-30;
    dialog[0].fg=dialog[1].fg=makecol(255,255,255);
    dialog[0].dp=name;
    score=s;
    player=init_dialog(dialog,0);
}

bool CHiscore::UpdateDialog(BITMAP *buf)
{
    if(score==-1) return false;

    BITMAP *scr=screen;
    screen=buf;
    dialog[0].flags|=D_DIRTY;
    dialog[1].flags|=D_DIRTY;
    for(int i=0; i<2; i++)
    {
        if(!update_dialog(player))
        {
            screen=scr;
            Add(name,score);
            shutdown_dialog(player);
            player=NULL;
            return false;
        }
    }

    screen=scr;
    return true;
}

void CHiscore::Draw(BITMAP *buf)
{
    int y=(SCREEN_H-200)/2,x=SCREEN_W-64;

    text_mode(-1);

    for(int i=0; i<10; i++)
    {
        if(items[i].score==0) break;

        int c=255-20*i;
        c=makecol(c,c,c);
        textout_right(buf,font,items[i].name,x-8,y,c);
        textprintf(buf,font,x,y,c,"%8d",items[i].score);
        Dirty(x-text_length(font,items[i].name)-12,y,text_length(font,items[i].name)+80,8);
        y+=20;
    }
}



class CStar
{
 public:

    class Star
    {
     public:

        Star() : x(rand()%SCREEN_W),y(itofix(rand()%SCREEN_H)),
            speed((rand()&0xffff)-0x7fff)
            { int c=rand()&255; color=makecol(c,c,c); }

        int x;
        fixed y,speed;
        int color;
    };

    CStar(int num) : numstars(num) { stars=new Star [num]; }
    ~CStar() { delete[] stars; }

    void Update(fixed speed);
    void Draw(BITMAP *buf);

 private:

    int numstars;
    Star *stars;
};

void CStar::Update(fixed speed)
{
    Star *p=stars;
    for(int i=numstars; i; i--,p++)
    {
        p->y+=speed+p->speed;
        if(fixtoi(p->y)>=SCREEN_H)
        {
            p->x=rand()%SCREEN_W;
            p->y=0;
            p->speed=(rand()&0xffff)-0x7fff;
            int c=rand()&255;
            p->color=makecol(c,c,c);
        }
    }
}

void CStar::Draw(BITMAP *buf)
{
    Star *p=stars;
    for(int i=numstars; i; i--,p++)
    {
        putpixel(buf,p->x,fixtoi(p->y),p->color);
        Dirty(p->x,fixtoi(p->y),1,1);
    }
}



class CParticle
{
 public:

    class Particle
    {
     public:

        Particle() : used(false) {}

        bool used;
        float x,y,xspeed,yspeed;
    };

    CParticle(int num);
    ~CParticle();

    void Update(fixed speed);
    void Draw(BITMAP *dbuf);

    void Kaboom(float x, float y, int num, float power, float a1=0, float a2=2*PI);
    void Kaboom(float x, float y, float xs, float ys, int num, float power, float a1=0, float a2=2*PI);

 private:

    int numparts;
    Particle *parts;
    RGB palette[256];
    unsigned char *buf;
    int *ytable;
};

CParticle::CParticle(int num) : numparts(num)
{
    int i;
    parts=new Particle [num];
    buf=new unsigned char [(SCREEN_W/2)*(SCREEN_H/2)];
    memset(buf,0,(SCREEN_W/2)*(SCREEN_H/2));

    ytable=new int [SCREEN_H/2];
    for(i=0; i<SCREEN_H/2; i++) ytable[i]=i*(SCREEN_W/2);

    memset(palette,0,sizeof(palette));
    for(i=0; i<64; i++) palette[i].r=i<<2;
    for(; i<128; i++) { palette[i].r=255; palette[i].g=i<<2; }
    for(; i<192; i++) { palette[i].r=palette[i].g=255; palette[i].b=i<<2; }
    for(; i<256; i++) { palette[i].r=palette[i].g=palette[i].b=255; }
}

CParticle::~CParticle()
{
    delete[] parts;
    delete[] buf;
    delete[] ytable;
}

void CParticle::Update(fixed speed)
{
    float fspeed=fixtof(speed/2);
    Particle *p=parts;
    for(int i=numparts; i; i--,p++)
        if(p->used)
        {
            p->x+=p->xspeed;
            p->y+=fspeed+p->yspeed;
            if(p->x<0 || (int)p->x>=SCREEN_W/2 || p->y<0 || (int)p->y>=SCREEN_H/2)
                { p->used=false; continue; }

            buf[(int)p->y*SCREEN_W/2+(int)p->x]=255;
        }

    int sx, sy, max_x = SCREEN_W/2-1, max_y = SCREEN_H/2-1,
            col_tot, col, row1, row2, row3, isp=fixtoi(speed/2);

    // move buf down to compensate speed
    for(sy=max_y; sy>=isp; sy--)
        memcpy(buf+ytable[sy],buf+ytable[sy-isp],SCREEN_W/2);

    //// blurring code came from `Paraclone'
    memset(buf,0,SCREEN_W/2);
    memset(buf+ytable[SCREEN_H/2-1],0,SCREEN_W/2);

    for (sy = 1; sy < max_y; sy++) {
    row2=ytable[sy];
    row1=ytable[sy-1];
    row3=ytable[sy+1];
    buf[row1++]=0;
    buf[row2++]=0;
    buf[row3++]=0;
    for (sx = 1; sx < max_x; sx++) {

     col_tot=buf[row1]+buf[row3];
     col_tot+=buf[row2-1];
     col_tot+=buf[row2+1];
     col=col_tot/4;
     col-=2;
        if (col < 0)
        col = 0;
     buf[row2]=col;
     row1++;
     row2++;
     row3++;
    }
    buf[row1++]=0;
    buf[row2++]=0;
    buf[row3++]=0;
    }
}

void CParticle::Draw(BITMAP *dbuf)
{
    switch(bitmap_color_depth(dbuf))
    {
        #define DRAW(dep,typ)   \
        {   \
            unsigned char *ptr2=buf;    \
            for(int y=0; y<SCREEN_H/2; y++)    \
            {   \
                bool dirt=false;    \
                typ *ptr=(typ *)dbuf->line[y<<1];   \
                for(int x=SCREEN_W/2; x; x--)    \
                {   \
                    if(*ptr2)   \
                    {   \
                        int r=getr##dep(*ptr);  \
                        int g=getg##dep(*ptr);  \
                        int b=getb##dep(*ptr);  \
                        RGB *pal=&palette[*ptr2];    \
                        r+=pal->r;    \
                        g+=pal->g;    \
                        b+=pal->b;    \
                        *ptr=makecol##dep(MIN(r,255),MIN(g,255),MIN(b,255));    \
                        ptr++;  \
                        r=getr##dep(*ptr);  \
                        g=getg##dep(*ptr);  \
                        b=getb##dep(*ptr);  \
                        r+=pal->r;    \
                        g+=pal->g;    \
                        b+=pal->b;    \
                        *ptr=makecol##dep(MIN(r,255),MIN(g,255),MIN(b,255));    \
                        ptr++;  \
                        if(!dirt) { Dirty((SCREEN_W-(x<<1))&~63,y<<1,64,1); dirt=true; } \
                    }   \
                    else ptr+=2;  \
                    ptr2++; \
                    if((x&31)==0) dirt=false;   \
                }   \
            }   \
        }

        case 15: DRAW(15, unsigned short); break;
        case 16: DRAW(16, unsigned short); break;
        default: ASSERT(false); break;
    }
};

void CParticle::Kaboom(float x, float y, int num, float power, float a1, float a2)
{
    x*=0.5;
    y*=0.5;
    power*=0.5;

    Particle *p=parts;
    for(int i=numparts,j=num; i && j; i--,p++)
        if(!p->used)
        {
            p->used=true;
            p->x=x;
            p->y=y;
            float a=a1+(float)rand()/RAND_MAX*(a2-a1);
            float b=(float)rand()/RAND_MAX*power;
            p->xspeed=cos(a)*b;
            p->yspeed=sin(a)*b;
            j--;
        }
}

void CParticle::Kaboom(float x, float y, float xs, float ys, int num, float power, float a1, float a2)
{
    x*=0.5;
    y*=0.5;
    xs*=0.5;
    ys*=0.5;
    power*=0.5;

    Particle *p=parts;
    for(int i=numparts,j=num; i && j; i--,p++)
        if(!p->used)
        {
            p->used=true;
            p->x=x;
            p->y=y;
            float a=a1+(float)rand()/RAND_MAX*(a2-a1);
            float b=(float)rand()/RAND_MAX*power;
            p->xspeed=xs+cos(a)*b;
            p->yspeed=ys+sin(a)*b;
            j--;
        }
}



class CEnemy
{
 public:

    typedef float (*Func)(int t);

    class Enemy
    {
     public:

        Enemy() : t(UNUSED) { bmp=create_bitmap_ex(8,ENEMY_SIZE,ENEMY_SIZE); }
        ~Enemy() { destroy_bitmap(bmp); }

        int t,x;
        fixed y,speed;
        Func f,bx[3],by[3];
        int rx,rbx[3],rby[3];
        int color;
        BITMAP *bmp;
    };

    CEnemy(int num) : numenemies(num),level(NUM_ENEMIES) { enemies=new Enemy [num];
        tmp=create_bitmap_ex(32,ENEMY_SIZE,ENEMY_SIZE); }
    ~CEnemy() { delete[] enemies; destroy_bitmap(tmp); }

    void Update(fixed speed);
    void Draw(BITMAP *buf);

    bool Collides(int x, int y);
    bool KillAt(int x, int y);
    int  KillAll();

    void Advance() { level++; }
    void SetLevel(int l) { level=l; }
    void Reset() { for(int i=0; i<numenemies; i++) enemies[i].t=UNUSED; level=5; }

 private:

    static float Zero(int) { return 0.0; }
    static float Sine(int t) { return sin((float)t/30*PI); }
    static float Cosine(int t) { return cos((float)t/30*PI); }
    static float Sine2(int t) { return sin((float)t/60*PI); }
    static float Cosine2(int t) { return cos((float)t/60*PI); }
    static float Sine3(int t) { return sin((float)t/90*PI); }
    static float Cosine3(int t) { return cos((float)t/90*PI); }

    static const Func patterns[];

    int numenemies;
    Enemy *enemies;
    int level;

    BITMAP *tmp;
};

const CEnemy::Func CEnemy::patterns[]=
{
    CEnemy::Zero,
    CEnemy::Sine3, CEnemy::Cosine3, 
    CEnemy::Sine2, CEnemy::Cosine2,
    CEnemy::Sine, CEnemy::Cosine,
};

void CEnemy::Update(fixed speed)
{
    int c=0;
    Enemy *p=enemies;
    for(int i=numenemies; i; i--,p++)
        if(p->t!=UNUSED)
        {
            p->y+=speed+p->speed;
            if(fixtoi(p->y)>=SCREEN_H+ENEMY_SIZE/2) { p->t=UNUSED; continue; }

            p->rx=p->x+(int)(SCREEN_W/8*p->f(p->t));
            for(int j=0; j<3; j++)
            {
                p->rbx[j]=(int)(ENEMY_SIZE/4*(p->bx[j])(p->t));
                p->rby[j]=(int)(ENEMY_SIZE/4*(p->by[j])(p->t));
            }
            p->t++;
            c++;
        }

    if(c<level)
    {
        p=enemies;
        for(int i=numenemies; i; i--,p++)
            if(p->t==UNUSED)
            {
                p->t=0;
                p->x=rand()%SCREEN_W;
                p->y=-ENEMY_SIZE/2;
                p->speed=(rand()&0x3ffff)-0x1ffff;
                int m=MIN(level,(int)(sizeof(patterns)/sizeof(Func)));
                p->f=patterns[rand()%m];
                for(int j=0; j<3; j++)
                {
                    m=(int)(sizeof(patterns)/sizeof(Func));
                    p->bx[j]=patterns[rand()%m];
                    p->by[j]=patterns[rand()%m];
                }
                m=rand()%6;
                switch(m)
                {   case 0: p->color=makecol32(255,0,0); break;
                    case 1: p->color=makecol32(0,255,0); break;
                    case 2: p->color=makecol32(0,0,255); break;
                    case 3: p->color=makecol32(255,255,0); break;
                    case 4: p->color=makecol32(0,255,255); break;
                    case 5: p->color=makecol32(255,0,255); break;
                }
                break;
            }
    }
}

void CEnemy::Draw(BITMAP *buf)
{
    Enemy *p=enemies;
    for(int i=numenemies; i; i--,p++)
        if(p->t!=UNUSED && p->t!=0)
        {
            unsigned char *ptr=(unsigned char *)p->bmp->dat;
            for(int y=0; y<ENEMY_SIZE; y++)
                for(int x=0; x<ENEMY_SIZE; x++)
                {
                    float sum=0.0;
                    for(int j=0; j<3; j++)
                        sum+=(float)M(ENEMY_BLOB_SIZE)/
                            (M(p->rbx[j]-x+ENEMY_SIZE/2)+
                            M(p->rby[j]-y+ENEMY_SIZE/2));

                    if(sum>=1.0) *ptr=255-(int)(255.0/sum);
                    else *ptr=0;
                    ptr++;
                }

            clear_to_color(tmp,p->color);
            set_write_alpha_blender();
            draw_trans_sprite(tmp,p->bmp,0,0);
            set_alpha_blender();
            draw_trans_sprite(buf,tmp,p->rx-ENEMY_SIZE/2,fixtoi(p->y)-ENEMY_SIZE/2);
            //rect(buf,p->rx-ENEMY_SIZE/2,fixtoi(p->y)-ENEMY_SIZE/2,p->rx+ENEMY_SIZE/2,fixtoi(p->y)+ENEMY_SIZE/2,-1);
            //text_mode(-1);
            //textprintf(buf,font,p->rx,fixtoi(p->y),-1,"%d",numenemies-i);
            Dirty(p->rx-ENEMY_SIZE/2,fixtoi(p->y)-ENEMY_SIZE/2,ENEMY_SIZE,ENEMY_SIZE);
        }
}

bool CEnemy::Collides(int x, int y)
{
    Enemy *p=enemies;
    for(int i=numenemies; i; i--,p++)
        if(p->t!=UNUSED && p->t!=0)
        {
            if(getpixel(p->bmp,x-p->rx+ENEMY_SIZE/2,y-fixtoi(p->y)+ENEMY_SIZE/2)>0)
                return true;
        }

    return false;
}

bool CEnemy::KillAt(int x, int y)
{
    bool ret=false;
    Enemy *p=enemies;
    for(int i=numenemies; i; i--,p++)
        if(p->t!=UNUSED && p->t!=0)
        {
            if(getpixel(p->bmp,x-p->rx+ENEMY_SIZE/2,y-fixtoi(p->y)+ENEMY_SIZE/2)>0)
            {
                parts->Kaboom((float)p->rx,fixtof(p->y),64,5);
                p->t=UNUSED;
                ret=true;
            }
        }

    return ret;
}

int  CEnemy::KillAll()
{
    int c=0;
    Enemy *p=enemies;
    for(int i=numenemies; i; i--,p++)
        if(p->t!=UNUSED && p->t!=0)
        {
            parts->Kaboom((float)p->rx,fixtof(p->y),64,5);
            p->t=UNUSED;
            c++;
        }

    level=0;
    return c;
}



class CPlayer
{
 public:

    class Ship
    {
     public:

        Ship() : x(itofix(SCREEN_W/2)),y(itofix(SCREEN_H/2)),xspeed(0),yspeed(0),bmp(NULL) {}

        fixed x,y,xspeed,yspeed;
        BITMAP *bmp;
    };

    CPlayer();
    ~CPlayer();

    fixed Update();//returns speed
    void Draw(BITMAP *buf);

 private:

    Ship ship;
    int score, moved, lives, level;
};

CPlayer::CPlayer() : ship(),score(0),moved(0),lives(5),level(0)
{
    ship.bmp=create_bitmap(40,40);
    clear_to_color(ship.bmp,makecol(255,0,255));
    triangle(ship.bmp,20,0,10,20,30,20,makecol(255,128,128));
    triangle(ship.bmp,0,20,10,20,10,39,makecol(192,128,128));
    triangle(ship.bmp,39,20,30,20,30,39,makecol(192,128,128));
}

CPlayer::~CPlayer()
{
    destroy_bitmap(ship.bmp);
}

fixed CPlayer::Update()
{
    ship.yspeed-=VERT_SLOWDOWN;
    if(ship.yspeed<itofix(4)) ship.yspeed=itofix(4);
    if(ship.yspeed>itofix(20)) ship.yspeed=itofix(20);

    if(ship.xspeed>=HORIZ_SLOWDOWN) ship.xspeed-=HORIZ_SLOWDOWN;
    else if(ship.xspeed<=HORIZ_SLOWDOWN) ship.xspeed+=HORIZ_SLOWDOWN;
    else ship.xspeed=0;

    ship.x+=ship.xspeed;
    if(fixtoi(ship.x)<0-ship.bmp->w/2) ship.x+=itofix(SCREEN_W+ship.bmp->w);
    if(fixtoi(ship.x)>=SCREEN_W+ship.bmp->w/2) ship.x-=itofix(SCREEN_W+ship.bmp->w);

    if(lives>=0)
    {
        if(key[KEY_LEFT]) ship.xspeed-=HORIZ_ACCEL;
        if(key[KEY_RIGHT]) ship.xspeed+=HORIZ_ACCEL;
        if(key[KEY_UP])
        {
            ship.yspeed+=VERT_ACCEL;
            parts->Kaboom(fixtof(ship.x),fixtof(ship.y)-15,3,1.0,0.0,PI);
        }
        if(key[KEY_DOWN])
        {
            ship.yspeed-=VERT_ACCEL;
            parts->Kaboom(fixtof(ship.x)-15,fixtof(ship.y),1,fixtof(ship.yspeed*2),1.4*PI,1.6*PI);
            parts->Kaboom(fixtof(ship.x)+15,fixtof(ship.y),1,fixtof(ship.yspeed*2),1.4*PI,1.6*PI);
        }
        if(enemies->KillAt(fixtoi(ship.x),fixtoi(ship.y)))
        {
            if(--lives<0)
            {
                ship.yspeed=itofix(4);
                ship.x=itofix(SCREEN_W/2);
                parts->Kaboom(fixtof(ship.x),fixtof(ship.y),128,20);
                parts->Kaboom(SCREEN_W/2,SCREEN_H/2,256,20);
                enemies->KillAll();
                hiscore->InitDialog(score);
            }
        }

        score+=level*(fixtoi(ship.yspeed)*fixtoi(ship.yspeed))/16;
        moved+=fixtoi(ship.yspeed);
        if(moved>1000) { enemies->Advance(); level++; moved=0; }
    }

    return ship.yspeed;
}

void CPlayer::Draw(BITMAP *buf)
{
    if(lives>=0)
        draw_sprite(buf,ship.bmp,fixtoi(ship.x)-ship.bmp->w/2,fixtoi(ship.y)-ship.bmp->h/2);
    text_mode(-1);
    textprintf_centre(buf,font,SCREEN_W/2,20,makecol(255,255,255)," Score: %d ",score);

    if(level==0) hiscore->Draw(buf);

    if(lives<0)
    {
        if(!hiscore->UpdateDialog(buf))
        {
            level=score=moved=0;
            lives=5;
            enemies->Reset();
        }
    }
}



//BITMAP *buffer;
volatile int speed_counter;
//volatile int fps,frame;

void inc_speed_counter(void)
{
    speed_counter++;
}
END_OF_FUNCTION(inc_speed_counter);

/*void fps_proc(void)
{
    fps=frame;
    frame=0;
}
END_OF_FUNCTION(fps_proc);

int main()
{
    srand((unsigned)time(NULL));

    allegro_init();
    install_keyboard();

    install_timer();
    LOCK_FUNCTION(inc_speed_counter);
    LOCK_VARIABLE(speed_counter);
    LOCK_FUNCTION(fps_proc);
    LOCK_VARIABLE(fps);
    LOCK_VARIABLE(frame);

    set_color_depth(16);
    if(set_gfx_mode(GFX_AUTODETECT,640,480,0,0)<0)
    {
        set_color_depth(15);
        if(set_gfx_mode(GFX_AUTODETECT,640,480,0,0)<0)
        {
            set_gfx_mode(GFX_TEXT,0,0,0,0);
            allegro_message("Error setting graphics mode: %s.",allegro_error);
            return 1;
        }
    }

    buffer=create_bitmap(SCREEN_W,SCREEN_H);

    hiscore = new CHiscore;
    if(!hiscore->Load()) hiscore->Reset();

    stars = new CStar(1024);
    parts = new CParticle(1024);
    enemies = new CEnemy(32);
    player = new CPlayer;

    install_int_ex(inc_speed_counter,BPS_TO_TIMER(25));
    install_int(fps_proc,1000);

    while(!key[KEY_ESC])
    {
        frame++;

        while(speed_counter>0)
        {
            fixed speed=player->Update();
            stars->Update(speed);
            parts->Update(speed);
            enemies->Update(speed);

            speed_counter--;
        }

        clear_bitmap(buffer);

        stars->Draw(buffer);
        enemies->Draw(buffer);
        player->Draw(buffer);
        parts->Draw(buffer);

        text_mode(-1);
        textprintf(buffer,font,0,0,makecol(255,255,255),"%d fps",fps);
        textout_centre(buffer,font,"Welcome to BlobLand",SCREEN_W/2,0,makecol(255,255,255));
        blit(buffer,screen,0,0,0,0,SCREEN_W,SCREEN_H);
    }

    hiscore->Save();
    delete hiscore;

    delete stars;
    delete parts;
    delete enemies;
    delete player;

    return 0;
}
END_OF_MAIN();*/

extern "C"
void blob_init()
{
    hiscore = new CHiscore;
    if(!hiscore->Load()) hiscore->Reset();

    //stars = new CStar(1024);
    parts = new CParticle(1024);
    enemies = new CEnemy(NUM_ENEMIES);
    //player = new CPlayer;

    LOCK_FUNCTION(inc_speed_counter);
    LOCK_VARIABLE(speed_counter);

    install_int_ex(inc_speed_counter,BPS_TO_TIMER(25));
}

extern "C"
void blob_shutdown()
{
    remove_int(inc_speed_counter);

    hiscore->Save();
    delete hiscore;

    //delete stars;
    delete parts;
    delete enemies;
    //delete player;
}

extern "C"
void blob_drawall(BITMAP *buffer)
{
    //stars->Draw(buffer);
    enemies->Draw(buffer);
    //player->Draw(buffer);
    parts->Draw(buffer);

    text_mode(-1);
    textout_centre(buffer,font,"Welcome to BlobLand II",SCREEN_W/2,10,makecol(255,255,255));
    Dirty(SCREEN_W/2-100,10,200,8);
}

extern "C"
void blob_update(fixed speed)
{
    while(speed_counter>0)
    {
        //fixed speed=player->Update();
        //stars->Update(speed);
        parts->Update(speed);
        enemies->SetLevel(NUM_ENEMIES);
        enemies->Update(speed);

        speed_counter--;
    }
}

extern "C"
void blob_kaboom(float x,float y,float xs,float ys,int n,float power)
{
    parts->Kaboom(x,y,xs,ys,n,power);
}

extern "C"
int  blob_killat(int x, int y)
{
    return enemies->KillAt(x,y)?TRUE:FALSE;
}

extern "C"
int blob_killall()
{
    return enemies->KillAll();
}

extern "C"
void blob_hiscore_init(int score)
{
    clear_keybuf();
    hiscore->InitDialog(score);
}

extern "C"
int blob_hiscore_update(BITMAP *b)
{
    return hiscore->UpdateDialog(b)?TRUE:FALSE;
}

extern "C"
void blob_hiscore_disp(BITMAP *b)
{
    hiscore->Draw(b);
}
