/* Author: Tobi Vollebregt */

/*  Bataafje -- A small game written for the Allegro SpeedHack 2003
 *  Copyright (C) 2003  Tobi Vollebregt
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  See `License.txt', which contains a verbatim copy of the
 *  GNU General Public License, for details.
 *
 *  Please send your reaction to: tobivollebregt@hotmail.com
 */

// This file implements the player objects

#include <allegro.h>
#include <math.h>
#include "anim.h"
#include "object.h"

#ifndef PI
# define PI 3.14159263
#endif

float global_vx, global_vy;

static int inc_ammo[16];

void PlayerIncAmmo (int w)
{
    ASSERT (w >= 0 && w < 16);
    ++inc_ammo[w];
}

class CPlayer : public IObject
{
    NO_COPY (CPlayer);
    DECLARE_CLASS (CPlayer);

 public:

    CPlayer () { anim=0; showweapon=0; }
    ~CPlayer () { if (anim) delete anim; if (showweapon) delete showweapon; }

    // this does also display a menu
    bool Initialize (int ix);

    void Draw (BITMAP* d, int dx);
    bool Update (int ms);

 private:

    CAnimation* anim;
    bool flip;
    int speed;
    int weapon;
    int ammo[16];//max 16 weapons
    int throwpressed; // is the throw key pressed? and how long?
    int maxthrow;
    bool nextwpressed; // is the nextweapon key pressed?
    bool prevwpressed; // prevweapon
    CAnimation* showweapon; // to display our selected weapon
    int angle;
    int angle_dist, angle_dist2;
};

IMPLEMENT_CLASS (CPlayer);

// callback for d_list_proc
static const char* GetPlayerText (int index, int* list_size)
{
    if (index < 0) { *list_size = get_config_int ("player.names", "N", 0); return 0; }
    char tmp[32];
    uszprintf (tmp, sizeof (tmp), "%d", index);
    return get_config_string ("player.names", tmp, 0);
}

static int g_control[7];

// callback for d_list_proc
static const char* GetControlText (int index, int* list_size)
{
    static const char* k_control_text[] = 
    {
        "Move Left",
        "Move Right",
        "Throw Object",
        "Next Weapon",
        "Previous Weapon",
        "Aim Higher",
        "Aim Lower",
    };
    static const char* k_key_text[] =
    {
        "no key",
        "KEY_A",
        "KEY_B",
        "KEY_C",
        "KEY_D",
        "KEY_E",
        "KEY_F",
        "KEY_G",
        "KEY_H",
        "KEY_I",
        "KEY_J",
        "KEY_K",
        "KEY_L",
        "KEY_M",
        "KEY_N",
        "KEY_O",
        "KEY_P",
        "KEY_Q",
        "KEY_R",
        "KEY_S",
        "KEY_T",
        "KEY_U",
        "KEY_V",
        "KEY_W",
        "KEY_X",
        "KEY_Y",
        "KEY_Z",
        "KEY_0",
        "KEY_1",
        "KEY_2",
        "KEY_3",
        "KEY_4",
        "KEY_5",
        "KEY_6",
        "KEY_7",
        "KEY_8",
        "KEY_9",
        "KEY_0_PAD",
        "KEY_1_PAD",
        "KEY_2_PAD",
        "KEY_3_PAD",
        "KEY_4_PAD",
        "KEY_5_PAD",
        "KEY_6_PAD",
        "KEY_7_PAD",
        "KEY_8_PAD",
        "KEY_9_PAD",
        "KEY_F1",
        "KEY_F2",
        "KEY_F3",
        "KEY_F4",
        "KEY_F5",
        "KEY_F6",
        "KEY_F7",
        "KEY_F8",
        "KEY_F9",
        "KEY_F10",
        "KEY_F11",
        "KEY_F12",
        "KEY_ESC",
        "KEY_TILDE",
        "KEY_MINUS",
        "KEY_EQUALS",
        "KEY_BACKSPACE",
        "KEY_TAB",
        "KEY_OPENBRACE",
        "KEY_CLOSEBRACE",
        "KEY_ENTER",
        "KEY_COLON",
        "KEY_QUOTE",
        "KEY_BACKSLASH",
        "KEY_BACKSLASH2",
        "KEY_COMMA",
        "KEY_STOP",
        "KEY_SLASH",
        "KEY_SPACE",
        "KEY_INSERT",
        "KEY_DEL",
        "KEY_HOME",
        "KEY_END",
        "KEY_PGUP",
        "KEY_PGDN",
        "KEY_LEFT",
        "KEY_RIGHT",
        "KEY_UP",
        "KEY_DOWN",
        "KEY_SLASH_PAD",
        "KEY_ASTERISK",
        "KEY_MINUS_PAD",
        "KEY_PLUS_PAD",
        "KEY_DEL_PAD",
        "KEY_ENTER_PAD",
        "KEY_PRTSCR",
        "KEY_PAUSE",
        "KEY_ABNT_C1",
        "KEY_YEN",
        "KEY_KANA",
        "KEY_CONVERT",
        "KEY_NOCONVERT",
        "KEY_AT",
        "KEY_CIRCUMFLEX",
        "KEY_COLON2",
        "KEY_KANJI",
        "KEY_LSHIFT",
        "KEY_RSHIFT",
        "KEY_LCONTROL",
        "KEY_RCONTROL",
        "KEY_ALT",
        "KEY_ALTGR",
        "KEY_LWIN",
        "KEY_RWIN",
        "KEY_MENU",
        "KEY_SCRLOCK",
        "KEY_NUMLOCK",
        "KEY_CAPSLOCK"
    };

    static char buf[128];
    if (index < 0) { *list_size = sizeof (k_control_text) / sizeof (k_control_text[0]); return 0; }
    uszprintf (buf, sizeof (buf), "%15s - %s", k_control_text[index], k_key_text[g_control[index]]);
    return buf;
}

static int get_config_color (const char* sect, int rdef, int gdef, int bdef)
{
    return makecol (get_config_int (sect, "r", rdef),
        get_config_int (sect, "g", gdef), get_config_int (sect, "b", bdef));
}

static int BindNewKey ()
{
    text_mode (gui_bg_color);
    textout_centre (screen, font, "PRESS A KEY", SCREEN_W / 2, SCREEN_H - 10, gui_fg_color);

    for (;;)
    {
        for (int i = 1; i < KEY_MAX; ++i)
            if (key[i]) return i;
    }
}

bool CPlayer::Initialize (int ix)
{
    static DIALOG dlg[] =
    /* int (*proc)(), int x, y, w, h, fg, bg, key, flags, d1, d2, void *dp, *dp2, *dp3 */
    {
        { d_clear_proc, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, NULL, NULL, NULL },
        { d_list_proc, 0, 20, 320, 40, 0, 0, 0, 0, 0, 0, (void*) GetPlayerText, NULL, NULL },
        { d_list_proc, 0, 100, 320, 65, 0, 0, 0, D_EXIT, 0, 0, (void*) GetControlText, NULL, NULL },
        { d_ctext_proc, 160, 0, 0, 0, 0, 0, 0, 0, 0, 0, (void*) "Select your hero:", NULL, NULL },
        { d_ctext_proc, 160, 70, 0, 0, 0, 0, 0, 0, 0, 0, (void*) "Select your controls:", NULL, NULL },
        { d_ctext_proc, 160, 80, 0, 0, 0, 0, 0, 0, 0, 0, (void*) "(doubleclick to change)", NULL, NULL },
        //{ d_check_proc, 0, 175, 320, 10, 0, 0, 0, 0, 1, 0, (void*) "I want extra large font because I can't read this", NULL, NULL },
        { d_button_proc, 0, 200, 150, 20, 0, 0, 13, D_EXIT, 0, 0, (void*) "Ok", NULL, NULL },
        { d_button_proc, 170, 200, 150, 20, 0, 0, 'c', D_EXIT, 0, 0, (void*) "&Cancel", NULL, NULL },
        { d_ctext_proc, 160, 230, 0, 0, 0, 0, 0, 0, 0, 0, (void*) "Programming by Tobi Vollebregt", NULL, NULL },
        { d_ctext_proc, 160, 240, 0, 0, 0, 0, 0, 0, 0, 0, (void*) "2D Graphics by Duco Vollebregt", NULL, NULL },
        { NULL, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, NULL, NULL, NULL }
    };

    g_control[0] = get_config_int ("controls", "move_left", 0);
    g_control[1] = get_config_int ("controls", "move_right", 0);
    g_control[2] = get_config_int ("controls", "throw_object", 0);
    g_control[3] = get_config_int ("controls", "next_weapon", 0);
    g_control[4] = get_config_int ("controls", "previous_weapon", 0);
    g_control[5] = get_config_int ("controls", "inc_angle", 0);
    g_control[6] = get_config_int ("controls", "dec_angle", 0);

    gui_fg_color = get_config_color ("gui_fg_color", 255, 255, 255);
    gui_mg_color = get_config_color ("gui_mg_color", 127, 127, 127);
    gui_bg_color = get_config_color ("gui_bg_color", 0, 0, 0);
    set_dialog_color (dlg, gui_fg_color, gui_bg_color);
    centre_dialog (dlg);

    int ret;
    do
    {
        ret = do_dialog (dlg, -1);
        switch (ret)
        {
            case 2:
            {
                TRACE ("doubleclick on control select dialog\n");
                g_control[dlg[2].d1] = BindNewKey ();
                break;
            }
            case -1:
            case 7: // cancel button
                return false;
        }
    }
    while (ret != 6);

    set_config_int ("controls", "move_left", g_control[0]);
    set_config_int ("controls", "move_right", g_control[1]);
    set_config_int ("controls", "throw_object", g_control[2]);
    set_config_int ("controls", "next_weapon", g_control[3]);
    set_config_int ("controls", "previous_weapon", g_control[4]);
    set_config_int ("controls", "inc_angle", g_control[5]);
    set_config_int ("controls", "dec_angle", g_control[6]);

    //if (dlg[6].flags & D_SELECTED) LoadBigFont ();

    speed = get_config_int (0, "player.speed", 1);
    x = ix;
    flip = true;
    weapon = 0;
    throwpressed=false;

    const char* s = get_config_string ("player.throws", "0", 0);
    if (!s) return false;
    if (!(s = ustrchr (s, ':'))) FatalError ("Expected colon");
    showweapon = new CAnimation (s + 1);

    char tmp[32];

    for (int i = 0; i < 16; ++i)
    {
        uszprintf (tmp, sizeof (tmp), "%d", i);
        ammo[i] = get_config_int ("player.throws", tmp, 0);
    }

    uszprintf (tmp, sizeof (tmp), "%d", dlg[1].d1);
    s = get_config_string ("player.animations", tmp, 0);
    if (!s) return false;
    anim = new CAnimation (s);
    angle = 45;
    maxthrow = get_config_int (0, "player.maxthrow", 1);
    angle_dist = get_config_int (0, "player.angle_dist", 0);
    angle_dist2 = get_config_int (0, "player.angle_dist2", 0);

    return true;
}

void CPlayer::Draw (BITMAP* d, int dx)
{
    anim->Draw (d, int (x) - anim->GetWidth () / 2 - dx,
        SCREEN_H - anim->GetHeight (), flip);
    int ax = int (x - dx + angle_dist * cos (angle * PI / 180.0) * throwpressed / maxthrow);
    int ay = int (SCREEN_H - anim->GetHeight () / 2 - angle_dist * sin (angle * PI / 180.0) * throwpressed / maxthrow);
    int ax2 = int (x - dx + angle_dist2 * cos (angle * PI / 180.0) * throwpressed / maxthrow);
    int ay2 = int (SCREEN_H - anim->GetHeight () / 2 - angle_dist2 * sin (angle * PI / 180.0) * throwpressed / maxthrow);
    line (d, ax, ay, ax2, ay2, 0);

    drawing_mode(DRAW_MODE_TRANS,0,0,0);
    set_trans_blender (0,0,0,128);
    rectfill (d, 10, 10, showweapon->GetWidth () + 30, showweapon->GetHeight () + 30, 0);
    solid_mode ();
    showweapon->Draw (d, 20, 20);
    textprintf_centre (d, game_font, 20 + showweapon->GetWidth () / 2, 11, makecol (255, 255, 255), "%dx", ammo[weapon]);
}

bool CPlayer::Update (int ms)
{
    bool walking = false;

    showweapon->Update (ms);

    for (int i =0; i < 16; ++i)
    {
        ammo[i] += inc_ammo[i];
        inc_ammo[i] = 0;
    }

    // walk left
    if (key[g_control[0]])
    {
        walking = true;
        flip = false;
        x -= speed;
    }

    // walk right
    if (key[g_control[1]])
    {
        walking = true;
        flip = true;
        x += speed;
    }

    // throw object
    if (key[g_control[2]])
    {
        if (throwpressed < maxthrow) ++throwpressed;
    }
    else
    {
        if (throwpressed && ammo[weapon] > 0)
        {
            char tmp[32];
            uszprintf (tmp, sizeof (tmp), "%d", weapon);
            const char* s = get_config_string ("player.throws", tmp, 0);
            if (s && (s = ustrchr (s, ':')))
            {
                IObject* obj = IObject::New ("CBullet");
                if (!obj || !obj->Initialize2 (x + anim->GetWidth () / 3, SCREEN_H - anim->GetHeight () / 2))
                    FatalError ("Couldn't initialize bullet");
                // a good example of bad coding:
                const char* name = ustrchr(s, '#');
                if (!name) name = s+1; else ++name;
                global_vx = get_config_float (name, "vx", 0.0) * cos (angle * PI / 180.0) * throwpressed / maxthrow;
                global_vy = get_config_float (name, "vy", 0.0) * -sin (angle * PI / 180.0) * throwpressed / maxthrow;
                //TRACE("%g; %g\n", global_vx, global_vy);
                obj->SetAnimation (s+1);
                AddObject (obj);
                --ammo[weapon];
            }
        }
        throwpressed=0;
    }

    // next weapon
    if (key[g_control[3]])
    {
        if (!nextwpressed)
        {
            char tmp[32];
            uszprintf (tmp, sizeof (tmp), "%d", weapon + 1);
            const char* s = get_config_string ("player.throws", tmp, 0);
            if (s && (s = ustrchr (s, ':')))
            {
                ++weapon;
                delete showweapon;
                showweapon = new CAnimation (s + 1);
            }
            nextwpressed=true;
        }
    }
    else nextwpressed = false;

    // previous weapon
    if (key[g_control[4]])
    {
        if (!prevwpressed)
        {
            char tmp[32];
            uszprintf (tmp, sizeof (tmp), "%d", weapon - 1);
            const char* s = get_config_string ("player.throws", tmp, 0);
            if (s && (s = ustrchr (s, ':')))
            {
                --weapon;
                delete showweapon;
                showweapon = new CAnimation (s + 1);
            }
            prevwpressed=true;
        }
    }
    else prevwpressed = false;

    // increase angle
    if (key[g_control[5]])
    {
        if (angle < 90) ++angle;
    }
    // decrease angle
    if (key[g_control[6]])
    {
        if (angle > 0) --angle;
    }

    if (walking) anim->Update (ms);
    return true;
}
