/* Author: Tobi Vollebregt */

/*  Bataafje -- A small game written for the Allegro SpeedHack 2003
 *  Copyright (C) 2003  Tobi Vollebregt
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  See `License.txt', which contains a verbatim copy of the
 *  GNU General Public License, for details.
 *
 *  Please send your reaction to: tobivollebregt@hotmail.com
 */

#ifndef __tjv_OBJECT_H__
#define __tjv_OBJECT_H__

#include "debug.h"

// a object is almost anything in the game except the map
//-- cheeses, ducks, your character, etc.

class IObject
{
    NO_COPY (IObject);

 public:

    // this is a system to register all objects at program startup,
    // so that each object can be created from a string containing its name.
    // (using the IObject::New()-function)
    struct Class
    {
        char name[32];
        Class* next;
        // A global function that creates an instance of the object.
        IObject* (*creator) ();
    };
    // Each object has a static Class object. At program startup, all
    // Class objects are linked to the linked list "classes".
    static Class* classes;

    static void DumpClassList ();    // debugging function
    static IObject* New (const char* name); // create instance of class <name>

 public:

    IObject () : next (0) {}
    virtual ~IObject () {}

    virtual bool Initialize (int x) = 0;
    virtual bool Initialize2 (int, int) { return false; }
    virtual void Draw (BITMAP* d, int x) = 0;
    // Returns false if the object wants to be destroyed.
    virtual bool Update (int ms) = 0;

    virtual void SetAnimation (const char*) {}
    
    // My machine hangs using dynamic_cast, so had to do it another way
    virtual bool IsEnemy () const { return false; }

    IObject* next; // to make a linked list of objects in the sectors

 protected:

    int x;
};

// Within the class declaration there must be DECLARE_CLASS(<classname>)
#define DECLARE_CLASS(cl) \
    private: static IObject* New##cl (); \
        static Class info; \
        friend struct tjv_Ctor##cl;

// somewhere must be the implementation: IMPLEMENT_CLASS(<classname>)
#define IMPLEMENT_CLASS(cl) \
    IObject* cl::New##cl () { return new cl; } \
    IObject::Class cl::info = { #cl, 0, 0 }; \
    static struct tjv_Ctor##cl { \
        tjv_Ctor##cl () { \
            cl::info.next = IObject::classes; \
            IObject::classes = &cl::info; \
            cl::info.creator = cl::New##cl; \
        }   \
    } tjv_Ctor##cl##_obj;

#endif
