/* Author: Tobi Vollebregt */

/*  Bataafje -- A small game written for the Allegro SpeedHack 2003
 *  Copyright (C) 2003  Tobi Vollebregt
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  See `License.txt', which contains a verbatim copy of the
 *  GNU General Public License, for details.
 *
 *  Please send your reaction to: tobivollebregt@hotmail.com
 */

// ripped this code from another game of myself and
// changed it to use it inside an existing allegro environment

#include <stdarg.h>
#include <string.h>
#include <time.h>
#include <allegro.h>

#ifdef DEBUGMODE
#define ASSERTP(nmap,x,y)  ASSERT((x)>=0 && (y)>=0 && (x)<(nmap)->w && (y)<(nmap)->h);
#else
#define ASSERTP(nmap,x,y)
#endif

typedef enum {false,true} bool;
#define SWAP(a,b)   {temp=a;a=b;b=temp;}

#ifdef ALLEGRO_DJGPP
#define srand       srandom
#define rand        random
#endif

#define D_WEST      1
#define D_NORTH     2
#define D_EAST      3
#define D_SOUTH     4

#define NIBBLE_OK       0
#define NIBBLE_SUCCESS  1
#define NIBBLE_FAILURE  2

typedef struct POINT {int x,y;} POINT;

typedef struct NIBBLE
{   int id,direction,died;
        int length,tolength;
        POINT p[512];
} NIBBLE;

typedef struct NMAP
{   int w,h;
        int xscale,yscale;
        struct {int count;POINT p,prevp;} food;
        unsigned char *dat;
        unsigned char *prevdat;
        unsigned char *line[0];
} NMAP;

void exitmsg(const char *fmt, ...);
void restart_game(bool,bool);

NMAP *the_nmap=NULL;
NIBBLE *the_nibble=NULL,*the_nibble2=NULL,*the_nibble3=NULL;
int player_lives,player2_lives,nibbles_level;
int player_score,player2_score,nibbles_speed=20;
bool player_ai=false,player2_ai=false;
int nibble_startx,nibble_starty,nibble_startdir;
bool nibble_wrap=false;
int vcard=0,vwidth=640,vheight=480,vdepth=8;
PALETTE palette;

NMAP *create_nmap(int w, int h, int xs, int ys)
{
        NMAP *nmap;
        int y;

        nmap=(NMAP *)malloc(sizeof(NMAP)+h*sizeof(unsigned char *));
        if(!nmap) return NULL;
        memset(nmap,0,sizeof(NMAP)+h*sizeof(unsigned char *));

        nmap->dat=(unsigned char *)malloc(2*w*h);
        if(!nmap->dat) {free(nmap); return NULL;}
        memset(nmap->dat,0,2*w*h);
        nmap->prevdat=nmap->dat+w*h;

        nmap->w=w;
        nmap->h=h;
        nmap->xscale=xs;
        nmap->yscale=ys;

        nmap->food.count=0;
        nmap->food.p.x=nmap->food.p.y=0;
        nmap->food.prevp.x=nmap->food.prevp.y=0;

        for(y=0; y<h; y++) nmap->line[y]=&nmap->dat[y*w];

        return nmap;
}

void destroy_nmap(NMAP *nmap)
{
        if(nmap)
        {   if(nmap->dat) free(nmap->dat);
                free(nmap);
        }
}

void draw_nmap(NMAP *nmap)
{
        int x,y,sx,sy,c;
        unsigned char *ptr,*pptr;
        char buf[4]={0,0,0,0};

        if(bitmap_color_depth(screen)==8)
        {   for(y=0,sy=16; y<nmap->h; y++,sy+=nmap->yscale)
                {   ptr=nmap->line[y];
                        pptr=&nmap->prevdat[y*nmap->w];
                        for(x=0,sx=16; x<nmap->w; x++,sx+=nmap->xscale,ptr++,pptr++)
                        {   if(*ptr!=*pptr)
                                {   *pptr=*ptr;
                                        rectfill(screen,sx,sy,sx+nmap->xscale-1,sy+nmap->yscale-1,*ptr);
                                }
                        }
                }
        }else
        {   for(y=0,sy=16; y<nmap->h; y++,sy+=nmap->yscale)
                {   ptr=nmap->line[y];
                        pptr=&nmap->prevdat[y*nmap->w];
                        for(x=0,sx=16; x<nmap->w; x++,sx+=nmap->xscale,ptr++,pptr++)
                        {   if(*ptr!=*pptr)
                                {   *pptr=*ptr;
                                        rectfill(screen,sx,sy,sx+nmap->xscale-1,sy+nmap->yscale-1,
                                                makecol(palette[*ptr].r<<2,palette[*ptr].g<<2,palette[*ptr].b<<2));
                                }
                        }
                }
        }
        if(nmap->food.p.x!=nmap->food.prevp.x || nmap->food.p.y!=nmap->food.prevp.y)
        {   sx=16+nmap->food.prevp.x*nmap->xscale;
                sy=16+nmap->food.prevp.y*nmap->yscale;
                if(bitmap_color_depth(screen)==8)
                        rectfill(screen,sx,sy,sx+nmap->xscale-1,sy+nmap->yscale-1,
                                nmap->line[nmap->food.prevp.y][nmap->food.prevp.x]);
                else
                {   c=nmap->line[nmap->food.prevp.y][nmap->food.prevp.x];
                        rectfill(screen,sx,sy,sx+nmap->xscale-1,sy+nmap->yscale-1,
                                makecol(palette[c].r<<2,palette[c].g<<2,palette[c].b<<2));
                }
                text_mode(-1);
                *buf='0'+nmap->food.count;
                textout(screen,font,buf,
                        16+nmap->food.p.x*nmap->xscale+(nmap->xscale-text_length(font,buf))/2,
                        16+nmap->food.p.y*nmap->yscale+(nmap->yscale-text_height(font))/2,
                        makecol(255,255,255));
                nmap->food.prevp=nmap->food.p;
        }
}

void create_new_food(NMAP *nmap)
{
        do
        {   nmap->food.p.x=rand()%nmap->w;
                nmap->food.p.y=rand()%nmap->h;
        }while(nmap->line[nmap->food.p.y][nmap->food.p.x]!=0);
        nmap->food.count++;
}

int has_food(NMAP *nmap, int x, int y)
{
        if(nmap->food.p.x==x && nmap->food.p.y==y) return 1;
        return 0;
}

void nmap_hline(NMAP *nmap, int x1, int y, int x2)
{
        int xbeg,xend,x,c,temp;

        if(x1>x2) SWAP(x1,x2);

        xbeg=x1;
        xend=(x2+x1)/2;
        for(x=xbeg; x<xend; x++)
        {   c=63+192*(x-xbeg)/(xend-xbeg);
                nmap->line[y][x]=makecol8(c,c,c);
        }
        xbeg=xend;
        xend=x2;
        for(x=xbeg; x<xend; x++)
        {   c=255-192*(x-xbeg)/(xend-xbeg);
                nmap->line[y][x]=makecol8(c,c,c);
        }
}

void nmap_vline(NMAP *nmap, int x, int y1, int y2)
{
        int ybeg,yend,y,c,temp;

        if(y1>y2) SWAP(y1,y2);

        ybeg=y1;
        yend=(y2+y1)/2;
        for(y=ybeg; y<yend; y++)
        {   c=63+192*(y-ybeg)/(yend-ybeg);
                nmap->line[y][x]=makecol8(c,c,c);
        }
        ybeg=yend;
        yend=y2;
        for(y=ybeg; y<yend; y++)
        {   c=255-192*(y-ybeg)/(yend-ybeg);
                nmap->line[y][x]=makecol8(c,c,c);
        }
}

bool nmap_hline_empty(NMAP *nmap, int x1, int y, int x2)
{
        int xbeg,xend,x,temp;

        if(x1>x2) SWAP(x1,x2);
        if(x2-x1<6) return false;

        xbeg=x1;
        xend=x2;
        for(x=xbeg; x<xend; x++)
        {   if(nmap->line[y][x]
                        || (x+1<nmap->w && nmap->line[y][x+1])
                        || (x-1>=0 && nmap->line[y][x-1])
                        || (y+1<nmap->h && nmap->line[y+1][x])
                        || (y-1>=0 && nmap->line[y-1][x])) return false;
        }
        return true;
}

bool nmap_vline_empty(NMAP *nmap, int x, int y1, int y2)
{
        int ybeg,yend,y,temp;

        if(y1>y2) SWAP(y1,y2);
        if(y2-y1<6) return false;

        ybeg=y1;
        yend=y2;
        for(y=ybeg; y<yend; y++)
        {   if(nmap->line[y][x]
                        || (x+1<nmap->w && nmap->line[y][x+1])
                        || (x-1>=0 && nmap->line[y][x-1])
                        || (y+1<nmap->h && nmap->line[y+1][x])
                        || (y-1>=0 && nmap->line[y-1][x])) return false;
        }
        return true;
}

void generate_level(NMAP *nmap, int n, int *xbeg, int *ybeg, int *dbeg)
{
        int i,x1,y1,x2,y2;

        memset(nmap->dat,0,nmap->w*nmap->h);

        *xbeg=7*nmap->w/8;
        *ybeg=7*nmap->h/8;
        *dbeg=D_WEST;

        nmap->food.count=0;
        nmap->food.p.x=nmap->food.p.y=0;

        switch(n)
        {   case 1:
                        break;
                case 2:
                        nmap_hline(nmap,nmap->w/4,nmap->h/2,3*nmap->w/4);
                        break;
                case 3:
                        *dbeg=D_NORTH;
                        nmap_vline(nmap,nmap->w/4,nmap->h/4,3*nmap->h/4);
                        nmap_vline(nmap,3*nmap->w/4,nmap->h/4,3*nmap->h/4);
                        break;
                case 4:
                        *dbeg=D_NORTH;
                        nmap_hline(nmap,0,3*nmap->h/4-1,nmap->w/2);
                        nmap_hline(nmap,nmap->w/2,nmap->h/4,nmap->w);
                        nmap_vline(nmap,nmap->w/4,0,nmap->h/2);
                        nmap_vline(nmap,3*nmap->w/4,nmap->h/2,nmap->h);
                        break;
                case 5:
                        nmap_hline(nmap,nmap->w/4,nmap->h/2,3*nmap->w/4);
                        nmap_vline(nmap,nmap->w/4,nmap->h/4,3*nmap->h/4);
                        nmap_vline(nmap,3*nmap->w/4,nmap->h/4,3*nmap->h/4);
                        break;
                case 6:
                        nmap_hline(nmap,nmap->w/4+4,nmap->h/4,3*nmap->w/4-4);
                        nmap_hline(nmap,nmap->w/4+4,3*nmap->h/4,3*nmap->w/4-4);
                        nmap_vline(nmap,nmap->w/4,nmap->h/4+4,3*nmap->h/4-4);
                        nmap_vline(nmap,3*nmap->w/4,nmap->h/4+4,3*nmap->h/4-4);
                        break;
                case 7:
                        nmap_hline(nmap,nmap->w/3,nmap->h/3,3*nmap->w/4);
                        nmap_hline(nmap,nmap->w/4,2*nmap->h/3,2*nmap->w/3);
                        nmap_vline(nmap,nmap->w/3,nmap->h/6,nmap->h/3);
                        nmap_vline(nmap,nmap->w/4,nmap->h/6,2*nmap->h/3);
                        nmap_vline(nmap,2*nmap->w/3,2*nmap->h/3,5*nmap->h/6);
                        nmap_vline(nmap,3*nmap->w/4,nmap->h/3,5*nmap->h/6);
                        break;
                case 8:
                        nmap_hline(nmap,nmap->w/4+4,nmap->h/4,3*nmap->w/4-4);
                        nmap_hline(nmap,nmap->w/4+4,3*nmap->h/4,3*nmap->w/4-4);
                        nmap_vline(nmap,nmap->w/4,nmap->h/4+4,3*nmap->h/4-4);
                        nmap_vline(nmap,3*nmap->w/4,nmap->h/4+4,3*nmap->h/4-4);
                        nmap_vline(nmap,nmap->w/2,nmap->h/4+1,3*nmap->h/4);
                        break;
                case 9:
                        nmap_hline(nmap,nmap->w/4+4,nmap->h/4,3*nmap->w/4-4);
                        nmap_hline(nmap,nmap->w/4+4,3*nmap->h/4,3*nmap->w/4-4);
                        nmap_vline(nmap,nmap->w/4,nmap->h/4+4,3*nmap->h/4-4);
                        nmap_vline(nmap,3*nmap->w/4,nmap->h/4+4,3*nmap->h/4-4);
                        nmap_vline(nmap,nmap->w/2,nmap->h/4+1,3*nmap->h/4);
                        nmap_hline(nmap,nmap->w/4+nmap->w/3/4,nmap->h/2,3*nmap->w/4-nmap->w/3/4);
                        break;
                default:
                        for(i=0; i<n; i++)
                        {   x1=rand()%nmap->w;
                                x2=rand()%nmap->w;
                                y1=rand()%nmap->h;
                                y2=rand()%nmap->h;
                                if(rand()&1) {if(nmap_hline_empty(nmap,x1,y1,x2)) nmap_hline(nmap,x1,y2,x2);}
                                else {if(nmap_vline_empty(nmap,x1,y1,y2)) nmap_vline(nmap,x1,y1,y2);}
                        }
                        break;
        }
}

NIBBLE *create_nibble(NMAP *nmap, int id, int x, int y, int d, int l)
{
        NIBBLE *nibble;

        nibble=(NIBBLE *)malloc(sizeof(NIBBLE));
        if(!nibble) return NULL;
        memset(nibble,0,sizeof(NIBBLE));

        nibble->id=id;
        nibble->died=false;
        nibble->direction=d;
        nibble->tolength=l;
        nibble->length=1;
        nibble->p[0].x=x;
        nibble->p[0].y=y;

        ASSERTP(nmap,x,y);
        nmap->line[y][x]=nibble->id;

        return nibble;
}

void destroy_nibble(NIBBLE *nibble)
{
        if(nibble) free(nibble);
}

int update_nibble(NMAP *nmap, NIBBLE *nibble, int d)
{
        int i;

        if(d && d!=((nibble->direction+1)&3)+1) nibble->direction=d;

        if(nibble->length<nibble->tolength) nibble->length++;
        else
        {   i=nibble->length-1;
                ASSERTP(nmap,nibble->p[i].x,nibble->p[i].y);
                nmap->line[nibble->p[i].y][nibble->p[i].x]=0;
        }

        for(i=nibble->length-1; i>0; i--)
        {   nibble->p[i]=nibble->p[i-1];
                ASSERTP(nmap,nibble->p[i].x,nibble->p[i].y);
                nmap->line[nibble->p[i].y][nibble->p[i].x]=nibble->id-MIN(i,40);
        }

        if(nibble->direction==D_WEST)
                if(--nibble->p[0].x<0)
                {   if(nibble_wrap) nibble->p[0].x+=nmap->w;
                        else {nibble->died=true;return NIBBLE_FAILURE;}
                }
        if(nibble->direction==D_EAST)
                if(++nibble->p[0].x>=nmap->w)
                {   if(nibble_wrap) nibble->p[0].x-=nmap->w;
                        else {nibble->died=true;return NIBBLE_FAILURE;}
                } 
        if(nibble->direction==D_NORTH)
                if(--nibble->p[0].y<0)
                {   if(nibble_wrap) nibble->p[0].y+=nmap->h;
                        else {nibble->died=true;return NIBBLE_FAILURE;}
                }
        if(nibble->direction==D_SOUTH)
                if(++nibble->p[0].y>=nmap->h)
                {   if(nibble_wrap) nibble->p[0].y-=nmap->h;
                        else {nibble->died=true;return NIBBLE_FAILURE;}
                }

        ASSERTP(nmap,nibble->p[0].x,nibble->p[0].y);
        if(nmap->line[nibble->p[0].y][nibble->p[0].x])
        {   nibble->died=true;
                return NIBBLE_FAILURE;
        }
        nmap->line[nibble->p[0].y][nibble->p[0].x]=nibble->id;

        if(has_food(nmap,nibble->p[0].x,nibble->p[0].y))
        {   nibble->tolength+=2*nmap->food.count;
                create_new_food(nmap);
                if(nmap->food.count>=10) return NIBBLE_SUCCESS;
        }

        return NIBBLE_OK;
}

bool is_possible(NMAP *nmap, int x, int y, int d)
{
        if(d==D_WEST)
        {   x--;
                if(nibble_wrap && x<0) x+=nmap->w;
                if(x<0 || nmap->line[y][x]) return false;
        }else if(d==D_EAST)
        {   x++;
                if(nibble_wrap && x>=nmap->w) x-=nmap->w;
                if(x>=nmap->w || nmap->line[y][x]) return false;
        }else if(d==D_NORTH)
        {   y--;
                if(nibble_wrap && y<0) y+=nmap->h;
                if(y<0 || nmap->line[y][x]) return false;
        }else if(d==D_SOUTH)
        {   y++;
                if(nibble_wrap && y>=nmap->h) y-=nmap->h;
                if(y>=nmap->h || nmap->line[y][x]) return false;
        }
        return true;
}

int count_surface(NMAP *nmap, BITMAP *d, int x, int y)
{
        if(nibble_wrap)
        {   if(x<0) x+=nmap->w;
                else if(x>=nmap->w) x-=nmap->w;
                if(y<0) y+=nmap->h;
                else if(y>=nmap->h) y-=nmap->h;
        }
        if(x>=0 && y>=0 && x<nmap->w && y<nmap->h && nmap->line[y][x]==0 && d->line[y][x]==0)
        {   d->line[y][x]=255;
                return count_surface(nmap,d,x-1,y)
                        +count_surface(nmap,d,x+1,y)
                        +count_surface(nmap,d,x,y-1)
                        +count_surface(nmap,d,x,y+1)+1
                        +((x==nmap->food.p.x && y==nmap->food.p.y) ? nmap->w*nmap->h : 0);
        }
        return 0;
}

int nibble_ai_get_direction(NMAP *nmap, NIBBLE *nibble)
{
        static BITMAP *dbitmap=NULL;
        int x=nibble->p[0].x,y=nibble->p[0].y;
        int d,dw=nibble->direction,done,surf,bestsurf=0,bestd=0;

        if(!dbitmap && !(dbitmap=create_bitmap_ex(8,nmap->w,nmap->h)))
                exitmsg("Not enough memory for nibbles AI.\n");

        if((nibble->direction&1)==0 && y==nmap->food.p.y)
        {   if(x<nmap->food.p.x) dw=D_EAST;
                else dw=D_WEST;
        }
        if((nibble->direction&1) && x==nmap->food.p.x)
        {   if(y<nmap->food.p.y) dw=D_SOUTH;
                else dw=D_NORTH;
        }
        if(!is_possible(nmap,x,y,dw))
        {   done=1<<(dw-1);
                while(done!=15)
                {   d=rand()&3;
                        while(done&(1<<d)) d=(d+1)&3;
                        done|=1<<d;
                        if(is_possible(nmap,x,y,++d))
                        {   clear(dbitmap);
                                surf=count_surface(nmap,dbitmap,
                                        (d==D_WEST)?x-1:((d==D_EAST)?x+1:x),
                                        (d==D_NORTH)?y-1:((d==D_SOUTH)?y+1:y));
                                if(surf>bestsurf) {bestsurf=surf;bestd=d;}
                        }
                }
                return bestd;
        }
        return dw;
}

static volatile int frame=0,fps=0,speed_counter=1;

static void fps_calculator(void)
{
        fps=frame;
        frame=0;
}
END_OF_STATIC_FUNCTION(fps_calculator);

static void inc_speed_counter(void)
{
        speed_counter++;
}
END_OF_STATIC_FUNCTION(inc_speed_counter);

static void cleanup(void)
{
        if(the_nmap) {destroy_nmap(the_nmap); the_nmap=NULL;}
        if(the_nibble) {destroy_nibble(the_nibble); the_nibble=NULL;}
        if(the_nibble2) {destroy_nibble(the_nibble2); the_nibble2=NULL;}
        if(rgb_map) {free(rgb_map); rgb_map=NULL;}
        //allegro_exit();
}

void exitmsg(const char *fmt, ...)
{
        va_list va;
        char buf[512];

        va_start(va,fmt);
        uvsprintf(buf,fmt,va);
        va_end(va);

        cleanup();

        allegro_exit();
        allegro_init();
        allegro_message(buf);
        allegro_exit();

        exit(1);
}

static void initialize(int argc, const char *const argv[])
{
        static int virgin = TRUE;

        bool twoplayer=false,third=false;
        int i;
        //RGB temp;

#ifdef STANDALONE_NIBBLES
        bool vmodesel=false;
        int j;

        srand((unsigned)time(NULL));

        for(i=1; i<argc; i++)
        {   if(argv[i][0]=='-')
                {   if(strcmp(argv[i],"-help")==0 || strcmp(argv[i],"-h")==0 || strcmp(argv[i],"-?")==0)
                        {   allegro_message(
                                "\nNibbles, by Tobi Vollebregt, nov/dec 2001\n"
                                "\nUsage: nibbles <options>, where these options are available:\n"
                                "\t-help,-h,-?  Display this help screen.\n"
                                "\t-one,-two    One player / Two players\n"
                                "\t-ai1,-ai2    Make player One/Two CPU controlled.\n"
                                "\t-third       Third (CPU-controlled) player.\n"
                                "\t-wrap        Wrap nibbles at the edges of the world.\n"
                                "\t-spN         Set game speed where N is a number greater than 20.\n"
                                "\t-WxH[xBPP]   Set video resolution to WxHxBPP, BPP is default 8.\n"
                                "\t-BPP         Set colordepth, if W,H _and_ BPP are given, no\n"
                                "\t             videomode selection dialog will be shown.\n"
                                "Game Control:\n"
                                "\tRed snake    arrow keys\n"
                                "\tBlue snake   A,S,D,W\n"
                                "Function keys:\n"
                                "\tF2           - cheat - Both players will die.\n"
                                "\tF3           - cheat - Next level.\n"
                                "\tF5           Restart game.\n"
                                "\tF6           Change wrapping mode.\n"
                                "\tF7           Switch between One-/Two-player mode.\n"
                                "\tF8           Change game speed.\n"
                                "\tF9,F10       Switch between human-/CPU-control of player One/Two.\n"
                                "\tF11          Include a third (CPU-controlled) player in the game?\n"
                                "\tESC          Quit, Shift+ESC: quit without warning.\n"
                                );
                                exit(0);
                        }else if(strcmp(argv[i],"-two")==0) twoplayer=true;
                        else if(strcmp(argv[i],"-one")==0) twoplayer=false;
                        else if(strcmp(argv[i],"-no-two")==0) twoplayer=false;
                        else if(strcmp(argv[i],"-no-one")==0) twoplayer=true;
                        else if(strcmp(argv[i],"-wrap")==0) nibble_wrap=true;
                        else if(strcmp(argv[i],"-no-wrap")==0) nibble_wrap=false;
                        else if(strncmp(argv[i],"-sp",3)==0)
                        {   nibbles_speed=atoi(argv[i]+3);
                                if(nibbles_speed<20) nibbles_speed=20;
                        }else if(strcmp(argv[i],"-ai1")==0) player_ai=true;
                        else if(strcmp(argv[i],"-ai2")==0) player2_ai=true;
                        else if(strcmp(argv[i],"-no-ai1")==0) player_ai=false;
                        else if(strcmp(argv[i],"-no-ai2")==0) player2_ai=false;
                        else if(strcmp(argv[i],"-third")==0) third=true;
                        else if(strcmp(argv[i],"-no-third")==0) third=false;
                        else
                        {   for(j=0; argv[i][j] && argv[i][j]!='x' && argv[i][j]!='X'; j++) ;
                                if(argv[i][j])
                                {   vwidth=atoi(&argv[i][1]);
                                        vheight=atoi(&argv[i][++j]);
                                        while(argv[i][j] && argv[i][j]!='x' && argv[i][j]!='X') j++;
                                        if(argv[i][j])
                                        {   vdepth=atoi(&argv[i][++j]);
                                                vmodesel=false;
                                        }
                                }else vdepth=atoi(&argv[i][1]);
                        }
                }
        }

        allegro_init();
        install_keyboard();

        install_timer();
#endif

        if (virgin)
        {
            LOCK_VARIABLE(fps);
            LOCK_VARIABLE(frame);
            LOCK_VARIABLE(speed_counter);
            LOCK_FUNCTION(fps_calculator);
            LOCK_FUNCTION(inc_speed_counter);
            virgin = FALSE;
        }

#ifdef STANDALONE_NIBBLES
        if(install_mouse()<0)
                exitmsg("Failed to install Allegro mouse driver.\n");

        if(vmodesel)
        {   set_gfx_mode(GFX_SAFE,320,200,0,0);
                set_palette(black_palette);
                clear_to_color(screen,64);

                for(i=0; i<256; i++) palette[i].r=palette[i].g=palette[i].b=63-(i>>2);
                select_palette(palette);
                set_mouse_sprite(NULL);
                unselect_palette();

                for(i=0; i<256; i++) palette[i].r=palette[i].g=palette[i].b=i>>2;
                set_palette(palette);

                gui_fg_color=0;
                gui_mg_color=96;
                gui_bg_color=192;

                if(!gfx_mode_select_ex(&vcard,&vwidth,&vheight,&vdepth))
                        exitmsg("I suggest you accidentally pressed the cancel-button?\n");
        }

        set_color_depth(vdepth);
        if(set_gfx_mode(vcard,vwidth,vheight,0,0)<0)
                exitmsg("Error setting %dx%dx%d graphics mode: %s.\n",vwidth,vheight,vdepth,allegro_error);
#else
        (void) argc; //suppress "unused" warnings
        (void) argv;

        vcard =GFX_AUTODETECT;
        vwidth = SCREEN_W;
        vheight = SCREEN_H;
        vdepth = bitmap_color_depth (screen);
#endif
        clear(screen);

        memset(palette,0,sizeof(palette));
        for(i=0; i<64; i++) palette[i].r=palette[i].g=palette[i].b=i;
        for(i=64; i<128; i++) palette[i].r=i-64;
        for(i=128; i<192; i++) palette[i].g=i-128;
        for(i=192; i<256; i++) palette[i].b=i-192;
        set_palette(palette);

        if(!(rgb_map=(RGB_MAP *)malloc(sizeof(RGB_MAP))))
                exitmsg("Not enough memory to allocate rgb mapping table.\n");
        create_rgb_table(rgb_map,palette,NULL);

        gui_fg_color=makecol(255,255,255);
        gui_bg_color=makecol(0,0,128);
        set_mouse_sprite(NULL);

        player_lives=player2_lives=5;
        player_score=player2_score=0;
        nibbles_level=1;

        if(!(the_nmap=create_nmap((vwidth-32)/8,(vheight-32)/8,8,8)))
        //if(!(the_nmap=create_nmap(80,50,(vwidth-32)/80,(vheight-32)/50)))
                exitmsg("Not enough memory to create Nibbles-map.\n");
        restart_game(twoplayer,third);

        install_int(fps_calculator,1000);
        install_int_ex(inc_speed_counter,BPS_TO_TIMER(nibbles_speed));
}

void draw_text(bool mainloop)
{
        static int fps0,player_lives0,player2_lives0,nibbles_level0;
        static int player_score0,player2_score0,nibbles_speed0;
        char buf[512];

        text_mode(0);
        if(mainloop)
        {   if(fps!=fps0)
                {   usprintf(buf,"%i fps   ",fps);
                        textout(screen,font,buf,0,0,makecol(255,255,255));
                        fps0=fps;
                }
        }else if(fps0)
        {   rectfill(screen,0,0,vwidth-1,text_height(font),0);
                nibbles_level0=fps0=0;
                nibbles_speed0=0;
        }
        if(nibbles_speed!=nibbles_speed0)
        {   usprintf(buf,"    %d sq/sec",nibbles_speed);
                textout_right(screen,font,buf,vwidth-1,0,makecol(255,255,255));
                nibbles_speed0=nibbles_speed;
        }
        if(nibbles_level!=nibbles_level0)
        {   usprintf(buf," LEVEL %i ",nibbles_level);
                textout_centre(screen,font,buf,vwidth/2,0,makecol(0,255,0));
                nibbles_level0=nibbles_level;
        }
        if(player_lives!=player_lives0 || player_score!=player_score0)
        {   usprintf(buf,"One: lives:%i; score:%X;",player_lives,player_score);
                textout(screen,font,buf,0,vheight-text_height(font),makecol(255,0,0));
                player_lives0=player_lives;
                player_score0=player_score;
        }
        if(the_nibble2)
        {   if(player2_lives!=player2_lives0 || player2_score!=player2_score0)
                {   usprintf(buf,"Two: lives:%i; score:%X;",player2_lives,player2_score);
                        textout(screen,font,buf,vwidth/2,vheight-text_height(font),makecol(0,0,255));
                        player2_lives0=player2_lives;
                        player2_score0=player2_score;
                }
        }else if(player2_lives)
        {   rectfill(screen,vwidth/2,vheight-text_height(font),vwidth-1,vheight-1,0);
                player2_lives=0;
        }
}

void restart_game(bool twoplayer, bool third)
{
        int x,y;

        if(the_nibble) destroy_nibble(the_nibble);
        if(the_nibble2) destroy_nibble(the_nibble2);
        if(the_nibble3) destroy_nibble(the_nibble3);
        the_nibble=the_nibble2=the_nibble3=NULL;
        generate_level(the_nmap,nibbles_level,&nibble_startx,&nibble_starty,&nibble_startdir);
        if(!(the_nibble=create_nibble(the_nmap,makecol8(255,0,0),nibble_startx,nibble_starty,nibble_startdir,15)))
                exitmsg("Not enough memory to create player-Nibble.\n");
        if(twoplayer && !(the_nibble2=create_nibble(the_nmap,makecol8(0,0,255),the_nmap->w-1-nibble_startx,the_nmap->h-1-nibble_starty,((nibble_startdir+1)&3)+1,15)))
                exitmsg("Not enough memory to create second player-Nibble.\n");
        if(third)
        {   do
                {   x=rand()%the_nmap->w;
                        y=rand()%the_nmap->h;
                }while(the_nmap->line[y][x] || ABS(nibble_startx-x)<15 || ABS(nibble_starty-y)<15
                        || (twoplayer && (ABS(the_nmap->w-nibble_startx-1-x)<15 || ABS(the_nmap->h-nibble_starty-1-y)<15)));
                if(!(the_nibble3=create_nibble(the_nmap,makecol8(0,255,0),x,y,1+(rand()&3),15)))
                        exitmsg("Not enough memory to create third player-Nibble.\n");
        }
        the_nmap->food.count=0;
        create_new_food(the_nmap);
        speed_counter=0;
}

int aalert3(const char *s1, const char *s2, const char *s3, const char *b1, const char *b2, const char *b3, int c1, int c2, int c3, int n)
{
        int ret;

        gui_fg_color=makecol(255,255,255);
        if(n==1) gui_bg_color=makecol(128,0,0);
        else if(n==2) gui_bg_color=makecol(0,0,128);
        else gui_bg_color=makecol(0,128,0);

        ret=alert3(s1,s2,s3,b1,b2,b3,c1,c2,c3);

        while(key[KEY_ESC]) ;
        speed_counter=0;

        return ret;
}

int aalert(const char *s1, const char *s2, const char *s3, const char *b1, const char *b2, int c1, int c2, int n)
{
        return aalert3(s1,s2,s3,b1,b2,NULL,c1,c2,0,n);
}

bool check_for_dead_nibbles(int i, int j, int n3)
{
        char buf[128];

        if(!the_nibble2) j=0;
        if(!the_nibble3) n3=0;

        if(i==NIBBLE_FAILURE || j==NIBBLE_FAILURE || n3==NIBBLE_FAILURE || key[KEY_F2])
        {   // Maybe both (or even three) died at once...
                if(the_nibble3)
                {   if(key[KEY_F2] || (the_nibble3->p[0].x==the_nibble->p[0].x && the_nibble3->p[0].y==the_nibble->p[0].y
                                && the_nibble3->p[0].x==the_nibble2->p[0].x && the_nibble3->p[0].y==the_nibble2->p[0].y))
                        {   ustrcpy(buf,"Three players died at once!!");
                                i=j=n3=NIBBLE_FAILURE;
                        }else if(the_nibble3->p[0].x==the_nibble->p[0].x && the_nibble3->p[0].y==the_nibble->p[0].y)
                        {   ustrcpy(buf,"Both player One and player Three died!");
                                i=n3=NIBBLE_FAILURE;
                        }else if(the_nibble3->p[0].x==the_nibble2->p[0].x && the_nibble3->p[0].y==the_nibble2->p[0].y)
                        {   ustrcpy(buf,"Both player Two and player Three died!");
                                j=n3=NIBBLE_FAILURE;
                        }else if(the_nibble->p[0].x==the_nibble2->p[0].x && the_nibble->p[0].y==the_nibble2->p[0].y)
                        {   ustrcpy(buf,"Both player One and player Two died!");
                                i=j=NIBBLE_FAILURE;
                        }else if(i==NIBBLE_FAILURE) ustrcpy(buf,"Player One died...");
                        else if(j==NIBBLE_FAILURE) ustrcpy(buf,"Player Two died...");
                        else if(n3==NIBBLE_FAILURE) ustrcpy(buf,"Player Three died...");
                }else
                {   if((key[KEY_F2] || the_nibble2) && (the_nibble->p[0].x==the_nibble2->p[0].x
                                && the_nibble->p[0].y==the_nibble2->p[0].y))
                        {   ustrcpy(buf,"Both players died...");
                                i=j=NIBBLE_FAILURE;
                        }else // but assume only one died
                                usprintf(buf,"Player %s died...",(i==NIBBLE_FAILURE)?"One":"Two");
                }
                if(i==NIBBLE_FAILURE) player_lives--;
                if(j==NIBBLE_FAILURE) player2_lives--;
                draw_text(false);
                aalert(buf,NULL,NULL,"Damn!",NULL,0,0,i==j?0:(i==NIBBLE_FAILURE?1:2));
                if(!player_lives || !(!the_nibble2 || player2_lives))
                {   if(aalert("GAME OVER",NULL,NULL,"&Play Again","&Quit",'p','q',i==j?0:(i==NIBBLE_FAILURE?2:1))==2) return true;
                        player_lives=player2_lives=5;
                        player_score=player2_score=0;
                        nibbles_level=1;
                }
                restart_game(the_nibble2?true:false,the_nibble3?true:false);
        }

        return false;
}

bool check_for_finishing_level(int i, int j, int n3)
{
        char buf[128];

        if(!the_nibble2) j=0;
        if(!the_nibble3) n3=0;

        if(i==NIBBLE_SUCCESS || j==NIBBLE_SUCCESS || n3==NIBBLE_SUCCESS || key[KEY_F3])
        {   draw_text(false);
                usprintf(buf,"Finished level %i!",nibbles_level);
                aalert(buf,NULL,NULL,"Yeah!",NULL,0,0,i==j?0:(i==NIBBLE_SUCCESS?1:2));
                player_lives++; player2_lives++;
                if(i==NIBBLE_SUCCESS) player_score+=nibbles_level*0xFFF;
                if(j==NIBBLE_SUCCESS) player2_score+=nibbles_level*0xFFF;
                nibbles_level++;
                restart_game(the_nibble2?true:false,the_nibble3?true:false);
        }

        return false;
}

int slider_callback(void *dp3, int d2)
{
        DIALOG *d=(DIALOG *)dp3;
        int ret;

        usprintf((char *)d[4].dp,"%i sq/sec",d2+20);
        scare_mouse();
        rectfill(screen,d[4].x-d[4].w,d[4].y,d[4].x+d[4].w,d[4].y+d[4].h,d[4].bg);
        ret=d[4].proc(MSG_DRAW,&d[4],0);
        unscare_mouse();

        return ret;
}

void do_speed_dialog(void)
{
        char slider_text[128];

        DIALOG dialog[] =
        {
           /* (proc)            (x)  (y) (w)  (h) (fg) (bg) (key) (flags) (d1) (d2) (dp)                          (dp2)            (dp3)   */
           { d_shadow_box_proc, 0,   0,  288, 80, 0,   0,   0,    0,      0,   0,   NULL,                         NULL,            NULL   },
           { d_button_proc,     64,  56, 80,  16, 0,   0,   'o',  D_EXIT, 0,   0,   "&Ok",                        NULL,            NULL   },
           { d_button_proc,     152, 56, 80,  16, 0,   0,   'c',  D_EXIT, 0,   0,   "&Cancel",                    NULL,            NULL   },
           { d_slider_proc,     16,  24, 256, 16, 0,   0,   0,    0,      180, 0,   NULL,                         slider_callback, dialog },
           { d_ctext_proc,      144, 40, 80,  8,  0,   0,   0,    0,      0,   0,   slider_text,                  NULL,            NULL   },
           { d_ctext_proc,      144, 8,  10,  8,  0,   0,   0,    0,      0,   0,   "Set Nibbles speed",          NULL,            NULL   },
           { NULL,              0,   0,  0,   0,  0,   0,   0,    0,      0,   0,   NULL,                         NULL,            NULL   }
        };

        gui_fg_color=makecol(255,255,255);
        gui_bg_color=makecol(0,128,0);
        set_dialog_color(dialog,gui_fg_color,gui_bg_color);
        centre_dialog(dialog);

        dialog[3].d2=MIN(200,nibbles_speed)-20;
        usprintf(slider_text,"%i sq/sec",MIN(nibbles_speed,200));

        if(popup_dialog(dialog,-1)==1)
        {   nibbles_speed=dialog[3].d2+20;
                install_int_ex(inc_speed_counter,BPS_TO_TIMER(nibbles_speed));
        }

        while(key[KEY_ESC]) ;
        speed_counter=0;
}

#ifdef STANDALONE_NIBBLES
int main(int argc, const char *const argv[]) {
#else
int do_nibbles () { int argc=0; const char* const argv[1]={""};
#endif
        bool done=false;
        int d=0,i=0,j=0,n3=0;

        initialize(argc,argv);

        rect(screen,15,15,vwidth-16,vheight-16,makecol(255,255,255));
        aalert("Press enter to play nibbles...","...by Tobi Vollebregt (nov 2001)","Congratulations, you found the easter egg","Play",NULL,13,0,0);

        while(!done)
        {   frame++;
                while(speed_counter>0 && !done)
                {   d=0;
                        if(player_ai) d=nibble_ai_get_direction(the_nmap,the_nibble);
                        else
                        {   if(key[KEY_LEFT]) d=D_WEST;
                                if(key[KEY_RIGHT]) d=D_EAST;
                                if(key[KEY_UP]) d=D_NORTH;
                                if(key[KEY_DOWN]) d=D_SOUTH;
                        }
                        i=update_nibble(the_nmap,the_nibble,d);
                        player_score+=the_nibble->length;

                        if(the_nibble2)
                        {   d=0;
                                if(player2_ai) d=nibble_ai_get_direction(the_nmap,the_nibble2);
                                else
                                {   if(key[KEY_A]) d=D_WEST;
                                        if(key[KEY_D]) d=D_EAST;
                                        if(key[KEY_W]) d=D_NORTH;
                                        if(key[KEY_S]) d=D_SOUTH;
                                }
                                j=update_nibble(the_nmap,the_nibble2,d);
                                player2_score+=the_nibble2->length;
                        }
                        if(the_nibble3)
                        {   d=nibble_ai_get_direction(the_nmap,the_nibble3);
                                n3=update_nibble(the_nmap,the_nibble3,d);
                        }

                        done=done
                                || check_for_dead_nibbles(i,j,n3)
                                || check_for_finishing_level(i,j,n3);

                        if(key[KEY_F5])
                        {   draw_text(false);
                                if(aalert("Quit current game and restart new?",NULL,NULL,"&Yes","&No",'y','n',0)==1)
                                {   player_lives=player2_lives=5;
                                        player_score=player2_score=0;
                                        nibbles_level=1;
                                        restart_game(the_nibble2?true:false,the_nibble3?true:false);
                                }
                        }
                        if(key[KEY_F6])
                        {   draw_text(false);
                                j=aalert3("Wrap nibbles on edge of the world?",NULL,NULL,"&Yes","&No","&Cancel",'y','n','c',0);
                                if(j==1) nibble_wrap=true;
                                else if(j==2) nibble_wrap=false;
                        }
                        if(key[KEY_F7])
                        {   draw_text(false);
                                i=the_nibble2?true:false;
                                j=aalert3("How many players do we have?","Warning: Needs restarting game!",NULL,"&One-player","&Two-player","&Cancel",'o','t','c',0);
                                if(j!=3)
                                {   player_lives=player2_lives=5;
                                        player_score=player2_score=0;
                                        nibbles_level=1;
                                        restart_game(j==1?false:true,the_nibble3?true:false);
                                }
                        }
                        if(key[KEY_F8]) do_speed_dialog();
                        if(key[KEY_F9])
                        {   draw_text(false);
                                j=aalert3("Should player One be CPU-controlled (AI)?",NULL,NULL,"&Yes","&No","&Cancel",'y','n','c',1);
                                if(j==1) player_ai=true;
                                else if(j==2) player_ai=false;
                        }
                        if(key[KEY_F10] && the_nibble2)
                        {   draw_text(false);
                                j=aalert3("Should player Two be CPU-controlled (AI)?",NULL,NULL,"&Yes","&No","&Cancel",'y','n','c',2);
                                if(j==1) player2_ai=true;
                                else if(j==2) player2_ai=false;
                        }
                        if(key[KEY_F11])
                        {   draw_text(false);
                                j=aalert3("Do we want an extra AI/CPU-nibble?","Warning: Needs restarting game!",NULL,"&Yes","&No","&Cancel",'y','n','c',0);
                                if(j!=3)
                                {   player_lives=player2_lives=5;
                                        player_score=player2_score=0;
                                        nibbles_level=1;
                                        restart_game(the_nibble2?true:false,j==1);
                                }
                        }
                        if(key[KEY_PLUS_PAD])
                        {   nibbles_speed++;
                                install_int_ex(inc_speed_counter,BPS_TO_TIMER(nibbles_speed));
                        }
                        if(key[KEY_MINUS_PAD])
                        {   if(nibbles_speed>20) nibbles_speed--;
                                install_int_ex(inc_speed_counter,BPS_TO_TIMER(nibbles_speed));
                        }
                        if(key[KEY_ESC])
                                if((key_shifts&KB_SHIFT_FLAG) || aalert("Do you really want to quit?",NULL,NULL,"&Yes","&No",'y','n',0)==1) done=true;
                        speed_counter--;
                }
                draw_nmap(the_nmap);
                draw_text(true);
        }

        cleanup();
        //allegro_exit();

        return 0;
}
#ifdef STANDALONE_NIBBLES
END_OF_MAIN();
#endif
