/* Author: Tobi Vollebregt */

/*  Bataafje -- A small game written for the Allegro SpeedHack 2003
 *  Copyright (C) 2003  Tobi Vollebregt
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  See `License.txt', which contains a verbatim copy of the
 *  GNU General Public License, for details.
 *
 *  Please send your reaction to: tobivollebregt@hotmail.com
 */

#include <allegro.h>
#include <stdarg.h>
#include <string.h>
#include <time.h>
#include "anim.h"
#include "object.h"

extern "C"
{
    // from nibbles.c
    int do_nibbles ();
    // from intro.c
    void do_intro ();
    void do_fractal ();
}

// -----globals-----
//int frame_id;
static IObject* objs;
int score;
int game_time;
extern "C" { FONT* game_font; }

// -----timing------
static volatile int speed_counter;
static void ticker () { ++speed_counter; }
END_OF_STATIC_FUNCTION (ticker);

// -----fps calc-----
static volatile int fps, frame;
static void fps_calc () { fps = frame; frame = 0; }
END_OF_STATIC_FUNCTION (fps_calc);

IObject* GetObjects () { return objs; }

void LoadBigFont ()
{
    DATAFILE* dat = load_datafile ("bigfont.dat");
    if (!dat) FatalError ("Couldn't load bigfont.dat: %s\n", strerror (errno));
    game_font = (FONT*) dat[0].dat;
}

void AddObject (IObject* obj)
{
    obj->next = objs;
    objs = obj;
}

void FatalError (const char* fmt, ...)
{
    char buf[128];
    va_list argp;
    va_start (argp, fmt);
    uvszprintf (buf, sizeof (buf), fmt, argp);
    va_end (argp);
    set_gfx_mode (GFX_TEXT, 0, 0, 0, 0);
    allegro_message ("%s\n", buf);
    exit (1);
}

int main ()
{
    static const char k_window_title[] = "Allegro SpeedHack-2003: Bataafje";
    static const char k_config_file[] = "bataafje.cfg";
    static const char k_config_internal_file[] = "bataafje.dat#bataafje_intern.cfg";
    static const char* k_fs_text[2] = { "windowed", "fullscreen" };

    srand ((unsigned) time (NULL));
    IObject::DumpClassList ();

    LOCK_FUNCTION (ticker);
    LOCK_VARIABLE (speed_counter);
    LOCK_FUNCTION (fps_calc);
    LOCK_VARIABLE (fps);
    LOCK_VARIABLE (frame);

    // initialize allegro
    allegro_init ();
    set_config_file (k_config_file);
    override_config_file (k_config_internal_file);
    set_window_title (k_window_title);

    // set graphics mode
    int w = get_config_int ("screen", "width", 640);
    int h = get_config_int ("screen", "height", 480);
    int bpp = get_config_int ("screen", "color_depth", 16);
    int fs = get_config_int ("screen", "fullscreen", 0);
    if (bpp == 8) FatalError ("Didn't have time to add 8 bit support");
    set_color_depth (bpp);
    if (set_gfx_mode (fs ? GFX_AUTODETECT_FULLSCREEN : GFX_AUTODETECT_WINDOWED, w, h, 0, 0) < 0)
        FatalError ("Error setting %dx%dx%d %s graphics mode:\n%s", w, h, bpp, k_fs_text[fs], allegro_error);

    if (install_mouse () < 0) FatalError ("Cannot install mouse.");
    if (install_timer () < 0) FatalError ("Cannot install timer.");
    if (install_keyboard () < 0) FatalError ("Cannot install keyboard.");

    game_font = font;

    // from intro.c
    if (get_config_int (0, "intro", 0))
    {
        do_intro ();
        while (key[KEY_ESC]) ;
        clear_keybuf ();
    }

    // create double buffer
    BITMAP* buffer = create_bitmap (w, h);
    if (!buffer) FatalError ("Not enough memory for double buffering.");

    // install sound
    /*if (install_sound (DIGI_AUTODETECT, MIDI_AUTODETECT) < 0)
        FatalError ("Error installing sound:\n%s", allegro_error);*/

    /*const char* first_level = get_config_string (0, "first_level", 0);
    if (!first_level) FatalError ("Couldn't read first_level from %s.", k_config_file);
    the_map = new CMap (first_level);*/

    /*CAnimation hein ("gfx/hein-lopen.cfg");
    CAnimation abra ("gfx/abra-lopen.cfg");
    CAnimInstance h1 (hein), h2 (abra);*/

    IObject* player = IObject::New ("CPlayer");
    if (!player || !player->Initialize (50))
        FatalError ("Couldn't initialize CPlayer object");
    objs = player;

    for (int i = 0;; ++i)
    {
        char tmp[128];
        uszprintf (tmp, sizeof (tmp), "%d", i);
        const char* s = get_config_string ("init.create", tmp, 0);
        if (!s) break;
        ustrzcpy (tmp, sizeof (tmp), s);
        char* colon = strchr (tmp, ':');
        if (!colon) FatalError ("Expected `:'.");
        *colon = 0;
        IObject* obj = IObject::New (tmp);
        obj->next = objs;
        objs = obj;
        if (!obj->Initialize (atoi (colon + 1)))
            FatalError ("Couldn't initialize %s object.", tmp);
        colon = strchr (colon + 1, ':');
        if (colon) obj->SetAnimation (colon + 1);
    }

    text_mode (-1);
    
    // install timers
    install_int_ex (ticker, BPS_TO_TIMER (50));
    install_int_ex (fps_calc, BPS_TO_TIMER (1));

    int createcount = 0, createdcount = 0;
    // main loop
    while (!key[KEY_ESC])
    {
        ++frame;

        // update everything
        while (speed_counter > 0)
        {
            //++frame_id;
            for (IObject* obj = objs, *prev=0; obj; prev = obj, obj = obj ? obj->next : objs)
            {
                if (!obj->Update (20))
                {
                    if (prev) prev->next = obj->next;
                    else objs = obj->next;
                    delete obj;
                    obj = prev;
                }
            }
            for (int i = 0;; ++i)
            {
                char tmp[128];
                uszprintf (tmp, sizeof (tmp), "%d", i);
                const char* s = get_config_string ("update.create", tmp, 0);
                if (!s) break;
                ustrzcpy (tmp, sizeof (tmp), s);
                char* colon = strchr (tmp, ':');
                if (!colon) FatalError ("Expected `:'.");
                *colon = 0;
                if ((rand () & 8191) < atoi (tmp)) ++createcount;
                if (createcount && createdcount == 0)
                {
                    char* cl = colon + 1;
                    colon = strchr (colon + 1, ':');
                    if (!colon) FatalError ("Expected `:'.");
                    *colon = 0;
                    IObject* obj = IObject::New (cl);
                    obj->next = objs;
                    objs = obj;
                    if (!obj->Initialize (atoi (colon + 1)))
                        FatalError ("Couldn't initialize %s object.", tmp);
                    colon = strchr (colon + 1, ':');
                    if (colon) obj->SetAnimation (colon + 1);
                    createdcount = 50;
                    --createcount;
                }
                if (createdcount) --createdcount;
            }

            //the_map->Update (20);
            /*h1.Update (20);
            h2.Update (20);*/
            --speed_counter;
            ++game_time;
        }

        // draw everything
        clear_to_color (buffer, makecol (170, 193, 255));

        for (IObject* obj = objs; obj; obj = obj->next)
            obj->Draw (buffer, 0);
        //the_map->Draw (buffer, the_map_x, the_map_y);
        /*h1.Draw (buffer, 100, 100, true);
        h2.Draw (buffer, 200, 100, false);*/

        if (fps)
        {
            text_mode (-1);
            textprintf (buffer, font, 0, 0, 0, "%d fps", fps);
        }
        textprintf_right (buffer, game_font, SCREEN_W-1, 0, 0, "score: %d", score);

        // take screenshot
        if (key[KEY_F12])
        {
            const char* s = get_config_string (0, "screenshot", "shot.bmp");
            if (save_bitmap (s, buffer, 0)) {
                TRACE ("Couldn't save screenshot to %s: %s\n", s, strerror (errno)); }
        }
        /*
        
            code for to enter the easter egg: a nibbles subgame
        
        */
        if (key[KEY_S] && key[KEY_N] && key[KEY_A] && key[KEY_K] && key[KEY_E])
        {
            remove_int (ticker);
            remove_int (fps_calc);
            do_nibbles ();
            while (key[KEY_ESC]) ;
            clear_keybuf ();
            install_int_ex (ticker, BPS_TO_TIMER (50));
            install_int_ex (fps_calc, BPS_TO_TIMER (1));
        }

        // blit buffer to screen
        blit (buffer, screen, 0, 0, 0, 0, w, h);
    }

    while (objs)
    {
        IObject* tmp = objs;
        objs = objs->next;
        delete tmp;
    }

    //delete the_map;
    destroy_bitmap (buffer);

    // from intro.c
    if (get_config_int (0, "fractal", 0)) do_fractal ();

    return 0;
}
END_OF_MAIN();
