/* Author: Tobi Vollebregt */

/*  Bataafje -- A small game written for the Allegro SpeedHack 2003
 *  Copyright (C) 2003  Tobi Vollebregt
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  See `License.txt', which contains a verbatim copy of the
 *  GNU General Public License, for details.
 *
 *  Please send your reaction to: tobivollebregt@hotmail.com
 */

#include <allegro.h>
#include <string.h>
#include "anim.h"

CAnimationData::CAnimationData (const char* file) : frames (0)
{
    TRACE ("Loading animation from %s.....", file);

    push_config_state ();
    set_config_file (file);

    num_frames = get_config_int (0, "num_frames", -1);
    if (num_frames <= 0) FatalError ("Couldn't read animation from %s.", file);

    frames = new Frame[num_frames];

    int greatest_x = 0;
    int greatest_y = 0;
    int greatest_x_flip = 0;
    int i;
    for (i = 0; i < num_frames; ++i)
    {
        char tmp[32];
        uszprintf (tmp, sizeof (tmp), "frame.%d", i);
        const char* s = get_config_string (tmp, "bmp", 0);
        frames[i].x = get_config_int (tmp, "x", -1);
        frames[i].y = get_config_int (tmp, "y", -1);
        frames[i].dt = get_config_int (tmp, "dt", -1); // dt is the time interval till the next frame
        if (!s || frames[i].x < 0 || frames[i].y < 0 || frames[i].dt < 0)
            FatalError ("%s: Couldn't read animation frame# %d.", file, i);
        if (!(frames[i].bmp = load_bitmap (s, 0)))
            FatalError ("%s: Couldn't load bitmap from %s: %s", file, s, strerror (errno));

        // find the greatest x and y values
        if (frames[i].x > greatest_x) greatest_x = frames[i].x;
        if (frames[i].y > greatest_y) greatest_y = frames[i].y;

        // find the greatest flipped x value
        frames[i].x_flip = frames[i].bmp->w - frames[i].x;
        if (frames[i].x_flip > greatest_x_flip) greatest_x_flip = frames[i].x_flip;
    }

    // the frames are translated so that the (x,y)-point of all frames overlap
    for (i = 0; i < num_frames; ++i)
    {
        frames[i].x = greatest_x - frames[i].x;
        frames[i].y = greatest_y - frames[i].y;
        frames[i].x_flip = greatest_x_flip - frames[i].x_flip;
    }

    // compute width and height of the animation
    width = 0;
    height = 0;
    for (i = 0; i < num_frames; ++i)
    {
        if (frames[i].x + frames[i].bmp->w > width) width = frames[i].x + frames[i].bmp->w;
        if (frames[i].y + frames[i].bmp->h > height) height = frames[i].y + frames[i].bmp->h;
    }

    pop_config_state ();

    TRACE ("done\n");
}

CAnimationData::~CAnimationData ()
{
    if (frames)
    {
        for (int i = 0; i < num_frames; ++i) destroy_bitmap (frames[i].bmp);
        delete[] frames;
    }
}

CAnimation::CAnimation (const char* file) : anim (0), t (0), pos (0)
{
    int i;
    for (i = 0; i < MAX_ANIMATIONS && anims[i].data; ++i)
        if (ustrcmp (file, anims[i].filename) == 0)
        {
            ++ref_count;
            anim = anims[i].data;
            ASSERT (anim);
            return;
        }
    //if (i == 0) memset (anims, 0, sizeof (anims));
    if (i >= MAX_ANIMATIONS)
        FatalError ("Too much anims. Increase MAX_ANIMATIONS in anim.h.");
    ustrzcpy (anims[i].filename, sizeof (anims[i].filename), file);
    anim = anims[i].data = new CAnimationData (file);
    ASSERT (anim);
    ++ref_count;
}

CAnimation::~CAnimation ()
{
    anim = 0;
    if (--ref_count == 0)
    {
        TRACE ("deleting all animations\n");
        for (int i = 0; i < MAX_ANIMATIONS; ++i)
        {
            delete anims[i].data;
            memset (anims+i, 0, sizeof (Anim));
        }
    }
}

void CAnimation::Draw (BITMAP* d, int x, int y, bool flip)
{
    ASSERT (anim);
    if (flip) draw_sprite_h_flip (d, anim->frames[pos].bmp, x + anim->frames[pos].x_flip, y + anim->frames[pos].y);
    else draw_sprite (d, anim->frames[pos].bmp, x + anim->frames[pos].x, y + anim->frames[pos].y);
}

// Draw the animation, but using draw_trans_sprite instead of draw_sprite.
void CAnimation::DrawTrans (BITMAP* d, int x, int y, bool flip)
{
    ASSERT (anim);
    draw_trans_sprite (d, anim->frames[pos].bmp, x + anim->frames[pos].x, y + anim->frames[pos].y);
}

void CAnimation::Update (int ms)
{
    ASSERT (anim);
    t += ms;
    if (t >= anim->frames[pos].dt)
    {
        t -= anim->frames[pos].dt;
        if (++pos == anim->num_frames) pos = 0;
    }
}

CAnimation::Anim CAnimation::anims[MAX_ANIMATIONS];
int CAnimation::ref_count;
