//
//  **************************************************************************
//
//  vector3d.h 
//
//  (C) 2003 Bosco K. Ho 
//  Adapted from Reduce (C) 1999 J. Michael Word
//
//  **************************************************************************
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU Lesser General Public License as published
//  by the Free Software Foundation; either version 2.1 of the License, or (at
//  your option) any later version.
//  
//  This program is distributed in the hope that it will be useful,  but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details. 
//  
//  You should have received a copy of the GNU Lesser General Public License 
//  along with this program; if not, write to the Free Software Foundation, 
//  Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
//  **************************************************************************
//

#ifndef _vector3d_h_
#define _vector3d_h_

#include <iostream>
#include <cmath>


/////////////////////////////////////////////
//////                               ////////
//////   ALL ANGLES ARE IN DEGREES   //////// 
//////                               ////////
/////////////////////////////////////////////


const double PI = 3.14159265358979323846;

// Conversion factors  
const double DEG2RAD = PI / 180.0; 
const double RAD2DEG = 180.0 / PI; 


class Matrix3d;


class Vector3d 
{ 
  public:
    Vector3d(): _x(0.0), _y(0.0), _z(0.0) {} 
	  Vector3d(double x, double y, double z): _x(x),_y(y),_z(z) {}
    Vector3d(const Vector3d& p): _x(p._x), _y(p._y), _z(p._z) {}
  
    ~Vector3d() {}
  
    Vector3d operator - (const Vector3d& p) const;
    Vector3d operator + (const Vector3d& p) const;
    Vector3d operator * (double s) const;
    Vector3d operator / (double s) const;
  
    Vector3d operator - () const;
  
    Vector3d& operator =  (const Vector3d& p);
    Vector3d& operator -= (const Vector3d&); 
    Vector3d& operator += (const Vector3d&);
    Vector3d& operator *= (double);
    Vector3d& operator /= (double);
  
    double x() const { return _x; }
    double y() const { return _y; }
    double z() const { return _z; }

    void x(double x0) { _x = x0; } 
    void y(double y0) { _y = y0; }
    void z(double z0) { _z = z0; }
  
    double     lengthSquared() const;
    double     length() const;
    Vector3d&  normalize();     // set length to unity
    Vector3d   normal() const;
    Vector3d&  scaleTo(double); // set length to a new value
    Vector3d   scaled(double) const;
  
    // rotate  (in degrees) our point around the vector(p1-p2)
    Vector3d   rotated(double, const Vector3d&) const;
    Vector3d&  rotate(double, const Vector3d&);

    Vector3d&  applyMatrix(const Matrix3d& matrix);

    Vector3d   parallel(const Vector3d&) const;
    double     parallelComponent(const Vector3d&) const;
    Vector3d   perpendicular(const Vector3d&) const;
    Vector3d   outPlane(const Vector3d&, const Vector3d&, const Vector3d&) const;
    Vector3d   inPlane(const Vector3d&, const Vector3d&, const Vector3d&) const;
    double     planeAngle(const Vector3d&, const Vector3d&, const Vector3d&) const; 

    friend std::ostream& operator << (std::ostream&, const Vector3d&);

  private:
 	  double _x, _y, _z;
};



// dot product
inline double dot(const Vector3d& a, const Vector3d& b) {
  return (a.x()*b.x()) + (a.y()*b.y()) + (a.z()*b.z());
}

// cross product
Vector3d cross(const Vector3d&, const Vector3d&);

// linear interpolation from lo (when a=0) to hi (when a=1)
inline double interpolate(double lo, double hi, double a) {
  return lo + a*(hi-lo);
}

// linear interpolation between two points or vectors
Vector3d interpolate(const Vector3d&, const Vector3d&, double);

// distance between two points
double distanceSquared(const Vector3d&, const Vector3d&);
double distance(const Vector3d&, const Vector3d&);

// calculate angles and dihedrals in degrees
double anglePos(const Vector3d&, const Vector3d&, const Vector3d&);
double angle(const Vector3d&, const Vector3d&);
double dihedralPos(const Vector3d&, const Vector3d&,
                   const Vector3d&, const Vector3d&);
double dihedral(const Vector3d&, const Vector3d&, const Vector3d&);

// projection algorithms
Vector3d rotatePos(double theta, const Vector3d& p1, const Vector3d& p2,
                   const Vector3d& p3);
Vector3d projectPos(double length, double angle, double dihedralAngle, const Vector3d& p1, 
                    const Vector3d& p2, const Vector3d& p3);

inline Vector3d planeNormal(const Vector3d& p1, const Vector3d& p2, 
                          const Vector3d& p3) 
{ 
  return cross(p2-p1, p3-p2);
}



// homogenous coordinates transformation matrix

class Matrix3d 
{ 
  public:
    Matrix3d() { makeIdentity(); }                  // identity matrix
    Matrix3d(double scale);                         // uniform scale matrix
    Matrix3d(double sx, double sy, double sz);      // scale matrix
    Matrix3d(const Vector3d& axis, double theta);   // rotation matrix
    Matrix3d(const Vector3d& axis, double theta, const Vector3d& center);
    Matrix3d(const Vector3d& p);                    // translation matrix
  
    ~Matrix3d() {}
  
    Matrix3d operator * (const Matrix3d&) const;      // matrix multiplication
    // to transform, post-multiply vectors (mat*vec)
    Vector3d operator * (const Vector3d&) const;
  
    Matrix3d& makeIdentity();

    friend std::ostream& operator << (std::ostream&, const Matrix3d&);
  
  private:
    double _element[4][4];      
};


#endif // _VECTOR3D_H_ 

