//
//  **************************************************************************
//
//  vector3d.cpp 
//
//  (C) 2003 Bosco K. Ho 
//
//  This is a 3 dimensional vector library intended to be used for analysing
//  protein molecules. As there are special operations in 3D that do not
//  generalise to n dimensional vectors, I decided to construct a special
//  class called Vector3d instead of using an off-the-shelf n-dimensional
//  library. A large number of functions are included, many of
//  which are trivial, which should aid the writing of clear code.
//
//  Angle functions return in DEGREES. This was chosen instead of radians as 
//  I found that I always converted the answer to degrees in the
//  output, so this saves one step in thinking. Anyway, conversion
//  constants are provided
//
//  Adapted from Reduce (C) 1999 J. Michael Word
//  I eliminated the difference between position vectors and vectors,
//  which is a somewhat artificial distinction as a position vector is always
//  better to be thought of in terms of a displacement vector relative to 
//  an origin. Fixed the translation matrix bug 
//
//  **************************************************************************
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU Lesser General Public License as published
//  by the Free Software Foundation; either version 2.1 of the License, or (at
//  your option) any later version.
//  
//  This program is distributed in the hope that it will be useful,  but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details. 
//  
//  You should have received a copy of the GNU Lesser General Public License 
//  along with this program; if not, write to the Free Software Foundation, 
//  Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
//  **************************************************************************
//

#include <cmath>
#include <iostream>
#include "vector3d.h"

using namespace std;



Vector3d Vector3d::operator-(const Vector3d& p) const 
{
  return Vector3d(_x - p._x, _y - p._y, _z - p._z);
}



Vector3d Vector3d::operator+(const Vector3d& p) const 
{
  return Vector3d(_x + p._x, _y + p._y, _z + p._z);
}



Vector3d Vector3d::operator*(double s) const 
{
  return Vector3d(_x*s, _y*s, _z*s);
}



Vector3d Vector3d::operator/(double s) const 
{
  return (s != 0.0) ? Vector3d(_x/s, _y/s, _z/s) : Vector3d(*this);
}



Vector3d Vector3d::operator-() const 
{
  return Vector3d(-_x, -_y, -_z);
}



Vector3d& Vector3d::operator=(const Vector3d& p) 
{ 
  _x = p._x; 
	_y = p._y; 
	_z = p._z;
  return *this;
}



double Vector3d::lengthSquared() const 
{ 
  return (_x*_x) + (_y*_y) + (_z*_z);
}



double Vector3d::length() const 
{
  return sqrt(lengthSquared());
}



// scale a vector to unit length
Vector3d& Vector3d::normalize() 
{
   return Vector3d::operator/=(length());
}



// generate a vector from a prev one scaled to unit length
Vector3d Vector3d::normal() const 
{
   return Vector3d::operator/(length());
}



// scale a vector to a given length
Vector3d& Vector3d::scaleTo(double newlen) 
{
	return normalize() *= newlen;
}



// generate a vector from a prev one scaled to a given length
Vector3d Vector3d::scaled(double newlen) const 
{
	return normal() * newlen;
}



Vector3d& Vector3d::operator-=(const Vector3d& p) 
{
  _x -= p._x;
  _y -= p._y;
  _z -= p._z;
  return *this;
}



Vector3d& Vector3d::operator+=(const Vector3d& p) 
{
  _x += p._x;
  _y += p._y;
  _z += p._z;
  return *this;
}



Vector3d& Vector3d::operator*=(double s) 
{
  _x *= s;
  _y *= s;
  _z *= s;
  return *this;
}



Vector3d& Vector3d::operator/=(double s) 
{
  if (s != 0.0) 
  {
     _x /= s;
     _y /= s;
     _z /= s;
  }
  return *this;
}


//
// --------- distances and vectors
//

Vector3d cross(const Vector3d& a, const Vector3d& b) 
{
  return Vector3d( (a.y()*b.z()) - (a.z()*b.y()),
                   (a.z()*b.x()) - (a.x()*b.z()),
                   (a.x()*b.y()) - (a.y()*b.x()) );
}



Vector3d interpolate(const Vector3d& lo, const Vector3d& hi, double alpha) 
{
  return Vector3d(interpolate(lo.x(), hi.x(), alpha),
                  interpolate(lo.y(), hi.y(), alpha),
                  interpolate(lo.z(), hi.z(), alpha));
}



double distanceSquared(const Vector3d& a, const Vector3d& b) 
{
	Vector3d c = a - b;
  return c.lengthSquared();
}



double distance(const Vector3d& a, const Vector3d& b) 
{
	return sqrt(distanceSquared(a, b));
}



// ---------- components of vectors



Vector3d Vector3d::parallel(const Vector3d& axis) const 
{
  return axis * (dot(*this, axis) / axis.lengthSquared());
}


				
double Vector3d::parallelComponent(const Vector3d& axis) const 
{
  return (dot(*this, axis) / axis.length());
}


				
Vector3d Vector3d::perpendicular(const Vector3d& axis) const 
{
  return *this - parallel(axis);
}



Vector3d Vector3d::outPlane(const Vector3d& p1, const Vector3d& p2, 
                            const Vector3d& p3) const 
{
	Vector3d d = cross (p2 - p1, p3 - p2);

	if ( (d.length() == 0) || (length() == 0) ) 
		return Vector3d(0,0,0);

	return parallel(d);
}


 
Vector3d Vector3d::inPlane(const Vector3d& p1, const Vector3d& p2, 
                           const Vector3d& p3) const 
{
	return *this - outPlane(p1, p2, p3);
}



// ---------- angles and dihedrals


double Vector3d::planeAngle (const Vector3d& p1, const Vector3d& p2, 
                             const Vector3d& p3) const 
{
  Vector3d d = outPlane(p1, p2, p3);
  double ang = RAD2DEG * asin( dot(normal(), d.normal()) );
  if (parallelComponent (d) > 0) 
    return ang; 
  else 
    return -ang;
}



Vector3d Vector3d::rotated(double theta, const Vector3d& axis) const 
{
	Matrix3d rotmat(axis, theta);
	return rotmat * (*this);
}



Vector3d& Vector3d::rotate(double theta, const Vector3d& axis) 
{
	Matrix3d rot(axis, theta);
	*this = rot * (*this);
	return (*this);

}



Vector3d& Vector3d::applyMatrix(const Matrix3d& transformMatrix) 
{
  *this = transformMatrix * (*this);
	return *this;
}



// rotate point p1 theta degrees around the p3->p2 axis to yield a new point
Vector3d rotatePos(double theta, const Vector3d& p1, const Vector3d& p2,
                   const Vector3d& p3) 
{
	Matrix3d rotmat(p2 - p3, theta);
	return (rotmat * (p1 - p2)) + p2;
}



// calculate the angle (degrees) between 2 vectors
double angle(const Vector3d& a, const Vector3d& b)
{
	double amag = a.length();
	double bmag = b.length();

	if (amag * bmag < 0.0001) return 0.0;

	double c = dot(a, b) / amag / bmag;

	if (c >=  1.0) return   0.0;
	if (c <= -1.0) return 180.0;

	return (RAD2DEG * acos(c));
}



// calculate the angle (degrees) between 3 points

double anglePos(const Vector3d& p1, const Vector3d& p2, const Vector3d& p3) 
{
	return angle(p1-p2, p3-p2);
}



// dihedral angle between a and c through axis
double dihedral(const Vector3d& a, const Vector3d& axis, const Vector3d& c) {
	Vector3d ap = a.perpendicular(axis);
	Vector3d cp = c.perpendicular(axis);

	double ang = angle(ap, cp);

	if ( dot( cross(ap, cp), axis ) > 0 ) 
		return ( -ang );
	else 
		return ( ang );
}



// calculate the dihedral angle (degrees) given 4 points
double dihedralPos(const Vector3d& p1, const Vector3d& p2,
                  const Vector3d& p3, const Vector3d& p4) 
{
	return dihedral(p1-p2, p2-p3, p4-p3);			
}



// projects, from p1-p2-p3, the vector position p4 
Vector3d projectPos(double length, double angle, double dihedAngle, 
                  const Vector3d& p1, const Vector3d& p2, const Vector3d& p3) {
  Vector3d norm = planeNormal(p1, p2, p3);
  Vector3d axis = p3 - p2;

  Vector3d vecDiff = axis.scaled (-length);
  vecDiff = vecDiff.rotated(-angle, norm);                   
  vecDiff = vecDiff.rotated(dihedAngle, axis);

  return p3 + vecDiff; 
}



std::ostream& operator<<(std::ostream& os, const Vector3d& p) {
   os << "{" << p.x() << ", " << p.y() << ", " << p.z() << "}";
   return os;
}




// ---------- matrix related



// uniform scale matrix
Matrix3d::Matrix3d(double s) 
{ 
   makeIdentity();
   _element[0][0] = s;
   _element[1][1] = s;
   _element[2][2] = s;
}



// scale matrix
Matrix3d::Matrix3d(double sx, double sy, double sz)  
{ 
   makeIdentity();
   _element[0][0] = sx;
   _element[1][1] = sy;
   _element[2][2] = sz;
}



// translation matrix
Matrix3d::Matrix3d(const Vector3d& p) 
{ 
   makeIdentity();
   _element[0][3] = p.x();
   _element[1][3] = p.y();
   _element[2][3] = p.z();
}



// build a theta (degrees) angle rotation matrix with vector as an axis
Matrix3d::Matrix3d(const Vector3d& axis, double theta)  
{
  Vector3d v = axis.normal();
	const double x = v.x();
	const double y = v.y();
	const double z = v.z();

	const double c = cos(theta*DEG2RAD);
	const double s = sin(theta*DEG2RAD);
	const double t = 1.0 - c;

	makeIdentity();

	_element[0][0] = t * x * x    +        c;
	_element[0][1] = t * x * y    +    z * s;
	_element[0][2] = t * x * z    -    y * s;

	_element[1][0] = t * y * x    -    z * s;
	_element[1][1] = t * y * y    +        c;
	_element[1][2] = t * y * z    +    x * s;

	_element[2][0] = t * z * x    +    y * s;
	_element[2][1] = t * z * y    -    x * s;
	_element[2][2] = t * z * z    +        c;
}



// build a theta (degrees) angle rotation matrix with vector as an axis
Matrix3d::Matrix3d(const Vector3d& axis, double theta, const Vector3d& center)  
{
  Matrix3d rot(axis, theta);
	Vector3d disp = center - rot*center;
	*this = rot;
  _element[0][3] = disp.x();
  _element[1][3] = disp.y();
  _element[2][3] = disp.z();
}



// build an identity matrix
Matrix3d& Matrix3d::makeIdentity() 
{ 
  _element[0][0] = 1.0;
  _element[0][1] = 0.0;
  _element[0][2] = 0.0;
  _element[0][3] = 0.0;

  _element[1][0] = 0.0;
  _element[1][1] = 1.0;
  _element[1][2] = 0.0;
  _element[1][3] = 0.0;

  _element[2][0] = 0.0;
  _element[2][1] = 0.0;
  _element[2][2] = 1.0;
  _element[2][3] = 0.0;

  _element[3][0] = 0.0;
  _element[3][1] = 0.0;
  _element[3][2] = 0.0;
  _element[3][3] = 1.0;

  return *this;
}



Vector3d Matrix3d::operator * (const Vector3d& v) const 
{
  double x, y, z;
  x = (v.x() * _element[0][0]) +
      (v.y() * _element[1][0]) +
      (v.z() * _element[2][0]) +
               _element[0][3]; 
  y = (v.x() * _element[0][1]) +
      (v.y() * _element[1][1]) +
      (v.z() * _element[2][1]) +
               _element[1][3]; 
  z = (v.x() * _element[0][2]) +
      (v.y() * _element[1][2]) +
      (v.z() * _element[2][2]) +
               _element[2][3]; 

  return Vector3d(x, y, z);
}



Matrix3d Matrix3d::operator * (const Matrix3d& m2) const 
{
  Matrix3d c;
  int i, j, k;

  for (i = 0; i < 3; i++) 
	{
    for (j = 0; j < 3; j++) 
    {
      c._element[i][j] = 0.0;
      for (k = 0; k < 3; k++) 
		  {
         c._element[i][j] += _element[i][k] * m2._element[k][j];
      }
    }
  }

  for (i = 0; i < 3; i++) 
	{
    c._element[i][3] = _element[i][3];
    for (j = 0; j < 3; j++) 
		{
      c._element[i][3] += _element[j][i] * m2._element[j][3];
    }
  }

  return c;
}



std::ostream& operator << (std::ostream& os, const Matrix3d& m) 
{
   os << "{{" << m._element[0][0] << ", " << m._element[0][1] << ", "
              << m._element[0][2] << ", " << m._element[0][3] << "},";
   os << " {" << m._element[1][0] << ", " << m._element[1][1] << ", "
              << m._element[1][2] << ", " << m._element[1][3] << "},";
   os << " {" << m._element[2][0] << ", " << m._element[2][1] << ", "
              << m._element[2][2] << ", " << m._element[2][3] << "},";
   os << " {" << m._element[3][0] << ", " << m._element[3][1] << ", "
              << m._element[3][2] << ", " << m._element[3][3] << "}}";
   return os;
}






