//
//  **************************************************************************
//
//  protein.cpp
//
//  (C) 2003 Bosco K. Ho 
//
//	The purpose of the Protein class is to provide a convenient 
//	structure to access the atoms via the Resblk structures without 
//	disturbing the underlying structure of the atoms.
//	The ResBlk structure allows you to access the atom by 
//	residue number and atom-type. 
//
//  There are basically three categories of routines
//  
//  a) routines to read a pdb file and extract dihedral angles
//  b) routines to rotate and change internal parameters of a loaded
//     pdb file
//  c) routines to generate the residues de novo
//	d) various diagnostics to check protein integrity
//
//	In terms of de-novo strucutre determination, the information
//	to generate the structures are kept in the code instead of using
//	extensive data structures, I find this is easier to read and 
//	highlights the similarity between various amino acids.
//
//	Another important question is how do we index the residues and
//	chi angles? In C++, one assumes the count to start from 0 whereas
//	in most other cases, one starts counting from one.
//
//  Adapted from the The Biomolecule Toolkit (C) 2001, Eric Alm 
//
//  Cleaned the abstraction interface. Introduced a key for atoms
//  as well as for amino acids. This avoids the clumsy method of
//  using strings to identify atoms. There is too much confusion in the
//  peptide class since there is much interface between the protein
//  and peptide class. Consequently, the peptide class has been changed
//  to a lookup table class with little intrinsic functionality. 
//  Got rid of templates.
//
//
//  24/2/2004 fixed the HA1 and HA2 hydrogen sprout
//
//  **************************************************************************
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU Lesser General Public License as published
//  by the Free Software Foundation; either version 2.1 of the License, or (at
//  your option) any later version.
//  
//  This program is distributed in the hope that it will be useful,  but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details. 
//  
//  You should have received a copy of the GNU Lesser General Public License 
//  along with this program; if not, write to the Free Software Foundation, 
//  Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
//  **************************************************************************
//

#include <iostream>
#include <iomanip>
#include <sstream>
#include <string>
#include <vector>
#include <deque>
#include <list>
#include "protein.h"
#include "vector3d.h"

using namespace std;



// copy constructor
Protein::Protein(const Protein& rhs) : Molecule(rhs), _resList() 
{
  scanResBlk();
}



// constructs the resBlk entries from the flat atom structures
// in the parent Molecule class
void Protein::scanResBlk() 
{
	_resList.clear();

  int thisResNo = atomBegin()->resNo; 
	int i = -1;
	ResBlk resBlk;
	for (AtomIter a = atomBegin(); a != atomEnd(); a++) {
    // check when residue number changes
    if ((a->resNo != thisResNo) || i == -1) 
		{ 
      // first set the atomEnd of the previous residue
			if (i>-1) 
			  _resList[i]._atomEnd = a;

      thisResNo              = a->resNo;
			resBlk.resNo           = a->resNo;  
			resBlk.resKey          = a->resKey;  
			resBlk.resName         = a->resName;  

      _resList.push_back(resBlk);
			++i;
    }
		_resList[i].pushAtomIter(a);
  }
	if (i>-1) _resList[i]._atomEnd = atomEnd();
}



// extracts the phi, psi, chi and omega dihedral angles 
void Protein::extractAngles() 
{
	for (int i = 1; i <= nRes(); i++) 
	{
		if (i > 1) 
		{
			res(i).phi   = dihedralPos(atom(i-1, C).pos, atom(i,  N).pos, 
			                           atom(i,  CA).pos, atom(i,  C).pos);
			res(i).omega = dihedralPos(atom(i-1,CA).pos, atom(i-1,C).pos, 
			                           atom(i,   N).pos, atom(i, CA).pos);
		}
		else 
		{
      if (res(1).resKey == PRO)
      {
     		if (hasAtom(1, CD1))
          res(1).phi   = 180.0 + dihedralPos(atom(1, CD1).pos, atom(1,  N).pos, 
			                           atom(1,  CA).pos, atom(1,  C).pos);
        else
          res(1).phi   = 0;
      }
      else
      {
     		if (hasAtom(1, H))
          res(1).phi   = 180.0 + dihedralPos(atom(1, H).pos, atom(1,  N).pos, 
			                     atom(1,  CA).pos, atom(1, C).pos);
        else
          res(1).phi   = 0;
      }
      
			res(1).omega = 180;
		}

		if (i <= nRes() - 1) 
		{
			res(i).psi   = dihedralPos(atom(i,  N).pos, atom(i,  CA).pos, 
			                           atom(i,  C).pos, atom(i+1, N).pos);
		}
		else 
		{
			res(i).psi = dihedralPos(atom(i,  N).pos, atom(i,  CA).pos, 
                               atom(i,  C).pos, atom(i, O).pos) - 180;
		}

		int& resKey = res(i).resKey;
		for (int n = 1; n <= getNChi(resKey); n++) 
		{
			Vector3d p0 = atom(i, resChiTetrad[resKey][n-1][0]).pos;
			Vector3d p1 = atom(i, resChiTetrad[resKey][n-1][1]).pos;
			Vector3d p2 = atom(i, resChiTetrad[resKey][n-1][2]).pos;
			Vector3d p3 = atom(i, resChiTetrad[resKey][n-1][3]).pos;

			res(i).chi(n) = dihedralPos(p0, p1, p2, p3);
		}
	}
}



// checks for missing atoms according to standard list
void Protein::checkMissingAtoms() 
{
  for (int i=1; i<=nRes(); i++) 
	{
    int resKey = _resList[i-1].resKey;
    for (int j = 0; resAllowedAtoms[resKey][j] != XX; j++) 
		{
      if (! hasAtom(i, resAllowedAtoms[resKey][j]))
        cerr << "Warning: Missing " 
             << getAtomName(resKey, resAllowedAtoms[resKey][j])
             << " in " << getResName3(resKey)
             << _resList[i-1].resNo << endl;
    }

    AtomIterList::iterator iter;
    for (iter = _resList[i-1]._atomIterList.begin(); 
         iter != _resList[i-1]._atomIterList.end(); iter++) 
		{
      if ( (!isAllowedAtom(resKey, (*iter)->atomKey))
         && (!isTitrableAtom(resKey, (*iter)->atomKey)) )
      {
			  cerr << "Warning: Extra " << (*iter)->atomName
             << " in " << getResName3(resKey)
             << _resList[i-1].resNo << endl;
			}
    }
  }
}



// -------- access functions



Atom& Protein::operator () (int resIndex, int atomKey) {
	return atom(resIndex, atomKey);
}



Atom& Protein::atom(int resIndex, int atomKey) 
{
  AtomIterList::iterator iter = _resList[resIndex-1]._atomIterList.begin(); 
  AtomIterList::iterator end  = _resList[resIndex-1]._atomIterList.end(); 
  for (iter; iter != end; iter++) 
		if ((*iter)->atomKey == atomKey) return *(*iter);
	return nullAtom;
}



ResBlk& Protein::res(int resIndex) 
{
  return _resList[resIndex-1];
}



bool Protein::hasAtom(int resIndex, int atomKey) 
{
  if ((resIndex <=0) || (resIndex > nRes())) return false;

  for (int i=0; i < _resList[resIndex-1].nAtoms(); i++)
    if (_resList[resIndex-1]._atomIterList[i]->atomKey == atomKey) 
      return true;

  return false;
}



// constructs a string of the form 'LYS:32-CA'
string Protein::fullAtomName(int i, int a)
{
  ostringstream os;
  os << res(i).resName << ":" << res(i).resNo; 
  string aName = res(i).atom(a).atomName;
	while (' ' == aName[0]) aName.erase(0,1);
	return os.str() + "-" + aName; 
}




double Protein::getMaxLength(void)
{
  double lengthSquared = 0.0;
  for (int i = 1; i <= nRes(); i++)
  {
    for (int j = i+1; j <= nRes(); j++)
    {
       double newLengthSquared = distanceSquared(atom(i, CA).pos, atom(j, CA).pos);
       if (newLengthSquared > lengthSquared)
         lengthSquared = newLengthSquared;
    }
  }
  return sqrt(lengthSquared);
}




// -------- insertion/deletion of atoms using resIndex





// inserts atom into the molecule and updates residue block
void Protein::insertAtom(int resIndex, Atom a) 
{
  a.resKey              = _resList[resIndex-1].resKey; 
  a.resName             = _resList[resIndex-1].resName; 
  a.resNo               = _resList[resIndex-1].resNo; 
  AtomIter& insertPoint = _resList[resIndex-1].atomEnd();

  AtomIter newAtomIter = _atomList.insert(insertPoint, a);
	_resList[resIndex-1].pushAtomIter(newAtomIter);
}


  
// inserts atom into the molecule using a key
void Protein::insertAtom(int resIndex, int atomKey) 
{
  Atom a;
  a.atomKey  = atomKey; 
  a.atomName = getAtomName(_resList[resIndex-1].resKey, atomKey); 
  insertAtom(resIndex, a);
}



// inserts atom into the molecule and updates residue block
void Protein::eraseAtom(int resIndex, int atomKey) 
{
	AtomIterList::iterator iter;
	for (iter = _resList[resIndex-1]._atomIterList.begin(); 
	  iter != _resList[resIndex-1]._atomIterList.end(); iter++) {
		 AtomIter a = *iter;
		 if (a->atomKey == atomKey) {
			 _resList[resIndex-1]._atomIterList.erase(iter);
			 _atomList.erase(a);
			 return;
		 }
	}
}



// -------- insertion/deletion of residues



// checks if resBlk has all the atoms in the standard 
// representation of the amino acids, if not, inserts the
// atoms
void Protein::initializeAtomsInRes(int resIndex) 
{
  int resKey = _resList[resIndex-1].resKey;
  for (int j = 0; resAllowedAtoms[resKey][j] != XX; j++) 
	{
    if (! hasAtom(resIndex, resAllowedAtoms[resKey][j])) 
		{
			Atom a;
			a.resKey   = resKey;
			a.resName  = _resList[resIndex-1].resName;
			a.resNo    = _resList[resIndex-1].resNo;
			a.atomKey  = resAllowedAtoms[resKey][j];
			a.atomName = getAtomName(resKey, a.atomKey);
			insertAtom(resIndex, a);
		}
	}
}



void Protein::eraseExcessAtomsInRes(int i) 
{
  int resKey = _resList[i-1].resKey;
	for (int j=_resList[i-1].nAtoms()-1; j>= 0; j--) 
	{
		AtomIter a = _resList[i-1]._atomIterList[j];
    if ( (!isAllowedAtom(resKey,  a->atomKey))
      && (!isTitrableAtom(resKey, a->atomKey)) ) 
		{
        eraseAtom(i, a->atomKey);
		}
  }
}

	

void Protein::mutateRes(int resIndex, int resKey) 
{
	if (resIndex > nRes()) return;
	res(resIndex).resKey = resKey;
  res(resIndex).resName = getResName3(resKey);

  for (int i=1; i<=res(resIndex).nAtoms(); i+=1)
  {
    AtomIter& a = res(resIndex)._atomIterList[i-1];
    a->resKey = res(resIndex).resKey;
    a->resName = res(resIndex).resName;
  }

  initializeAtomsInRes(resIndex);
  eraseExcessAtomsInRes(resIndex);
}



void Protein::insertRes(int i, int resKey) 
{

	if (i > nRes()) return;

	ResBlk r;
	_resList.push_back(r); 

  // copies every residue from i to i+1 and updates resNo 
	if (nRes() == 0) 
	{
		_resList[0].resNo = 1;
	}
	else
	{
  	for (int j = nRes()-1; j >= i; j--) 
	  {
  		_resList[j] = _resList[j-1];
	  	_resList[j].resNo += 1;
		  for (AtomIter a = _resList[j].atomBegin(); a != _resList[j].atomEnd(); a++)
  		{
	  	  a->resNo += 1;
		  }
    }
	}

	_resList[i-1].resKey  = resKey;
	_resList[i-1].resName = getResName3(resKey);

  // intialize dihedral angles
	_resList[i-1].phi     = 180;
	_resList[i-1].psi     = 180;
	_resList[i-1].omega   = 180;
	_resList[i-1]._chi[0] = 180;
	_resList[i-1]._chi[1] = 180;
	_resList[i-1]._chi[2] = 180;
	_resList[i-1]._chi[3] = 180;
	_resList[i-1]._chi[4] = 180;

  // makes sure that atomEnd matches the next resBlk 
  _resList[i-1]._atomEnd = _resList[i].atomBegin();

  // create new entries 
	_resList[i-1]._atomIterList.clear();
	initializeAtomsInRes(i);

  // makes sure that the prev atomEnd is
	// pointing in the right place
  if (i > 1)
	  _resList[i-2]._atomEnd = _resList[i-1].atomBegin();
}



void Protein::appendRes(int resKey) 
{
	ResBlk r;

	r.resKey = resKey;
	r.resName = getResName3(resKey);

	if (nRes() == 0)
		r.resNo = 1;
	else
		r.resNo = _resList[nRes()-1].resNo + 1;

	r.phi     = 180;
	r.psi     = 180;
	r.omega   = 180;
	r._chi[0]	=	180;
	r._chi[1]	=	180;
	r._chi[2]	=	180;
	r._chi[3]	=	180;
	r._chi[4]	=	180;
		
	_resList.push_back(r);

	_resList[nRes()-1]._atomEnd = atomEnd();
	initializeAtomsInRes(nRes());
  if (nRes() > 1)
  	_resList[nRes()-2]._atomEnd = _resList[nRes()-1].atomBegin();
}



void Protein::eraseRes(int resIndex) 
{
	if (resIndex > nRes()) return;

	AtomIter first = _resList[resIndex-1].atomBegin();
  AtomIter last = _resList[resIndex-1].atomEnd();
	_atomList.erase(first, last);
	
	for (int n = resIndex; n < nRes(); n++) 
	{
		_resList[n-1] = _resList[n];			
		_resList[n-1].resNo -= 1;			
	  for (AtomIter a = _resList[n-1].atomBegin(); a != _resList[n-1].atomEnd(); a++)
 		{
  	  a->resNo -= 1;
	  }
	}
	_resList.pop_back();

	if (resIndex > 1) 
		_resList[resIndex-2]._atomEnd = last;
} 



// -------- de novo build routines



// standard covalent bond lengths
#define N_C			      1.325
#define N_H			      1.000
#define C_O			      1.230
#define C_H			      1.100
#define CA_C		      1.530
#define CA_N		      1.453
#define CA_CB		      1.530
#define CB_S		      1.810
#define CB_HB		      1.100
#define C_N			      1.300
#define PRO_CA_CB		  1.500
#define PRO_CB_CG		  1.470
#define PRO_CD_N	    1.470

// standard covalent bond angles
#define C_C_S			    112.7
#define CA_C_O				120.5
#define CA_N_H				115.0
#define CA_C_N				115.0
#define CA_N_C				121.0
#define C_CA_N				109.3
#define CA_CB_HB			117.0
#define CA_CB_CG			113.8
#define PRO_CA_N_CD		113.0
#define PRO_CA_CB_CG	106.6
#define PRO_CHI1		  -6.11



void Protein::rebuildBackboneFromScratch(int i) 
{
	atom(i,CA).pos = Vector3d(0.0, 0.0, 1.2);

	Vector3d w(-1.0, 0.0, 0.0);

	Vector3d ca_n(0.0, 0.0, -1.0);
	ca_n.rotated(-C_CA_N * 0.5, w);
	ca_n.scaleTo(CA_N);

	atom(i,N).pos = atom(i,CA).pos + ca_n;

	Vector3d ca_c = ca_n.rotated(+C_CA_N, w);
	ca_c.scaleTo(CA_C);
	atom(i,C).pos = atom(i,CA).pos + ca_c;
}



void Protein::sproutBackboneFromPrev(int i) 
{
  res(i).atom(N).pos  = projectPos(1.325, 115.0, res(i-1).psi, 
                                                 res(i-1).atom(N).pos, 
                                                 res(i-1).atom(CA).pos, 
                                                 res(i-1).atom(C).pos);

  res(i).atom(CA).pos = projectPos(1.453, 121.0, res(i).omega, 
                                                 res(i-1).atom(CA).pos, 
                                                 res(i-1).atom(C).pos, 
                                                 res(i).atom(N).pos);
  
  res(i).atom(C).pos  = projectPos(1.530, 109.3, res(i).phi, 
                                                 res(i-1).atom(C).pos, 
                                                 res(i).atom(N).pos, 
                                                 res(i).atom(CA).pos);
}

  
	
inline void Protein::projectAtom(int i, int a1, double length, double angle, double dihed,
        												 int a2, int a3, int a4) 
{
	if (hasAtom(i,a1))
		res(i).atom(a1).pos  = projectPos(length, angle, dihed,
		  res(i).atom(a2).pos, res(i).atom(a3).pos, res(i).atom(a4).pos);
}



void Protein::sproutSidechain(int i, int n) 
{
	switch (n) 
	{
	  case 0: 
      {
				projectAtom(i, H, N_H, CA_N_H, 180 + res(i).phi, C, CA, N);
				projectAtom(i, H2,N_H, CA_N_H, 180 + res(i).phi + 120, C, CA, N);
				projectAtom(i, H3,N_H, CA_N_H, 180 + res(i).phi + 240, C, CA, N);

        projectAtom(i, O, C_O, CA_C_O, 180 + res(i).psi, N, CA, C);
        projectAtom(i, OXT, C_O, CA_C_O, 180 + res(i).psi + 180, N, CA, C);
        projectAtom(i, HXT, 1.00, CA_C_O, 180, N, CA, C);

        Vector3d ca_n = atom(i,N).pos - atom(i,CA).pos;
        Vector3d ca_c = atom(i,C).pos - atom(i,CA).pos;
				Vector3d v    = ca_n.scaleTo(1.0) + ca_c.scaleTo(1.0);
				Vector3d w    = ca_n.scaleTo(1.0) - ca_c.scaleTo(1.0);

				Vector3d ca_ha = v.rotated(-125.5, w).scaleTo(1.1);
				atom(i,HA1).pos = atom(i,CA).pos + ca_ha;

				Vector3d ca_cb = v.rotated(125.5, w);

				if (res(i).resKey == GLY)
				{
          atom(i,HA2).pos = atom(i,CA).pos + ca_cb.scaleTo(CB_HB);
        }
				else if (res(i).resKey == PRO) 
				{
          atom(i,CB).pos = atom(i,CA).pos + ca_cb.scaleTo(1.53);
					projectAtom(i,  CG1, 1.470, 106.6,  -6.11,  N, CA, CB);
					projectAtom(i,  HB1, C_H, 109.0,  -6.11 + 120,  N, CA, CB);
					projectAtom(i,  HB2, C_H, 109.0,  -6.11 + 240,  N, CA, CB);

					projectAtom(i,  HG11, C_H, 109.0, 120, CA, CB, CG1);
					projectAtom(i,  HG12, C_H, 109.0, 240, CA, CB, CG1);

					projectAtom(i,  CD1, 1.460,  113.0,  0.0, CB, CA,  N);
					projectAtom(i,  HD11, C_H, 109.0,  120, CA,  N, CD1);
					projectAtom(i,  HD12, C_H, 109.0,  240, CA,  N, CD1);
        }
				else if (res(i).resKey == ALA) 
				{
          atom(i,CB).pos = atom(i,CA).pos + ca_cb.scaleTo(CA_CB);
					projectAtom(i, HB1, 1.1, 109.0,  60, N, CA, CB);
					projectAtom(i, HB2, 1.1, 109.0, 180, N, CA, CB);
					projectAtom(i, HB3, 1.1, 109.0, 300, N, CA, CB);
				}
        else
				{
          atom(i,CB).pos = atom(i,CA).pos + ca_cb.scaleTo(CA_CB);
				}

      }
			break;
		case 1:
      // step one for chi(1)
			switch (res(i).resKey) 
			{
				case ASP:
				case ASN:
				case HIS:
				case LEU:
				case GLN:
				case GLU:
				case ARG:
				case LYS:
				case TRP:
				case TYR:
				case PHE:
				case MET:
					projectAtom(i,  CG1, 1.530, 111.0, res(i).chi(1), N, CA, CB);
					break;
				case ILE:
					projectAtom(i,  CG1, 1.530, 111.0, res(i).chi(1),        N, CA, CB);
					projectAtom(i,  HB1, C_H, 109.0, res(i).chi(1) + 120,  N, CA, CB);
					projectAtom(i,  CG2, 1.530, 111.0, res(i).chi(1) + 240,  N, CA, CB);
					break;
				case SER:
					projectAtom(i,  OG1, 1.425, 112.0, res(i).chi(1),        N, CA, CB);
					projectAtom(i,  HB1, C_H, 109.0, res(i).chi(1) + 120,  N, CA, CB);
					projectAtom(i,  HB2, C_H, 109.0, res(i).chi(1) + 240,  N, CA, CB);

					projectAtom(i, HG11, 1.000, 110.0,                 180, CA, CB, OG1);
					break;
				case CYS:
					projectAtom(i,  SG1, 1.830, 110.0, res(i).chi(1),  N, CA,  CB);
					projectAtom(i, HG11, 1.350,  96.0,             0, CA, CB, SG1);
					break;
				case THR:
					projectAtom(i,  OG1, 1.425, 104.0, res(i).chi(1),        N, CA, CB);
					projectAtom(i,  HB1, C_H, 109.0, res(i).chi(1) + 120,  N, CA, CB);
					projectAtom(i,  CG2, 1.530, 111.0, res(i).chi(1) + 240,  N, CA, CB);
					projectAtom(i, HG11, 1.000, 110.0, 180,                 CA, CB, OG1);
					break;
				case VAL:
					projectAtom(i,  CG1, 1.530, 111.0, res(i).chi(1),        N, CA, CB);
					projectAtom(i,  HB1, C_H, 109.0, res(i).chi(1) + 240,  N, CA, CB);
					projectAtom(i,  CG2, 1.530, 111.0, res(i).chi(1) + 120,  N, CA, CB);


					projectAtom(i, HG11, C_H, 109.0,  60, CA, CB, CG1);
					projectAtom(i, HG12, C_H, 109.0, 180, CA, CB, CG1);
					projectAtom(i, HG13, C_H, 109.0, 300, CA, CB, CG1);
					break;
			}
      // step two for chi(1)
			switch (res(i).resKey) 
			{
				case ASP:
				case ASN:
				case HIS:
				case LEU:
				case GLN:
				case GLU:
				case LYS:
				case ARG:
				case TRP:
				case TYR:
				case PHE:
				case MET:
				case CYS:
					projectAtom(i,  HB1, C_H, 109.0, res(i).chi(1) + 120,  N, CA, CB);
					projectAtom(i,  HB2, C_H, 109.0, res(i).chi(1) + 240,  N, CA, CB);
					break;
				case THR:
				case ILE:
        case VAL:
					projectAtom(i, HG21, C_H, 109.0,  60, CA, CB, CG2);
					projectAtom(i, HG22, C_H, 109.0, 180, CA, CB, CG2);
					projectAtom(i, HG23, C_H, 109.0, 300, CA, CB, CG2);
					break;
			}
			break;
		case 2:
      // step one for chi(2)
			switch (res(i).resKey) 
			{
				case ASN:
					projectAtom(i,  OD1, 1.220, 120.0, res(i).chi(2)      , CA, CB, CG1);
					projectAtom(i,  ND2, 1.330, 120.0, res(i).chi(2) + 180, CA, CB, CG1);

					projectAtom(i, HD21, 1.000, 120.0,    0, CB, CG1, ND2);
					projectAtom(i, HD22, 1.000, 120.0,  180, CB, CG1, ND2);
					break;
				case ASP:
					projectAtom(i,  OD1, 1.220, 120.0, res(i).chi(2)      , CA, CB, CG1);
					projectAtom(i,  OD2, 1.220, 120.0, res(i).chi(2) + 180, CA, CB, CG1);
					break;
				case ILE:
					projectAtom(i,  CD1, 1.530, 111.0, res(i).chi(2),       CA, CB, CG1);
					projectAtom(i, HG11, C_H, 109.0, res(i).chi(2) + 120, CA, CB, CG1);
					projectAtom(i, HG12, C_H, 109.0, res(i).chi(2) + 240, CA, CB, CG1);

					projectAtom(i, HD11, C_H, 109.0,   60, CB, CG1, CD1);

					projectAtom(i, HD12, C_H, 109.0,  180, CB, CG1, CD1);
					projectAtom(i, HD13, C_H, 109.0,  300, CB, CG1, CD1);
					break;
				case LEU:
					projectAtom(i,  CD1, 1.530, 111.0, res(i).chi(2),       CA, CB, CG1);
					projectAtom(i,  CD2, 1.530, 111.0, res(i).chi(2) + 120, CA, CB, CG1);
					projectAtom(i, HG11, C_H, 109.0, res(i).chi(2) + 240, CA, CB, CG1);

					projectAtom(i, HD11, C_H, 109.0,   60, CB, CG1, CD1);
					projectAtom(i, HD12, C_H, 109.0,  180, CB, CG1, CD1);
					projectAtom(i, HD13, C_H, 109.0,  300, CB, CG1, CD1);

					projectAtom(i, HD21, C_H, 109.0,   60, CB, CG1, CD2);
					projectAtom(i, HD22, C_H, 109.0,  180, CB, CG1, CD2);
					projectAtom(i, HD23, C_H, 109.0,  300, CB, CG1, CD2);
					break;
				case GLN:
				case GLU:
				case LYS:
				case ARG:
					projectAtom(i,  CD1, 1.530, 111.0, res(i).chi(2),       CA, CB, CG1);
					projectAtom(i, HG11, C_H, 109.0, res(i).chi(2) + 120, CA, CB, CG1);
					projectAtom(i, HG12, C_H, 109.0, res(i).chi(2) + 240, CA, CB, CG1);
					break;
				case MET:

					projectAtom(i,  SD1, 1.530, 109.0, res(i).chi(2),       CA, CB, CG1);
					projectAtom(i, HG11, C_H, 109.0, res(i).chi(2) + 120, CA, CB, CG1);
					projectAtom(i, HG12, C_H, 109.0, res(i).chi(2) + 240, CA, CB, CG1);
					break;
				case TRP:
					projectAtom(i,  CD1, 1.340,  128.0,   res(i).chi(2),   CA, CB, CG1);
					projectAtom(i,  CD2, 1.443, -126.6,   res(i).chi(2),   CA, CB, CG1);

					projectAtom(i,  NE1, 1.380,  111.5, 180,   CB, CG1, CD1);
					projectAtom(i,  CE2, 1.390,  107.4,   0,  CG1, CD1, NE1);

					projectAtom(i,  CE3, 1.410,  129.6,   0,   CB, CG1, CD2);
					projectAtom(i,  CZ3, 1.400,  115.4, 180,  CG1, CD2, CE3);
					projectAtom(i,  CH2, 1.390,  124.6,   0,  CD2, CE3, CZ3);
					projectAtom(i,  CZ2, 1.400,  129.6, 180,  CD1, NE1, CE2);

					projectAtom(i, HD11, C_H,  124.3,   0,   CB, CG1, CD1);
					projectAtom(i, HE11, 1.000,  126.8, 180,  CG1, CD1, NE1);
					projectAtom(i, HZ21, C_H,  125.8, 180,  CD2, CE2, CZ2);
					projectAtom(i, HH21, C_H,  125.8, 180,  CE2, CZ2, CH2);
					projectAtom(i, HZ31, C_H,  125.8, 180,  CD2, CE3, CZ3);
					projectAtom(i, HE31, C_H,  125.8,   0,  CG1, CD2, CE3);
					break;
				case HIS:
					projectAtom(i,  ND1, 1.384,  122.8,   res(i).chi(2),   CA, CB, CG1);
					projectAtom(i,  CE1, 1.325,  108.4,   180,   CB, CG1, ND1);
					projectAtom(i,  NE2, 1.364,  108.4,     0,  CG1, ND1, CE1);
					projectAtom(i,  CD2, 1.530,  108.4,     0,  ND1, CE1, NE2);

					projectAtom(i, HD11, C_H,  125.8,     0,   CB, CG1, ND1);
					projectAtom(i, HE11, 1.000,  125.8,   180,  CG1, ND1, CE1);
					projectAtom(i, HD21, C_H,  125.8,     0,   CB, CG1, CD2);
					projectAtom(i, HE21, 1.000,  125.8,   180,  ND1, CE1, NE2);
					break; 
				case TYR:
				case PHE:
					projectAtom(  i,  CD1, 1.360, 120.0,  res(i).chi(2),  CA,  CB, CG1);
					projectAtom(  i,  CD2, 1.360,-120.0,  res(i).chi(2),  CA,  CB, CG1);
					projectAtom(  i,  CE1, 1.420, 120.0,  180,  CB, CG1, CD1);
					projectAtom(  i,  CE2, 1.420, 120.0,  180,  CB, CG1, CD2);
					projectAtom(  i,  CZ1, 1.360, 120.0,    0, CG1, CD1, CE1);

					projectAtom(  i, HD11, C_H, 120.0,    0,  CB, CG1, CD1);
					projectAtom(  i, HD21, C_H, 120.0,    0,  CB, CG1, CD2);

					projectAtom(  i, HE11, C_H, 120.0,  180, CG1, CD1, CE1);
					projectAtom(  i, HE21, C_H, 120.0,  180, CG1, CD2, CE2);

					if (res(i).resKey == PHE)
						projectAtom(i, HZ11, C_H, 120.0,  180, CD1, CE1, CZ1);
					else 
					{
						projectAtom(i,  OH1, 1.360, 120.0,  180, CD1, CE1, CZ1);
						projectAtom(i, HH11, C_H, 110.0,  180, CE1, CZ1, OH1);
          }

					break;
			}
			break;
		case 3:
			switch (res(i).resKey) 
			{
				case GLN:
					projectAtom(i,  OE1, 1.220, 120.0, res(i).chi(3)      , CB, CG1, CD1);
					projectAtom(i,  NE2, 1.330, 120.0, res(i).chi(3) + 180, CB, CG1, CD1);

					projectAtom(i, HE21, 1.000, 120.0,    0, CG1, CD1, NE2);
					projectAtom(i, HE22, 1.000, 120.0,  180, CG1, CD1, NE2);
					break;
				case GLU:
					projectAtom(i,  OE1, 1.240, 120.0, res(i).chi(3)      , CB, CG1, CD1);
					projectAtom(i,  OE2, 1.240, 120.0, res(i).chi(3) + 180, CB, CG1, CD1);
					break;
				case ARG:
					projectAtom(i,  NE1, 1.530, 109.0, res(i).chi(3)      , CB, CG1, CD1);
					projectAtom(i, HD11, C_H, 109.0, res(i).chi(3) + 120, CB, CG1, CD1);
					projectAtom(i, HD12, C_H, 109.0, res(i).chi(3) + 240, CB, CG1, CD1);
					break;
				case LYS:
					projectAtom(i,  CE1, 1.530, 109.0, res(i).chi(3)      , CB, CG1, CD1);
					projectAtom(i, HD11, C_H, 109.0, res(i).chi(3) + 120, CB, CG1, CD1);
					projectAtom(i, HD12, C_H, 109.0, res(i).chi(3) + 240, CB, CG1, CD1);
					break;
				case MET:
					projectAtom(i,  CE1, 1.530, 109.0, res(i).chi(3)      , CB, CG1, SD1);
					projectAtom(i, HD11, C_H, 109.0, res(i).chi(3) + 120, CB, CG1, SD1);
					projectAtom(i, HD12, C_H, 109.0, res(i).chi(3) + 240, CB, CG1, SD1);

					projectAtom(i, HE11, C_H, 109.0,  60, CG1, SD1, CE1);
					projectAtom(i, HE12, C_H, 109.0, 180, CG1, SD1, CE1);
					projectAtom(i, HE13, C_H, 109.0, 300, CG1, SD1, CE1);
					break;
			}
			break;
		case 4:
			switch (res(i).resKey) 
			{
				case ARG:
					projectAtom(i,  CZ1, 1.530, 120.0, res(i).chi(4), CG1, CD1, NE1);
					projectAtom(i, HE11, C_H, 240.0, res(i).chi(4), CG1, CD1, NE1);

					projectAtom(i,  NH1, 1.530, 120.0,   0,  CD1, NE1, CZ1);
					projectAtom(i,  NH2, 1.530, 120.0, 180,  CD1, NE1, CZ1);

					projectAtom(i, HH11, 1.000, 120.0,   0,  NE1, CZ1, NH1);
					projectAtom(i, HH12, 1.000, 120.0, 180,  NE1, CZ1, NH1);

					projectAtom(i, HH21, 1.000, 120.0,   0,  NE1, CZ1, NH2);
					projectAtom(i, HH22, 1.000, 120.0, 180,  NE1, CZ1, NH2);

					break;
				case LYS:
					projectAtom(i,  NZ1, 1.530, 111.0, res(i).chi(4),       CG1, CD1, CE1);
					projectAtom(i, HE11, C_H, 109.0, res(i).chi(4) + 120, CG1, CD1, CE1);
					projectAtom(i, HE12, C_H, 109.0, res(i).chi(4) + 240, CG1, CD1, CE1);

					projectAtom(i, HZ11, C_H, 109.0,   60, CD1, CE1, NZ1);
					projectAtom(i, HZ12, C_H, 109.0,  180, CD1, CE1, NZ1);
					projectAtom(i, HZ13, C_H, 109.0,  300, CD1, CE1, NZ1);
					break;
			}
			break;
	}
}



void Protein::sproutWholeSidechain(int i) 
{
  sproutSidechain(i, 0);
	for (int n = 1; n <= getNChi(res(i).resKey); n++)
	  sproutSidechain(i, n);
}


	
	
inline void Protein::projectHydrogen(
                         int i, int a1, 
												 double length, double angle, double dihed,
                         int a2, int a3, int a4) 
{
	if (!hasAtom(i,a1)) 
	{
    insertAtom(i, a1);
  	if (hasAtom(i,a2) && hasAtom(i,a3) && hasAtom(i,a4))
  		res(i).atom(a1).pos  = projectPos(length, angle, dihed,
	  	  res(i).atom(a2).pos, res(i).atom(a3).pos, res(i).atom(a4).pos);
  }
}       



void Protein::sproutHydrogen(int i) 
{
  if (res(i).resKey == XXX) return;
  
  projectHydrogen(i, H, N_H, CA_N_H, 180 + res(i).phi, C, CA, N);

  Vector3d ca_n = atom(i,N).pos - atom(i,CA).pos;
  Vector3d ca_c = atom(i,C).pos - atom(i,CA).pos;
	Vector3d v    = ca_n.scaleTo(1.0) + ca_c.scaleTo(1.0);
	Vector3d w    = ca_n.scaleTo(1.0) - ca_c.scaleTo(1.0);

	Vector3d ca_ha = v.rotated(-125.5, w).scaleTo(1.1);
	if (!hasAtom(i,HA1))
	{
    insertAtom(i, HA1);
    atom(i,HA1).pos = atom(i,CA).pos + ca_ha;
  }

	Vector3d ca_cb = v.rotated(125.5, w);
	if (res(i).resKey == GLY)
  {
  	if (!hasAtom(i,HA2))
	  {
      insertAtom(i, HA2);
  	  atom(i,HA2).pos = atom(i,CA).pos + ca_cb.scaleTo(CB_HB);
    }
	}
	else if (res(i).resKey == PRO) 
	{
		projectHydrogen(i,  HB1, C_H, 109.0,  -6.11 + 120,  N, CA, CB);
		projectHydrogen(i,  HB2, C_H, 109.0,  -6.11 + 240,  N, CA, CB);

		projectHydrogen(i,  HG11, C_H, 109.0, 120, CA, CB, CG1);

		projectHydrogen(i,  HG12, C_H, 109.0, 240, CA, CB, CG1);

		projectHydrogen(i,  HD11, C_H, 109.0,  120, CA,  N, CD1);
		projectHydrogen(i,  HD12, C_H, 109.0,  240, CA,  N, CD1);
  }
	else if (res(i).resKey == ALA) 
	{
		projectHydrogen(i, HB1, 1.1, 109.0,  60, N, CA, CB);
		projectHydrogen(i, HB2, 1.1, 109.0, 180, N, CA, CB);
		projectHydrogen(i, HB3, 1.1, 109.0, 300, N, CA, CB);
	}
  
	switch (res(i).resKey) 
	{
		case ILE:
			projectHydrogen(i,  HB1, C_H, 109.0, res(i).chi(1) + 120,  N, CA, CB);
			break;
		case SER:
			projectHydrogen(i,  HB1, C_H, 109.0, res(i).chi(1) + 120,  N, CA, CB);
			projectHydrogen(i,  HB2, C_H, 109.0, res(i).chi(1) + 240,  N, CA, CB);
			projectHydrogen(i, HG11, 1.000, 110.0,                 180, CA, CB, OG1);
			break;
		case THR:
			projectHydrogen(i,  HB1, C_H, 109.0, res(i).chi(1) + 120,  N, CA, CB);
			projectHydrogen(i, HG11, 1.000, 110.0, 180,                 CA, CB, OG1);
			break;
		case VAL:
			projectHydrogen(i,  HB1, C_H, 109.0, res(i).chi(1) + 240,  N, CA, CB);
			projectHydrogen(i, HG11, C_H, 109.0,  60, CA, CB, CG1);
			projectHydrogen(i, HG12, C_H, 109.0, 180, CA, CB, CG1);
			projectHydrogen(i, HG13, C_H, 109.0, 300, CA, CB, CG1);
			break;
	}

	switch (res(i).resKey) 
	{
		case ASP:
		case ASN:
		case HIS:
		case LEU:
		case GLN:
		case GLU:
		case LYS:
		case ARG:
		case TRP:
		case TYR:
		case PHE:
		case MET:
		case CYS:
			projectHydrogen(i,  HB1, C_H, 109.0, res(i).chi(1) + 120,  N, CA, CB);
			projectHydrogen(i,  HB2, C_H, 109.0, res(i).chi(1) + 240,  N, CA, CB);
			break;
		case THR:
		case ILE:
    case VAL:
			projectHydrogen(i, HG21, C_H, 109.0,  60, CA, CB, CG2);
			projectHydrogen(i, HG22, C_H, 109.0, 180, CA, CB, CG2);
			projectHydrogen(i, HG23, C_H, 109.0, 300, CA, CB, CG2);
			break;
	}

	switch (res(i).resKey) 
	{
		case ASN:
			projectHydrogen(i, HD21, 1.000, 120.0,    0, CB, CG1, ND2);
			projectHydrogen(i, HD22, 1.000, 120.0,  180, CB, CG1, ND2);
			break;
			break;
		case ILE:
			projectHydrogen(i, HG11, C_H, 109.0, res(i).chi(2) + 120, CA, CB, CG1);
			projectHydrogen(i, HG12, C_H, 109.0, res(i).chi(2) + 240, CA, CB, CG1);

			projectHydrogen(i, HD11, C_H, 109.0,   60, CB, CG1, CD1);
			projectHydrogen(i, HD12, C_H, 109.0,  180, CB, CG1, CD1);

			projectHydrogen(i, HD13, C_H, 109.0,  300, CB, CG1, CD1);
			break;
		case LEU:
			projectHydrogen(i, HG11, C_H, 109.0, res(i).chi(2) + 240, CA, CB, CG1);

			projectHydrogen(i, HD11, C_H, 109.0,   60, CB, CG1, CD1);
			projectHydrogen(i, HD12, C_H, 109.0,  180, CB, CG1, CD1);
			projectHydrogen(i, HD13, C_H, 109.0,  300, CB, CG1, CD1);

			projectHydrogen(i, HD21, C_H, 109.0,   60, CB, CG1, CD2);
			projectHydrogen(i, HD22, C_H, 109.0,  180, CB, CG1, CD2);
			projectHydrogen(i, HD23, C_H, 109.0,  300, CB, CG1, CD2);
			break;
		case GLN:
		case GLU:
		case LYS:
		case ARG:
			projectHydrogen(i, HG11, C_H, 109.0, res(i).chi(2) + 120, CA, CB, CG1);
			projectHydrogen(i, HG12, C_H, 109.0, res(i).chi(2) + 240, CA, CB, CG1);
			break;
		case MET:
			projectHydrogen(i, HG11, C_H, 109.0, res(i).chi(2) + 120, CA, CB, CG1);
			projectHydrogen(i, HG12, C_H, 109.0, res(i).chi(2) + 240, CA, CB, CG1);

			break;
		case TRP:
  		projectHydrogen(i, HD11, C_H,  124.3,   0,   CB, CG1, CD1);
			projectHydrogen(i, HE11, 1.000,  126.8, 180,  CG1, CD1, NE1);
			projectHydrogen(i, HZ21, C_H,  125.8, 180,  CD2, CE2, CZ2);
			projectHydrogen(i, HH21, C_H,  125.8, 180,  CE2, CZ2, CH2);
			projectHydrogen(i, HZ31, C_H,  125.8, 180,  CD2, CE3, CZ3);
			projectHydrogen(i, HE31, C_H,  125.8,   0,  CG1, CD2, CE3);
			break;
		case HIS:
			projectHydrogen(i, HD11, C_H,  125.8,     0,   CB, CG1, ND1);
			projectHydrogen(i, HE11, 1.000,  125.8,   180,  CG1, ND1, CE1);
			projectHydrogen(i, HD21, C_H,  125.8,     0,   CB, CG1, CD2);
			projectHydrogen(i, HE21, 1.000,  125.8,   180,  ND1, CE1, NE2);
			break; 
		case TYR:
		case PHE:
			projectHydrogen(  i, HD11, C_H, 120.0,    0,  CB, CG1, CD1);
			projectHydrogen(  i, HD21, C_H, 120.0,    0,  CB, CG1, CD2);

			projectHydrogen(  i, HE11, C_H, 120.0,  180, CG1, CD1, CE1);
			projectHydrogen(  i, HE21, C_H, 120.0,  180, CG1, CD2, CE2);

			if (res(i).resKey == PHE)
				projectHydrogen(i, HZ11, C_H, 120.0,  180, CD1, CE1, CZ1);
			else {
				projectHydrogen(i, HH11, C_H, 110.0,  180, CE1, CZ1, OH1);
      }

			break;
	}

	switch (res(i).resKey) 
	{
		case GLN:
			projectHydrogen(i, HE21, 1.000, 120.0,    0, CG1, CD1, NE2);
			projectHydrogen(i, HE22, 1.000, 120.0,  180, CG1, CD1, NE2);
			break;
		case ARG:
			projectHydrogen(i, HD11, C_H, 109.0, res(i).chi(3) + 120, CB, CG1, CD1);
			projectHydrogen(i, HD12, C_H, 109.0, res(i).chi(3) + 240, CB, CG1, CD1);
			break;
		case LYS:

			projectHydrogen(i, HD11, C_H, 109.0, res(i).chi(3) + 120, CB, CG1, CD1);
			projectHydrogen(i, HD12, C_H, 109.0, res(i).chi(3) + 240, CB, CG1, CD1);
			break;
		case MET:
			projectHydrogen(i, HE11, C_H, 109.0,  60, CG1, SD1, CE1);
			projectHydrogen(i, HE12, C_H, 109.0, 180, CG1, SD1, CE1);
			projectHydrogen(i, HE13, C_H, 109.0, 300, CG1, SD1, CE1);
			break;
	}

	switch (res(i).resKey) 
	{
		case ARG:
			projectHydrogen(i, HE11, C_H, 240.0, res(i).chi(4), CG1, CD1, NE1);

			projectHydrogen(i, HH11, 1.000, 120.0,   0,  NE1, CZ1, NH1);
			projectHydrogen(i, HH12, 1.000, 120.0, 180,  NE1, CZ1, NH1);

			projectHydrogen(i, HH21, 1.000, 120.0,   0,  NE1, CZ1, NH2);
			projectHydrogen(i, HH22, 1.000, 120.0, 180,  NE1, CZ1, NH2);

			break;
		case LYS:
			projectHydrogen(i, HE11, C_H, 109.0, res(i).chi(4) + 120, CG1, CD1, CE1);
			projectHydrogen(i, HE12, C_H, 109.0, res(i).chi(4) + 240, CG1, CD1, CE1);

			projectHydrogen(i, HZ11, C_H, 109.0,   60, CD1, CE1, NZ1);
			projectHydrogen(i, HZ12, C_H, 109.0,  180, CD1, CE1, NZ1);
			projectHydrogen(i, HZ13, C_H, 109.0,  300, CD1, CE1, NZ1);
			break;
	}
}



// --------- dihedral angle rotations



void Protein::rotatePhi(int i, double phi) 
{
	Vector3d center  = atom(i,N).pos;
	Vector3d axis    = atom(i,N).pos - atom(i,CA).pos;
	double dPhi      = phi - res(i).phi;

	Matrix3d rotMatrix(axis, dPhi, center);

	AtomIter prevAtomIter = res(i).atomBegin();
	if (prevAtomIter != atomBegin()) 
    ::applyMatrix(atomBegin(), prevAtomIter, rotMatrix);

	if (res(i).resKey != PRO) 
  {  
		atom(i,H).pos.applyMatrix(rotMatrix);
		if (hasAtom(i, H2)) atom(i,H2).pos.applyMatrix(rotMatrix);
		if (hasAtom(i, H3)) atom(i,H3).pos.applyMatrix(rotMatrix);
  }

	res(i).phi = phi;
}



void Protein::rotatePsi(int i, double psi) 
{
	Vector3d center  = atom(i,C).pos;
	Vector3d axis    = atom(i,C).pos - atom(i,CA).pos;
	double dPsi      = psi - res(i).psi;

	Matrix3d rotMatrix(axis, dPsi, center);

	if (i < nRes()) 
    ::applyMatrix(res(i).atomEnd(), atomEnd(), rotMatrix);

  atom(i,O).pos.applyMatrix(rotMatrix);
	if (hasAtom(i, OXT)) atom(i,OXT).pos.applyMatrix(rotMatrix);
	if (hasAtom(i, HXT)) atom(i,HXT).pos.applyMatrix(rotMatrix);

	res(i).psi = psi;
}



void Protein::rotateChi(int i, int n, double chi) 
{
	int resKey        = res(i).resKey;
	int pivotAtomKey  = resChiTetrad[resKey][n-1][1];
	int centerAtomKey = resChiTetrad[resKey][n-1][2];

	Vector3d pivot    = atom(i, pivotAtomKey).pos;
	Vector3d center   = atom(i, centerAtomKey).pos;
	Vector3d axis     = center - pivot;
	double   dChi     = chi - res(i).chi(n);

	Matrix3d rotMatrix(axis, dChi, center);

  for (int j = 0; j < res(i).nAtoms(); j++) 
	{
		Atom& a = *res(i)._atomIterList[j];
		if (a.atomKey > centerAtomKey)
			a.pos.applyMatrix(rotMatrix);
	}

	res(i).chi(n) = chi;
}



// --------- helper functions



ostream& operator << (ostream & os, Protein& p) 
{
	int i = 1;
	for (AtomIter iter = p.atomBegin(); iter != p.atomEnd(); iter++) {
		Atom& a = *iter;
		os << "ATOM " 
			 << setw(6) << i          << " "
			 << setw(4) << a.atomName << " " 
			 << setw(3) << a.resName  << " "
			 << setw(5) << a.resNo    << "    "
       << fixed << setw(8) << setprecision(3) << a.pos.x() 
			 << fixed << setw(8) << setprecision(3) << a.pos.y() 
			 << fixed << setw(8) << setprecision(3) << a.pos.z() 
       << fixed << setw(6) << setprecision(2) << 1.00 
			 << fixed << setw(6) << setprecision(2) << 0.0 << endl;
		i++;
	}
  return os;
}



void Protein::deleteRes(int leftRes, int rightRes)
{

  if ( (leftRes > 1) && (rightRes < nRes()) )
    spliceSegment(leftRes-1, rightRes+1, nRes());
  if (nRes() > 0)
  {
    for (int j = leftRes; j <= rightRes; j+=1)
    {
      if (nRes() == 1) break;
      eraseRes(leftRes);
    }
  }
}


void Protein::spliceSegment(int target, int seg1, int seg2)
{
  // finds the transform so that the residues
  // rightRes+1 to rightRes+1 + rightRes-leftRes
  // is attached to the C-terminal of rightRes
  // now since we just made a copy, rightRes+1
  // should have the same conformation as left_Res
  int cRes = seg1;
  Vector3d pivot  = res(cRes).atom(CA).pos;
  Vector3d ca2_n2 = res(cRes).atom(N).pos
                   -res(cRes).atom(CA).pos;
  Vector3d n2_c1;

  // at the N-terminus of cRes, must project ghost atoms
  Vector3d c1 = projectPos(1.325, 121.9, res(cRes).phi,
                                  res(cRes).atom(C).pos,
                                  res(cRes).atom(CA).pos,
                                  res(cRes).atom(N).pos);
  n2_c1  = c1 - res(cRes).atom(N).pos;

  // at the C-terminus of nRes, must project ghost atoms
  int nRes = target;
  Vector3d ca_n;
  Vector3d n_c;
  Vector3d pivot2;
  Vector3d n = projectPos(1.325, 115.0, res(nRes).psi,
                                        res(nRes).atom(N).pos,
                                        res(nRes).atom(CA).pos,
                                        res(nRes).atom(C).pos);

  Vector3d ca = projectPos(1.453, 121.0, 180.0,
                                        res(nRes).atom(CA).pos,
                                        res(nRes).atom(C).pos,
                                        n);
  ca_n = n - ca;
  n_c  = res(nRes).atom(C).pos - n;
  pivot2 = n;

  Vector3d axis1  =  cross(ca_n, ca2_n2);
  double angle1   =  angle(ca_n, ca2_n2);
  Matrix3d matrix(axis1.normal(), -angle1);

  n2_c1   =  matrix * n2_c1;
  double angle2   =  dihedral(n2_c1, ca_n, n_c);
  Matrix3d matrix2(ca_n.normal(), -angle2);

  for (AtomIter i = res(cRes).atomBegin(); i != res(seg2).atomEnd(); i++)
  {
    i->pos -= pivot;
    i->pos =  matrix * i->pos;
    i->pos -= ca_n;
    i->pos =  matrix2 * i->pos;
    i->pos += pivot2;
  }
}



void Protein::copyResToRight(int leftRes, int rightRes)
{
  int active_size = rightRes - leftRes + 1;

  // copies residues leftRes to rightRes to a
  // position immediately adjacent to rightRes
  // i.e. rightRes+1 to rightRes+1 + rightRes-leftRes
  // is a copy of leftRes to rightRes
  int j = rightRes;
  int k;
  for (k = rightRes; k >= leftRes; k--)
  {
    int resKey = res(j).resKey;
    insertRes(leftRes, resKey);
    j+=1;
    res(leftRes).phi    = res(j).phi;
    res(leftRes).psi    = res(j).psi;
    res(leftRes).chi(1) = res(j).chi(1);
    res(leftRes).chi(2) = res(j).chi(2);
    res(leftRes).chi(3) = res(j).chi(3);
    res(leftRes).chi(4) = res(j).chi(4);
    res(leftRes).omega  = res(j).omega;
    j-=1;
  }
  for (k = leftRes; k <= rightRes; k++)
  {
    for (int l = 0; l < res(k).nAtoms(); l++)
    {
      Atom& a = * (res(k)._atomIterList[l]);
      if (!hasAtom(k + active_size, a.atomKey))
      insertAtom(k + active_size, a.atomKey);
      a.pos = res(k + active_size).atom(a.atomKey).pos;
    }
  }
  spliceSegment(rightRes, rightRes+1, nRes());
}

