//
//  **************************************************************************
//
//  molecule.cpp
//
//  (C) 2003 Bosco K. Ho 
//
//  Molecule is a rigid-body molecule class, meaning that it can
//  represent molecules as fixed collections of atoms/points that
//  may be translated, rotated and transformed like any other fixed
//  set of points in space. As Molecule does not contain any more
//  chemical information, you cannot change the conformation of a 
//  Molecule.
//
//  The core piece of information stored by Molecule is a set
//  of atoms, so how we express this is going to have a significant 
//  impact on the library. I've used a double linked list from the
//  STL to represent the atoms. This allows atoms to be inserted or deleted 
//  without destroying the validity of any other iterator.
//
//  Adapted from The Biomolecule Toolkit (C) 2001, Eric Alm 
//  Cleaned the abstraction interface by removing all traces
//  of protein structure at this level. 
//
//  ****************************************************************************
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU Lesser General Public License as published
//  by the Free Software Foundation; either version 2.1 of the License, or (at
//  your option) any later version.
//  
//  This program is distributed in the hope that it will be useful,  but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details. 
//  
//  You should have received a copy of the GNU Lesser General Public License 
//  along with this program; if not, write to the Free Software Foundation, 
//  Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
//  ****************************************************************************
//

#include <stdio.h>
#include <iostream>
#include <fstream>
#include "atoms.h"
#include "molecule.h"

using namespace std;



// default constructor
Molecule::Molecule(char id, const std::string& name): _atomList(0), 
             _principalAxes(3), _id(id), _name(name), _centerOfMass() 
{
  // Intialize principal axes to orthogonal set
  _principalAxes[0] = Vector3d(1.0, 0.0, 0.0);
  _principalAxes[1] = Vector3d(0.0, 1.0, 0.0);
  _principalAxes[2] = Vector3d(0.0, 0.0, 1.0);
}


		
AtomIter Molecule::getAtomIter(int index) 
{
	AtomIter iter = _atomList.begin();
	for (int i=0; i<index; i++)
		iter++;
	return iter;
}



void Molecule::insert(int i, Molecule& molecule) 
{
	_atomList.insert(getAtomIter(i), molecule._atomList.begin(), molecule._atomList.end());	
}



void Molecule::erase(int i, int j) 
{
	_atomList.erase(getAtomIter(i), getAtomIter(j));	
}



Molecule Molecule::extract(int i, int j) 
{
	Molecule fragment;
	fragment._atomList.insert(fragment._atomList.begin(), getAtomIter(i), getAtomIter(j));
  fragment._id = _id;
  fragment._name = _name;

	return fragment;
}



void Molecule::translate(const Vector3d& delta) 
{
	AtomIter j = _atomList.end();
  for (AtomIter i = _atomList.begin(); i != j; i++) 
    (*i).pos += delta;
}
 


void Molecule::computeCenterOfMass() 
{
  double nAtoms = 0;

  _centerOfMass = Vector3d(0.0, 0.0, 0.0);
	AtomIter j = _atomList.end();
  for (AtomIter i = _atomList.begin(); i != j; i++)
  {
    _centerOfMass += i->pos;
    nAtoms += 1.0;
  }

  if (0.0 == nAtoms) return;
  _centerOfMass /= nAtoms;
}



double Molecule::getMaxLength(void)
{
  double lengthSquared = 0.0;
  for (AtomIter i = _atomList.begin(); i != _atomList.end(); i++)
  {
    for (AtomIter j = i; j != _atomList.end(); j++)
    {
      if (i != j)
      {
         double newLengthSquared = distanceSquared(i->pos, j->pos);
         if (newLengthSquared > lengthSquared)
           lengthSquared = newLengthSquared;
      }
    }
  }
  return sqrt(lengthSquared);
}



void Molecule::rotate(const Vector3d& axis, double theta,
											const Vector3d& center) 
{
  Matrix3d rotMatrix(axis, theta, center);

	AtomIter j = _atomList.end();
  for (AtomIter i = _atomList.begin(); i != j; i++)
    i->pos.applyMatrix(rotMatrix);

  for (int k = 0; k < 3; k++)
    _principalAxes[k].applyMatrix(rotMatrix);
}



void Molecule::applyMatrix(const Matrix3d& transformMatrix) 
{
	AtomIter j = _atomList.end();
  for (AtomIter i = _atomList.begin(); i != j; i++)
    (*i).pos  = transformMatrix * ((*i).pos);

  for (int k = 0; k < 3; k++)
    _principalAxes[k] = transformMatrix * _principalAxes[k];
}



/*
void Molecule::calculate_principal_axes()
{
  BTKEigensystem axes;
  principal_axes(_atomList.begin(),_atomList.end(),axes,_centerOfMass);

  _principalAxes[0] = axes[0].eigenvector;
  _principalAxes[1] = axes[1].eigenvector;
  _principalAxes[2] = axes[2].eigenvector;
}

*/
