//
//  ***************************************************************************
//
//  bond.cpp
//  (C) 2004 Bosco K. Ho 
//
//  HbondList takes a Protein object, counts the h-bonds, and stores the
//  h-bond information in a list for easy access later on.
//
//  ***************************************************************************
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU Lesser General Public License as published
//  by the Free Software Foundation; either version 2.1 of the License, or (at
//  your option) any later version.
//  
//  This program is distributed in the hope that it will be useful,  but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details. 
//  
//  You should have received a copy of the GNU Lesser General Public License 
//  along with this program; if not, write to the Free Software Foundation, 
//  Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
//  ****************************************************************************
//


#include "bond.h"


using namespace std;


const double CA_DISTANCE_SQ          =  9.0 * 9.0;
const double Q_SQ                    = -27888.0; // 331.2 * 0.30 * 0.30
const double HBOND_ENERGY_LIMIT      = -9900;    // cal/mol
const double HBOND_ENERGY_HIGH       = -500;     // cal/mol
const double WEAK_HBOND_ENERGY_HIGH  = -1000;     // cal/mol
const double DISULFIDE_DISTANCE_SQ   = 2.5 * 2.5;    //  angstroms



void HbondList::checkHbond(Protein* protein, int i, int c, int o, int j, int n, int h)
{

  if (distanceSquared(protein->atom(j, CA).pos, protein->atom(i, CA).pos) > CA_DISTANCE_SQ)
    return;

  if (!protein->hasAtom(i, c) || !protein->hasAtom(i, o)
   || !protein->hasAtom(j, n) || !protein->hasAtom(j, h))
    return;

  double ho = sqrt(distanceSquared(protein->atom(j, h).pos, protein->atom(i, o).pos));
  double hc = sqrt(distanceSquared(protein->atom(j, h).pos, protein->atom(i, c).pos));
  double no = sqrt(distanceSquared(protein->atom(j, n).pos, protein->atom(i, o).pos));
  double nc = sqrt(distanceSquared(protein->atom(j, n).pos, protein->atom(i, c).pos));

  // make sure it is a hydrogen bond, i.e. the h is between the donors
  if (ho > nc) return;

  // the kabsch-sanders h-bond function!!!!
  double energy = (long)floor(Q_SQ/ho - Q_SQ/hc + Q_SQ/nc - Q_SQ/no + 0.5);

  if (energy > HBOND_ENERGY_HIGH) return;

  // add this h-bond to the list
  HbondStruct hbond(i,c,o,j,n,h);
  list.push_back(hbond);
}



void HbondList::countHbondDonors(Protein* protein, int i, int c, int o, int j)
{
  checkHbond(protein, i, c, o, j, N, H);

  switch (protein->res(j).resKey)
  {
  case SER:
  case THR:
    checkHbond(protein, i, c, o, j, OG1, HG11);
    break;
  case ARG:
    checkHbond(protein, i, c, o, j, NE2, HE11);
    checkHbond(protein, i, c, o, j, NH1, HH11);
    checkHbond(protein, i, c, o, j, NH1, HH12);
    checkHbond(protein, i, c, o, j, NH2, HH21);
    checkHbond(protein, i, c, o, j, NH2, HH21);
    break;
  case LYS:
    checkHbond(protein, i, c, o, j, NZ1, HZ11);
    checkHbond(protein, i, c, o, j, NZ1, HZ12);
    checkHbond(protein, i, c, o, j, NZ1, HZ13);
    break;
  case ASN:
    checkHbond(protein, i, c, o, j, ND2, HD21);
    checkHbond(protein, i, c, o, j, ND2, HD22);
    break;
  case GLN:
    checkHbond(protein, i, c, o, j, NE2, HE21);
    checkHbond(protein, i, c, o, j, NE2, HE22);
    break;
  case HIS:
    checkHbond(protein, i, c, o, j, ND1, HD11);
    checkHbond(protein, i, c, o, j, NE2, HE21);
    break;
  case TRP:
    checkHbond(protein, i, c, o, j, NE1, HE11);
    break;
  case TYR:
    checkHbond(protein, i, c, o, j, OH1, HH11);
    break;
  }
}



void HbondList::countHbondsBetweenResidues(Protein* protein, int i, int j)
{
  countHbondDonors(protein, i, C, O, j);

  switch (protein->res(i).resKey)
  {
  case SER:
  case THR:
    countHbondDonors(protein, i, CB, OG1, j);
    break;
  case ASP:
    countHbondDonors(protein, i, CG1, OD1, j);
    countHbondDonors(protein, i, CG1, OD2, j);
    break;
  case ASN:
    countHbondDonors(protein, i, CG1, OD1, j);
    break;
  case GLU:
    countHbondDonors(protein, i, CD1, OE1, j);
    countHbondDonors(protein, i, CD1, OE2, j);
    break;
  case GLN:
    countHbondDonors(protein, i, CD1, OE1, j);
    break;
  case TYR:
    countHbondDonors(protein, i, CZ1, OH1, j);
    break;
  }
}




void HbondList::countBackboneHbondsBetweenResidues(Protein* protein, int res1, int res2)
{
  checkHbond(protein, res1, C, O, res2, N, H);
}



void HbondList::countHbondsWithinRes(Protein* protein, int i)
{
  switch (protein->res(i).resKey)
  {
  case SER:
  case THR:
    checkHbond(protein, i, C, O, i, OG1, HG11);
    break;
  case ASP:
    checkHbond(protein, i, CG1, OD1, i, N, H);
    checkHbond(protein, i, CG1, OD2, i, N, H);
    break;
  case ASN:
    checkHbond(protein, i, CG1, OD1, i, N, H);
    checkHbond(protein, i, C, O, i, ND2, HD21);
    checkHbond(protein, i, C, O, i, ND2, HD22);
    break;
  case GLU:
    checkHbond(protein, i, CD1, OE1, i, N, H);
    checkHbond(protein, i, CD1, OE2, i, N, H);
    break;
  case GLN:
    checkHbond(protein, i, CD1, OE1, i, N, H);
    checkHbond(protein, i, C, O, i, NE2, HE21);
    checkHbond(protein, i, C, O, i, NE2, HE22);
    break;
  case TYR:
    checkHbond(protein, i, CZ1, OH1, i, N, H);
    checkHbond(protein, i, C, O, i, OH1, HH11);
    break;
    break;
  case ARG:
    checkHbond(protein, i, C, O, i, NE2, HE11);
    checkHbond(protein, i, C, O, i, NH1, HH11);
    checkHbond(protein, i, C, O, i, NH1, HH12);
    checkHbond(protein, i, C, O, i, NH2, HH21);
    checkHbond(protein, i, C, O, i, NH2, HH21);
    break;
  case LYS:
    checkHbond(protein, i, C, O, i, NZ1, HZ11);
    checkHbond(protein, i, C, O, i, NZ1, HZ12);
    checkHbond(protein, i, C, O, i, NZ1, HZ13);
    break;
  case HIS:
    checkHbond(protein, i, C, O, i, ND1, HD11);
    checkHbond(protein, i, C, O, i, NE2, HE21);
    break;
  case TRP:
    checkHbond(protein, i, C, O, i, NE1, HE11);
    break;
  }
}



// weak HBondList



void WeakHbondList::checkWeakHbond(Protein* protein, int i, int c, int o, int j, int n, int h)
{

  if (distanceSquared(protein->atom(j, CA).pos, protein->atom(i, CA).pos) > CA_DISTANCE_SQ)
    return;

  if (!protein->hasAtom(i, c) || !protein->hasAtom(i, o)
   || !protein->hasAtom(j, n) || !protein->hasAtom(j, h))
    return;

  double ho = sqrt(distanceSquared(protein->atom(j, h).pos, protein->atom(i, o).pos));
  double hc = sqrt(distanceSquared(protein->atom(j, h).pos, protein->atom(i, c).pos));
  double no = sqrt(distanceSquared(protein->atom(j, n).pos, protein->atom(i, o).pos));
  double nc = sqrt(distanceSquared(protein->atom(j, n).pos, protein->atom(i, c).pos));

  // make sure it is a hydrogen bond, i.e. the h is between the donors
  if (ho > nc) return;

  // the kabsch-sanders h-bond function!!!!
  double energy = (long)floor(Q_SQ/ho - Q_SQ/hc + Q_SQ/nc - Q_SQ/no + 0.5);

  if (energy > WEAK_HBOND_ENERGY_HIGH) return;

  // add this h-bond to the list
  HbondStruct hbond(i,c,o,j,n,h);
  list.push_back(hbond);
}



void WeakHbondList::countHbondDonors(Protein* protein, int i, int c, int o, int j)
{
  checkWeakHbond(protein, i, c, o, j, CA, HA1);

  if (protein->res(j).resKey == PRO)
    checkWeakHbond(protein, i, c, o, j, CD1, HD11);
  
  if (protein->res(j).resKey == GLY)
    checkWeakHbond(protein, i, c, o, j, CA, HA2);

}


void WeakHbondList::countHbondsBetweenResidues(Protein* protein, int i, int j)
{
  countHbondDonors(protein, i, C, O, j);

  switch (protein->res(i).resKey)
  {
  case SER:
  case THR:
    countHbondDonors(protein, i, CB, OG1, j);
    break;
  case ASP:
    countHbondDonors(protein, i, CG1, OD1, j);
    countHbondDonors(protein, i, CG1, OD2, j);
    break;
  case ASN:
    countHbondDonors(protein, i, CG1, OD1, j);
    break;
  case GLU:
    countHbondDonors(protein, i, CD1, OE1, j);
    countHbondDonors(protein, i, CD1, OE2, j);
    break;
  case GLN:
    countHbondDonors(protein, i, CD1, OE1, j);
    break;
  case TYR:
    countHbondDonors(protein, i, CZ1, OH1, j);
    break;
  }
}



void WeakHbondList::countBackboneHbondsBetweenResidues(Protein* protein, int res1, int res2)
{
  checkWeakHbond(protein, res1, C, O, res2, CA, HA1);
  if (protein->res(res2).resKey == PRO)
    checkWeakHbond(protein, res1, C, O, res2, CD1, HD11);
  if (protein->res(res2).resKey == GLY)
    checkWeakHbond(protein, res1, C, O, res2, CA, HA2);
}



// weak HBondList




void DisulfideBondList::countBondsBetweenResidues(Protein* protein, int i, int j)
{
  if (protein->res(i).resKey != CYS) return;
  if (protein->res(j).resKey != CYS) return;

  if (distanceSquared(protein->atom(j, SG1).pos, protein->atom(i, SG1).pos)  
     > DISULFIDE_DISTANCE_SQ)
    return;

  // add this h-bond to the list
  HbondStruct hbond(i,CB, SG1,j,CB, SG1);
  list.push_back(hbond);
}

