/***************************************************************************
                          hbond.cpp  -  description
                             -------------------
    begin                : Thu Mar 25 2004
    copyright            : (C) 2004 by bosco
    email                : bosco@helmholtz.compbio.ucsf.edu

    hbondList is an object that takes a protein object and counts the
    number of h-bonds, the information of which, is stored in a list
    for easy access later on.
    
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


#include "hbond.h"


using namespace std;


const double CA_DISTANCE_SQ =  9.0 * 9.0;
const double Q_SQ           = -27888.0; // 331.2 * 0.30 * 0.30
const double HB_LOW         = -9900;    // cal/mol
const double HB_HIGH        = -500;     // cal/mol
const double WEAK_HB_HIGH   = -1000;     // cal/mol


void HbondList::checkHbond(Protein* protein, int i, int c, int o, int j, int n, int h)
{

  if (distanceSquared(protein->atom(j, CA).pos, protein->atom(i, CA).pos) > CA_DISTANCE_SQ)
    return;

  if (!protein->hasAtom(i, c) || !protein->hasAtom(i, o)
   || !protein->hasAtom(j, n) || !protein->hasAtom(j, h))
    return;

  double ho = sqrt(distanceSquared(protein->atom(j, h).pos, protein->atom(i, o).pos));
  double hc = sqrt(distanceSquared(protein->atom(j, h).pos, protein->atom(i, c).pos));
  double no = sqrt(distanceSquared(protein->atom(j, n).pos, protein->atom(i, o).pos));
  double nc = sqrt(distanceSquared(protein->atom(j, n).pos, protein->atom(i, c).pos));

  // make sure it is a hydrogen bond, i.e. the h is between the donors
  if (ho > nc) return;

  // the kabsch-sanders h-bond function!!!!
  double energy = (long)floor(Q_SQ/ho - Q_SQ/hc + Q_SQ/nc - Q_SQ/no + 0.5);

  if (energy > HB_HIGH) return;

  // add this h-bond to the list
  HbondStruct hbond(i,c,o,j,n,h);
  hbondList.push_back(hbond);
}



void HbondList::checkWeakHbond(Protein* protein, int i, int c, int o, int j, int n, int h)
{

  if (distanceSquared(protein->atom(j, CA).pos, protein->atom(i, CA).pos) > CA_DISTANCE_SQ)
    return;

  if (!protein->hasAtom(i, c) || !protein->hasAtom(i, o)
   || !protein->hasAtom(j, n) || !protein->hasAtom(j, h))
    return;

  double ho = sqrt(distanceSquared(protein->atom(j, h).pos, protein->atom(i, o).pos));
  double hc = sqrt(distanceSquared(protein->atom(j, h).pos, protein->atom(i, c).pos));
  double no = sqrt(distanceSquared(protein->atom(j, n).pos, protein->atom(i, o).pos));
  double nc = sqrt(distanceSquared(protein->atom(j, n).pos, protein->atom(i, c).pos));

  // make sure it is a hydrogen bond, i.e. the h is between the donors
  if (ho > nc) return;

  // the kabsch-sanders h-bond function!!!!
  double energy = (long)floor(Q_SQ/ho - Q_SQ/hc + Q_SQ/nc - Q_SQ/no + 0.5);

  if (energy > WEAK_HB_HIGH) return;

  // add this h-bond to the list
  HbondStruct hbond(i,c,o,j,n,h);
  hbondList.push_back(hbond);
}



void HbondList::countHbondDonors(Protein* protein, int res1, int c, int o, int res2)
{
  checkHbond(protein, res1, c, o, res2, N, H);

  checkWeakHbond(protein, res1, c, o, res2, CA, HA1);
  if (protein->res(res2).resKey == PRO)
    checkWeakHbond(protein, res1, c, o, res2, CD1, HD11);
  if (protein->res(res2).resKey == GLY)
    checkWeakHbond(protein, res1, c, o, res2, CA, HA2);

  switch (protein->res(res2).resKey)
  {
  case SER:
  case THR:
    checkHbond(protein, res1, c, o, res2, OG1, HG11);
    break;
  case ARG:
    checkHbond(protein, res1, c, o, res2, NE2, HE11);
    checkHbond(protein, res1, c, o, res2, NH1, HH11);
    checkHbond(protein, res1, c, o, res2, NH1, HH12);
    checkHbond(protein, res1, c, o, res2, NH2, HH21);
    checkHbond(protein, res1, c, o, res2, NH2, HH21);
    break;
  case LYS:
    checkHbond(protein, res1, c, o, res2, NZ1, HZ11);
    checkHbond(protein, res1, c, o, res2, NZ1, HZ12);
    checkHbond(protein, res1, c, o, res2, NZ1, HZ13);
    break;
  case ASN:
    checkHbond(protein, res1, c, o, res2, ND2, HD21);
    checkHbond(protein, res1, c, o, res2, ND2, HD22);
    break;
  case GLN:
    checkHbond(protein, res1, c, o, res2, NE2, HE21);
    checkHbond(protein, res1, c, o, res2, NE2, HE22);
    break;
  case HIS:
    checkHbond(protein, res1, c, o, res2, ND1, HD11);
    checkHbond(protein, res1, c, o, res2, NE2, HE21);
    break;
  case TRP:
    checkHbond(protein, res1, c, o, res2, NE1, HE11);
    break;
  case TYR:
    checkHbond(protein, res1, c, o, res2, OH1, HH11);
    break;
  }
}


void HbondList::countHbondsBetweenResidues(Protein* protein, int res1, int res2)
{
  countHbondDonors(protein, res1, C, O, res2);

  switch (protein->res(res1).resKey)
  {
  case SER:
  case THR:
    countHbondDonors(protein, res1, CB, OG1, res2);
    break;
  case ASP:
    countHbondDonors(protein, res1, CG1, OD1, res2);
    countHbondDonors(protein, res1, CG1, OD2, res2);
    break;
  case ASN:
    countHbondDonors(protein, res1, CG1, OD1, res2);
    break;
  case GLU:
    countHbondDonors(protein, res1, CD1, OE1, res2);
    countHbondDonors(protein, res1, CD1, OE2, res2);
    break;
  case GLN:
    countHbondDonors(protein, res1, CD1, OE1, res2);
    break;
  case TYR:
    countHbondDonors(protein, res1, CZ1, OH1, res2);
    break;
  }
}


void HbondList::countHbondsWithinRes(Protein* protein, int i)
{
  switch (protein->res(i).resKey)
  {
  case SER:
  case THR:
    checkHbond(protein, i, C, O, i, OG1, HG11);
    checkHbond(protein, i, CB, OG1, i, N, H);
    break;
  case ASP:
    checkHbond(protein, i, CG1, OD1, i, N, H);
    checkHbond(protein, i, CG1, OD2, i, N, H);
    break;
  case ASN:
    checkHbond(protein, i, CG1, OD1, i, N, H);
    checkHbond(protein, i, C, O, i, ND2, HD21);
    checkHbond(protein, i, C, O, i, ND2, HD22);
    break;
  case GLU:
    checkHbond(protein, i, CD1, OE1, i, N, H);
    checkHbond(protein, i, CD1, OE2, i, N, H);
    break;
  case GLN:
    checkHbond(protein, i, CD1, OE1, i, N, H);
    checkHbond(protein, i, C, O, i, NE2, HE21);
    checkHbond(protein, i, C, O, i, NE2, HE22);
    break;
  case TYR:
    checkHbond(protein, i, CZ1, OH1, i, N, H);
    checkHbond(protein, i, C, O, i, OH1, HH11);
    break;
    break;
  case ARG:
    checkHbond(protein, i, C, O, i, NE2, HE11);
    checkHbond(protein, i, C, O, i, NH1, HH11);
    checkHbond(protein, i, C, O, i, NH1, HH12);
    checkHbond(protein, i, C, O, i, NH2, HH21);
    checkHbond(protein, i, C, O, i, NH2, HH21);
    break;
  case LYS:
    checkHbond(protein, i, C, O, i, NZ1, HZ11);
    checkHbond(protein, i, C, O, i, NZ1, HZ12);
    checkHbond(protein, i, C, O, i, NZ1, HZ13);
    break;
  case HIS:
    checkHbond(protein, i, C, O, i, ND1, HD11);
    checkHbond(protein, i, C, O, i, NE2, HE21);
    break;
  case TRP:
    checkHbond(protein, i, C, O, i, NE1, HE11);
    break;
  }
}



void HbondList::countBackboneHbondsBetweenResidues(Protein* protein, int res1, int res2)
{
  checkHbond(protein, res1, C, O, res2, N, H);

  checkWeakHbond(protein, res1, C, O, res2, CA, HA1);
  if (protein->res(res2).resKey == PRO)
    checkWeakHbond(protein, res1, C, O, res2, CD1, HD11);
  if (protein->res(res2).resKey == GLY)
    checkWeakHbond(protein, res1, C, O, res2, CA, HA2);
}

