//
//  ***************************************************************************
//
//  control.cpp
//  (C) 2004 Bosco K. Ho 
//
//  Various decorator widgets for DisplayBox that manipulate
//  the protein embedded in DisplayBox.
//
//  ***************************************************************************
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU Lesser General Public License as published
//  by the Free Software Foundation; either version 2.1 of the License, or (at
//  your option) any later version.
//  
//  This program is distributed in the hope that it will be useful,  but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details. 
//  
//  You should have received a copy of the GNU Lesser General Public License 
//  along with this program; if not, write to the Free Software Foundation, 
//  Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
//  ****************************************************************************
//


#include "control.h"



void normalise(double& out)
{
	while (out < -180) out+=360;
	while (out >  180) out-=360;
}


/// Ramachandran plot



int RamaBox::handle(int event)
{
  if ((event == WGT_MOUSE_B1_DOWN) || (event == WGT_MOUSE_B1_DRAG))
  {
    double phi = ((mouse_x - _x - 2) / ((double)_w - 4) * 360.0) - 180.0;

    double psi = -(((mouse_y - _y - 2) / ((double)_h - 4) * 360.0) - 180.0);

    normalise(phi);
    normalise(psi);

    int i;
    for (i=_box->active_left_res; i<=_box->active_right_res; i+=1)
    {
      _box->protein->rotatePhi(i, phi);
      _box->protein->rotatePsi(i, psi);
    }

    _box->setFocusWithSameOrientation();
    return 1;
  }

  return 0;
}



void RamaBox::draw(BITMAP *bmp)
{
 	text_mode(-1);

	rect(bmp, _x, _y, _x + _w - 1, _y + _w - 1, GREEN);

	line(bmp, _x + (_w-1)/2, _y + 1, _x + (_w-1)/2, _y + _h - 2, DARK_GREY);
	line(bmp, _x + 1, _y + (_h-1)/2, _x + _w - 2, _y + (_h-1)/2, DARK_GREY);

	int i;
	for (i=_box->active_left_res; i<=_box->active_right_res; i+=1)
	{
  	int r_x = (int)(( _box->protein->res(i).phi+180) / 360.0 * (_w-4) + _x  + 2);
  	int r_y = (int)((-_box->protein->res(i).psi+180) / 360.0 * (_h-4) + _y  + 2);

  	line(bmp, r_x, _y + 1, r_x, _y + _h - 2, DARK_GREEN);
  	line(bmp, _x + 1, r_y, _x + _w - 2, r_y, DARK_GREEN);
	}

	for (i=_box->active_left_res; i<=_box->active_right_res; i+=1)
	{
  	float phi = (float) _box->protein->res(i).phi;
	  float psi = (float) _box->protein->res(i).psi;

  	int r_x = (int)(( phi+180) / 360.0 * (_w-4) + _x + 2);
  	int r_y = (int)((-psi+180) / 360.0 * (_h-4) + _y + 2);

    textprintf(bmp, font, _x + _w - 65, _y +  5, GREEN, "phi % 4.0f",(phi));
    textprintf(bmp, font, _x + _w - 65, _y + 17, GREEN, "psi % 4.0f",(psi));
	  textprintf(bmp, font, _x + 5, _y + 5, GREEN, "%s:%d",
		  getResName3(_box->protein->res(i).resKey).c_str(), _box->protein->res(i).resNo);

  	circlefill(bmp, r_x, r_y, 4, BLACK);
	  circlefill(bmp, r_x, r_y, 3, GREEN);
  }
}



/// Chi angle control box



bool ChiBox::isInside(int x, int y)
{
	if (_box->active_left_res != _box->active_right_res) return false;
	if (getNChi(_box->protein->res(_box->active_left_res).resKey) < 1) return false;

	int actual_height = getNChi(_box->protein->res(_box->active_left_res).resKey)*_h;
	return((mouse_x >= _x) && (mouse_x <(_x + _w))
		    && (mouse_y >= _y + 4*_h - actual_height)
			  && (mouse_y <= _y + 4*_h) );
}



int ChiBox::handle(int event)
{
  switch (event)
  {
		case WGT_MOUSE_B1_DOWN:
		  active_chi = 0;
	  case WGT_MOUSE_B1_DRAG:
		{
     	if (_box->active_left_res != _box->active_right_res) return 0;

			if (active_chi == 0)
				active_chi = (int)( 4 - (mouse_y - _y)/(double)_h + 1 );

      if ((active_chi > 0) && (active_chi <= getNChi(_box->protein->res(_box->active_left_res).resKey)))
			{
    		double chi = ( 360.0 * (mouse_x - _x - 2) / (double)(_w - 4) + -180.0 );
		  	normalise(chi);
  			_box->protein->rotateChi(_box->active_left_res, active_chi, chi);
	  		_box->setFocusWithSameOrientation();
			}
		}
		break;
		default:
		  return 0;
	}
  return 1;
}



void ChiBox::draw(BITMAP *bmp)
{
	if (_box->active_left_res != _box->active_right_res) return;
	if (getNChi(_box->protein->res(_box->active_left_res).resKey) < 1) return;

	text_mode(-1);

	for (int j=1; j <= getNChi(_box->protein->res(_box->active_left_res).resKey); j+=1)
	{
		float chi = _box->protein->res(_box->active_left_res).chi(j);
		int r_x = (int)( (_w - 4) * (chi - (-180.0)) / 360.0 + _x + 2 );
		int r_y = _y + _h * (4 - j);
		textprintf(bmp, font, _x + _w - 65, r_y + 5, GREEN, "chi%d% 4.0f", j, chi);
		rect(bmp, _x, r_y, _x + _w - 1, r_y + _h, GREEN);
		circlefill(bmp, r_x, r_y + _h/2, 4, BLACK);
		circlefill(bmp, r_x, r_y + _h/2, 3, GREEN);
	}
}







bool SequenceBox::isInside(int x, int y)

{
	return((x >= _x) && (x <(_x + _box->protein->nRes() * _w)) && (y >= _y) && (y <(_y + _h)) );
}



int SequenceBox::handle(int event)
{
  switch (event)
	{
	  case WGT_MOUSE_B1_DOWN:
		{
			int n = convertedSeqNo(mouse_x);

			if (n != 0)
			{
				_box->active_left_res   = n;
				_box->active_right_res  = n;
				_anchor_res  = n;
			}
	  }
		break;

	  case WGT_MOUSE_B1_DRAG:
		{
			int n;
			n = convertedSeqNo(mouse_x);
			if (n != 0)
      {
				if (n >= _anchor_res)
        {
				  _box->active_right_res = n;
				  _box->active_left_res  = _anchor_res;
				}
				else
        {
				  _box->active_left_res  = n;
				  _box->active_right_res = _anchor_res;
				}
				
        if (VIEW_SIZE+1 < _box->protein->nRes())
        {
          if ( (n >= _box->view_right_res) && (n <= _box->protein->nRes()) )
          {
             _box->view_right_res = n;
             _box->view_left_res = n - VIEW_SIZE + 1;
          }
					if ( (n <= _box->view_left_res) && (n > 0 ) )
          {
             _box->view_left_res = n;
             _box->view_right_res = n + VIEW_SIZE - 1;
          }
        }
			}
		}
		return 1;
    break;

	  case WGT_MOUSE_B2_DOWN:
		{
			int	n = convertedSeqNo(mouse_x);
			if ((n < _box->active_left_res) || (n > _box->active_right_res) )
      {
				_box->active_left_res  = n;
				_box->active_right_res = n;
			}
			_mutate_flag = true;
			_mutate_res  = n;
		}
  	break;

	  case WGT_MOUSE_B2_RELEASE:
		{

			if (!_mutate_flag) return 0;

			_mutate_flag = false;

			int m_x = (_mutate_res - _box->view_left_res) * _w + _x;
	    int m_y = _y;

			int i = (int)((mouse_x - m_x) /(double)_w + 1 );
			int j = (int)((mouse_y - m_y) /(double)_h );

			int n = (j-1) * 5 + i;

			if ((n < 0) && (n > 24))
			  return 0;

			if ((n >= 1) && (n <= 20))
			{
				for (i = _box->active_left_res; i <= _box->active_right_res; i++)
				{
					_box->protein->mutateRes(i, n);
          _box->protein->sproutWholeSidechain(i);
				}
				return 1;
			}

			if ((n >= 21) && (n <= 22))
      // copy residues
			{
         _box->protein->copyResToRight(_box->active_left_res, _box->active_right_res);
         if (_box->focus_res > _box->active_right_res)
           _box->focus_res += _box->active_right_res - _box->active_left_res + 1;
			}

 	    if ((n >= 23) && (n <= 24))
      // delete residues
      {
        if ( (_box->focus_res >= _box->active_left_res) && (_box->focus_res <= _box->active_right_res) )
          _box->focus_res = _box->active_left_res;
        else if (_box->focus_res > _box->active_right_res)
          _box->focus_res -= _box->active_right_res - _box->active_left_res + 1;

        _box->protein->deleteRes(_box->active_left_res, _box->active_right_res);

        if (_box->active_left_res > _box->protein->nRes())
          _box->active_left_res = _box->protein->nRes();
        _box->active_right_res = _box->active_left_res;
        if (_box->focus_res > _box->protein->nRes())
          _box->focus_res = _box->protein->nRes();
			}

      checkBounds();

			_box->setFocusWithSameOrientation();
			return 1;
  	}
  	break;
	}
  return 0;
}


void SequenceBox::checkBounds()
{
  if ( _box->protein->nRes() <= VIEW_SIZE )
	{
    _box->view_left_res = 1;
    _box->view_right_res = _box->protein->nRes();
  }

  while (_box->view_right_res > _box->protein->nRes())
    _box->view_right_res--;


  while (((_box->view_right_res - _box->view_left_res + 1) < VIEW_SIZE) && (VIEW_SIZE < _box->protein->nRes()) )
    if (_box->view_right_res != _box->protein->nRes())
      _box->view_right_res++;
    else
      _box->view_left_res--;
}



int SequenceBox::convertedSeqNo(int x)
{
	int j = (int)((x - _x) /(double)_w + _box->view_left_res);
	if ( (j>=1) && (j <= _box->protein->nRes()) ) return j;
	return 0;
}



void SequenceBox::draw(BITMAP *bmp)
{
	text_mode(-1);
  int max_res = _box->view_right_res - _box->view_left_res + 1;
	int i;
	for (i = 1; i <= max_res; i+=1)

  {
		int r_x = (i-1)*_w + _x;

    int actual_res = i + _box->view_left_res - 1;
		if ((actual_res >= _box->active_left_res) && (actual_res <= _box->active_right_res) )
    {
			rectfill(bmp, r_x, _y, r_x + _w, _y + _h, GREEN);
			rect(bmp, r_x, _y, r_x + _w, _y + _h, WHITE);
			textprintf(bmp, font, r_x + 3, _y + 3, BLACK, "%c", getResChar(_box->protein->res(actual_res).resKey));
		}
		else
    {
			rect(bmp, r_x, _y, r_x+ _w, _y + _h, WHITE);
			textprintf(bmp, font, r_x+3, _y+3, WHITE, "%c", getResChar(_box->protein->res(actual_res).resKey));
		}

	}

  if (!_mutate_flag) return;

  int m_x = (_mutate_res - _box->view_left_res)*_w + _x;
	int m_y = _y;

	int j;
	for (j=1; j<=4; j+=1)
		for (i=1; i<=5; i+=1)
    {
			int r_x = m_x + (i-1)*_w;

			int r_y = m_y + j*_h;
			if ((mouse_x >= r_x) && (mouse_x <(r_x + _w))
		     && (mouse_y >= r_y) && (mouse_y <(r_y + _h)) )
      {
				rectfill(bmp, r_x, r_y, r_x+_w, r_y+_h, GREEN);

				textprintf(bmp, font, r_x+3, r_y+3, BLACK, "%c", getResChar((j-1)*5+i));
			}
			else
      {
				rectfill(bmp, r_x, r_y, r_x+_w, r_y+_h, BLACK);
				textprintf(bmp, font, r_x+3, r_y+3, GREEN, "%c", getResChar((j-1)*5+i));
			}
			rect(bmp, r_x, r_y, r_x+_w, r_y+_h, GREEN);

		}

	int r_x = m_x;
	int r_y = m_y + 5*_w;
	if ( (mouse_x >= r_x) && (mouse_x < (r_x + _w*2))
	  && (mouse_y >= r_y) && (mouse_y < (r_y+_h)) )
  {
		rectfill(bmp, r_x, r_y, r_x+_w*2, r_y+_h, GREEN);

		textprintf(bmp, font, r_x+3, r_y+3, BLACK, "CPY");
	}
	else
  {
		rectfill(bmp, r_x, r_y, r_x+_w*2, r_y+_h, BLACK);
		textprintf(bmp, font, r_x+3, r_y+3, GREEN, "CPY");
	}
	rect(bmp, r_x, r_y, r_x+_w*2, r_y+_h, GREEN);

	r_x = m_x + _w*2;
	r_y = m_y + 5*_h;
	if ((mouse_x >= r_x) && (mouse_x <(r_x + _w*2))
	 && (mouse_y >= r_y) && (mouse_y <(r_y+_h)) )
  {
		rectfill(bmp, r_x, r_y, r_x+_w*2, r_y+_h, GREEN);
		textprintf(bmp, font, r_x+3, r_y+3, BLACK, "DEL");
	}
	else
  {
		rectfill(bmp, r_x, r_y, r_x+_w*2, r_y+_h, BLACK);

		textprintf(bmp, font, r_x+3, r_y+3, GREEN, "DEL");
	}
	rect(bmp, r_x, r_y, r_x+_w*2, r_y+_h, GREEN);


}




int SeqFocusBox::handle(int event)
{
  switch (event)
	{
	  case WGT_MOUSE_B1_DOWN:
	  case WGT_MOUSE_B1_DRAG:
		{
	    int j = convertedSeqNo(mouse_x);
	    if ( (j >= 1) && (j <= _box->protein->nRes()) )
			{
		    _box->focus_res = j;
		    _box->focus_atom = CA;
		    _box->setFocus();
        if (j > _box->view_right_res)
				{
          _box->view_right_res = j;
          _box->view_left_res = _box->view_right_res - VIEW_SIZE + 1;
        }

        if (j < _box->view_left_res)
				{
          _box->view_left_res = j;
          _box->view_right_res = _box->view_left_res + VIEW_SIZE - 1;
        }
	    }
   	}

		return 1;
    break;
	}
  return 0;
}



void SeqFocusBox::draw(BITMAP *bmp)
{
	text_mode(-1);
  int max_res = _box->view_right_res - _box->view_left_res + 1;
	int i;
	for (i = 1; i <= max_res; i+=1)
	{
		int r_x = (i-1)*_w + _x;
    int actual_res = i + _box->view_left_res - 1;
		if (actual_res == _box->focus_res)
		{
			rectfill(bmp, r_x, _y, r_x + _w, _y + _h, PURPLE);
			rect(bmp, r_x, _y, r_x + _w, _y + _h, WHITE);
		}
		else
		{
			rect(bmp, r_x, _y, r_x + _w, _y + _h, WHITE);
		}
	}
}






bool ScrollBar::isInside(int x, int y)
{
  if (_box->protein->nRes() <= VIEW_SIZE) return false;
	return((x >= _x) && (x <(_x + _box->protein->nRes() * _w))
		   && (y >= _y) && (y <(_y + _h)) );
}



int ScrollBar::handle(int event)
{
  switch (event)

	{
	  case WGT_MOUSE_B1_DOWN:
		{
	    int residueClicked = (int)((mouse_x - _x + 1) /((double)_w - 4.0) * _box->protein->nRes());

      if ((residueClicked >= _box->view_left_res) && (residueClicked <= _box->view_right_res) )

        _centreResidueOffset = residueClicked -(_box->view_right_res + _box->view_left_res) / 2;
      else
        _centreResidueOffset = 0;
    }

	  case WGT_MOUSE_B1_DRAG:
		{
	    int new_centre_residue = (int)(1.0 * (mouse_x - _x + 1) /(_w - 4) * _box->protein->nRes() - _centreResidueOffset);
	    int centre_residue = (int)(1.0 * (_box->view_right_res + _box->view_left_res) / 2);
      int disp = new_centre_residue - centre_residue;


      _box->view_left_res += disp;
      _box->view_right_res += disp;

      if (_box->view_left_res < 1)
			{
        _box->view_left_res = 1;
        _box->view_right_res = VIEW_SIZE;
      }
      if (_box->view_right_res > _box->protein->nRes())
			{
        _box->view_right_res = _box->protein->nRes();
        _box->view_left_res = _box->view_right_res - VIEW_SIZE + 1;
      }


/*      if (_box->showState != SHOW_ALL)
      {
        if (_box->focus_res > _box->view_right_res)
				{
          _box->focus_res = _box->view_right_res;
          _box->focus_atom = CA;
        }
        if (_box->focus_res < _box->view_left_res)
				{
          _box->focus_res = _box->view_left_res;
          _box->focus_atom = CA;
        }
      }
*/
      
      _box->setFocus();

		}
		return 1;
    break;
	}
  return 0;
}



void ScrollBar::draw(BITMAP *bmp)
{

  if (_box->protein->nRes() <= VIEW_SIZE) return;

  int bar_size      = (int)(1.0 * (_box->view_right_res - _box->view_left_res + 1) / _box->protein->nRes() * (_w-4));
  int bar_offset    = (int)(1.0 * (_box->view_left_res - 1) / _box->protein->nRes() * (_w-4));
  int focus_offset  = (int)(1.0 * (_box->focus_res - 1) / _box->protein->nRes() * (_w-4));
  int focus_offset2 = (int)(1.0 * (_box->focus_res) / _box->protein->nRes() * (_w-4));
  int active_offset = (int)(1.0 * (_box->active_left_res - 1) / _box->protein->nRes() * (_w-4));
  int active_offset2= (int)(1.0 * (_box->active_right_res) / _box->protein->nRes() * (_w-4));

  rect(bmp, _x + bar_offset + 1, _y + 1, _x + bar_offset + bar_size - 1, _y + _h - 2, BLACK);
  rectfill(bmp, _x + bar_offset + 2, _y + 2, _x + bar_offset + bar_size + 2, _y + _h - 2, WHITE);


  line(bmp, _x, _y, _x, _y +_h - 1, WHITE);
  line(bmp, _x +_w - 1, _y, _x +_w - 1, _y +_h - 1, WHITE);
  line(bmp, _x, _y + _h/2, _x +_w - 1, _y + _h/2, WHITE);

  rectfill(bmp, _x + active_offset + 2, _y + 3, _x + active_offset2 + 1, _y + _h - 5, GREEN);
  rectfill(bmp, _x + focus_offset + 2, _y + 5, _x + focus_offset2 + 1, _y + _h - 3, PURPLE);
}


