// REDKARMA LOGO: Copyright 2002,2003 Kenneth Thornton
//                All rights reserved.
//
// PURPOSE: Contains standard code for RedKarma animated intro logo
//
// NOTE: The code herein *should* be written to allow for easy insertion
//       into new projects as they arise. Timers will be assumed to be
//       available, and resolution and color mode will be locked at 320x2X0,
//       8bpp. This shouldn't be too difficult to change if it becomes
//       a necessity.



#include "ha_demo.h"
#include <stdlib.h>
#include <time.h>

#define H_MAX_DELAY   7
#define MAX_CELLS     5000      // Theoretical maximum number of active cells
                                // Being conservative may help slow CPUs



// NOTE: This is the standard construct for storing the effect's pertinent
//       data. These variables are considered PUBLIC, and may be raped by
//       any routine that is within scope.
//
typedef struct _RK
{
  BITMAP *texture;              // Holds texture for shading logo
  bool finished;                // boolean: TRUE if time out or user abort
  short timeout;                // Time left before intro automatically aborts
  
  short h_x_pos;                // X offset of highlight layer
  short h_y_pos;                // Y offset of highlight layer
  short h_delay;                // Delay between moving highlight layer
  
  short cells[MAX_CELLS][5];    // X, Y, Color, Velocity, Distance
  short in_use;                 // number of cells actually in use
  short mode;                   // Mode of animation. Modes are:
                                // 1 - assembling picture
                                // 2 - highlighting picture
                                // 3 - reserved for future use
} _RK;



_RK karma;                      // Go ahead and define our container
COLOR_MAP karma_map;            // We'll need a color map for this effect




// NOTE: Since this effect is processor independent, we need to use a timer
//       to control the speed of the animation. Stock settings are for 70hz,
//       to match the refresh rate of mode 13h. It should still look pretty
//       good in other video modes, however. We're moving pixels, not
//       scrolling, so we have some considerable leeway.
//
void RK_animate_cells()
{
 short i;                         // Loop control variable

 // Check for cells that need to be moved
 for(i=0; i<karma.in_use; i++) {
    if(karma.cells[i][4]) {
        karma.cells[i][1] += karma.cells[i][3];
        karma.cells[i][4]--;
                          }
                               }

 // Check for highlight layers that need to be moved
 karma.h_delay++;
 if(karma.h_delay >= H_MAX_DELAY) {
    karma.h_delay = 0;
    karma.h_x_pos++; karma.h_x_pos &= 0xFF;
    karma.h_y_pos++; karma.h_y_pos &= 0xFF;
                                  }
                                  
 // Check for effect timeout (i.e. forced to continue)
 karma.timeout--;
 if(karma.timeout <= 0) { karma.finished = 1; }

 // Check for effect cancel (i.e. user abort)
 if(key[KEY_ESC]) { karma.finished = 1; }
 if(key[KEY_ALT]) { karma.finished = 1; }

 return;
}

END_OF_FUNCTION(RK_animate_cells);



// NOTE: Logo is composed of evenly spaced pixels, or cells. Instead of
//       painstakingly defining these cells in a list or array, we can read
//       their position from within a bitmap stored on the HD. Since we know
//       the pixels of the graphic will show up from 1-15, we can look for
//       them, and capture their position on-screen.
//
void RK_read_source_graphic(char *name)
{
 short i,j;                       // Variables for loop control
 BITMAP *loader;                  // Stores data while we analyze it

 // Let's initialize our data before starting
 for(i=0; i<MAX_CELLS; i++) {
    for(j=0; j<5; j++) { karma.cells[i][j] = 0; }
                            }

 karma.mode     = 1;
 karma.in_use   = 0;
 karma.timeout  = 490;            // 7 seconds at 70 cycles per second
 karma.finished = 0;
 karma.h_x_pos  = 0;
 karma.h_y_pos  = 0;
 karma.texture  = create_bitmap(278,31);

 // Load texture
 extract_resource("res.dat", 418070, 4765);
 _32x_textures = load_pcx("tempola.xyz",palette);
 remove("tempola.xyz");
 
 // Load bitmap
 //
 extract_resource("res.dat", 412247, 5823);
 loader = load_pcx("tempola.xyz",palette);
 remove("tempola.xyz");

 // Loop through bitmap, searching for pixels
 for(i=0; i<200; i++) {
    for(j=0; j<320; j++) {
       if(loader->line[i][j] > 0 &&
          loader->line[i][j] < 32) {
             karma.cells[karma.in_use][0] = j;
             karma.cells[karma.in_use][1] = i;
             karma.cells[karma.in_use][2] = loader->line[i][j];
             karma.in_use++;
                                   }
                         }
                      }

 // For this effect we'll use a single light map
 create_light_table(&karma_map, palette, 0, 0, 0, NULL);

 // Hack color map so entry 0 returns same color (just in case it doesn't)
 for(i=0; i<256; i++) { karma_map.data[0][i] = i; }

 // Clean up our mess
 destroy_bitmap(loader);

 return;
}



// NOTE: Cells need to be animated, so we'll randomly scatter them along the
//       Y axis, and provide them with both a distance to travel, and a set
//       velocity. This makes the logo appear to assemble itself from a
//       plethora of random pixels. Take each cell and choose a random
//       direction (up or down), then determine the velocity, and a random
//       distance.
//
void RK_scatter_cells()
{
 short i,j;                              // Variables for loop control
 short _d;                               // For creating low-scope random numbers
 short _v;                               // For creating low-scope random numbers

 srand(time(0));                         // Plant the seed

 for(i=0; i<karma.in_use; i++) {
    // Choose random velocity for cell
    _d = ((rand() % 100) / 50);
    _v = ((rand() % 90) / 30) + 1;

    // Set velocities based on direction
    if(_d) { karma.cells[i][3] = 0 - _v; }
    else   { karma.cells[i][3] = _v;     }

    // Set distance, and offset cell Y value
    karma.cells[i][4] = (rand() % 200);
    if(karma.cells[i][3] > 0) { karma.cells[i][1] -= abs(karma.cells[i][3]) * karma.cells[i][4]; }
    else                      { karma.cells[i][1] += abs(karma.cells[i][3]) * karma.cells[i][4]; }

                               }


 return;
}



// NOTE: Renders a single frame using the data provided in the KARMA
//       container. Assembly buffer, palette, color map(s), and other graphic
//       resources should be considered as standard for any project, so basic
//       assumptions can be made. We'll provide the destination buffer just to
//       be safe!
//
void RK_render_frame(BITMAP *buf)
{
 short i,j;                       // Variables for loop control

 // Clear screen (we have CPU cycles to spare, so...)
 clear(buf);

 // Loop through cells, drawing them individually
 for(i=0; i<karma.in_use; i++) {
     putpixel(buf, karma.cells[i][0], karma.cells[i][1]+20, karma.cells[i][2]);
                               }


 // Any additional digital filters should go here; ideally this should
 // be unique to each application, just for kicks. Remember how LucasArts
 // always mocks their own logo in their games...
 //
 color_map = &karma_map;

 // I know; I said I hate sub-bitmaps. Sue me.
 //
 subs = create_sub_bitmap(_32x_textures, 0, 32, 32, 32);
 drawing_mode(DRAW_MODE_COPY_PATTERN, subs, karma.h_x_pos, karma.h_y_pos);
 rectfill(karma.texture, 0, 0, 277, 31 ,0);
 drawing_mode(DRAW_MODE_SOLID, NULL, 0, 0);
 destroy_bitmap(subs);
 draw_trans_sprite(buf, karma.texture, 22, 96);

 // Send completed buffer to the screen, using native transfer routine
 HA_final_pass(screenbuf);

 return;
}




// NOTE: This is the routine that coordinates the party. Call this after
//       you've initialized your graphics, and let it rock n' roll. You
//       should also have your palette and screen zero'd before calling this.
//
void RK_intro()
{

 RK_read_source_graphic("redkarma.pcx");
 RK_scatter_cells();

 // Hacked in for testing purposes
 clear(screen);
 set_palette(palette);

 LOCK_FUNCTION(RK_animate_cells);

 // Install intro handler at 60hz
 install_int_ex(RK_animate_cells,BPS_TO_TIMER(60));


 do {
   RK_render_frame(screenbuf);
    } while(!karma.finished);

 // Normally I fade while stuff is animating, but I feel lazy...
 fade_out(4);

 // Clean up the mess
 remove_int(RK_animate_cells);
 destroy_bitmap(karma.texture);
 
 return;
}



