/* GIVE PEACE A CHANCE */

/*
	input.c

	This is a groovy little input handler that can be quite
	useful. It is a bit cumbersome at present, but with a bit
	of modification, it could go a long way.

	You can create joysticks, keyboards, cpu, and log-read
	input handlers. The CPU input handler has a little 
	variable function that it calls to stay flexible.

	Logging is saved into a compressed packfile. I dunno
	how big of a performance hit that is.
*/

#include <allegro.h>

#include "main.h"
#include "input.h"

/*
	This creates an input handler. It is used to make
	a universal way to handle key presses.
*/
INPUT_HANDLER *input_handler_create(int type, int device)
{
	INPUT_HANDLER *ih = malloc(sizeof(INPUT_HANDLER));
	int i;

	ih->type = type;
	for (i=0; i!= MAX_INPUTS; i++)
	{
		ih->map[i] = 0;
		ih->status[i] = IH_STATUS_UP;
		ih->down[i] = 0;
	}
	
	switch (type)
	{
		case HANDLER_JOYSTICK:
			ih->device = device;
			ih->map[IH_BUTTON_A] = 0;
			ih->map[IH_BUTTON_B] = 1;
		break;

		case HANDLER_KEYBOARD:
			ih->map[IH_UP] = KEY_UP;
			ih->map[IH_DOWN] = KEY_DOWN;
			ih->map[IH_LEFT] = KEY_LEFT;
			ih->map[IH_RIGHT] = KEY_RIGHT;
			ih->map[IH_BUTTON_A] = KEY_SPACE;
			ih->map[IH_BUTTON_B] = KEY_TAB;
		break;
	}

	ih->log = 0;
	ih->cpu_proc = 0;

	return ih;
}

void input_handler_destroy(INPUT_HANDLER *ih)
{
	free(ih);
	ih = 0;

	return;
}

/*
	Opens or closes a log. Pass a valid filename to
	start logging keypresses. (You can even log a log!)
	Pass NULL to stop the logging and close the file.
*/
void input_handler_log(INPUT_HANDLER *ih, char *filename)
{	
	if (filename)
	{
		if (ih->type != HANDLER_LOG)
			ih->log = pack_fopen(filename, "pw");
		else
			ih->log = pack_fopen(filename, "pr");
	}
	else if (ih->log)
	{
		pack_fclose(ih->log);
		ih->log = 0;		
	}
	return;
}

/*
	The main function here, it checks for key presses and returns
	various status flags
*/

void input_handler_poll(INPUT_HANDLER *ih)
{
	int i;

	switch (ih->type)
	{
		/* the joystick: the first D-Pad is used + any defined buttons */
		case HANDLER_JOYSTICK:
			poll_joystick();
			
			ih->down[IH_UP] = joy[ih->device].stick[0].axis[1].d1;
			ih->down[IH_DOWN] = joy[ih->device].stick[0].axis[1].d2;
			ih->down[IH_LEFT] = joy[ih->device].stick[0].axis[0].d1;
			ih->down[IH_RIGHT] = joy[ih->device].stick[0].axis[0].d2;

			for (i=IH_BUTTON_A; i!=MAX_INPUTS; i++)
				ih->down[i] = joy[ih->device].button[ih->map[i]].b;
		break;

		/* check for key presses */
		case HANDLER_KEYBOARD:
			for (i=0; i!=MAX_INPUTS; i++)
				ih->down[i] = key[ih->map[i]];			
		break;

		/* reads from a log file */
		case HANDLER_LOG:
			if (ih->log)
			{
				/* we can return after reading, because the actual STATUS flags have been stored */
				pack_fread(ih->status, sizeof(ih->status), ih->log);
				return;
			}
		break;

		/* a hook to allow cpu input */
		case HANDLER_CPU:
			ih->cpu_proc(ih->down);
		break;
	}

	/* proess the down[] array (similar to the allegro key[]) and convert it to useful flags */
	for (i=0; i!=MAX_INPUTS; i++)
	{
		if (ih->down[i])
		{
			if (ih->status[i] == IH_STATUS_UP)
				ih->status[i] = IH_STATUS_PRESSED;
			else
				ih->status[i] = IH_STATUS_DOWN;				
		}
		else
		{
			if (ih->status[i] == IH_STATUS_DOWN)
				ih->status[i] = IH_STATUS_RELEASED;
			else
				ih->status[i] = IH_STATUS_UP;
		}
	}

	/* if logging, add to the file */
	if (ih->log) 
		pack_fwrite(ih->status, sizeof(ih->status), ih->log);					
}

