/*******************************************************************
 *
 *  ttapi.c
 *
 *    High-level interface implementation
 *
 *  Copyright 1996-1999 by
 *  David Turner, Robert Wilhelm, and Werner Lemberg.
 *
 *  This file is part of the FreeType project, and may only be used,
 *  modified, and distributed under the terms of the FreeType project
 *  license, LICENSE.TXT.  By continuing to use, modify, or distribute
 *  this file you indicate that you have read the license and
 *  understand and accept it fully.
 *
 *  Notes:
 *
 *    This file is used to implement most of the functions that are
 *    defined in the file "freetype.h". However, two functions are
 *    implemented elsewhere :
 *
 *     TT_MulDiv and TT_MulFix  are in ttcalc.h/ttcalc.c
 *
 ******************************************************************/

#include "ttconfig.h"

#include "freetype.h"
#include "ttengine.h"
#include "ttcalc.h"
#include "ttmemory.h"
#include "ttcache.h"
#include "ttfile.h"
#include "ttobjs.h"
#include "ttload.h"
#include "ttgload.h"
#include "ttraster.h"
#include "ttextend.h"


/* required by the tracing mode */
#undef  TT_COMPONENT
#define TT_COMPONENT  trace_api


#ifdef TT_STATIC_RASTER
#define RAS_OPS  /* void */
#define RAS_OP   /* void */
#else
#define RAS_OPS  ((TRaster_Instance*)_engine->raster_component),
#define RAS_OP   ((TRaster_Instance*)_engine->raster_component)
#endif /* TT_STATIC_RASTER */


#define RENDER_Glyph( glyph, target ) \
          Render_Glyph( RAS_OPS  glyph, target )

#define RENDER_Gray_Glyph( glyph, target, palette ) \
          Render_Gray_Glyph( RAS_OPS  glyph, target, palette )



/*******************************************************************
 *
 *  Function    :  TT_FreeType_Version
 *
 *  Description :  Returns the major and minor version of the library.
 *
 *  Input  :  major, minor addresses
 *
 *  Output :  Error code.
 *
 *  MT-Note : YES!
 *
 ******************************************************************/

  EXPORT_FUNC
  TT_Error  TT_FreeType_Version( int  *major, int  *minor )
  {
    if ( !major || !minor )
      return TT_Err_Invalid_Argument;

    *major = TT_FREETYPE_MAJOR;
    *minor = TT_FREETYPE_MINOR;

    return TT_Err_Ok;
  }


/*******************************************************************
 *
 *  Function    : TT_Init_FreeType
 *
 *  Description : The library's engine initializer.  This function
 *                must be called prior to any call.
 *
 *  Input  :  engine        pointer to a FreeType engine instance
 *
 *  Output :  Error code.
 *
 *  MT-Note : This function should be called each time you want
 *            to create a TT_Engine.  It is not necessarily thread
 *            safe depending on the implementations of ttmemory,
 *            ttfile and ttmutex, so take care.  Their default
 *            implementations are safe, however.
 *
 ******************************************************************/

  EXPORT_FUNC
  TT_Error  TT_Init_FreeType( TT_Engine*  engine )
  {
    PEngine_Instance  _engine;

    TT_Error  error;
    int       n;


    /* first of all, initialize memory sub-system */
    error = TTMemory_Init();
    if ( error )
      return error;

    /* Allocate engine instance */
    if ( ALLOC( _engine, sizeof ( TEngine_Instance ) ) )
      return error;

#undef  TT_FAIL
#define TT_FAIL( x )  ( error = x (_engine) ) != TT_Err_Ok

    /* Initalize components */
    if ( TT_FAIL( TTFile_Init  )  ||
         TT_FAIL( TTCache_Init )  ||
#ifdef TT_CONFIG_OPTION_EXTEND_ENGINE
         TT_FAIL( TTExtend_Init ) ||
#endif
         TT_FAIL( TTObjs_Init )   ||
         TT_FAIL( TTRaster_Init ) )
       goto Fail;

#undef TT_FAIL

    /* set the gray palette defaults: 0 to 4 */
    for ( n = 0; n < 5; n++ )
      _engine->raster_palette[n] = (Byte)n;  /* Conversion ok, some warn */

    /* create the engine lock */
    MUTEX_Create( _engine->lock );

    HANDLE_Set( *engine, _engine );
    return TT_Err_Ok;

  Fail:
    TT_Done_FreeType( *engine );
    HANDLE_Set( *engine, NULL );
    return error;
  }


/*******************************************************************
 *
 *  Function    : TT_Done_FreeType
 *
 *  Description : The library's engine finalizer.  This function
 *                will discard all active face and glyph objects
 *                from the heap.
 *
 *  Input  :  engine        FreeType engine instance
 *
 *  Output :  Error code.
 *
 *  MT-Note : Destroys an engine.  Not necessarily thread-safe
 *            depending on the implementations of ttmemory,
 *            ttfile and ttmutex.  The default implementations
 *            are safe, however.
 *
 ******************************************************************/

  EXPORT_FUNC
  TT_Error  TT_Done_FreeType( TT_Engine  engine )
  {
    PEngine_Instance  _engine = HANDLE_Engine( engine );


    if ( !_engine )
      return TT_Err_Ok;

    MUTEX_Destroy( _engine->lock );

    TTRaster_Done( _engine );
    TTObjs_Done  ( _engine );
#ifdef TT_CONFIG_OPTION_EXTEND_ENGINE
    TTExtend_Done( _engine );
#endif
    TTCache_Done ( _engine );
    TTFile_Done  ( _engine );
    FREE( _engine );

    TTMemory_Done();

    return TT_Err_Ok;
  }


#ifdef TT_CONFIG_OPTION_GRAY_SCALING

/*******************************************************************
 *
 *  Function    :  TT_Set_Raster_Gray_Palette
 *
 *  Description :  Sets the gray-levels palette used for font
 *                 smoothing.
 *
 *  Input  :  engine        FreeType engine instance
 *            palette       address of palette (a 5 byte array)
 *
 *  Output :  Invalid argument if 'palette' is NULL.
 *
 *  MT-Note:  NO!  Unprotected modification of an engine's palette.
 *
 ******************************************************************/

  EXPORT_FUNC
  TT_Error  TT_Set_Raster_Gray_Palette( TT_Engine  engine,
                                        Byte*      palette )
  {
    int  i;


    if ( !palette )
      return TT_Err_Invalid_Argument;

    for ( i = 0; i < 5; i++ )
      HANDLE_Engine( engine )->raster_palette[i] = (Byte)palette[i];

    return TT_Err_Ok;
  }

#endif /* TT_CONFIG_OPTION_GRAY_SCALING */


/*******************************************************************
 *
 *  Function    :  TT_Open_Face
 *
 *  Description :  Creates a new face object from a given font file.
 *
 *  Input  :  engine        FreeType engine instance
 *            fontPathName  the font file's pathname
 *            face          adress of returned face handle
 *
 *  Output :  Error code.
 *
 *  Note :    The face handle is set to NULL in case of failure.
 *
 *  MT-Note : YES!
 *
 ******************************************************************/

  EXPORT_FUNC
  TT_Error  TT_Open_Face( TT_Engine       engine,
                          const TT_Text*  fontPathName,
                          TT_Face*        face )
  {
    PEngine_Instance  _engine = HANDLE_Engine( engine );

    TFont_Input  input;
    TT_Error     error;
    TT_Stream    stream;
    PFace        _face;


    if ( !_engine )
      return TT_Err_Invalid_Engine;

    /* open the file */
    error = TT_Open_Stream( fontPathName, &stream );
    if ( error )
      return error;

    input.stream    = stream;
    input.fontIndex = 0;
    input.engine    = _engine;

    /* Create and load the new face object - this is thread-safe */
    error = CACHE_New( _engine->objs_face_cache,
                       _face,
                       &input );

    /* Set the handle */
    HANDLE_Set( *face, _face );

    if ( error )
      goto Fail;

    return TT_Err_Ok;

  Fail:
    TT_Close_Stream( &stream );
    return error;
  }


/*******************************************************************
 *
 *  Function    :  TT_Open_Collection
 *
 *  Description :  Creates a new face object from a given font file.
 *
 *  Input  :  engine                FreeType engine instance
 *            collectionPathName    the font file's pathname
 *            fontIndex             index of font in TrueType collection
 *            face                  adress of returned face handle
 *
 *  Output :  Error code.
 *
 *  Note :    The face handle is set to NULL in case of failure.
 *
 *  MT-Note : YES!
 *
 ******************************************************************/

  EXPORT_FUNC
  TT_Error  TT_Open_Collection( TT_Engine       engine,
                                const TT_Text*  collectionPathName,
                                TT_ULong        fontIndex,
                                TT_Face*        face )
  {
    PEngine_Instance  _engine = HANDLE_Engine( engine );

    TFont_Input  input;
    TT_Error     error;
    TT_Stream    stream;
    PFace        _face;


    if ( !_engine )
      return TT_Err_Invalid_Engine;

    /* open the file */
    error = TT_Open_Stream( collectionPathName, &stream );
    if ( error )
      return error;

    input.stream    = stream;
    input.fontIndex = fontIndex;
    input.engine    = _engine;

    /* Create and load the new face object - this is thread-safe */
    error = CACHE_New( _engine->objs_face_cache,
                       _face,
                       &input );

    /* Set the handle */
    HANDLE_Set( *face, _face );

    if ( error )
      goto Fail;

    return TT_Err_Ok;

  Fail:
    TT_Close_Stream( &stream );

    return error;
  }


/*******************************************************************
 *
 *  Function    :  TT_Get_Face_Properties
 *
 *  Description :  Returns face properties.
 *
 *  Input  :  face          the face handle
 *            properties    address of target properties record
 *
 *  Output :  Error code.
 *
 *  Note :    Currently, max_Faces is always set to 0.
 *
 *  MT-Note : YES!  Reads only permanent data.
 *
 ******************************************************************/

  EXPORT_FUNC
  TT_Error  TT_Get_Face_Properties( TT_Face              face,
                                    TT_Face_Properties*  properties )
  {
    PFace _face = HANDLE_Face( face );


    if ( !_face )
      return TT_Err_Invalid_Face_Handle;

    properties->num_Glyphs   = _face->numGlyphs;
    properties->max_Points   = _face->maxPoints;
    properties->max_Contours = _face->maxContours;
    properties->num_CharMaps = _face->numCMaps;
    properties->num_Names    = _face->nameTable.numNameRecords;

    if ( _face->ttcHeader.DirCount == 0 )
      properties->num_Faces = 1;
    else
      properties->num_Faces = _face->ttcHeader.DirCount;

    properties->header       = &_face->fontHeader;
    properties->horizontal   = &_face->horizontalHeader;

    if ( _face->verticalInfo )
      properties->vertical   = &_face->verticalHeader;
    else
      properties->vertical   = NULL;

    properties->os2          = &_face->os2;
    properties->postscript   = &_face->postscript;
    properties->hdmx         = &_face->hdmx;

    return TT_Err_Ok;
  }


/*******************************************************************
 *
 *  Function    :  TT_Set_Face_Pointer
 *
 *  Description :  Each face object has one pointer, which use is
 *                 reserved to client applications.  The TrueType
 *                 engine never accesses or uses this field.
 *
 *                 This function is used to set the pointer.
 *
 *  Input  :  face    the given face handle
 *            data    the generic pointer value
 *
 *  Output :  Error code.
 *
 *  MT-Note : NO!  But this function is reserved to "enlightened"
 *            developers, so it shouldn't be a problem.
 *
 ******************************************************************/

  EXPORT_FUNC
  TT_Error  TT_Set_Face_Pointer( TT_Face  face,
                                 void*    data )
  {
    PFace  faze = HANDLE_Face( face );


    if ( !faze )
      return TT_Err_Invalid_Face_Handle;
    else
      faze->generic = data;

    return TT_Err_Ok;
  }


/*******************************************************************
 *
 *  Function    :  TT_Get_Face_Pointer
 *
 *  Description :  Each face object has one pointer, which use is
 *                 reserved to client applications.  The TrueType
 *                 engine never access or use this field.
 *
 *                 This function is used to read the pointer.
 *
 *  Input  :  face    the given face handle
 *            data    the generic pointer value
 *
 *  Output :  Error code.
 *
 *  MT-Note : NO!  But this function is reserved to "enlightened"
 *            developers, so it shouldn't be a problem.
 *
 ******************************************************************/

  EXPORT_FUNC
  void*  TT_Get_Face_Pointer( TT_Face  face )
  {
    PFace  faze = HANDLE_Face( face );


    if ( !faze )
      return NULL;
    else
      return faze->generic;
  }


/*******************************************************************
 *
 *  Function    :  TT_Get_Face_Metrics
 *
 *  Description :  This function returns the original horizontal AND
 *                 vertical metrics as found in the "hmtx" and "vmtx"
 *                 tables.  These are the glyphs' left-side-bearings
 *                 and advance widths (horizontal), as well as top
 *                 side bearings and advance heights (vertical).
 *
 *                 All are expressed in FONT UNITS, a.k.a. EM
 *                 units.
 *
 *  Input  :     face  The given face handle.
 *              first  Index of first glyph in table.
 *               last  Index of last glyph in table.
 *
 *       leftBearings  A pointer to an array of TT_Shorts where the
 *                     left side bearings for the glyphs 'first'
 *                     to 'last' will be returned.  If these metrics
 *                     don't interest you, simply set it to NULL.
 *
 *             widths  A pointer to an array of TT_UShorts
 *                     where the advance widths for the glyphs
 *                     'first' to 'last' will be returned.  If these
 *                     metrics don't interest you, simply set it
 *                     to NULL.
 *
 *        topBearings  A pointer to an array of TT_Shorts where the
 *                     top side bearings for the glyphs 'first'
 *                     to 'last' will be returned.  If these metrics
 *                     don't interest you, simply set it to NULL.
 *
 *            heights  A pointer to an array of TT_UShorts
 *                     where the advance heights for the glyphs
 *                     'first' to 'last' will be returned.  If these
 *                     metrics don't interest you, simply set it
 *                     to NULL.
 *
 *  Output :  Error code.
 *
 *  IMPORTANT NOTE :
 *
 *  As vertical metrics are optional in a TrueType font, this
 *  function will return an error ( TT_Err_No_Vertical_Data )
 *  whenever this function is called on such a face with non-NULL
 *  'topBearings' or 'heights' arguments.
 *
 *  When a font has no vertical data, the 'vertical' field in its
 *  properties structure is set to NULL.
 *
 *  MT-Note : YES!  Reads only permanent data.
 *
 ******************************************************************/

  EXPORT_FUNC
  TT_Error  TT_Get_Face_Metrics( TT_Face     face,
                                 TT_UShort   firstGlyph,
                                 TT_UShort   lastGlyph,
                                 TT_Short*   leftBearings,
                                 TT_UShort*  widths,
                                 TT_Short*   topBearings,
                                 TT_UShort*  heights )
  {
    PFace   _face = HANDLE_Face( face );
    UShort  num;


    if ( !_face )
      return TT_Err_Invalid_Face_Handle;

    /* Check the glyph range */
    if ( lastGlyph >= _face->numGlyphs || firstGlyph > lastGlyph )
      return TT_Err_Invalid_Argument;

    num = lastGlyph - firstGlyph;   /* number of elements-1 in each array */

    /* store the left side bearings and advance widths first */
    {
      UShort  n;
      Short   left_bearing;
      UShort  advance_width;


      for ( n = 0; n <= num; n++ )
      {
        TT_Get_Metrics( &_face->horizontalHeader,
                        firstGlyph + n, &left_bearing, &advance_width );

        if ( leftBearings )  leftBearings[n] = left_bearing;
        if ( widths )        widths[n]       = advance_width;
      }
    }

    /* check for vertical data if topBearings or heights is non-NULL */
    if ( !topBearings && !heights )
      return TT_Err_Ok;

    if ( !_face->verticalInfo )
      return TT_Err_No_Vertical_Data;

    /* store the top side bearings */
    {
      UShort  n;
      Short   top_bearing;
      UShort  advance_height;

      for ( n = 0; n <= num; n++ )
      {
        TT_Get_Metrics( (TT_Horizontal_Header*)&_face->verticalHeader,
                        firstGlyph + n, &top_bearing, &advance_height );

        if ( topBearings )  topBearings[n] = top_bearing;
        if ( heights )      heights[n]     = advance_height;
      }
    }

    return TT_Err_Ok;
  }


/*******************************************************************
 *
 *  Function    :  TT_Flush_Face
 *
 *  Description :  This function is used to close an active face's
 *                 file handle or descriptor.  This is useful to save
 *                 system resources, if your application uses tons
 *                 of fonts.
 *
 *  Input  :  face    the given face handle
 *
 *  Output :  Error code.
 *
 *  MT-Note : YES!  (If ttfile is.)
 *
 ******************************************************************/

  EXPORT_FUNC
  TT_Error  TT_Flush_Face( TT_Face  face )
  {
    PFace  faze = HANDLE_Face( face );


    if ( !faze )
      return TT_Err_Invalid_Face_Handle;
    else
      return TT_Flush_Stream( &faze->stream );
  }


/*******************************************************************
 *
 *  Function    :  TT_Close_Face
 *
 *  Description :  Closes an opened face object.  This function
 *                 will destroy all objects associated to the
 *                 face, except the glyphs.
 *
 *  Input  :  face    the given face handle
 *
 *  Output :  Error code.
 *
 *  NOTE   :  The handle is set to NULL on exit.
 *
 *  MT-Note : YES!
 *
 ******************************************************************/

  EXPORT_FUNC
  TT_Error  TT_Close_Face( TT_Face  face )
  {
    PFace  _face = HANDLE_Face( face );


    if ( !_face )
      return TT_Err_Invalid_Face_Handle;

    TT_Close_Stream( &_face->stream );

    /* delete the face object -- this is thread-safe */
    return CACHE_Done( _face->engine->objs_face_cache, _face );
  }


/*******************************************************************
 *
 *  Function    :  TT_New_Instance
 *
 *  Description :  Creates a new instance from a given face.
 *
 *  Input  :  face        parent face handle
 *            instance    address of instance handle
 *
 *  Output :  Error code.
 *
 *  Note   :  The handle is set to NULL in case of failure.
 *
 *  MT-Note : YES!
 *
 ******************************************************************/

  EXPORT_FUNC
  TT_Error  TT_New_Instance( TT_Face       face,
                             TT_Instance*  instance )
  {
    TT_Error   error;
    PFace      _face = HANDLE_Face( face );
    PInstance  _ins;


    if ( !_face )
      return TT_Err_Invalid_Face_Handle;

    /* get a new instance from the face's cache -- this is thread-safe */
    error = CACHE_New( &_face->instances, _ins, _face );

    HANDLE_Set( *instance, _ins );

    if ( !error )
    {
      error = Instance_Init( _ins );
      if ( error )
      {
        HANDLE_Set( *instance, NULL );
        CACHE_Done( &_face->instances, _ins );
      }
    }

    return error;
  }


/*******************************************************************
 *
 *  Function    :  TT_Set_Instance_Resolutions
 *
 *  Description :  Resets an instance to a new device resolution.
 *
 *  Input  :  instance      the instance handle
 *            xResolution   new horizontal device resolution in dpi
 *            yResolution   new vertical device resolution in dpi
 *
 *  Output :  Error code.
 *
 *  Note :    There is no check for overflow; with other words,
 *            the product of glyph dimensions times the device
 *            resolutions must have reasonable values.
 *
 *  MT-Note : You should set the charsize or pixel size immediately
 *            after this call in multi-threaded programs.  This will
 *            force the instance data to be resetted.  Otherwise, you
 *            may encounter corruption when loading two glyphs from
 *            the same instance concurrently!
 *
 *            Happily, 99.99% will do just that :-)
 *
 ******************************************************************/

  EXPORT_FUNC
  TT_Error  TT_Set_Instance_Resolutions( TT_Instance  instance,
                                         TT_UShort    xResolution,
                                         TT_UShort    yResolution )
  {
    PInstance  ins = HANDLE_Instance( instance );


    if ( !ins )
      return TT_Err_Invalid_Instance_Handle;

    ins->metrics.x_resolution = xResolution;
    ins->metrics.y_resolution = yResolution;
    ins->valid                = FALSE;

    /* In the case of a thread-safe implementation, we immediately    */
    /* call Instance_Reset in order to change the instance's variable */

    /* In the case of a non-threaded build, we simply set the 'valid' */
    /* flag to FALSE, which will force the instance's resetting at    */
    /* the next glyph loading                                         */

    return TT_Err_Ok;
  }


/*******************************************************************
 *
 *  Function    :  TT_Set_Instance_CharSizes
 *
 *  Description :  Resets an instance to new point size.
 *
 *  Input  :  instance      the instance handle
 *            charWidth     the new width in 26.6 char points
 *            charHeight    the new height in 26.6 char points
 *
 *  Output :  Error code.
 *
 *  Note :    There is no check for overflow; with other words,
 *            the product of glyph dimensions times the device
 *            resolution must have reasonable values.
 *
 *  MT-Note : NO!  This should be called only when setting/resetting
 *            instances, so there is no need to protect.
 *
 ******************************************************************/

  EXPORT_FUNC
  TT_Error  TT_Set_Instance_CharSizes( TT_Instance  instance,
                                       TT_F26Dot6   charWidth,
                                       TT_F26Dot6   charHeight )
  {
    PInstance  ins = HANDLE_Instance( instance );


    if ( !ins )
      return TT_Err_Invalid_Instance_Handle;

    if ( charWidth < 1 * 64 )
      charWidth = 1 * 64;

    if ( charHeight < 1 * 64 )
      charHeight = 1 * 64;

    ins->metrics.x_scale1 = ( charWidth * ins->metrics.x_resolution ) / 72;
    ins->metrics.x_scale2 = ins->owner->fontHeader.Units_Per_EM;

    ins->metrics.y_scale1 = ( charHeight * ins->metrics.y_resolution ) / 72;
    ins->metrics.y_scale2 = ins->owner->fontHeader.Units_Per_EM;

    if ( ins->owner->fontHeader.Flags & 8 )
    {
      ins->metrics.x_scale1 = (ins->metrics.x_scale1+32) & -64;
      ins->metrics.y_scale1 = (ins->metrics.y_scale1+32) & -64;
    }

    ins->metrics.x_ppem = ins->metrics.x_scale1 / 64;
    ins->metrics.y_ppem = ins->metrics.y_scale1 / 64;

    if ( charWidth > charHeight )
      ins->metrics.pointSize = charWidth;
    else
      ins->metrics.pointSize = charHeight;

    ins->valid  = FALSE;

    return Instance_Reset( ins );
  }


/*******************************************************************
 *
 *  Function    :  TT_Set_Instance_CharSize
 *
 *  Description :  Resets an instance to new point size.
 *
 *  Input  :  instance      the instance handle
 *            charSize      the new character size in 26.6 char points
 *
 *  Output :  Error code.
 *
 *  Note :    There is no check for overflow; with other words,
 *            the product of glyph dimensions times the device
 *            resolution must have reasonable values.
 *
 *  MT-Note : NO!  This should be called only when setting/resetting
 *            instances, so there is no need to protect.
 *
 ******************************************************************/

  EXPORT_FUNC
  TT_Error  TT_Set_Instance_CharSize( TT_Instance  instance,
                                      TT_F26Dot6   charSize )
  {
    return TT_Set_Instance_CharSizes( instance, charSize, charSize );
  }


/*******************************************************************
 *
 *  Function    :  TT_Set_Instance_PixelSizes
 *
 *  Description :  Resets an instance to new pixel sizes
 *
 *  Input  :  instance      the instance handle
 *            pixelWidth    the new width in pixels
 *            pixelHeight   the new height in pixels
 *
 *  Output :  Error code.
 *
 *  Note :    There is no check for overflow; with other words,
 *            the product of glyph dimensions times the device
 *            resolution must have reasonable values.
 *
 *  MT-Note : NO!  This should be called only when setting/resetting
 *            instances, so there is no need to protect.
 *
 ******************************************************************/

  EXPORT_FUNC
  TT_Error  TT_Set_Instance_PixelSizes( TT_Instance  instance,
                                        TT_UShort    pixelWidth,
                                        TT_UShort    pixelHeight,
                                        TT_F26Dot6   pointSize )
  {
    PInstance  ins = HANDLE_Instance( instance );

    if ( !ins )
      return TT_Err_Invalid_Instance_Handle;

    if ( pixelWidth  < 1 ) pixelWidth = 1;
    if ( pixelHeight < 1 ) pixelHeight = 1;

    ins->metrics.x_ppem    = pixelWidth;
    ins->metrics.y_ppem    = pixelHeight;
    ins->metrics.pointSize = pointSize;

    ins->metrics.x_scale1 = ins->metrics.x_ppem * 64L;
    ins->metrics.x_scale2 = ins->owner->fontHeader.Units_Per_EM;
    ins->metrics.y_scale1 = ins->metrics.y_ppem * 64L;
    ins->metrics.y_scale2 = ins->owner->fontHeader.Units_Per_EM;

    ins->valid = FALSE;

    return Instance_Reset( ins );
  }


/*******************************************************************
 *
 *  Function    :  TT_Set_Instance_Transform_Flags
 *
 *  Description :  Informs the interpreter about the transformations
 *                 that will be applied to the rendered glyphs.
 *
 *  Input  :  instance      the instance handle
 *            rotated       set to TRUE if the glyph are rotated
 *            stretched     set to TRUE if the glyph are stretched
 *
 *  Output :  Error code.
 *
 *  Note :    This function is deprecated!  It's much better to
 *            control hinting manually when calling TT_Load_Glyph
 *            than relying on the font programs...
 *
 *            Never use it, unless calling for trouble ;-)
 *
 *  MT-Note : NO!  This should be called only when setting/resetting
 *            instances, so there is no need to protect.
 *
 ******************************************************************/

  EXPORT_FUNC
  TT_Error  TT_Set_Instance_Transform_Flags( TT_Instance  instance,
                                             TT_Bool      rotated,
                                             TT_Bool      stretched )
  {
    PInstance  ins = HANDLE_Instance( instance );


    if ( !ins )
      return TT_Err_Invalid_Instance_Handle;

    ins->metrics.rotated   = rotated;
    ins->metrics.stretched = stretched;
    ins->valid             = FALSE;

    return TT_Err_Ok;
  }


/*******************************************************************
 *
 *  Function    :  TT_Get_Instance_Metrics
 *
 *  Description :  Returns instance metrics.
 *
 *  Input  :  instance      the instance handle
 *            metrics       address of target instance metrics record
 *
 *  Output :  Error code.
 *
 *  MT-Note : YES!  Reads only semi-permanent data.
 *
 ******************************************************************/

  EXPORT_FUNC
  TT_Error  TT_Get_Instance_Metrics( TT_Instance           instance,
                                     TT_Instance_Metrics*  metrics )
  {
    PInstance  ins = HANDLE_Instance( instance );


    if ( !ins )
     return TT_Err_Invalid_Instance_Handle;

    if ( !ins->valid )
      Instance_Reset( ins );

    metrics->pointSize    = ins->metrics.pointSize;

    metrics->x_scale      = TT_MulDiv( 0x10000,
                                       ins->metrics.x_scale1,
                                       ins->metrics.x_scale2 );

    metrics->y_scale      = TT_MulDiv( 0x10000,
                                       ins->metrics.y_scale1,
                                       ins->metrics.y_scale2 );

    metrics->x_resolution = ins->metrics.x_resolution;
    metrics->y_resolution = ins->metrics.y_resolution;
    metrics->x_ppem       = ins->metrics.x_ppem;
    metrics->y_ppem       = ins->metrics.y_ppem;

    return TT_Err_Ok;
  }


/*******************************************************************
 *
 *  Function    :  TT_Set_Instance_Pointer
 *
 *  Description :  Each instance has one pointer, which use is
 *                 reserved to client applications.  The TrueType
 *                 engine never accesses or uses this field.
 *
 *                 This function is used to set the pointer.
 *
 *  Input  :  face    the given face handle
 *            data    the generic pointer value
 *
 *  Output :  Error code.
 *
 *  MT-Note : NO!
 *
 ******************************************************************/

  EXPORT_FUNC
  TT_Error  TT_Set_Instance_Pointer( TT_Instance  instance,
                                     void*        data )
  {
    PInstance  ins = HANDLE_Instance( instance );


    if ( !ins )
      return TT_Err_Invalid_Instance_Handle;
    else
      ins->generic = data;

    return TT_Err_Ok;
  }


/*******************************************************************
 *
 *  Function    :  TT_Get_Instance_Pointer
 *
 *  Description :  Each instance has one pointer, which use is
 *                 reserved to client applications.  The TrueType
 *                 engine never accesses or uses this field.
 *
 *                 This function is used to read the pointer.
 *
 *  Input  :  face    the given face handle
 *            data    the generic pointer value
 *
 *  Output :  Error code.
 *
 *  MT-Safe : NO!
 *
 ******************************************************************/

  EXPORT_FUNC
  void*  TT_Get_Instance_Pointer( TT_Instance  instance )
  {
    PInstance  ins = HANDLE_Instance( instance );


    if ( !ins )
      return NULL;
    else
      return ins->generic;
  }


/*******************************************************************
 *
 *  Function    :  TT_Done_Instance
 *
 *  Description :  Closes a given instance.
 *
 *  Input  :  instance      address of instance handle
 *
 *  Output :  Error code.
 *
 *  MT-Safe : YES!
 *
 ******************************************************************/

  EXPORT_FUNC
  TT_Error  TT_Done_Instance( TT_Instance  instance )
  {
    PInstance  ins = HANDLE_Instance( instance );


    if ( !ins )
      return TT_Err_Invalid_Instance_Handle;

    /* delete the instance -- this is thread-safe */
    return CACHE_Done( &ins->owner->instances, ins );
  }


/*******************************************************************
 *
 *  Function    :  TT_New_Glyph
 *
 *  Description :  Creates a new glyph object related to a given
 *                 face.
 *
 *  Input  :  face       the face handle
 *            glyph      address of target glyph handle
 *
 *  Output :  Error code.
 *
 *  MT-Safe : YES!
 *
 ******************************************************************/

  EXPORT_FUNC
  TT_Error  TT_New_Glyph( TT_Face    face,
                          TT_Glyph*  glyph )
  {
    TT_Error  error;
    PFace     _face = HANDLE_Face( face );
    PGlyph    _glyph;


    if ( !_face )
      return TT_Err_Invalid_Face_Handle;

    /* get a new glyph from the face's cache -- this is thread-safe */
    error = CACHE_New( &_face->glyphs, _glyph, _face );

    HANDLE_Set( *glyph, _glyph );

    return error;
  }


/*******************************************************************
 *
 *  Function    :  TT_Done_Glyph
 *
 *  Description :  Destroys a given glyph object.
 *
 *  Input  :  glyph  the glyph handle
 *
 *  Output :  Error code.
 *
 *  MT-Safe : YES!
 *
 ******************************************************************/

  EXPORT_FUNC
  TT_Error  TT_Done_Glyph( TT_Glyph  glyph )
  {
    PGlyph  _glyph = HANDLE_Glyph( glyph );


    if ( !_glyph )
      return TT_Err_Invalid_Glyph_Handle;

    /* delete the engine -- this is thread-safe */
    return CACHE_Done( &_glyph->face->glyphs, _glyph );
  }


/*******************************************************************
 *
 *  Function    :  TT_Load_Glyph
 *
 *  Description :  Loads a glyph.
 *
 *  Input  :  instance      the instance handle
 *            glyph         the glyph handle
 *            glyphIndex    the glyph index
 *            loadFlags     flags controlling how to load the glyph
 *                          (none, scaled, hinted, both)
 *
 *  Output :  Error code.
 *
 *  MT-Safe : YES!
 *
 ******************************************************************/

  EXPORT_FUNC
  TT_Error  TT_Load_Glyph( TT_Instance  instance,
                           TT_Glyph     glyph,
                           TT_UShort    glyphIndex,
                           TT_UShort    loadFlags   )
  {
    PInstance  _ins;
    PGlyph     _glyph;
    TT_Error   error;


    _ins = HANDLE_Instance( instance );

    if ( !_ins )
      loadFlags &= ~(TTLOAD_SCALE_GLYPH | TTLOAD_HINT_GLYPH);

    if ( (loadFlags & TTLOAD_SCALE_GLYPH) == 0 )
      _ins = 0;

    _glyph = HANDLE_Glyph( glyph );
    if ( !_glyph )
      return TT_Err_Invalid_Glyph_Handle;

    if ( _ins )
    {
      if ( _ins->owner != _glyph->face )
        return TT_Err_Invalid_Face_Handle;

      if ( !_ins->valid )
      {
        /* This code can only be called in non thread-safe builds */
        error = Instance_Reset( _ins );
        if ( error )
          return error;
      }
    }

    return Load_TrueType_Glyph( _ins, _glyph, glyphIndex, loadFlags );
  }


/*******************************************************************
 *
 *  Function    :  TT_Get_Glyph_Outline
 *
 *  Description :  Returns the glyph's outline data.
 *
 *  Input  :  glyph     the glyph handle
 *            outline   address where the glyph outline will be returned
 *
 *  Output :  Error code.
 *
 *  MT-Safe : YES!  Reads only semi-permanent data.
 *
 ******************************************************************/

  EXPORT_FUNC
  TT_Error  TT_Get_Glyph_Outline( TT_Glyph     glyph,
                                  TT_Outline*  outline )
  {
    PGlyph  _glyph = HANDLE_Glyph( glyph );


    if ( !_glyph )
      return TT_Err_Invalid_Glyph_Handle;

    *outline = _glyph->outline;
    outline->owner = FALSE;

    return TT_Err_Ok;
  }


/*******************************************************************
 *
 *  Function    :  TT_Get_Glyph_Metrics
 *
 *  Description :  Extracts the glyph's horizontal metrics information.
 *
 *  Input  :  glyph       glyph object handle
 *            metrics     address where metrics will be returned
 *
 *  Output :  Error code.
 *
 *  MT-Safe : NO!  Glyph containers can't be shared.
 *
 ******************************************************************/

  EXPORT_FUNC
  TT_Error  TT_Get_Glyph_Metrics( TT_Glyph           glyph,
                                  TT_Glyph_Metrics*  metrics )
  {
    PGlyph  _glyph = HANDLE_Glyph( glyph );


    if ( !_glyph )
      return TT_Err_Invalid_Glyph_Handle;

    metrics->bbox     = _glyph->metrics.bbox;
    metrics->bearingX = _glyph->metrics.horiBearingX;
    metrics->bearingY = _glyph->metrics.horiBearingY;
    metrics->advance  = _glyph->metrics.horiAdvance;

    return TT_Err_Ok;
  }


/*******************************************************************
 *
 *  Function    :  TT_Get_Glyph_Big_Metrics
 *
 *  Description :  Extracts the glyph's big metrics information.
 *
 *  Input  :  glyph       glyph object handle
 *            metrics     address where big metrics                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         